package com.thebeastshop.pegasus.channelservice.adapter.cart;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import com.thebeastshop.cart.enums.CartPackSourceEnum;
import com.thebeastshop.pegasus.channelservice.convert.ScConvert;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.BooleanUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import com.alibaba.fastjson.JSON;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.thebeastshop.cart.ProductPack;
import com.thebeastshop.cart.resp.CartCustomization;
import com.thebeastshop.cart.resp.CartProductPack;
import com.thebeastshop.common.enums.AccessWayEnum;
import com.thebeastshop.common.enums.MemberLevelEnum;
import com.thebeastshop.member.service.MemberQueryService;
import com.thebeastshop.member.vo.MemberVO;
import com.thebeastshop.pegasus.channelservice.adapter.ipos.IposAdapter;
import com.thebeastshop.pegasus.channelservice.adapter.price.IposPriceAdapter;
import com.thebeastshop.pegasus.channelservice.adapter.price.PriceAdapter;
import com.thebeastshop.pegasus.channelservice.adapter.product.IposSpvAdapter;
import com.thebeastshop.pegasus.channelservice.adapter.product.SpvCommonAdapter;
import com.thebeastshop.pegasus.channelservice.service.LabelService;
import com.thebeastshop.pegasus.component.discount.DiscountType;
import com.thebeastshop.pegasus.merchandise.cond.PsMatchCampaignCond;
import com.thebeastshop.pegasus.merchandise.cond.PsProductCond;
import com.thebeastshop.pegasus.merchandise.service.McPsProductService;
import com.thebeastshop.pegasus.merchandise.service.McPsSpvService;
import com.thebeastshop.pegasus.merchandise.vo.OpChannelVO;
import com.thebeastshop.pegasus.merchandise.vo.PsCampaignVO;
import com.thebeastshop.pegasus.merchandise.vo.PsProductVO;
import com.thebeastshop.pegasus.merchandise.vo.PsSkuCombinationVO;
import com.thebeastshop.pegasus.merchandise.vo.PsSpvVO;
import com.thebeastshop.price.vo.PrsAdditionalPackVO;
import com.thebeastshop.price.vo.PrsCampaignVO;
import com.thebeastshop.price.vo.PrsCartSettleLabelVO;
import com.thebeastshop.price.vo.PrsPacksPriceResultVO;
import com.thebeastshop.support.Label;
import com.thebeastshop.support.enums.CartPackSource;
import com.thebeastshop.support.exception.WrongArgException;
import com.thebeastshop.support.vo.cart.CartProductPackVO;
import com.thebeastshop.support.vo.cart.RequestPack;
import com.thebeastshop.support.vo.customize.UserSingleCustomization;
import com.thebeastshop.support.vo.order.OrderConfirmVO.BdayVO;
import com.thebeastshop.support.vo.product.SpvVO;

/**
 * IPOS 商品包 适配器
 * 
 * @Class Name IposProductPackAdapter
 * @Author Xiang.Gao
 * @Create In 2018-12-26
 */
@Component
public class IposPackAdapter {

	private Logger logger = LoggerFactory.getLogger(this.getClass());

	@Autowired
	McPsProductService mcPsProductService;

	@Autowired
	McPsSpvService mcPsSpvService;

	@Autowired
	IposSpvAdapter iposSpvAdapter;

	@Autowired
	CartCommonAdapter cartCommonAdapter;

	@Autowired
	SpvCommonAdapter spvCommonAdapter;

	@Autowired
	PriceAdapter priceAdapter;

	@Autowired
	IposAdapter iposAdapter;

	@Autowired
	IposPriceAdapter iposPriceAdapter;

	@Autowired
	LabelService labelService;
	
	@Autowired
	MemberQueryService memberQueryService;

	/**
	 * 请求商品包转换
	 * 
	 * @Methods Name setProdAndSpvCartPack
	 * @Create In 2018-12-27 By Xiang.Gao
	 * @return List<CartProductPack>
	 */
	public List<CartProductPack> setProdAndSpvCartPack(List<RequestPack> packs, AccessWayEnum accessWay,
			MemberVO member, String channelCode) {
		List<CartProductPack> prodPacks = Lists.newArrayList();

		MemberLevelEnum memberLevel = MemberLevelEnum.getEnumByCode(member.getMemberLevel());
		List<String> productIds = Lists.newArrayListWithExpectedSize(packs.size());

		Map<Long, PsSpvVO> spvMap = Maps.newHashMapWithExpectedSize(packs.size());
		Map<String, PsProductVO> prodMap = Maps.newHashMapWithExpectedSize(packs.size());

		List<Long> spvIds = Lists.newArrayListWithExpectedSize(packs.size());
		for (RequestPack pack : packs) {
			if (CollectionUtils.isNotEmpty(pack.getGiftPacks())) {
				for (RequestPack giftPack : pack.getGiftPacks()) {
					spvIds.add(giftPack.getSpvId());
				}
			}
			spvIds.add(pack.getSpvId());
		}

		if (CollectionUtils.isNotEmpty(spvIds)) {
			// 获取spv信息
			PsMatchCampaignCond spvCond = new PsMatchCampaignCond();
			spvCond.setChannelCode(channelCode);
			spvCond.setAccessWay(accessWay.getCode());
			spvCond.setMemberLevel(member.getMemberLevel());
			spvCond.setMatchCampaign(true);
			spvCond.setDiscount(memberLevel.getLevelAmountCondition());
			spvCond.setPagenum(Integer.MAX_VALUE);
			spvCond.setBdDiscount(member.getBdayDiscountRatio());
			List<PsSpvVO> spvs = mcPsSpvService.findBySpvIdsWithStock(spvIds, spvCond, channelCode);
			for (PsSpvVO spv : spvs) {
				spvMap.put(spv.getSpvId(), spv);
				productIds.add(spv.getProductId().toString());
			}
		}

		if (CollectionUtils.isNotEmpty(productIds)) {
			PsProductCond cond = new PsProductCond();
			cond.setIds(productIds);
			cond.setMemberCode(member.getCode());
			cond.setChannelCode(channelCode);
			cond.setAccessWay(accessWay.getCode());
			cond.setMemberLevel(member.getMemberLevel());
			cond.setMatchCampaign(true);
			cond.setDiscount(memberLevel.getLevelAmountCondition());
			cond.setPagenum(Integer.MAX_VALUE);
			cond.setBdDiscount(member.getBdayDiscountRatio());
			List<PsProductVO> products = mcPsProductService.findByCond(cond, true);
			for (PsProductVO psProductVO : products) {
				prodMap.put(psProductVO.getId(), psProductVO);
			}
		}

		Iterator<RequestPack> iterator = packs.iterator();
		while (iterator.hasNext()) {
			RequestPack pack = iterator.next();
			PsSpvVO spv = spvMap.get(pack.getSpvId());
			if (spv == null) {
				throw new WrongArgException("商品信息有误");
			}
			PsProductVO product = prodMap.get(spv.getProductId().toString());
			if (product == null) {
				throw new WrongArgException("商品信息有误");
			}
			CartProductPack prodPack = new CartProductPack();
			prodPack.setCount(pack.getCount());
			prodPack.setSource(CartPackSourceEnum.getEnumByCode(pack.getSource().getId()));
			prodPack.setSpvId(spv.getSpvId());
			prodPack.setProductId(spv.getProductId());
			prodPack.setStoreCount(pack.getStoreCount());
			prodPack.setWarehouseCount(pack.getWarehouseCount());
			prodPack.setDeliverys(ScConvert.convert2ScDelivery(pack.getDeliverys()));
			prodPack.setSpv(spv);
			prodPack.setProduct(product);
			if (CartPackSource.RAW.equals(pack.getSource())) {
				prodPack.setFactProductPrice(spv.getSalesPrice());
				if (CollectionUtils.isNotEmpty(pack.getGiftPacks())) {
					List<CartProductPack> giftPackList = Lists.newArrayList();
					for (RequestPack giftPack : pack.getGiftPacks()) {
						PsSpvVO giftSpv = spvMap.get(giftPack.getSpvId());
						if (giftSpv == null) {
							throw new WrongArgException("赠品信息有误");
						}
						CartProductPack giftProdPack = new CartProductPack();
						giftProdPack.setCount(giftPack.getCount());
						giftProdPack.setSource(CartPackSourceEnum.getEnumByCode(giftPack.getSource().getId()));
						giftProdPack.setStoreCount(giftPack.getStoreCount());
						giftProdPack.setWarehouseCount(giftPack.getWarehouseCount());
						giftProdPack.setDeliverys(ScConvert.convert2ScDelivery(giftPack.getDeliverys()));
						giftProdPack.setSpvId(giftSpv.getSpvId());
						giftProdPack.setProductId(giftSpv.getProductId());
						giftProdPack.setSpv(giftSpv);
						giftProdPack.setProduct(prodMap.get(giftSpv.getProductId().toString()));
						giftPackList.add(giftProdPack);
					}
					prodPack.setGiftPacks(giftPackList);
				}
			}
			if (CollectionUtils.isNotEmpty(pack.getCustomizes())) {
				prodPack.setCustomize(true);
				List<CartCustomization> cts = new ArrayList<>();
				for (UserSingleCustomization userSingleCustomization : pack.getCustomizes()) {
					CartCustomization cartCustomization = new CartCustomization();
					cartCustomization.setCustomDimensionId(userSingleCustomization.getCode());
					cartCustomization.setCustomDimensionName(userSingleCustomization.getName());
					cartCustomization.setCustomDimensionValue(userSingleCustomization.getValue());
					cts.add(cartCustomization);
				}
				prodPack.setCustomizeList(cts);
			}
			prodPacks.add(prodPack);
		}
		return prodPacks;
	}

	/**
	 * 获取订单 换购包
	 * 
	 * @Methods Name getOrderAddPack
	 * @Create In 2018-12-28 By Xiang.Gao
	 * @return CartProductPackVO
	 */
	private CartProductPackVO getOrderAddPack(CartProductPack pack, boolean containtWh,
			Map<String, Date> spvPresaleDateMap) {

		int left = iposSpvAdapter.countSpvStock(pack.getSpv(), containtWh);
		SpvVO spvVO = iposSpvAdapter.buildingAddSpvVO(pack.getProduct(), pack.getSpv(), left);
		Date presaleDate = spvPresaleDateMap.get(pack.getSpv().getSkuCode());
		spvVO.setPrice(pack.getSpv().getDiscountPrice());
		spvVO.setPreDeliver(iposSpvAdapter.getPreDeliverVO(pack, presaleDate));

		CartProductPackVO packVo = new CartProductPackVO();
		packVo.setCount(pack.getCount());
		packVo.setStoreCount(pack.getStoreCount());
		packVo.setWarehouseCount(pack.getWarehouseCount());
		packVo.setDeliverys(ScConvert.convert4ScDelivery(pack.getDeliverys()));
		packVo.setLabels(iposSpvAdapter.getSpvLabel(pack.getSpv()));
		packVo.setHasVariants(pack.getProduct().getSpvList().size() > 1);
		packVo.setSource(CartPackSource.valueOf(pack.getSource().name()));
		packVo.setProductId(pack.getProduct().getCode());
		packVo.setSpv(spvVO);
		packVo.setOwned(true);
		packVo.setIsJit(pack.getSpv().isJit());
		Label label = new Label();
		label.setType("换购");
		packVo.getLabels().add(label);
		return packVo;
	}

	/**
	 * 获取 订单多赠品包
	 * 
	 * @Methods Name getOrderMutiGiftPack
	 * @Create In 2018-12-28 By Xiang.Gao
	 * @return CartProductPackVO
	 */
	private CartProductPackVO getOrderMutiGiftPack(CartProductPack pack, boolean containtWh,
			Map<String, Date> spvPresaleDateMap) {
		if (!pack.getSource().equals(CartPackSourceEnum.GIFT)) {
			return null;
		}

		Date presaleDate = spvPresaleDateMap.get(pack.getSpv().getSkuCode());
		int left = iposSpvAdapter.countSpvStock(pack.getSpv(), containtWh);
		SpvVO spvVO = iposSpvAdapter.buildingAddSpvVO(pack.getProduct(), pack.getSpv(), left);
		spvVO.setPreDeliver(iposSpvAdapter.getPreDeliverVO(pack, presaleDate));
		spvVO.setPrice(BigDecimal.ZERO);

		CartProductPackVO packVo = new CartProductPackVO();
		packVo.setCount(pack.getCount());
		packVo.setLabels(iposSpvAdapter.getSpvLabel(pack.getSpv()));
		packVo.setHasVariants(pack.getProduct().getSpvList().size() > 1);
		packVo.setSource(CartPackSource.valueOf(pack.getSource().name()));
		packVo.setStoreCount(pack.getStoreCount());
		packVo.setWarehouseCount(pack.getWarehouseCount());
		packVo.setDeliverys(ScConvert.convert4ScDelivery(pack.getDeliverys()));
		packVo.setProductId(pack.getProduct().getCode());
		packVo.setSpv(spvVO);
		packVo.setIsJit(pack.getSpv().isJit());
		packVo.setOwned(true);
		Label label = new Label();
		label.setType("赠品");
		packVo.getLabels().add(label);
		return packVo;
	}

	/**
	 * 购物车 主品包 库存分配
	 * 
	 * @Methods Name allotRawPackVOStock
	 * @Create In 2018-12-28 By Xiang.Gao
	 */
	public void allotRawPackVOStock(CartProductPackVO packVO, PsSpvVO spvStockVo) {
		// 可售数量不足
		if (spvStockVo.getCanUseQuantityOffLine() < packVO.getCount()) {
			return;
		}
		// 本地库存
		int localNum = iposSpvAdapter.countSpvLocalStock(spvStockVo);
		// 仓库库存
		int warehouseNum = iposSpvAdapter.countSpvWarehouseStock(spvStockVo);
		// 非购买数量变更 且所选库存足够 不再重新分配
		if (localNum >= packVO.getStoreCount() && warehouseNum >= packVO.getWarehouseCount()
				&& packVO.getCount() == packVO.getStoreCount() + packVO.getWarehouseCount()) {
			return;
		}
		// 库存分配
		packVO.setStoreCount(packVO.getCount() > localNum ? localNum : packVO.getCount());
		packVO.setWarehouseCount(packVO.getCount() > localNum ? packVO.getCount() - localNum : 0);
	}

	/**
	 * 换购库存分配
	 * 
	 * @param packVO
	 * @param clientPack
	 * @param spvStockVo
	 */
	public void allotAddPackStock(CartProductPackVO packVO, CartProductPack clientPack, PsSpvVO spvStockVo) {
		if (BooleanUtils.isFalse(packVO.getOwned())) {
			return;
		}
		// 本地库存
		int localNum = iposSpvAdapter.countSpvLocalStock(spvStockVo);
		// 仓库库存
		int warehouseNum = iposSpvAdapter.countSpvWarehouseStock(spvStockVo);
		// 非购买数量变更 且所选库存足够 不再重新分配
		if (clientPack != null && clientPack.getCount() == packVO.getCount()
				&& clientPack.getCount() == clientPack.getStoreCount() + clientPack.getWarehouseCount()
				&& localNum >= clientPack.getStoreCount() && warehouseNum >= clientPack.getWarehouseCount()) {
			packVO.setStoreCount(clientPack.getStoreCount());
			packVO.setWarehouseCount(clientPack.getWarehouseCount());
			return;
		}
		// 分配库存
		packVO.setStoreCount(packVO.getCount() > localNum ? localNum : packVO.getCount());
		packVO.setWarehouseCount(packVO.getCount() > localNum ? packVO.getCount() - localNum : 0);
	}

	/**
	 * 赠品库存分配
	 * 
	 * @Methods Name allotSubPackStock
	 * @Create In 2018-12-29 By Xiang.Gao
	 * @return DeliveryVO
	 */
	public void allotGiftPackStock(CartProductPackVO packVO, PsSpvVO spvStockVo) {
		if (BooleanUtils.isFalse(packVO.getOwned())) {
			return;
		}
		// 本地库存
		int localNum = iposSpvAdapter.countSpvLocalStock(spvStockVo);
		// 分配库存
		packVO.setStoreCount(packVO.getCount() > localNum ? localNum : packVO.getCount());
		packVO.setWarehouseCount(packVO.getCount() > localNum ? packVO.getCount() - localNum : 0);
	}
	
	/**
	 * 购物车 获取主品包
	 * 
	 * @Methods Name getRawPackWithCart
	 * @Create In 2018-12-27 By Xiang.Gao
	 * @return List<CartProductPackVO>
	 */
	public List<CartProductPackVO> getCartRawPack(List<CartProductPack> packList, PrsPacksPriceResultVO priceResult,
			MemberVO member, Map<Long, PsSpvVO> spvStock, Map<String, PsProductVO> addProdMap,
			Map<Long, List<PrsAdditionalPackVO>> giftMap) {
		List<CartProductPackVO> rawPack = new ArrayList<>();

		List<PsSpvVO> spvs = iposSpvAdapter.featchPsSpvVO(packList);
		Map<String, Date> spvPresaleDateMap = iposSpvAdapter.mapSkuCodePresale(spvs); // 预售日期
		
		// 主品包
		for (CartProductPack pack : packList) {
			if (!CartPackSourceEnum.RAW.equals(pack.getSource())) {
				continue;
			}
			Date presaleDate = spvPresaleDateMap.get(pack.getSpv().getSkuCode());
			// 包裹信息
			CartProductPackVO packVO = buildingRawPack(pack, member, presaleDate);
			// 主品库存分配
			allotRawPackVOStock(packVO, pack.getSpv());
			// 单品赠品不为空 
			List<PrsAdditionalPackVO> giftList = giftMap.get(pack.getSpv().getSpvId());
			if (CollectionUtils.isNotEmpty(giftList)) {
				Map<String, PrsCampaignVO> prodCamp = iposPriceAdapter.getProdCampMap(priceResult.getPriceStepVOs());
				packVO.setGiftPacks(getCartGiftPacks(giftList, addProdMap, prodCamp, spvStock, pack.getWarehouseCount() != 0));
			}
			// 单品活动标签
			packVO.setLabels(getPackSignLabels(priceResult.getCartSettleLabelList(), pack.getSpvId()));
			rawPack.add(packVO);
		}
		return rawPack;
	}

	/**
	 * ipos 购物车 赠品包
	 * 
	 * @Methods Name signGiftPacks
	 * @Create In 2018-12-20 By Xiang.Gao
	 * @return List<CartProductPackVO>
	 */
	public List<CartProductPackVO> getCartGiftPacks(List<PrsAdditionalPackVO> giftList,
			Map<String, PsProductVO> addProdMap, Map<String, PrsCampaignVO> prodCamp,
			Map<Long, PsSpvVO> spvStockMap, boolean containtWh) {
		
		List<CartProductPackVO> resultGiftPack = Lists.newArrayList();
		
		// 活动限量
		Map<String, Integer> limitMap = iposPriceAdapter.getCspLimitMap(giftList, prodCamp, addProdMap);
		// 预售日期
		List<PsSpvVO> spvs = iposSpvAdapter.getAddSpvByProduct(giftList, addProdMap);
		Map<String, Date> spvPresaleDateMap = iposSpvAdapter.mapSkuCodePresale(spvs);
		// 区间赠送数量
		Map<Long, Integer> sectionGiftMap = Maps.newHashMap();
		// 赠品总数 只有N选1才有
		Integer giftCount = null;
		// 无库存的区间商品包
		Map<Long, PrsAdditionalPackVO> noStockSectionAddMap = Maps.newHashMap();
		
		/** 开始封装赠品包 **/
		// 添加库存足够的赠品包
		for (PrsAdditionalPackVO add : giftList) {
			PsProductVO product = addProdMap.get(add.getProductId().toString());
			if (product == null) {
				logger.error("product isNull key:[{}]", add.getProductId());
				continue;
			}
			PrsCampaignVO campaign = prodCamp.get(add.getGroupId() + "-" + add.getProductId());
			if (campaign == null) {
				logger.error("campaign isNull key:[{}]", add.getGroupId() + "-" + add.getProductId());
				continue;
			}
			// 已达限量上限
			String limitKey = campaign.getCode() + product.getCode();
			if (limitMap.get(limitKey) != null && limitMap.get(limitKey) < add.getCount()) {
				logger.warn("campaign limit key:[{}]", limitKey);
				continue;
			}
			// N选1 可累加
			boolean cumulative = campaign.getOptionFlag() == 2 && campaign.getCumulative() == 1;
			// 包裹数
			int packSize = cumulative ? add.getCount() : 1;
			// N选1
			if (campaign.getOptionFlag() == 2) {
				giftCount = add.getCount();	// N选1 最大可选数
				add.setCount(1);			// N选1 默认基数为1
			}
			
			// 添加有库存的SPV
			for (int i = 0; i < packSize; i++) {
				Integer sectionCount = sectionGiftMap.get(add.getGroupId());
				// 随机选赠品
				int left = 0;
				PsSpvVO psSpvVO = null;
				for (PsSpvVO psSpv : product.getSpvList()) {
					psSpvVO = spvStockMap.get(psSpv.getSpvId());
					left = iposSpvAdapter.countSpvStock(psSpvVO, containtWh);
					
					if (left >= add.getCount()) {
						break;
					} else {
						psSpvVO = null;
					}
				}
				// 库存不足则记录 
				if (psSpvVO == null) {
					if (noStockSectionAddMap.get(add.getGroupId()) == null) {
						noStockSectionAddMap.put(add.getGroupId(), add);
					}
					break;
				}
				boolean appendPack = false;
				// 全部赠送
				if (campaign.getOptionFlag() == 0) {
					appendPack = true;
				}
				// 依次赠送
				if (campaign.getOptionFlag() == 1 && sectionCount == null) {
					appendPack = true;
					sectionCount = add.getCount();
				}
				// N选1
				if (campaign.getOptionFlag() == 2 && (sectionCount == null || sectionCount.intValue() < giftCount)) {
					appendPack = true;
					if (sectionCount == null) {
						sectionCount = 0;
					}
					sectionCount++;
				}
				// 添加赠品
				if (appendPack) {
					Date presaleDate = spvPresaleDateMap.get(psSpvVO.getSkuCode());
					CartProductPackVO packVO = buildingGiftPack(campaign, product, psSpvVO, add, presaleDate, left);
					allotGiftPackStock(packVO, psSpvVO);
					resultGiftPack.add(packVO);
					sectionGiftMap.put(add.getGroupId(), sectionCount);
				}
			}
		}
		
		// 添加已赠完商品包
		Collection<PrsAdditionalPackVO> noStockSectionAddList = noStockSectionAddMap.values();
		for (PrsAdditionalPackVO add : noStockSectionAddList) {
			// 过滤同一区间已赠送的商品包
			if(sectionGiftMap.get(add.getGroupId()) != null) {
				continue;
			}
			PsProductVO product = addProdMap.get(add.getProductId().toString());
			PrsCampaignVO campaign = prodCamp.get(add.getGroupId() + "-" + add.getProductId());
			
			Long firstSpvId = product.getSpvList().get(0).getSpvId();
			// spvStockMap中的spv带库存信息及SKU对象
			PsSpvVO psSpvVO = spvStockMap.get(firstSpvId);
			int left = iposSpvAdapter.countSpvStock(psSpvVO, containtWh);
			Date presaleDate = spvPresaleDateMap.get(psSpvVO.getSkuCode());
			
			CartProductPackVO packVO = buildingGiftPack(campaign, product, psSpvVO, add, presaleDate, left);
			resultGiftPack.add(packVO);
		}
		return resultGiftPack;
	}

	/**
	 * 可换赠品
	 */
	public List<CartProductPackVO> getGiftPacksPool(Map<String, PrsCampaignVO> prodCamp, boolean containtWh,
			Map<Long, List<PrsAdditionalPackVO>> singleGiftMap, Map<Long, List<PrsAdditionalPackVO>> mutiGiftMap,
			Map<String, PsProductVO> addProdMap, Map<Long, PsSpvVO> spvStockMap) {
		List<CartProductPackVO> giftPacksPool = Lists.newArrayList();
		// 单品赠品包
		List<PrsAdditionalPackVO> giftList = mergeGiftPack(singleGiftMap, mutiGiftMap);
		// 活动限量
		Map<String, Integer> limitMap = iposPriceAdapter.getCspLimitMap(giftList, prodCamp, addProdMap);
		// 预售日期
		List<PsSpvVO> spvs = iposSpvAdapter.getAddSpvByProduct(giftList, addProdMap);
		Map<String, Date> spvPresaleDateMap = iposSpvAdapter.mapSkuCodePresale(spvs);

		for (PrsAdditionalPackVO add : giftList) {
			PsProductVO product = addProdMap.get(add.getProductId().toString());
			PrsCampaignVO campaign = prodCamp.get(add.getGroupId() + "-" + add.getProductId());
			if (product == null) {
				logger.error("product isNull key:[{}]", add.getProductId().toString());
				continue;
			}
			if (campaign == null) {
				logger.error("campaign isNull key:[{}]", add.getGroupId() + "-" + add.getProductId());
				continue;
			}
			String limitKey = campaign.getCode() + product.getCode();
			// 已达限量上限
			if (limitMap.get(limitKey) != null && limitMap.get(limitKey) < add.getCount()) {
				continue;
			}
			// 只有N选1 才有可换赠品池
			if (campaign.getOptionFlag() != 2) {
				continue;
			}
			add.setCount(1);
			// 单阶段赠品累加
			for (PsSpvVO psSpv : product.getSpvList()) {
				PsSpvVO psSpvVO = spvStockMap.get(psSpv.getSpvId());
				Date presaleDate = spvPresaleDateMap.get(psSpvVO.getSkuCode());
				int left = iposSpvAdapter.countSpvStock(psSpvVO, containtWh);
				CartProductPackVO backGiftPack = buildingGiftPack(campaign, product, psSpvVO, add, presaleDate, left);
				this.allotGiftPackStock(backGiftPack, psSpvVO);
				giftPacksPool.add(backGiftPack);
			}
		}
		return giftPacksPool;
	}

	/**
	 * 合并赠品包
	 * 
	 * @param singleGiftMap
	 * @param mutiGiftMap
	 * @return
	 */
	private List<PrsAdditionalPackVO> mergeGiftPack(Map<Long, List<PrsAdditionalPackVO>> singleGiftMap,
			Map<Long, List<PrsAdditionalPackVO>> mutiGiftMap) {
		List<PrsAdditionalPackVO> packList = Lists.newArrayList();
		for (Entry<Long, List<PrsAdditionalPackVO>> entry : singleGiftMap.entrySet()) {
			packList.addAll(entry.getValue());
		}
		for (Entry<Long, List<PrsAdditionalPackVO>> entry : mutiGiftMap.entrySet()) {
			packList.addAll(entry.getValue());
		}
		return packList;
	}

	/**
	 * 购物车 换购包
	 */
	public List<CartProductPackVO> getAddPacks(Map<Long, CartProductPack> clientAddPack,
			List<PrsAdditionalPackVO> addList, Map<String, PsProductVO> addProdMap, Map<String, PrsCampaignVO> prodCamp,
			Map<Long, PsSpvVO> spvStockMap, boolean containtWh) {

		List<CartProductPackVO> addPackVOlist = new ArrayList<CartProductPackVO>();
		if (CollectionUtils.isEmpty(addList)) {
			return addPackVOlist;
		}
		Map<String, Integer> limitMap = iposPriceAdapter.getCspLimitMap(addList, prodCamp, addProdMap); // 活动限量
		List<PsSpvVO> spvs = iposSpvAdapter.getAddSpvByProduct(addList, addProdMap); // 换购SPV
		Map<String, Date> spvPresaleDateMap = iposSpvAdapter.mapSkuCodePresale(spvs); // 预计发货信息
		// 封装换购包
		for (PrsAdditionalPackVO add : addList) {
			PsProductVO product = addProdMap.get(add.getProductId().toString());
			if (product == null) {
				logger.error("product isNull key:[{}]", add.getProductId().toString());
				continue;
			}
			PrsCampaignVO campaign = prodCamp.get(add.getGroupId() + "-" + add.getProductId());
			if (campaign == null) {
				logger.error("campaign isNull key:[{}]", add.getGroupId() + "-" + add.getProductId());
				continue;
			}
			String limitKey = campaign.getCode() + product.getCode();
			// 已达限量上限
			if (limitMap.get(limitKey) != null && limitMap.get(limitKey) < add.getCount()) {
				continue;
			}
			// 选取 换购SPV
			PsSpvVO spvLeft = null;
			int left = 0;
			CartProductPack clientPack = clientAddPack.get(add.getProductId());
			if (clientPack != null && clientPack.getCount() == add.getCount()) {
				// 取已选换购
				PsSpvVO psSpvVO = spvStockMap.get(clientPack.getSpvId());
				left = iposSpvAdapter.countSpvStock(psSpvVO, containtWh);
				if (left >= clientPack.getCount()) {
					spvLeft = psSpvVO;
				}
			} else {
				// 随机取库存足够的SPV
				for (PsSpvVO psSpv : product.getSpvList()) {
					PsSpvVO psSpvVO = spvStockMap.get(psSpv.getSpvId());
					left = iposSpvAdapter.countSpvStock(psSpvVO, containtWh);
					if (left > 0 && left >= add.getCount()) {
						spvLeft = psSpvVO;
						break;
					}
				}
			}
			// 库存足够
			if (spvLeft != null) {
				SpvVO spvVO = iposSpvAdapter.buildingAddSpvVO(product, spvLeft, left);
				spvVO.setPrice(add.getDiscountPrice());
				Date presaleDate = spvPresaleDateMap.get(spvLeft.getSkuCode());
				spvVO.setPreDeliver(iposSpvAdapter.getPreDeliverVO(product, add.getCount(), spvLeft, presaleDate));
				// 包裹信息
				CartProductPackVO packVO = new CartProductPackVO();
				packVO.setCount(add.getCount());
				packVO.setSource(CartPackSource.ADDIBLE);
				packVO.setProductId(product.getCode());
				packVO.setSpv(spvVO);
				packVO.setHasVariants(product.getSpvList().size() > 1);
				packVO.setOwned(clientPack != null);
				packVO.setIsJit(spvLeft.isJit());
				packVO.setCampaign(iposPriceAdapter.getPackCampaign(campaign, left <= 0));
				// 发货库存分配
				this.allotAddPackStock(packVO, clientPack, spvLeft);
				addPackVOlist.add(packVO);
			}
		}
		return addPackVOlist;
	}

	/**
	 * 订单 预览包裹
	 */
	public List<CartProductPackVO> getPackVOWithOrder(PrsPacksPriceResultVO priceResult, List<CartProductPack> packList,
			MemberVO member, OpChannelVO opChannel, AccessWayEnum accessWay) {
		if (CollectionUtils.isEmpty(packList)) {
			return Lists.newArrayList();
		}
		// spv
		List<PsSpvVO> psSpvVoList = iposSpvAdapter.getPsSpvVOByPack(packList);
		// 预售
		Map<String, Date> spvPresaleDateMap = iposSpvAdapter.mapSkuCodePresale(psSpvVoList);
		// 包含仓库发货
		boolean containtWh = this.containWhDeliveryOrder(packList);
		
		List<CartProductPackVO> list = new ArrayList<>();
		for (CartProductPack pack : packList) {
			// 主品
			if (pack.getSource().equals(CartPackSourceEnum.RAW)) {
				CartProductPackVO rawPackVO = getOrderRawPack(pack, spvPresaleDateMap, member);
				List<Label> signPackLabel = getPackSignLabels(priceResult.getCartSettleLabelList(), pack.getSpvId());
				rawPackVO.setLabels(signPackLabel);
				list.add(rawPackVO);
			}
			// 赠品
			if (pack.getSource().equals(CartPackSourceEnum.GIFT)) {
				list.add(getOrderMutiGiftPack(pack, containtWh, spvPresaleDateMap));
			}
			// 换购
			if (pack.getSource().equals(CartPackSourceEnum.ADDIBLE)) {
				list.add(getOrderAddPack(pack, containtWh, spvPresaleDateMap));
			}
		}
		return list;
	}

	/**
	 * 订单预览 主品信息
	 */
	private CartProductPackVO getOrderRawPack(CartProductPack pack, 
			Map<String, Date> spvPresaleDateMap, MemberVO member) {

		Date presaleDate = spvPresaleDateMap.get(pack.getSpv().getSkuCode());
		CartProductPackVO packVO = buildingRawPack(pack, member, presaleDate);

		// 单品赠品
		if (CollectionUtils.isNotEmpty(pack.getGiftPacks())) {
			List<CartProductPackVO> signGiftPackList = Lists.newArrayList();
			for (CartProductPack signGiftPack : pack.getGiftPacks()) {

				PsSpvVO psSpvVO = signGiftPack.getSpv();
				int left = iposSpvAdapter.countSpvStock(psSpvVO, pack.getWarehouseCount() > 0);
				SpvVO giftSpvVO = iposSpvAdapter.buildingAddSpvVO(signGiftPack.getProduct(), signGiftPack.getSpv(), left);
				giftSpvVO.setPrice(BigDecimal.ZERO);
				Date giftPresaleDate = spvPresaleDateMap.get(signGiftPack.getSpv().getSkuCode());
				giftSpvVO.setPreDeliver(iposSpvAdapter.getPreDeliverVO(signGiftPack, giftPresaleDate));

				CartProductPackVO signGiftPackVO = new CartProductPackVO();
				signGiftPackVO.setCount(signGiftPack.getCount());
				signGiftPackVO.setStoreCount(signGiftPack.getStoreCount());
				signGiftPackVO.setWarehouseCount(signGiftPack.getWarehouseCount());
				signGiftPackVO.setDeliverys(ScConvert.convert4ScDelivery(signGiftPack.getDeliverys()));
				signGiftPackVO.setSource(CartPackSource.valueOf(signGiftPack.getSource().name()));
				signGiftPackVO.setProductId(signGiftPack.getProduct().getCode());
				signGiftPackVO.setSpv(giftSpvVO);
				signGiftPackVO.setIsJit(signGiftPack.getSpv().isJit());
				signGiftPackVO.setOwned(signGiftPack.getCount() != 0);
				signGiftPackVO.setHasVariants(signGiftPack.getProduct().getSpvList().size() > 1);
				signGiftPackList.add(signGiftPackVO);
			}
			packVO.setGiftPacks(signGiftPackList);
		}
		return packVO;
	}

	/**
	 * 是否定制
	 */
	private boolean spvCustomizable(PsSpvVO psSpvVO) {
		return psSpvVO.getSkuVO() != null && Integer.valueOf(1).equals(psSpvVO.getSkuVO().getCanCustomize());
	}

	/**
	 * 定制信息
	 */
	private List<UserSingleCustomization> getCustomization(List<CartCustomization> customizeList) {
		List<UserSingleCustomization> userCustomizes = Lists.newArrayList();
		if (CollectionUtils.isEmpty(customizeList)) {
			return userCustomizes;
		}
		for (CartCustomization cartCustomization : customizeList) {
			UserSingleCustomization userSingleCustomization = new UserSingleCustomization();
			userSingleCustomization.setCode(cartCustomization.getCustomDimensionId());
			userSingleCustomization.setName(cartCustomization.getCustomDimensionName());
			userSingleCustomization.setValue(cartCustomization.getCustomDimensionValue());
			userCustomizes.add(userSingleCustomization);
		}
		return userCustomizes;
	}

	/**
	 * 单品赠品标签
	 */
	public List<Label> getPackSignLabels(List<PrsCartSettleLabelVO> labels, Long spvId) {
		List<Label> labelList = Lists.newArrayList();
		if (CollectionUtils.isEmpty(labels)) {
			return labelList;
		}
		List<Label> singleLabels = new ArrayList<Label>();
		for (PrsCartSettleLabelVO label : labels) {
			if (label == null || label.getDiscountType() == null) {
				continue;
			}
			if (!label.getIsSingleScopeCampaign() || label.getCampaignId().equals(-1L)) {
				continue;
			}
			if (label.isFit() && label.getSingleScopeSpvId().equals(spvId)) {
				singleLabels.add(priceAdapter.setupLabel(label));
			}
		}
		return singleLabels;
	}
	
	/**
	 * 多品赠品标签
	 */
	public List<Label> getPackMutilLabels(List<PrsCartSettleLabelVO> labels) {
		List<Label> mutiLabels = Lists.newArrayList();
		if (CollectionUtils.isEmpty(labels)) {
			return mutiLabels;
		}
		for (PrsCartSettleLabelVO label : labels) {
			if (label == null || label.getDiscountType() == null) {
				continue;
			}
			if (label.getIsSingleScopeCampaign() || label.getCampaignId().equals(-1L)) {
				continue;
			}
			mutiLabels.add(priceAdapter.setupLabel(label));
		}
		return mutiLabels;
	}

	/**
	  *  商品包匹配 
	  *  订单预览有可能产品新的额外商品 此方法禁用
	 * 
	 * @Methods Name clickPackMatching
	 * @Create In 2018-12-26 By Xiang.Gao
	 */
	@Deprecated
	public void clickPackMatching(List<CartProductPack> clientAddPacks,
			Map<Long, List<PrsAdditionalPackVO>> singleGiftMap, Map<Long, List<PrsAdditionalPackVO>> mutiGiftMap,
			List<PrsAdditionalPackVO> addList) {
		// 活动单品满赠商品Id
		List<Long> signGiftProdIds = Lists.newArrayList();
		for (Entry<Long, List<PrsAdditionalPackVO>> entry : singleGiftMap.entrySet()) {
			List<PrsAdditionalPackVO> packList = entry.getValue();
			for (PrsAdditionalPackVO signGiftPack : packList) {
				signGiftProdIds.add(signGiftPack.getProductId());
			}
		}
		// 活动多品满赠商品Id
		List<Long> multiGiftProdIds = Lists.newArrayList();
		for (Entry<Long, List<PrsAdditionalPackVO>> entry : mutiGiftMap.entrySet()) {
			List<PrsAdditionalPackVO> packList = entry.getValue();
			for (PrsAdditionalPackVO multiGiftPack : packList) {
				multiGiftProdIds.add(multiGiftPack.getProductId());
			}
		}
		logger.info("clickPackMatching giftProdIds :{}", JSON.toJSONString(multiGiftProdIds));
		// 活动换购商品Id
		List<Long> addProIds = Lists.newArrayList();
		for (PrsAdditionalPackVO addPack : addList) {
			addProIds.add(addPack.getProductId());
		}
		logger.info("clickPackMatching addProdIds :{}", JSON.toJSONString(addProIds));
		for (CartProductPack cartPack : clientAddPacks) {
			// 单品赠品
			if (CartPackSourceEnum.RAW.equals(cartPack.getSource())) {
				if (CollectionUtils.isEmpty(cartPack.getGiftPacks())) {
					continue;
				}
				for (CartProductPack giftPack : cartPack.getGiftPacks()) {
					if (!CartPackSourceEnum.GIFT.equals(giftPack.getSource())) {
						continue;
					}
					if (!signGiftProdIds.contains(giftPack.getProductId())) {
						String productName = giftPack.getProduct().getNameCn();
						logger.error("clickPackMatching cart pack productId not match :[{}]", giftPack.getProductId());
						throw new WrongArgException("赠品[" + productName + "]已失效,请重新选购");
					}
				}
			}
			// 多品赠品
			if (CartPackSourceEnum.GIFT.equals(cartPack.getSource())) {
				if (!multiGiftProdIds.contains(cartPack.getProductId())) {
					logger.error("clickPackMatching cart pack productId not match :[{}]", cartPack.getProductId());
					String productName = cartPack.getProduct().getNameCn();
					throw new WrongArgException("赠品[" + productName + "]已失效,请重新选购");
				}
			}
			// 换购
			if (CartPackSourceEnum.ADDIBLE.equals(cartPack.getSource())) {
				if (!addProIds.contains(cartPack.getProductId())) {
					logger.error("clickPackMatching cart pack productId not match :[{}]", cartPack.getProductId());
					String productName = cartPack.getProduct().getNameCn();
					throw new WrongArgException("换购商品[" + productName + "]已失效,请重新选购");
				}
			}
		}
	}

	/**
	 * 整理单品赠品多品赠品以及加价购
	 * 
	 * @param resultMap
	 * @param singleGiftMap
	 * @param mutiGiftMap
	 * @param addList
	 */
	public void getGiftMapAndAdditionList(Map<PrsCampaignVO, List<PrsAdditionalPackVO>> resultMap,
			Map<Long, List<PrsAdditionalPackVO>> singleGiftMap, Map<Long, List<PrsAdditionalPackVO>> mutiGiftMap,
			List<PrsAdditionalPackVO> addList) {
		if (resultMap.isEmpty()) {
			return;
		}
		for (Map.Entry<PrsCampaignVO, List<PrsAdditionalPackVO>> entry : resultMap.entrySet()) {
			List<PrsAdditionalPackVO> adPacks = entry.getValue();
			if (CollectionUtils.isEmpty(adPacks)) {
				continue;
			}
			for (PrsAdditionalPackVO adPack : adPacks) {
				if (entry.getKey().getDiscountTypeId().equals(DiscountType.GIFT.id)) {
					if (adPack.getIsSingleScopeCampaign()) {
						List<PrsAdditionalPackVO> giftSingleList = new ArrayList<PrsAdditionalPackVO>();
						if (singleGiftMap.containsKey(adPack.getBindSpvIds().get(0))) {
							giftSingleList.addAll(singleGiftMap.get(adPack.getBindSpvIds().get(0)));
						}
						giftSingleList.add(adPack);
						singleGiftMap.put(adPack.getBindSpvIds().get(0), giftSingleList);
					} else {
						List<PrsAdditionalPackVO> giftSingleList = new ArrayList<PrsAdditionalPackVO>();
						if (mutiGiftMap.containsKey(adPack.getBindSpvIds().get(0))) {
							giftSingleList.addAll(mutiGiftMap.get(adPack.getBindSpvIds().get(0)));
						}
						giftSingleList.add(adPack);
						mutiGiftMap.put(adPack.getBindSpvIds().get(0), giftSingleList);
					}
				} else {
					addList.add(adPack);
				}
			}
		}
	}

	/**
	 * 构建主品包
	 * 
	 * @param pack
	 * @param member
	 * @param presaleDate
	 * @return
	 */
	private CartProductPackVO buildingRawPack(CartProductPack pack, MemberVO member, Date presaleDate) {
		// spv
		PsSpvVO psSpvVO = pack.getSpv();
		SpvVO spvVO = iposSpvAdapter.buildingRawSpvVO(pack.getProduct(), pack.getSpv(), member);
		spvVO.setPreDeliver(iposSpvAdapter.getPreDeliverVO(pack, presaleDate)); // 预计发货信息

		CartProductPackVO packVO = new CartProductPackVO();
		packVO.setCount(pack.getCount());
		packVO.setStoreCount(pack.getStoreCount());
		packVO.setWarehouseCount(pack.getWarehouseCount());
		packVO.setSource(CartPackSource.valueOf(pack.getSource().name()));
		packVO.setProductId(pack.getProduct().getCode());
		packVO.setDeliverys(ScConvert.convert4ScDelivery(pack.getDeliverys()));
		packVO.setSpv(spvVO);
		packVO.setIsJit(psSpvVO.isJit());
		packVO.setHasVariants(pack.getProduct().getSpvList().size() > 1);
		packVO.setLabels(iposSpvAdapter.getSpvLabel(psSpvVO));
		packVO.setCustomizable(spvCustomizable(psSpvVO));
		packVO.setCustomized(pack.isCustomize());
		packVO.setCustomizes(getCustomization(pack.getCustomizeList()));
		return packVO;

	}

	/**
	 * 构建赠品包
	 */
	private CartProductPackVO buildingGiftPack(PrsCampaignVO campaign, PsProductVO product, PsSpvVO psSpvVO,
			PrsAdditionalPackVO add, Date presaleDate, int left) {
		CartProductPackVO packVO = new CartProductPackVO();
		boolean owned = left >= add.getCount();
		// spv
		SpvVO spvVO = iposSpvAdapter.buildingAddSpvVO(product, psSpvVO, left);
		spvVO.setPrice(BigDecimal.ZERO);
		spvVO.setPreDeliver(iposSpvAdapter.getPreDeliverVO(product, add.getCount(), psSpvVO, presaleDate));
		// 包裹信息
		packVO.setCount(add.getCount());
		packVO.setSource(CartPackSource.GIFT);
		packVO.setProductId(product.getCode());
		packVO.setSpv(spvVO);
		packVO.setHasVariants(product.getSpvList().size() > 1);
		packVO.setCampaign(iposPriceAdapter.getPackCampaign(campaign, left < add.getCount()));
		packVO.setIsJit(psSpvVO.isJit());
		packVO.setOwned(owned);
		return packVO;
	}

	/**
	 * 是否有大家具SKU
	 */
	public boolean furnitureSku(List<CartProductPack> packs) {
		boolean whetherLogistics = false;
		for (CartProductPack pack : packs) {
			PsSpvVO spv = pack.getSpv();
			if (pack.getSpv().getCombined()) {
				if (CollectionUtils.isNotEmpty(pack.getSpv().getSkuCombinationList())) {
					for (PsSkuCombinationVO psc : pack.getSpv().getSkuCombinationList()) {
						if (Integer.valueOf(1).equals(psc.getSubSkuVO().getWhetherLogistics())) {
							whetherLogistics = true;
							break;
						}
					}
				}
			} else {
				if (spv.getSkuVO() != null) {
					if (Integer.valueOf(1).equals(spv.getSkuVO().getWhetherLogistics())) {
						whetherLogistics = true;
					}
				}
			}
		}
		return whetherLogistics;
	}

	/**
	 * 获取生日信息
	 */
	public BdayVO getBday(MemberVO member, List<CartProductPack> packs) {
		if (!MemberLevelEnum.NORMAL.getCode().equals(member.getMemberLevel()) && member.isInBdayDiscountTime()) {
			BdayVO bday = new BdayVO();
			bday.setWithin(member.isInBdayDiscountTime());
			bday.setUsed(member.isBdayDiscountUsed());
			bday.setRatio(member.getBdayDiscountRatio());
			int count = 0;
			for (CartProductPack paramPack : packs) {
				PsProductVO prod = paramPack.getProduct();
				if (prod.getAllowBdDiscount() == 1 && CartPackSourceEnum.RAW.equals(paramPack.getSource())) {
					int size = 0;
					List<PsCampaignVO> campList = paramPack.getSpv().getCampaignList();// 抢购判断
					if (null != campList && campList.size() > 0) {
						for (PsCampaignVO camp : campList) {
							if (camp.getDiscountTypeId() == 6) {
								size++;
							}
						}
					}
					if (size == 0) {
						count++;
					}
				}
				if (prod.getAllowBdDiscount() == 1 && CartPackSourceEnum.BENEFIT.equals(paramPack.getSource())) {
					if (paramPack.getSpv().getDiscountPrice().compareTo(BigDecimal.ZERO) > 0) {
						count++;
					}
				}
			}
			if (count > 0) {
				return bday;
			}
		}
		return null;
	}

	/**
	 * 已选购商品包 Map
	 */
	public Map<Long, CartProductPack> getClientAddiblePack(List<CartProductPack> addiblePacks) {
		Map<Long, CartProductPack> clientAddiblePack = new HashMap<>();
		if (CollectionUtils.isNotEmpty(addiblePacks)) {
			for (CartProductPack addiblePack : addiblePacks) {
				if (CartPackSourceEnum.ADDIBLE.equals(addiblePack.getSource())) {
					clientAddiblePack.put(addiblePack.getProductId(), addiblePack);
				}
			}
		}
		return clientAddiblePack;
	}

	/**
	 * 是否包含仓库发货
	 */
	public boolean containWhDeliveryOrder(List<CartProductPack> rawPacks) {
		for (CartProductPack pack : rawPacks) {
			if (CartPackSourceEnum.RAW.equals(pack.getSource())) {
				if (pack.getWarehouseCount() != 0) {
					return true;
				}
			}
		}
		return false;
	}

	/**
	 * 是否包含仓库发货
	 */
	public boolean containWhDeliveryCart(List<CartProductPackVO> rawPacks) {
		for (CartProductPackVO pack : rawPacks) {
			if (pack.getWarehouseCount() != 0) {
				return true;
			}
		}
		return false;
	}

	/**
	 * 是否包含海淘商品
	 */
	public boolean hasOverSea(List<CartProductPack> packs) {
		Boolean hasOverSea = false;
		for (CartProductPack productPack : packs) {
			if (!CartPackSourceEnum.ADDIBLE.equals(productPack.getSource())) {
				hasOverSea = cartCommonAdapter.hasOversea(productPack);
			}
		}
		return hasOverSea;
	}

	/**
	 * 是否含有定制商品
	 */
	public boolean containtCustomize(List<CartProductPack> packs) {
		for (CartProductPack cartProductPack : packs) {
			if (BooleanUtils.isTrue(cartProductPack.isCustomize())) {
				return true;
			}
		}
		return false;
	}

	/**
	 * 是否包含海淘商品
	 * 
	 * @param packs
	 * @return
	 */
	public boolean isOversea(List<? extends ProductPack> packs) {
		for (ProductPack pack : packs) {
			PsProductVO product = pack.getProduct();
			if (Integer.valueOf(1).equals(product.getCrossBorderFlag())) {
				return true;
			}
		}
		return false;
	}

	/**
	 * 转换赠品包
	 * 
	 * @param mutiGiftMap
	 * @return
	 */
	public List<PrsAdditionalPackVO> getMutilGiftList(Map<Long, List<PrsAdditionalPackVO>> mutiGiftMap) {
		List<PrsAdditionalPackVO> packList = Lists.newArrayList();
		for (Entry<Long, List<PrsAdditionalPackVO>> entry : mutiGiftMap.entrySet()) {
			packList.addAll(entry.getValue());
		}
		return packList;
	}

}
