package com.thebeastshop.pegasus.channelservice.adapter.cart;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.thebeastshop.cart.ProductPack;
import com.thebeastshop.cart.enums.CartPackSourceEnum;
import com.thebeastshop.cart.resp.CartProductPack;
import com.thebeastshop.cart.resp.TempCartProductPack;
import com.thebeastshop.common.enums.MemberLevelEnum;
import com.thebeastshop.common.utils.EmptyUtil;
import com.thebeastshop.common.utils.NumberUtil;
import com.thebeastshop.kit.prop.PropConstants;
import com.thebeastshop.pegasus.channelservice.adapter.product.ForecastAdapter;
import com.thebeastshop.pegasus.merchandise.vo.*;
import com.thebeastshop.support.enums.CartGroupType;
import com.thebeastshop.support.enums.CartPackSource;
import com.thebeastshop.support.exception.WrongArgException;
import com.thebeastshop.support.vo.cart.CartProductPackVO;
import com.thebeastshop.support.vo.cart.GroupsVO;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;

/**
 * 购物车组装通用适配器
 *
 * @author Wang Yawei
 */
@Component
public class CartCommonAdapter {
	private Logger logger = LoggerFactory.getLogger(getClass());

	@Autowired
	private ForecastAdapter forecastAdapter;

	/**
	 * 拆分有效无效商品包： 权益商品为有效商品
	 *
	 * @param packs
	 * @param validPacks
	 * @param invalidPacks
	 * @param opChannel
	 */
	public void separateInvalid(List<CartProductPack> packs, List<CartProductPack> validPacks,
			List<CartProductPack> invalidPacks, OpChannelVO opChannel, MemberLevelEnum memberLevelEnum) {
		if (CollectionUtils.isNotEmpty(packs)) {
			for (CartProductPack pack : packs) {
				if (CartPackSourceEnum.BENEFIT.equals(pack.getSource())) {
					// 权益 商品不判断库存
					validPacks.add(pack);
				} else if (isCanSee(pack.getProduct(), opChannel) && hasStockByPsSpv(pack.getSpv())) {
					// 在售 上架 切有库存为有效商品
					validPacks.add(pack);
				} else if (isCanSee(pack.getProduct(), opChannel)
						&& forecastAdapter.checkForecastProduct(pack.getProduct().getCode(), memberLevelEnum)) {
					// 在售上架 可见 是预告商品，不校验库存
					validPacks.add(pack);
				} else {
					invalidPacks.add(pack);
				}
			}
		}
	}

	/**
	 * 是否可见：可售 且上架的商品可见
	 *
	 * @param product
	 * @param opChannel
	 * @return
	 */
	public boolean isCanSee(PsProductVO product, OpChannelVO opChannel) {
		if (CollectionUtils.isEmpty(product.getCanSaleSku(opChannel.getCode()))) {
			// 不可售
			return false;
		}

		boolean isOnShelf = true; // 是否上架
		List<String> onShelfChnCodes = product.getOnShelfChnCodes();
		if (EmptyUtil.isEmpty(onShelfChnCodes))
			isOnShelf = false;
		if (!onShelfChnCodes.contains(opChannel.getCode()))
			isOnShelf = false;
		if (!isOnShelf) {
			return false;
		}

		return true;
	}

	/**
	 * psSpv是否有库存
	 *
	 * @param spvVO
	 * @return
	 */
	public boolean hasStockByPsSpv(PsSpvVO spvVO) {

		if (CollectionUtils.isNotEmpty(spvVO.getCampaignList())) {
			for (PsCampaignVO psCampaignVO : spvVO.getCampaignList()) {
				// 抢购活动
				if (psCampaignVO.getDiscountTypeId() == 6) {
					return true;
				}

			}
		}
		// 库存为空或为负数返回false
		if (NumberUtil.isNullOrZero(spvVO.getCanUseQuantity()) || spvVO.getCanUseQuantity() < 0) {
			return false;
		}

		return true;
	}

	public List<GroupsVO> separateGroupPack(List<? extends ProductPack> packs, List<CartProductPackVO> validPackVOs) {
		List<GroupsVO> groupsVOList = new ArrayList<>();
		if (CollectionUtils.isEmpty(packs)) {
			return groupsVOList;
		}
		List<CartProductPackVO> overSeaPackVOs = new ArrayList<CartProductPackVO>();
		List<CartProductPackVO> normalPackVOs = new ArrayList<CartProductPackVO>();
		// 拆分海淘/普通商品
		separateOverSeaPack(validPackVOs, overSeaPackVOs, normalPackVOs, packs);

		PsProductVO product = packs.get(0).getProduct();
		if (product != null && product.getCrossBorderFlag() != null && product.getCrossBorderFlag().equals(1)) {
			if (CollectionUtils.isNotEmpty(overSeaPackVOs)) {
				groupsVOList.add(getGroupVO(overSeaPackVOs, 1));
			}
			if (CollectionUtils.isNotEmpty(normalPackVOs)) {
				groupsVOList.add(getGroupVO(normalPackVOs, 2));
			}
		} else {
			if (CollectionUtils.isNotEmpty(normalPackVOs)) {
				groupsVOList.add(getGroupVO(normalPackVOs, 2));
			}
			if (CollectionUtils.isNotEmpty(overSeaPackVOs)) {
				groupsVOList.add(getGroupVO(overSeaPackVOs, 1));
			}
		}

		logger.info("设置分组");
		// 设置分组 名称 针对一个
		setGroupsVOOne(groupsVOList);
		return groupsVOList;
	}

	/**
	 * 将商品包里面的商品拆分成两个 海淘和 普通商品
	 *
	 * @param packVOs
	 *            购物车中的商品包
	 * @param overSeaPackVOs
	 *            是海外购的商品
	 * @param normalPacks
	 *            不是海外购的商品 普通商品
	 * @author lijing
	 */
	private void separateOverSeaPack(List<CartProductPackVO> packVOs, List<CartProductPackVO> overSeaPackVOs,
			List<CartProductPackVO> normalPacks, List<? extends ProductPack> packs) {
		if (null == packVOs || null == overSeaPackVOs || null == normalPacks) {
			logger.info("有一个参数为空  ，    组合拆分 是否是海淘的商品  方法结束");
			return;
		}
		Map<String, PsProductVO> productMap = Maps.newHashMapWithExpectedSize(packs.size());
		for (ProductPack pack : packs) {
			productMap.put(pack.getProduct().getCode().toString(), pack.getProduct());
		}
		for (CartProductPackVO cartProductPack : packVOs) {
			if (null != cartProductPack && null != cartProductPack.getProductId()) {
				PsProductVO productVO = productMap.get(cartProductPack.getProductId());
				if (productVO != null && productVO.getCrossBorderFlag() != null
						&& productVO.getCrossBorderFlag().equals(1)) {
					overSeaPackVOs.add(cartProductPack);
				} else {
					normalPacks.add(cartProductPack);
				}
			}
		}
	}

	/**
	 * 判断 当前分组 只有一个 的时候 是设置标题和描述
	 *
	 * @param groupsVOOne
	 * @auhtro lijing 产品需求
	 */
	public void setGroupsVOOne(List<GroupsVO> groupsVOOne) {
		if (CollectionUtils.isNotEmpty(groupsVOOne) && 1 == groupsVOOne.size()) {
			groupsVOOne.get(0).setDescription("");
			groupsVOOne.get(0).setTitle("全选");
		}
	}

	/**
	 * 组合成需要的数据group的
	 *
	 * @param packs
	 * @param type
	 *            1 是海外购 2是普通
	 * @return
	 */
	private GroupsVO getGroupVO(List<CartProductPackVO> packs, int type) {
		if (type == 1) {
			GroupsVO groupsVO = new GroupsVO();
			groupsVO.setExclusive(true);
			groupsVO.setTitle("海外购商品");
			groupsVO.setType(CartGroupType.OVERSEA);
			groupsVO.setPacks(packs);
			String description = PropConstants.getProperties("haitaodesc", "应海关要求，必须和普通商品分开下单，敬请谅解");
			groupsVO.setDescription(description);
			return groupsVO;
		}
		if (type == 2) {
			GroupsVO groupsVO = new GroupsVO();
			groupsVO.setExclusive(false);
			groupsVO.setType(CartGroupType.NORMAL);
			groupsVO.setTitle("普通商品");
			groupsVO.setPacks(packs);
			String description = PropConstants.getProperties("putongdesc", "应海关要求，必须和跨境商品分开下单，敬请谅解");
			groupsVO.setDescription(description);
			return groupsVO;
		}
		return null;

	}

	/**
	 * 校验商品包： 1、商品包统一为海淘或非海淘
	 *
	 * @param packs
	 * @return
	 */
	public void checkPacks(List<? extends ProductPack> packs, Boolean hasOversea) {
		for (ProductPack productPack : packs) {
			if (hasOversea != hasOversea(productPack)) {
				throw new WrongArgException("预览接口必须全是海淘或是普通的！");
			}

		}
	}

	/**
	 * 判断商品包是否海淘
	 *
	 * @param pack
	 * @return
	 */
	public boolean hasOversea(ProductPack pack) {
		PsProductVO product = pack.getProduct();
		if (product.getCrossBorderFlag() != null && product.getCrossBorderFlag().equals(1)) {
			return true;
		} else {
			return false;
		}
	}

	/**
	 * 获取加价购商品
	 *
	 * @param packs
	 * @return
	 */
	public List<ProductPack> getOwnedAddiblePackPrice(List<? extends ProductPack> packs) {

		List<ProductPack> addiblePacks = Lists.newArrayList();

		if (CollectionUtils.isNotEmpty(packs)) {
			for (ProductPack pack : packs) {
				if (pack.getSource().equals(CartPackSourceEnum.ADDIBLE)) {
					addiblePacks.add(pack);
				}
			}
		}

		return addiblePacks;
	}

	/**
	 * 判断商品是否有规格，并组成map
	 *
	 * @param products
	 * @param opChannel
	 * @return
	 */
	public Map<PsProductVO, Boolean> mapCheckHasVariants(Collection<PsProductVO> products, OpChannelVO opChannel) {
		String channelCode = opChannel.getCode(); // 默认线下渠道
		if (CollectionUtils.isNotEmpty(products)) {
			Map<PsProductVO, Boolean> map = Maps.newHashMapWithExpectedSize(products.size());
			for (PsProductVO psProductVO : products) {
				List<PsSpvVO> spvs = psProductVO.getSpvList();
				if (CollectionUtils.isNotEmpty(spvs) && spvs.size() > 1) {
					List<PsSpvVO> newSpvs = Lists.newArrayListWithExpectedSize(spvs.size());
					Map<String, String> canSaleSkuMap = getCanSaleSkuMap(channelCode, psProductVO);
					Map<String, String> canSeeSkuMap = getCanSeeSkuMap(channelCode, psProductVO);
					for (PsSpvVO spvVO : spvs) {
						// 判断是否可用
						if (isSpvAvailable(canSaleSkuMap, canSeeSkuMap, spvVO)) {
							newSpvs.add(spvVO);
						}
					}
					if (CollectionUtils.isNotEmpty(newSpvs) && newSpvs.size() > 1) {
						map.put(psProductVO, true);
					} else {
						map.put(psProductVO, false);
					}
				} else {
					map.put(psProductVO, false);
				}

			}
			return map;
		}
		return null;
	}

	private Map<String, String> getCanSaleSkuMap(String chnCode, PsProductVO psProductVO) {
		Map<String, String> canSaleSkuMap = Maps.newHashMap();
		List<PsChnProdSaleSkuVO> canSaleSkuList = psProductVO.getChnCanSaleSKu();
		if (CollectionUtils.isEmpty(canSaleSkuList))
			return canSaleSkuMap;
		PsChnProdSaleSkuVO canSaleSku = null;
		for (PsChnProdSaleSkuVO chnProdSaleSkuVO : canSaleSkuList) {
			if (org.apache.commons.lang.StringUtils.equals(chnProdSaleSkuVO.getChnCode(), chnCode)) {
				canSaleSku = chnProdSaleSkuVO;
				break;
			}
		}
		if (canSaleSku != null) {
			for (String skuCode : canSaleSku.getCanSkus()) {
				canSaleSkuMap.put(skuCode, skuCode);
			}
		}
		return canSaleSkuMap;
	}

	private Map<String, String> getCanSeeSkuMap(String chnCode, PsProductVO psProductVO) {
		Map<String, String> canSeeSkuMap = Maps.newHashMap();
		List<PsChnProdSaleSkuVO> canSeeSkuList = psProductVO.getChnCanSeeSKu();
		if (CollectionUtils.isEmpty(canSeeSkuList))
			return canSeeSkuMap;
		PsChnProdSaleSkuVO canSeeSku = null;
		for (PsChnProdSaleSkuVO chnProdSaleSkuVO : canSeeSkuList) {
			if (org.apache.commons.lang.StringUtils.equals(chnProdSaleSkuVO.getChnCode(), chnCode)) {
				canSeeSku = chnProdSaleSkuVO;
				break;
			}
		}
		if (canSeeSku != null) {
			for (String skuCode : canSeeSku.getCanSkus()) {
				canSeeSkuMap.put(skuCode, skuCode);
			}
		}
		return canSeeSkuMap;
	}

	private boolean isSpvAvailable(Map<String, String> canSaleMap, Map<String, String> canSeeMap, PsSpvVO spvVO) {
		if (spvVO.getCombined()) { // 组合
			List<PsSkuCombinationVO> skuCombinationList = spvVO.getSkuCombinationList();
			for (PsSkuCombinationVO skuCombinationVO : skuCombinationList) {
				if (!isSkuAvailable(canSaleMap, canSeeMap, skuCombinationVO.getSubSkuCode())) {
					return false;
				}
			}
			return true;
		} else { // 单品
			return isSkuAvailable(canSaleMap, canSeeMap, spvVO.getSkuCode());
		}
	}

	private boolean isSkuAvailable(Map<String, String> canSaleMap, Map<String, String> canSeeMap, String skuCode) {
		return canSaleMap.containsKey(skuCode) && canSeeMap.containsKey(skuCode);
	}

	/**
	 * 拆分有效无效商品包： 权益商品为有效商品
	 *
	 * @param packs
	 * @param validPacks
	 * @param invalidPacks
	 * @param opChannel
	 */
	public void separateInvalidByTemp(List<TempCartProductPack> packs, Collection<TempCartProductPack> validPacks,
			List<TempCartProductPack> invalidPacks, OpChannelVO opChannel) {
		if (CollectionUtils.isNotEmpty(packs)) {
			for (TempCartProductPack pack : packs) {
				if (CartPackSourceEnum.BENEFIT.equals(pack.getSource())) {
					// 权益 商品不判断库存
					validPacks.add(pack);
				} else if (isCanSee(pack.getProduct(), opChannel) && hasStockByPsSpv(pack.getSpv())) {
					// 在售 上架 切有库存为有效商品
					validPacks.add(pack);
				} else if (isCanSee(pack.getProduct(), opChannel)
						&& forecastAdapter.checkForecastProduct(pack.getProduct().getCode(), MemberLevelEnum.NORMAL)) {
					// 在售上架 可见 是预告商品，不校验库存
					validPacks.add(pack);
				} else {
					invalidPacks.add(pack);
				}
			}
		}
	}

	public void cartPackSeparateInvalidMini(List<CartProductPack> packs, List<CartProductPack> validPacks,
			List<CartProductPack> invalidPacks, OpChannelVO opChannel) {
		for (CartProductPack pack : packs) {
			PsProductVO product = pack.getProduct();
			PsSpvVO spv = pack.getSpv();
			if (null != product && null != spv) {
				boolean isValid = isProductValid(product, spv.getCanUseQuantity(), opChannel);
				if (isValid) {
					validPacks.add(pack);
				} else {
					invalidPacks.add(pack);
				}
			}
		}
	}

	private boolean isProductValid(PsProductVO productVO, Integer spvStock, OpChannelVO opChannel) {
		boolean isCanSaleSku = EmptyUtil.isNotEmpty(productVO.getCanSaleSku(opChannel.getCode()));
		boolean isOnShelf = true;
		List<String> onShelfChnCodes = productVO.getOnShelfChnCodes();
		if (EmptyUtil.isEmpty(onShelfChnCodes))
			isOnShelf = false;
		else if (!onShelfChnCodes.contains(opChannel.getCode()))
			isOnShelf = false;
		if (!isProductValid(isCanSaleSku, isOnShelf)) {
			return false;
		}

		// 是否售罄
		if (spvStock == null || spvStock <= 0) {
			return false;
		}
		return true;
	}

	private boolean isProductValid(boolean isCanSaleSku, boolean isOnShelf) {
		return isCanSaleSku && isOnShelf;
	}

	public void tempCartPackSeparateInvalidMini(List<TempCartProductPack> packs, List<TempCartProductPack> validPacks,
			List<TempCartProductPack> invalidPacks, OpChannelVO opChannel) {
		for (TempCartProductPack pack : packs) {
			PsProductVO product = pack.getProduct();
			PsSpvVO spv = pack.getSpv();
			if (null != product && null != spv) {
				boolean isValid = isProductValid(product, spv.getCanUseQuantity(), opChannel);
				if (isValid) {
					validPacks.add(pack);
				} else {
					invalidPacks.add(pack);
				}
			}
		}
	}

}
