package com.thebeastshop.pegasus.component.coupon.service.impl;

import com.alibaba.fastjson.JSONObject;
import com.google.common.collect.Lists;
import com.thebeastshop.common.kafka.KafkaProducerClient;
import com.thebeastshop.pegasus.component.coupon.condition.CouponCodeExchangeCondition;
import com.thebeastshop.pegasus.component.coupon.condition.CouponCondition;
import com.thebeastshop.pegasus.component.coupon.condition.CouponSetCondition;
import com.thebeastshop.pegasus.component.coupon.dao.CouponDao;
import com.thebeastshop.pegasus.component.coupon.domain.*;
import com.thebeastshop.pegasus.component.coupon.domain.impl.CouponWrapper;
import com.thebeastshop.pegasus.component.coupon.domain.impl.DefaultCouponCodeImpl;
import com.thebeastshop.pegasus.component.coupon.domain.impl.DefaultCouponImpl;
import com.thebeastshop.pegasus.component.coupon.domain.impl.DefaultCouponSampleImpl;
import com.thebeastshop.pegasus.component.coupon.dto.VerificationCouponDTO;
import com.thebeastshop.pegasus.component.coupon.enums.CouponSampleState;
import com.thebeastshop.pegasus.component.coupon.model.CouponEntity;
import com.thebeastshop.pegasus.component.coupon.model.CouponMemberCount;
import com.thebeastshop.pegasus.component.coupon.model.VerificationCoupon;
import com.thebeastshop.pegasus.component.coupon.service.*;
import com.thebeastshop.pegasus.component.coupon.util.ConnectionFactory;
import com.thebeastshop.pegasus.component.coupon.util.ConsJson;
import com.thebeastshop.pegasus.component.coupon.util.HttpKit;
import com.thebeastshop.pegasus.component.coupon.util.PropertyConfigurer;
import com.thebeastshop.support.exception.WrongArgException;
import com.thebeastshop.support.exception.WrongStateException;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.text.SimpleDateFormat;
import java.util.*;

/**
 * 
 * @author Paul-xiong
 *
 */
@Service
public class CouponServiceImpl extends BaseService implements CouponService {

	private static final Logger logger = LoggerFactory.getLogger(CouponServiceImpl.class);

	@Autowired
	private CouponSampleService couponSampleService;

	@Autowired
	private CouponDao couponDao;

	@Autowired
	private PropertyConfigurer propertyConfigurer;

	@Autowired
	private CouponSetService couponSetService;

	@Autowired
	CouponCodeService couponCodeService;

	@Autowired
	CouponCodeExchangeService couponCodeExchangeService;

	@Autowired
	KafkaProducerClient kafkaProducerClient;


	@Override
	public Coupon getById(long id) {
		Coupon obj = this.couponDao.getById(id);
		if (obj == null) {
			throw new WrongArgException("优惠券模板不存在", "id", id);
		}
		return obj;
	}

	@Override
	public List<Coupon> getCoupons(long ownerId) {
		return this.couponDao.getCoupons(ownerId);
	}

	@Override
	public List<Coupon> getCoupons(long ownerId, Long couponSampleId) {
		return this.couponDao.getCoupons(ownerId, couponSampleId);
	}

	@Override
	public List<Coupon> getUsableCoupons(final long ownerId) {
		return this.couponDao.getUsableCoupons(ownerId);
	}

	@Override
	public List<Coupon> getExpiredCoupons(final long ownerId) {
		return this.couponDao.getExpiredCoupons(ownerId);
	}

	@Override
	public List<Coupon> getUsedCoupons(final long ownerId) {
		return this.couponDao.getUsedCoupons(ownerId);
	}

	@Override
	public List<Coupon> getCouponsByCondition(CouponCondition condition) {
		return this.couponDao.getCouponsByCondition(condition);
	}

	@Override
	public CouponCount queryCouponCount(CouponCondition condition) {
		return this.couponDao.countCouponByCondition(condition);
	}

	@Override
	public Coupon create(final Coupon coupon) {
		return this.couponDao.create(coupon);
	}

	@Override
	public void invalid(final Coupon coupon) {
		Coupon couponImpl = new CouponWrapper<Coupon>(coupon) {
			@Override
			public boolean isValid() {
				return false;
			}
		};
		this.update(couponImpl);
	}

	@Transactional
	@Override
	public Coupon send(CouponSample couponSample, long memberId) {
		if (couponSample == null) {
			throw new WrongArgException("发送优惠券失败", "couponSample", couponSample);
		}
		if (couponSample.getState() == null) {
			throw new WrongArgException("发送优惠券失败", "couponSample.state", couponSample.getState());
		}
		if (!CouponSampleState.ONLINE.getId().equals(couponSample.getState().getId())) {
			throw new WrongArgException("优惠券未通过审批或已废弃");
		}
		// 检查剩余数量
		if (couponSample.getBalanceCount() <= 0) {
			throw new WrongArgException("优惠券已发完");
		}
		// 检查每人最多获得的上限限制
		Integer perMemberCount = this.getCouponsCount(memberId, couponSample.getId());
		if (perMemberCount >= couponSample.getMaxPerMember()) {
			throw new WrongArgException("您已领取过该优惠券！");
		}
		// 优惠券剩余数量减一
		DefaultCouponSampleImpl couponSampleImpl = (DefaultCouponSampleImpl) couponSample;
		boolean ret = this.couponSampleService.decBalanceCount(couponSampleImpl);
		if (!ret) {
			throw new WrongArgException("优惠券已抢光！");
		}
		// 创建优惠券
		DefaultCouponImpl coupon = new DefaultCouponImpl();
		coupon.setCouponSampleId(couponSample.getId());
		if(couponSample.getDateWay() != null && couponSample.getDateWay() == 2){
			try{
				Date currDate = new Date();
				coupon.setStartTime(currDate);
				SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd");
				String currDateStr = dateFormat.format(currDate);
				dateFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
				Date date = dateFormat.parse(currDateStr + " 23:59:51");
				Calendar cal = Calendar.getInstance();
				cal.setTime(date);
				cal.add(Calendar.DATE, couponSample.getDateNum());
				coupon.setExpireTime(cal.getTime());

			}catch (Exception e){}

		} else {
			coupon.setStartTime(couponSample.getStartTime());
			coupon.setExpireTime(couponSample.getExpireTime());
		}
		coupon.setNote(couponSample.getNote());
		coupon.setOwnerId(memberId);
		coupon.setUsed(false);
		coupon.setValid(true);
		return this.create(coupon);
	}

	@Transactional
	@Override
	public CouponCodeExchange sendOffline(CouponSample couponSample, long memberId) {
		Coupon coupon = send(couponSample, memberId);
		String code = "";
		//同一个模板循环到不存在相同的优惠码
		while (true){
			code = String.valueOf((int)(Math.random() * 9000) + 1000);
			Map<String, Object> map = new HashMap<String, Object>();
			map.put("csid", couponSample.getId());
			map.put("code", code);
			if (couponCodeService.getValidByCode(code) != null) {
			}else{
				break;
			}
		}

		DefaultCouponCodeImpl couponCode = new DefaultCouponCodeImpl();
		couponCode.setCouponSampleId(couponSample.getId());
		couponCode.setCode(code);
		couponCode.setCreatorId(couponSample.getCreatorId());
		couponCode.setCreateTime(new Date());
		couponCode.setNote(couponSample.getNote());
		couponCodeService.create(couponCode);

		CouponCodeExchange createObj = new CouponCodeExchange();
		createObj.setCode(code);
		createObj.setCouponId(coupon.getId());
		createObj.setCouponSampleId(couponSample.getId());
		createObj.setOwnerId(memberId);
		return couponCodeExchangeService.create(createObj);
	}

	@Transactional
	@Override
	public Coupon checkOffline(String code, long memberId,String storeId) {
		if (StringUtils.isBlank(code)) {
			throw new WrongArgException("兑换码不能为空", "code", code);
		}
		if (StringUtils.isBlank(storeId)) {
			throw new WrongArgException("店铺代码不能为空", "storeId", storeId);
		}

		// 校验优惠券口令是否有效
		CouponCodeExchangeCondition couponCodeExchangeCondition = new CouponCodeExchangeCondition();
		couponCodeExchangeCondition.setOwnerId(memberId);
		couponCodeExchangeCondition.setCode(code);
		List<CouponCodeExchange> couponCodeExchangeList = couponCodeExchangeService.getByCondition(couponCodeExchangeCondition);
		if (CollectionUtils.isEmpty(couponCodeExchangeList)) {
			throw new WrongArgException("兑换码不存在", "code", code);
		}

		Map map = new HashMap<String, Object>();
		map.put("memberId",memberId);
		map.put("code",code);
		map.put("storeId", storeId);
		CouponEntity obj = this.selectValidByMap(map);
		if(obj == null){
			throw new WrongArgException("该优惠码已经被使用过", "code", code);
		}else{
			String offlineStoreIds = obj.getOfflineStoreIds();
			if(offlineStoreIds != null && offlineStoreIds.contains(storeId)){
				throw new WrongArgException("优惠码在该线下门店不能使用", "code", code);
			}
		}

		final CouponSample couponSample = this.couponSampleService.getById(obj.getCouponSampleId());
		// 校验优惠券模板是否可领取
		if(couponSample.getDateWay() != null && couponSample.getDateWay() == 2) {

		}else{
			if (!CouponSampleState.ONLINE.equals(couponSample.getState()) || couponSample.isExpired()) {
				throw new WrongArgException("兑换码已过期", "state", couponSample.getState());
			}
		}
		// 校验优惠券是否超过用户限领数量
		CouponCodeExchangeCondition condition = new CouponCodeExchangeCondition();
		condition.setOwnerId(memberId);
		condition.setCouponSampleId(couponSample.getId());
		condition.setCode(code);
		List<CouponCodeExchange> exchanges = couponCodeExchangeService.getByCondition(condition);
		if (exchanges.size() >= couponSample.getMaxPerMember()) {
			throw new WrongArgException("优惠券超过限领数量", "code", code);
		}
		return this.getById(obj.getId());
	}


	@Transactional
	@Override
	public Boolean sendCheck(CouponSample couponSample, long memberId) {
		if (couponSample == null) {
			return  false;
		}
		if (couponSample.getState() == null) {
			return  false;
		}
		if (!CouponSampleState.ONLINE.getId().equals(couponSample.getState().getId())) {
			return  false;
		}
		// 检查剩余数量
		if (couponSample.getBalanceCount() <= 0) {
			return  false;
		}
		// 检查每人最多获得的上限限制
		Integer perMemberCount = this.getCouponsCount(memberId, couponSample.getId());
		if (perMemberCount >= couponSample.getMaxPerMember()) {
			return  false;
		}
		// 优惠券剩余数量减一
		DefaultCouponSampleImpl couponSampleImpl = (DefaultCouponSampleImpl) couponSample;
		boolean ret = this.couponSampleService.decBalanceCount(couponSampleImpl);
		if (!ret) {
			return  false;
		}
		// 创建优惠券
		DefaultCouponImpl coupon = new DefaultCouponImpl();
		coupon.setCouponSampleId(couponSample.getId());
//		coupon.setExpireTime(couponSample.getExpireTime());
		coupon.setNote(couponSample.getNote());
		coupon.setOwnerId(memberId);
//		coupon.setStartTime(couponSample.getStartTime());
		if(couponSample.getDateWay() != null && couponSample.getDateWay() == 2){
			try{
				Date currDate = new Date();
				coupon.setStartTime(currDate);
				SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd");
				String currDateStr = dateFormat.format(currDate);
				dateFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
				Date date = dateFormat.parse(currDateStr + " 23:59:51");
				Calendar cal = Calendar.getInstance();
				cal.setTime(date);
				cal.add(Calendar.DATE, couponSample.getDateNum());
				coupon.setExpireTime(cal.getTime());

			}catch (Exception e){}
		} else {
			coupon.setStartTime(couponSample.getStartTime());
			coupon.setExpireTime(couponSample.getExpireTime());
		}
		coupon.setUsed(false);
		coupon.setValid(true);
		this.create(coupon);
		return true;
	}

	@Transactional
	@Override
	public Boolean sendCouponList(CouponSample couponSample,Collection<Long> memberIds) {
		if (couponSample == null) {
			throw new WrongArgException("发送优惠券失败", "couponSample", couponSample);
		}
		if (couponSample.getState() == null) {
			throw new WrongArgException("发送优惠券失败", "couponSample.state", couponSample.getState());
		}
		if (!CouponSampleState.ONLINE.getId().equals(couponSample.getState().getId())) {
			throw new WrongArgException("优惠券未通过审批或已废弃");
		}
		// 检查剩余数量
		if (couponSample.getBalanceCount() <= 0 || couponSample.getBalanceCount() < memberIds.size()) {
			throw new WrongArgException("优惠券不足");
		}

		//map 判断(memberId,Excel表中出现的次数)
		List<Coupon> clist = new ArrayList<Coupon>();
		for(Long memberId:memberIds) {
//			if(map.containsKey(memberId)){
//				Integer value = map.get(memberId);
//				map.put(memberId,value+1);
//			}else{
//				map.put(memberId,1);
//			}
			DefaultCouponImpl coupon = new DefaultCouponImpl();
			coupon.setCouponSampleId(couponSample.getId());
			coupon.setNote(couponSample.getNote());
			coupon.setOwnerId(memberId);
			if(couponSample.getDateWay() != null && couponSample.getDateWay() == 2){
				try{
					Date currDate = new Date();
					coupon.setStartTime(currDate);
					SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd");
					String currDateStr = dateFormat.format(currDate);
					dateFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
					Date date = dateFormat.parse(currDateStr + " 23:59:51");
					Calendar cal = Calendar.getInstance();
					cal.setTime(date);
					cal.add(Calendar.DATE, couponSample.getDateNum());
					coupon.setExpireTime(cal.getTime());

				}catch (Exception e){}
			} else {
				coupon.setStartTime(couponSample.getStartTime());
				coupon.setExpireTime(couponSample.getExpireTime());
			}
			coupon.setUsed(false);
			coupon.setValid(true);
			clist.add(coupon);
		}
		Long startTime = System.currentTimeMillis();
		//去掉判断是否超过用户领取限制 影响效率
		//数据库用户优惠券个数+Excel优惠券个数 大于用户限制 则失败
		//不用List是因为map去除了menberId的重复
		// 这样可以减少数据库的查询提高效率 by Lk
//		Map<Long,Integer> map = new HashMap<Long,Integer>();
//		for (Map.Entry<Long, Integer> entry : map.entrySet()) {
//			Long memberId = entry.getKey();
//			Integer value = entry.getValue();
//			Integer perMemberCount = this.getCouponsCount(memberId, couponSample.getId());
//			if ((perMemberCount + value) > couponSample.getMaxPerMember()) {
//				throw new WrongStateException("获得" + couponSample.getName() + "的优惠券已达到上限。",
//						"member{id=" + memberId + "}.coupons{couponSampleId=" + couponSample.getId() + "}.size",
//						couponSample.getMaxPerMember());
//			}
//		}

		Connection conn = null;
		PreparedStatement ps = null;
		try {
			conn = ConnectionFactory.getConnection(propertyConfigurer);
			conn.setAutoCommit(false);
			StringBuffer sb = new StringBuffer();
			sb.append("INSERT INTO COUPON (COUPON_SAMPLE_ID, OWNER_ID, IS_USED,")
					.append(" IS_VALID, NOTE, START_TIME,EXPIRE_TIME, CREATE_TIME) ")
					.append(" VALUES (?,?,?,?,?,?,?,?) ");
			int i1 =1,i2=2,i3=3,i4=4,i5=5,i6=6,i7=7,i8=8;
			ps = conn.prepareStatement(sb.toString());
			final int batchSize = 1000;
			int count = 0;
			for (Coupon o: clist) {
				ps.setLong(i1, o.getCouponSampleId());
				ps.setLong(i2, o.getOwnerId());
				ps.setInt(i3,o.isUsed() == true ? 1 : 0);
				ps.setInt(i4, o.isValid()== true ? 1 : 0);
				ps.setString(i5, o.getNote());
				ps.setTimestamp(i6, new java.sql.Timestamp(o.getStartTime()!=null?o.getStartTime().getTime():null));
				ps.setTimestamp(i7, new java.sql.Timestamp(o.getExpireTime()!=null?o.getExpireTime().getTime():null));
				ps.setTimestamp(i8, new java.sql.Timestamp(new java.util.Date().getTime()));
				ps.addBatch();
				if(++count % batchSize == 0) {
					ps.executeBatch();
					ps.clearBatch();
				}
			}
			ps.executeBatch(); // insert remaining records
			conn.commit();
			//更新优惠券剩余数量 by Lk
			DefaultCouponSampleImpl couponSampleImpl = (DefaultCouponSampleImpl) couponSample;
			couponSampleImpl.setBalanceCount(couponSample.getBalanceCount() - clist.size());
			boolean ret = this.couponSampleService.update(couponSampleImpl);
			Long endTime = System.currentTimeMillis();
			System.out.println("OK,用时：" + (endTime - startTime)/1000);
			ps.close();
			conn.close();
			return true;
		} catch (Exception e) {
			e.printStackTrace();
			try {
				//同时成功和失败 失败则全部回滚
				conn.rollback();
			} catch (Exception e1) {
				System.out.println("批处理发生异常"+ e.getMessage());
			}
			return false;
		} finally{
			//finally block used to close resources
			try{
				if(ps!=null)
					ps.close();
			}catch(Exception se2){
			}// nothing we can do
			try{
				if(conn!=null)
					conn.close();
			}catch(Exception se){
				se.printStackTrace();
			}//end finally try
		}//end try
	}

	@Override
	public Boolean update(Coupon obj) {
		return this.couponDao.update(obj);
	}

	@Override
	public void use(Coupon obj) {
		if (obj != null) {
			Coupon coupon = new CouponWrapper<Coupon>(obj) {
				@Override
				public boolean isUsed() {
					return true;
				}
			};
			this.update(coupon);
		}
	}

	public void couponUseEzr(Long couponId,String shopCode,String salesNo,double salesMoney) {
		logger.info("优惠券couponId:" + couponId + "shopCode:" + shopCode + "salesNo:" + salesNo + "salesMoney:" + salesMoney + "已经使用过，将通过ezr核销");
		String ezrCouponCode=couponDao.getEZRCouponCodeByCid(couponId);
		if(StringUtils.isNotEmpty(ezrCouponCode)){
			VerificationCouponDTO dto = new VerificationCouponDTO();
			dto.setSalesMoney(salesMoney);
			dto.setSalesNo(salesNo);
			dto.setShopCode(shopCode);
			String[] couponNos ={ezrCouponCode};
			dto.setCouponNos(couponNos);
			boolean flag = kafkaProducerClient.send(dto);
			if(!flag){
				logger.error("优惠券couponId:" + couponId + "shopCode:" + shopCode + "salesNo:" + salesNo + "salesMoney:" + salesMoney +" 发送kafka失败");
			}
		}else{
			logger.info("优惠券couponId:" + couponId + "shopCode:" + shopCode + "salesNo:" + salesNo + "salesMoney:" + salesMoney + " 核销失败");
		}
	}

	private VerificationCoupon covertVerificationCouponForEzr(String ezrCouponCode){
		VerificationCoupon vo = new VerificationCoupon();
		vo.setSalesNo("ezr0000011");
		vo.setShopCode("CHN2018");
		String[] couponNos ={ezrCouponCode};
		vo.setCouponNos(couponNos);
		return vo;
	}


	@Override
	public void use(Long couponId) {
		use(getById(couponId));
	}

	@Override
	public int getCouponsCount(long ownerId) {
		return this.getCoupons(ownerId).size();
	}

	@Override
	public int getCouponsCount(long ownerId, Long couponSampleId) {
		return this.getCoupons(ownerId, couponSampleId).size();
	}

	@Override
	public int getUsableCouponsCount(long ownerId) {
		return getUsableCoupons(ownerId).size();
	}

	@Override
	public int getExpiredCouponsCount(long ownerId) {
		return getExpiredCoupons(ownerId).size();
	}
	
	@Override
	public List<Coupon> getNotExpiredCoupons(long ownerId) {
		return this.couponDao.getNotExpiredCoupons(ownerId);
	}

	@Override
	public int getNotExpiredCouponsCount(long ownerId) {
		return getNotExpiredCoupons(ownerId).size();
	}

	@Override
	public int getUsedCouponsCount(long ownerId) {
		return getUsedCoupons(ownerId).size();
	}

	@Override
	public Collection<Coupon> create(Collection<Coupon> coupons) {
		List<Coupon> list = Lists.newArrayList();
		for (Coupon coupon : coupons) {
			list.add(create(coupon));
		}
		return list;
	}

	@Override
	public void invalid(Collection<Coupon> coupons) {
		for (Coupon coupon : coupons) {
			invalid(coupon);
		}

	}

	@Override
	@Transactional
	public Collection<Coupon> send(CouponSample couponSample, Collection<Long> memberIds) {
		List<Coupon> list = Lists.newArrayList();
		for (Long memberId : memberIds) {
			list.add(send(couponSample, memberId));
		}
		return list;
	}

	@Override
	@Transactional
	public Boolean sendCouponsToMemberIds(Collection<Long> ids, Collection<Long> memberIds) {
		CouponSetCondition condition = new CouponSetCondition();
		condition.setIds(ids);
		List<Long> couponSampleList = couponSetService.getCouponSampleIdsByCondition(condition);
		//查询数据库通过套券ID拿到
		if(CollectionUtils.isNotEmpty(couponSampleList)){
			for (Long couponSampleId : couponSampleList) {
				for(Long memberId : memberIds){
					CouponSample couponSample = couponSampleService.getById(couponSampleId);
					sendCheck(couponSample, memberId);
				}
			}
		}
		return true;
	}

	@Override
	@Transactional
	public String getEZRCouponCodeByCid(Long couponId) {
		return couponDao.getEZRCouponCodeByCid(couponId);
	}

	@Override
	public Boolean mergeCoupon(Long mainId, Long subId) {
		Boolean result = couponDao.mergeCoupon(mainId, subId) > 0;
		if(result) {
			logger.info("合并会员编号为" + subId + "的优惠券到编号为:" + mainId + "的会员账户");
		}
		return result;
	}

	@Override
	public void unuse(Long couponId) {
		Coupon obj = getById(couponId);
		if (obj != null) {
			Coupon coupon = new CouponWrapper<Coupon>(obj) {
				@Override
				public boolean isUsed() {
					return false;
				}
			};
			this.update(coupon);
		}
	}

	@Override
	public List<CouponMemberCount> getAllCouponsByExpireTime(Date expireTime, Integer currentPage, Integer pageSize) {
		return couponDao.getAllCouponsByExpireTime(expireTime, currentPage, pageSize);
	}

	@Override
	public CouponEntity selectValidByMap(Map<String, Object> map) {
		return couponDao.selectValidByMap(map);
	}

}
