package com.thebeastshop.pegasus.component.coupon.service;

import java.util.Collection;
import java.util.List;

import com.thebeastshop.pegasus.component.coupon.condition.CouponCondition;
import com.thebeastshop.pegasus.component.coupon.domain.Coupon;
import com.thebeastshop.pegasus.component.coupon.domain.CouponCount;
import com.thebeastshop.pegasus.component.coupon.domain.CouponSample;
import com.thebeastshop.support.exception.WrongArgException;

/**
 * @author Liang Wenjian
 */
public interface CouponService {

	/**
	 * 根据id获取优惠券
	 * 
	 * @param id
	 * @return
	 */
	Coupon getById(final long id);

	/**
	 * 获取优惠券列表
	 * 
	 * @param ownerId
	 * @return
	 */
	List<Coupon> getCoupons(final long ownerId);

	/**
	 * 获取优惠券的数量
	 * 
	 * @param ownerId
	 * @return
	 */
	int getCouponsCount(final long ownerId);

	/**
	 * 根据优惠券模板id获取优惠券列表
	 * 
	 * @param ownerId
	 * @return
	 */
	List<Coupon> getCoupons(final long ownerId, final Long couponSampleId);

	/**
	 * 根据优惠券模板id获取优惠券的数量
	 * 
	 * @param ownerId
	 * @return
	 */
	int getCouponsCount(final long ownerId, final Long couponSampleId);

	/**
	 * 获取可用的优惠券列表
	 *
	 * @param ownerId
	 * @return
	 */
	List<Coupon> getUsableCoupons(final long ownerId);

	/**
	 * 获取可用的优惠券的数量
	 *
	 * @param ownerId
	 * @return
	 */
	int getUsableCouponsCount(final long ownerId);

	/**
	 * 获取过期的优惠券列表
	 *
	 * @param ownerId
	 * @return
	 */
	List<Coupon> getExpiredCoupons(final long ownerId);

	/**
	 * 获取未过期的优惠券列表
	 *
	 * @param ownerId
	 * @return
	 */
	List<Coupon> getNotExpiredCoupons(final long ownerId);

	/**
	 * 获取未过期的优惠券数量
	 *
	 * @param ownerId
	 * @return
	 */
	int getNotExpiredCouponsCount(final long ownerId);

	/**
	 * 获取已过期的优惠券数量
	 *
	 * @param ownerId
	 * @return
	 */
	int getExpiredCouponsCount(final long ownerId);

	/**
	 * 获取已用的优惠券列表
	 *
	 * @param ownerId
	 * @return
	 */
	List<Coupon> getUsedCoupons(final long ownerId);

	/**
	 * 获取已使用的优惠券数量
	 *
	 * @param ownerId
	 * @return
	 */
	int getUsedCouponsCount(final long ownerId);

	/**
	 * 根据条件获取优惠券列表
	 * 
	 * @param condition
	 * @return
	 */
	List<Coupon> getCouponsByCondition(CouponCondition condition);

	/**
	 * 根据条件统计优惠券
	 *
	 * @param condition
	 * @return
	 */
	CouponCount queryCouponCount(CouponCondition condition);

	/**
	 * 创建一个优惠券.
	 *
	 * @param coupon
	 * @return
	 * @throws WrongArgException
	 *             若所创建的优惠券的样板对每个会员的数量有限制，且要发给的会员已达到限制
	 */
	Coupon create(Coupon coupon);

	/**
	 * 创建一批优惠券。
	 *
	 * @param coupons
	 * @return
	 */
	Collection<Coupon> create(final Collection<Coupon> coupons);

	/**
	 * 令一个优惠券失效。
	 *
	 * @param coupon
	 */
	void invalid(Coupon coupon);

	/**
	 * 令一批优惠券失效。
	 *
	 * @param coupons
	 */
	void invalid(final Collection<Coupon> coupons);

	/**
	 * 给一批会员发放某种优惠券一个。
	 *
	 * @param couponSample
	 * @param memberIds
	 * @return
	 */
	Collection<Coupon> send(final CouponSample couponSample, final Collection<Long> memberIds);

	/**
	 * 给一个会员发放某种套券。
	 *
	 * @param couponSample
	 * @param memberId
	 * @return
	 */
	Boolean sendCheck(final CouponSample couponSample, final long memberId);

	/**
	 * 给多个会员发放多种套券。
	 *
	 * @param ids
	 * @param memberIds
	 * @return
	 */
	Boolean sendCouponsToMemberIds(final Collection<Long> ids, final Collection<Long> memberIds);


	/**
	 * 给一个会员发放某种优惠券一个。
	 *
	 * @param couponSample
	 * @param memberId
	 * @return
	 */
	Coupon send(final CouponSample couponSample, final long memberId);

	/**
	 * 批量给会员发放某种优惠券。
	 *
	 * @return
	 */
	Boolean sendCouponList(final CouponSample couponSample, final Collection<Long> memberIds);

	/**
	 * 更新优惠券
	 * 
	 * @param obj
	 * @return
	 */
	Boolean update(Coupon obj);

	/**
	 * 使用优惠券
	 * 
	 * @param obj
	 * @return
	 */
	void use(Coupon obj);

	/**
	 * 使用优惠券
	 * 
	 * @param couponId
	 * @return
	 */
	void use(Long couponId);
	
	/**
	 * 退优惠券
	 * 
	 * @param couponId
	 */
	void unuse(Long couponId);

	/**
	 * 获取EZR优惠券编码
	 *
	 * @param couponId
	 * @return
	 */
	String getEZRCouponCodeByCid(Long couponId);

	Boolean mergeCoupon(Long mainId, Long subId);
}
