package com.thebeastshop.pegasus.component.coupon.service.impl;

import java.math.BigDecimal;
import java.util.*;

import com.thebeastshop.pegasus.component.coupon.condition.PendingCouponCondition;
import com.thebeastshop.pegasus.component.coupon.domain.CouponCode;
import com.thebeastshop.pegasus.component.coupon.domain.CouponContent;
import com.thebeastshop.pegasus.component.coupon.domain.impl.CouponSampleTemplate;
import com.thebeastshop.pegasus.component.coupon.domain.impl.DefaultCouponCodeImpl;
import com.thebeastshop.pegasus.component.coupon.domain.impl.FullCutPrice;
import com.thebeastshop.pegasus.component.coupon.enums.CouponType;
import com.thebeastshop.pegasus.component.coupon.service.CouponCodeService;
import com.thebeastshop.support.exception.BusinessLogicNotExpectedException;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import com.google.common.collect.Lists;
import com.thebeastshop.pegasus.component.coupon.condition.CouponSampleCondition;
import com.thebeastshop.pegasus.component.coupon.dao.CouponSampleDao;
import com.thebeastshop.pegasus.component.coupon.domain.CouponSample;
import com.thebeastshop.pegasus.component.coupon.enums.CouponSampleState;
import com.thebeastshop.pegasus.component.coupon.enums.CouponSendWay;
import com.thebeastshop.pegasus.component.coupon.service.BaseService;
import com.thebeastshop.pegasus.component.coupon.service.CouponSampleService;
import com.thebeastshop.support.AccessWay;
import com.thebeastshop.support.exception.NoSuchResourceException;
import com.thebeastshop.support.exception.WrongArgException;
import com.thebeastshop.support.mark.State;
import org.springframework.transaction.annotation.Transactional;

/**
 * 
 * @author Paul-xiong
 *
 */
@Service
public class CouponSampleServiceImpl extends BaseService implements CouponSampleService {
	@Autowired
	private CouponSampleDao couponSampleDao;

	@Autowired
	private CouponCodeService couponCodeService;

	private void validate(CouponSample obj) {
		if (obj.getCouponType() == null) {
			throw new WrongArgException("未选择优惠券类型");
		}
		if (StringUtils.isEmpty(obj.getName())) {
			throw new WrongArgException("未填优惠券名称");
		}
		if (StringUtils.isEmpty(obj.getTitle())) {
			throw new WrongArgException("未填优惠券标题");
		}
		if (obj.getBalanceCount() == null) {
			throw new WrongArgException("未填剩余数量");
		}
		if (obj.getMaxCount() == null) {
			throw new WrongArgException("未填生成数量");
		}
		if (obj.getMaxPerMember() == null) {
			throw new WrongArgException("未填用户限领数量");
		}
		if (CollectionUtils.isEmpty(obj.getAccessWays())) {
			throw new WrongArgException("未选择适用平台");
		}
		if (obj.getCouponSendWay() == null) {
			throw new WrongArgException("未选择优惠券派发方式");
		}
		if(obj.getDateWay() != null && obj.getDateWay() == 1) {
			if (obj.getStartTime() == null) {
				throw new WrongArgException("未选择优惠券开始日期");
			}
			if (obj.getExpireTime() == null) {
				throw new WrongArgException("未选择优惠券截止日期");
			}
			if (!obj.getExpireTime().after(obj.getStartTime())) {
				throw new WrongArgException("优惠券截止日期必须在开始日期之后");
			}
		}
		if (obj.getCouponContent() == null) {
			throw new WrongArgException("未填优惠券优惠内容");
		}
		CouponContent content = obj.getCouponContent();
		if (obj.getCouponType().getId() == CouponType.CHEAPEN_LINE.getId()) {
			FullCutPrice fullCutPrice = (FullCutPrice) content;
			if (fullCutPrice.getFullPrice() == null) {
				throw new WrongArgException("未填优惠满足条件");
			}
			if (fullCutPrice.getCutPrice() == null) {
				throw new WrongArgException("未填优惠金额");
			}
			if (fullCutPrice.getFullPrice().compareTo(BigDecimal.ZERO) != 0 && fullCutPrice.getCutPrice().compareTo(fullCutPrice.getFullPrice()) > 0) {
				throw new WrongArgException("优惠金额必须小于等于满足条件");
			}
		}

		if (obj.getCouponSendWay().getId() == CouponSendWay.CODE.getId()) {
			// 优惠口令合法性判断
			String code = obj.getCouponCode();
			if (StringUtils.isEmpty(code)) {
				throw new WrongArgException("优惠口令不能为空");
			}
			// 判断是否存在相同可用的优惠口令
			if(obj.getId() == null){
				if (couponCodeService.isExistsSameCode(code)) {
					throw new WrongArgException("存在相同优惠口令");
				}
			}else {
				Map<String,Object> map = new HashMap<String,Object>();
				map.put("csid",obj.getId());
				map.put("code",code);
				if (couponCodeService.isExistsSameCodeByMap(map)) {
					throw new WrongArgException("存在相同优惠口令");
				}
			}
		}
	}

	@Transactional
	@Override
	public CouponSample create(CouponSample obj) {
		validate(obj);
		return this.couponSampleDao.create(obj);
	}

	private CouponCode buildCouponCode(CouponSample domain) {
		DefaultCouponCodeImpl couponCode = new DefaultCouponCodeImpl();
		couponCode.setCouponSampleId(domain.getId());
		couponCode.setCode(domain.getCouponCode());
		couponCode.setCreatorId(domain.getCreatorId());
		couponCode.setCreateTime(new Date());
		couponCode.setNote(domain.getNote());
		return couponCode;
	}

	@Transactional
	@Override
	public Boolean createCouponSample(CouponSample domain) {
		validate(domain);
		CouponSample couponSample = couponSampleDao.create(domain);
		if(null == couponSample || null == couponSample.getId()) {
			throw new BusinessLogicNotExpectedException("添加优惠模板出错");
		}
		if (domain.getCouponSendWay().getId().equals(CouponSendWay.CODE.getId())) {
			couponSample.setCouponCode(domain.getCouponCode());
			CouponCode couponCode = buildCouponCode(couponSample);
			if(domain.getId() == null){
				CouponCode result = couponCodeService.create(couponCode);
				if(null == result || null == result.getId()) {
					throw new BusinessLogicNotExpectedException("添加优惠码出错");
				}
			}else{
				Boolean flag = couponCodeService.update(couponCode);
				if(!flag) {
					throw new BusinessLogicNotExpectedException("编辑优惠码出错");
				}
			}

		}
		return true;
	}

	@Transactional
	@Override
	public Boolean updateCouponSample(CouponSample domain) {
		validate(domain);
		boolean flag = couponSampleDao.update(domain);
		if(!flag) {
			throw new BusinessLogicNotExpectedException("编辑优惠模板出错");
		}
		if (domain.getCouponSendWay().getId().equals(CouponSendWay.CODE.getId())) {
			CouponCode couponCode = buildCouponCode(domain);
			boolean flag2 = couponCodeService.update(couponCode);
			if(!flag2) {
				throw new BusinessLogicNotExpectedException("编辑优惠码出错");
			}
		}
		return true;
	}

	@Override
	public boolean update(CouponSample obj) {
		return this.couponSampleDao.update(obj);
	}

	public boolean decBalanceCount(CouponSample obj) {
		return this.couponSampleDao.decBalanceCount(obj);
	}

	@Override
	public boolean delete(Long id) {
		return false;
	}

	@Override
	public CouponSample getById(final long id) {
		CouponSample obj = this.couponSampleDao.getById(id);
		if (obj == null) {
			throw new WrongArgException("优惠券模板不存在", "id", id);
		}
		return obj;
	}

	@Override
	public List<State> getStates() {
		return Lists.<State> newArrayList(CouponSampleState.values());
	}

	@Override
	public List<CouponSendWay> getCouponSendWays() {
		return Arrays.asList(CouponSendWay.values());
	}

	@Override
	public List<AccessWay> getAccessWays() {
		return Arrays.asList(AccessWay.values());
	}

	@Override
	public List<CouponSample> getByCondition(CouponSampleCondition condition) {
		return this.couponSampleDao.getByCondition(condition);
	}

	@Override
	public List<CouponSample> getBySelectCondition(CouponSampleCondition condition) {
		return couponSampleDao.getBySelectCondition(condition);
	}

	@Override
	public List<CouponSample> getFirstPendingCouponSample(PendingCouponCondition cond) {
		Integer type = cond.getType();
		if(null == type) {
			return null;
		}
		if(1 == type) {
			//返回已经过第一级审批的数据集合
			return couponSampleDao.getFirstApprovaledCouponSample();
		}else if (0 == type) {
			//返回未经过第一级审批的数据集合
			return couponSampleDao.getFirstPendingCouponSample();
		}
		return null;
	}

	@Override
	public List<CouponSample> getSecondPendingCouponSample(PendingCouponCondition cond) {
		Integer type = cond.getType();
		if(null == type) {
			return null;
		}
		if(1 == type) {
			//返回已经过第二级审批的数据集合
			return couponSampleDao.getSecondApprovaledCouponSample();
		}else if(0 == type) {
			//返回未经过第二级审批的数据集合
			return couponSampleDao.getSecondPendingCouponSample();
		}
		return null;
	}

	@Override
	public Integer submitApproval(Long id) {
		return couponSampleDao.submitApproval(id);
	}

	@Override
	public CouponSample checkById(long id) {
		final CouponSample one = getById(id);
		if (one == null) {
			throw new NoSuchResourceException("优惠券样板", id);
		}
		return one;
	}

}
