package com.thebeastshop.pegasus.component.coupon.dao.impl;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.List;

import com.thebeastshop.pegasus.component.coupon.enums.CouponType;
import org.apache.commons.lang.ArrayUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import com.google.common.collect.Lists;
import com.thebeastshop.pegasus.component.coupon.condition.CouponCondition;
import com.thebeastshop.pegasus.component.coupon.dao.CouponDao;
import com.thebeastshop.pegasus.component.coupon.dao.mapper.CouponEntityMapper;
import com.thebeastshop.pegasus.component.coupon.domain.Coupon;
import com.thebeastshop.pegasus.component.coupon.domain.CouponCount;
import com.thebeastshop.pegasus.component.coupon.domain.impl.DefaultCouponImpl;
import com.thebeastshop.pegasus.component.coupon.domain.impl.DefaultCouponSampleImpl;
import com.thebeastshop.pegasus.component.coupon.enums.CouponSendWay;
import com.thebeastshop.pegasus.component.coupon.model.CouponCountEntity;
import com.thebeastshop.pegasus.component.coupon.model.CouponEntity;
import com.thebeastshop.pegasus.component.coupon.model.CouponEntityExample;
import com.thebeastshop.pegasus.component.coupon.model.CouponEntityExample.Criteria;
import com.thebeastshop.support.AccessWay;

/**
 * @author Liang Wenjian
 */
@Repository
public class CouponDaoImpl implements CouponDao {
	private Logger logger = LoggerFactory.getLogger(this.getClass());

	@Autowired
	private CouponEntityMapper mapper;

	/**
	 * Coupon to CouponEntity
	 * 
	 * @param obj
	 *            Coupon对象
	 * @return
	 */
	private CouponEntity domain2Entity(Coupon obj) {
		CouponEntity entity = new CouponEntity();
		entity.setCouponSampleId(obj.getCouponSampleId());
		entity.setCreateTime(obj.getCreateTime());
		entity.setExpireTime(obj.getExpireTime());
		entity.setId(obj.getId());
		entity.setIsUsed(obj.isUsed());
		entity.setIsValid(obj.isValid());
		entity.setNote(obj.getNote());
		entity.setOwnerId(obj.getOwnerId());
		entity.setStartTime(obj.getStartTime());
		return entity;
	}

	/**
	 * CouponEntity to Coupon
	 * 
	 * @param entity
	 *            CouponEntity对象
	 * @return
	 */
	private Coupon entity2Domain(CouponEntity entity) {
		DefaultCouponImpl obj = new DefaultCouponImpl();
		obj.setCouponSampleId(entity.getCouponSampleId());
		obj.setCreateTime(entity.getCreateTime());
		obj.setExpireTime(entity.getExpireTime());
		obj.setId(entity.getId());
		obj.setUsed(entity.getIsUsed());
		obj.setValid(entity.getIsValid());
		obj.setNote(entity.getNote());
		obj.setOwnerId(entity.getOwnerId());
		obj.setStartTime(entity.getStartTime());
		obj.setNickName(entity.getNickName());
		String sampleName = entity.getSampleName();
		try {
			if (StringUtils.isNotEmpty(sampleName)) {
				DefaultCouponSampleImpl couponSample = new DefaultCouponSampleImpl();
				couponSample.setId(entity.getCouponSampleId());
				couponSample.setName(sampleName);
				couponSample.setTitle(entity.getSampleTitle());
				couponSample.setCouponContent(entity.getCouponContent());
				couponSample.setBalanceCount(entity.getBalanceCount());
				couponSample.setStateId(entity.getSampleStateId());
				couponSample.setDescription(entity.getSampleDescription());
				couponSample.setMaxCount(entity.getMaxCount());
				couponSample.setMaxPerMember(entity.getMaxPerMember());
				couponSample.setNote(entity.getSampleNote());
				couponSample.setCreateTime(entity.getSampleCreateTime());
				couponSample.setCreatorId(entity.getSampleCreatorId());
				couponSample.setExpireTime(entity.getSampleExpireTime());

				if (entity.getCouponType() != null) {
					couponSample.setCouponType(CouponType.valueOf(entity.getCouponType()));
				}

				String accessWays = entity.getAccessWays();
				if (StringUtils.isNotEmpty(accessWays)) {
					String[] accessWayIds = accessWays.split(",");
					if (ArrayUtils.isNotEmpty(accessWayIds)) {
						List<AccessWay> accessWayList = new ArrayList<>();
						for (String accessWayId : accessWayIds) {
							if (StringUtils.isEmpty(accessWayId))
								continue;
							try {
								if (StringUtils.isNotEmpty(accessWayId)) {
									AccessWay accessWay = AccessWay.valueOf(Integer.valueOf(accessWayId));
									accessWayList.add(accessWay);
								}
							} catch (Exception e) {
								logger.error(e.getMessage());
							}
						}
						couponSample.setAccessWays(accessWayList);
					}
				}

				String sendWay = entity.getCouponSendWay();
				if (StringUtils.isNotEmpty(sendWay)) {
					couponSample.setCouponSendWay(CouponSendWay.valueOf(Integer.valueOf(sendWay)));
				}

				String productIds = entity.getProductIds();
				if (StringUtils.isNotEmpty(productIds)) {
					String[] productIdStrList = productIds.split(",");
					if (ArrayUtils.isNotEmpty(productIdStrList)) {
						List<Long> productIdList = new ArrayList<>();
						for (String productIdStr : productIdStrList) {
							try {
								if (StringUtils.isNotEmpty(productIdStr)) {
									productIdList.add(Long.valueOf(productIdStr));
								}
							} catch (Exception e) {
								logger.error(e.getMessage());
							}
						}
						couponSample.setProductIds(productIdList);
					}
				}

				String categoryIds = entity.getCategoryIds();
				if (StringUtils.isNotEmpty(categoryIds)) {
					String[] categoryIdStrList = categoryIds.split(",");
					if (ArrayUtils.isNotEmpty(categoryIdStrList)) {
						List<Long> categoryIdList = new ArrayList<>();
						for (String categoryIdStr : categoryIdStrList) {
							try {
								categoryIdList.add(Long.valueOf(categoryIdStr));
							} catch (Exception e) {
								logger.error(e.getMessage());
							}
						}
						couponSample.setCategoryIds(categoryIdList);
					}
				}
				obj.setCouponSample(couponSample);
			}
		} catch (Exception e) {
			logger.error(e.getMessage());
		}
		return obj;
	}

	private List<Coupon> entity2Domain(List<CouponEntity> entities) {
		List<Coupon> domains = Lists.newArrayList();
		for (CouponEntity entity : entities) {
			domains.add(entity2Domain(entity));
		}
		return domains;
	}

	private void fillCriteria(Criteria criteria, CouponCondition condition) {
		if (condition.getOwnerId() != null) {
			criteria.andOwnerIdEqualTo(condition.getOwnerId());
		}
		if (condition.getCouponSampleId() != null) {
			criteria.andCouponSampleIdEqualTo(condition.getCouponSampleId());
		}
		if (condition.getIsUsed() != null) {
			criteria.andIsUsedEqualTo(condition.getIsUsed());
		}
		if (condition.getIsValid() != null) {
			criteria.andIsValidEqualTo(condition.getIsValid());
		}
	}

	@Override
	public Coupon getById(long id) {
		return this.entity2Domain(this.mapper.selectById(id));
	}

	@Override
	public List<Coupon> getCoupons(long ownerId) {
		CouponCondition condition = new CouponCondition();
		condition.setOwnerId(ownerId);
		return this.getCouponsByCondition(condition);
	}

	@Override
	public List<Coupon> getCoupons(long ownerId, Long couponSampleId) {
		CouponCondition condition = new CouponCondition();
		condition.setOwnerId(ownerId);
		condition.setCouponSampleId(couponSampleId);
		return this.getCouponsByCondition(condition);
	}

	@Override
	public List<Coupon> getUsableCoupons(final long ownerId) {
		CouponEntityExample example = new CouponEntityExample();
		example.createCriteria().andIsValidEqualTo(true).andIsUsedEqualTo(false).andOwnerIdEqualTo(ownerId)
				.andStartTimeLessThanOrEqualTo(new Date()).andExpireTimeGreaterThanOrEqualTo(new Date());
		List<CouponEntity> entityList = this.mapper.selectByExample(example);

		return entity2Domain(entityList);
	}

	@Override
	public List<Coupon> getExpiredCoupons(final long ownerId) {
		CouponEntityExample example = new CouponEntityExample();
		example.createCriteria().andOwnerIdEqualTo(ownerId).andIsValidEqualTo(true).andIsUsedEqualTo(false)
				.andExpireTimeLessThan(new Date());
		List<CouponEntity> entityList = this.mapper.selectByExample(example);

		return entity2Domain(entityList);
	}

	@Override
	public List<Coupon> getUsedCoupons(final long ownerId) {
		CouponCondition condition = new CouponCondition();
		condition.setOwnerId(ownerId);
		condition.setIsValid(true);
		condition.setIsUsed(true);
		return this.getCouponsByCondition(condition);
	}

	@Override
	public List<Coupon> getCouponsByCondition(CouponCondition condition) {
		// CouponEntityExample example = new CouponEntityExample();
		// this.fillCriteria(example.createCriteria(), condition);
		List<CouponEntity> entityList = this.mapper.selectByCondition(condition);
		return entity2Domain(entityList);
	}

	@Override
	public CouponCount countCouponByCondition(CouponCondition condition) {
		CouponCountEntity couponCountEntity = this.mapper.countCouponByCondition(condition);
		if (couponCountEntity == null) {
			CouponCount emptyCoupontCount = new CouponCount();
			emptyCoupontCount.setUnusedCount(0);
			emptyCoupontCount.setUsedCount(0);
			emptyCoupontCount.setExpiredCount(0);
			emptyCoupontCount.setUnexpiredCount(0);
			return emptyCoupontCount;
		}
		CouponCount couponCount = new CouponCount();
		couponCount.setUnusedCount(couponCountEntity.getUnusedCount());
		couponCount.setUsedCount(couponCountEntity.getUsedCount());
		couponCount.setExpiredCount(couponCountEntity.getExpiredCount());
		couponCount.setUnexpiredCount(couponCountEntity.getUnexpiredCount());
		couponCount.setUnusedAndUnexpiredCount(couponCountEntity.getUnusedAndUnexpiredCount());
		couponCount.setExpiredAndUnusedCount(couponCountEntity.getExpiredAndUnusedCount());
		return couponCount;
	}

	@Override
	public Coupon create(final Coupon coupon) {
		logger.info("Creating Coupon: {}", coupon);

		CouponEntity entity = this.domain2Entity(coupon);
		if (entity != null) {
			entity.setCreateTime(new Date());
			int count = this.mapper.insertSelective(entity);

			if (count > 0) {
				logger.info("Created Coupon: {}", coupon);
				return this.entity2Domain(entity);
			}
		}

		return null;
	}

	@Override
	public Boolean update(Coupon obj) {
		logger.info("Updating Coupon: {}", obj);

		CouponEntity entity = this.domain2Entity(obj);
		if (entity != null) {
			int count = this.mapper.updateByPrimaryKeySelective(entity);

			if (count > 0) {
				logger.info("Updated Coupon: {}", obj);
				return true;
			}
		}

		return false;
	}

	@Override
	public int getCouponsCount(long ownerId) {
		return this.getCoupons(ownerId).size();
	}

	@Override
	public int getCouponsCount(long ownerId, Long couponSampleId) {
		return this.getCoupons(ownerId, couponSampleId).size();
	}

	@Override
	public int getUsableCouponsCount(long ownerId) {
		return getUsableCoupons(ownerId).size();
	}

	@Override
	public int getExpiredCouponsCount(long ownerId) {
		return getExpiredCoupons(ownerId).size();
	}

	@Override
	public int getUsedCouponsCount(long ownerId) {
		return getUsedCoupons(ownerId).size();
	}

	@Override
	public Collection<Coupon> create(Collection<Coupon> coupons) {
		List<Coupon> list = Lists.newArrayList();
		for (Coupon coupon : coupons) {
			list.add(create(coupon));
		}
		return list;
	}

	@Override
	public Integer mergeCoupon(Long mainId, Long subId) {
		CouponEntity entity = new CouponEntity();
		entity.setOwnerId(mainId);
		CouponEntityExample example = new CouponEntityExample();
		example.createCriteria().andOwnerIdEqualTo(subId);
		return mapper.updateByExampleSelective(entity, example);
	}
}
