package com.thebeastshop.stock.vo;

import com.thebeastshop.common.BaseDO;
import com.thebeastshop.stock.dto.SAlwaysCanUse;
import com.thebeastshop.stock.enums.SSkuStockTypeEnum;
import com.thebeastshop.stock.enums.SStockQueryTypeEnum;

/**
 * SKU维度库存的抽象类
 * 包含所有SKU级别库存的一些公用属性
 */
public abstract class SStockOfSku<SELF extends SStockOfSku<SELF>> extends BaseDO implements SAlwaysCanUse<SELF> {

    /**
     * SKU库存类型
     * <p>
     * 用于标识SKU维度库存的类型，
     * 调用方需要先判断类型，在做相应的库存处理
     * <p>
     * 其包含如下几种类型：
     * <ul>
     *     <li>DEFAULT: 普通库存</li>
     *     <li>PRESALE: 预售库存</li>
     *     <li>JIT: 来单库存</li>
     * </ul>
     */
    protected final SSkuStockTypeEnum stockType;

    protected SStockQueryTypeEnum queryType;

    /**
     * 是否来单
     */
    protected Boolean jit;

    /**
     * 是否预告
     */
    protected Boolean announce;

    /**
     * 是否定金
     */
    protected Boolean deposit;

    /**
     * SKU编号
     */
    protected String skuCode;

    /**
     * 仓库编号
     */
    protected String warehouseCode;


    protected SStockOfSku(SSkuStockTypeEnum stockType) {
        this.stockType = stockType;
    }

    public SStockQueryTypeEnum getQueryType() {
        return queryType;
    }

    public void setQueryType(SStockQueryTypeEnum queryType) {
        this.queryType = queryType;
    }

    public Boolean getJit() {
        return jit;
    }

    public SELF setJit(Boolean jit) {
        this.jit = jit;
        return (SELF) this;
    }

    public Boolean getAnnounce() {
        return announce;
    }

    public SELF setAnnounce(Boolean announce) {
        this.announce = announce;
        return (SELF) this;
    }

    public Boolean getDeposit() {
        return deposit;
    }

    public SELF setDeposit(Boolean deposit) {
        this.deposit = deposit;
        return (SELF) this;
    }

    public void setSkuCode(String skuCode) {
        this.skuCode = skuCode;
    }

    public void setWarehouseCode(String warehouseCode) {
        this.warehouseCode = warehouseCode;
    }

    /**
     * 获取SKU库存类型
     *
     * @return {@link SSkuStockTypeEnum}枚举实例
     */
    public SSkuStockTypeEnum getStockType() {
        return stockType;
    }

    /**
     * 是否为普通库存
     *
     * @return {@code true}: 是普通库存，{@code false}: 不是
     */
    public boolean isDefault() {
        return stockType == SSkuStockTypeEnum.DEFAULT;
    }

    /**
     * 是否为预售库存
     *
     * @return {@code true}: 是预售库存，{@code false}: 不是
     */
    public boolean isPresale() {
        return stockType == SSkuStockTypeEnum.PRESALE;
    }


    /**
     * 获取SKU编号
     *
     * @return SKU编号
     */
    public String getSkuCode() {
        return skuCode;
    }

    /**
     * 获取仓库编号
     *
     * @return 仓库编号
     */
    public String getWarehouseCode() {
        return warehouseCode;
    }

    /**
     * 获取SKU库存总数
     *
     * @return SKU库存总数
     */
    public abstract Integer getTotalQuantity();

    /**
     * 获取SKU库存可用数量
     *
     * @return SKU库存可用数量
     */
    public abstract Integer getCanUseQuantity();

    /**
     * 获取SKU库存真实可用数量
     * @return
     */
    public abstract Integer getRealCanUseQuantity();

}
