package com.thebeastshop.stock;

import com.thebeastshop.stock.dto.SCombinationDTO;
import com.thebeastshop.stock.dto.SIPosSpuStockDetailQueryDTO;
import com.thebeastshop.stock.dto.SAbstractSpuStockDetailQueryDTO;
import com.thebeastshop.stock.dto.SIPosSpvCombinedStockDetailQueryDTO;
import com.thebeastshop.stock.dto.SIPosSpvStockDetailQueryDTO;
import com.thebeastshop.stock.dto.SSpuStockDetailQueryDTO;
import com.thebeastshop.stock.dto.SSpvCombinedStockDetailQueryDTO;
import com.thebeastshop.stock.dto.SSpvStockDetailQueryDTO;
import com.thebeastshop.stock.enums.SStockChannelTypeEnum;
import com.thebeastshop.stock.vo.SStockChannelVO;

/**
 * 库存服务快速接口
 * <p>
 * 帮助快速构建一些常用查询对象
 */
public abstract class SStocks {

    /**
     * ============================= 普通 SPU/SPV 维度商品库存明细查询对象构造方法 =============================
     */

    /**
     * 构建 SPU 库存明细查询对象
     *
     * @param prodCode SPU编号
     * @return SPU 库存明细查询 {@link SSpuStockDetailQueryDTO} 对象
     */
    public static SSpuStockDetailQueryDTO spuDetail(String prodCode) {
        return new SSpuStockDetailQueryDTO().setProdCode(prodCode);
    }

    /**
     * 构建普通 SPV 库存明细查询对象
     *
     * @param spvId SPV ID
     * @return SPV 库存明细查询 {@link SSpvStockDetailQueryDTO} 对象
     */
    public static SSpvStockDetailQueryDTO spvDetail(Long spvId) {
        return new SSpvStockDetailQueryDTO().setSpvId(spvId);
    }

    /**
     * 构建组合 SPV 库存明细查询对象
     *
     * @param spvId 组合 SPV ID
     * @return 组合 SPV 库存明细查询 {@link SSpvStockDetailQueryDTO} 对象
     */
    public static SSpvCombinedStockDetailQueryDTO spvCombinedDetail(Long spvId) {
        return new SSpvCombinedStockDetailQueryDTO().setSpvId(spvId);
    }


    /**
     * ============================= iPos 下的 SPU/SPV 维度商品库存明细查询对象构造方法 =============================
     */


    /**
     * 构建 iPos 下的 SPU 库存明细查询对象
     *
     * @param prodCode SPU编号
     * @return iPos 下的 SPU 库存明细查询 {@link SIPosSpuStockDetailQueryDTO} 对象
     */
    public static SIPosSpuStockDetailQueryDTO iPosSpuDetail(String prodCode) {
        return new SIPosSpuStockDetailQueryDTO().setProdCode(prodCode);
    }

    /**
     * 构建 iPos 下的 SPV 库存明细查询对象
     *
     * @param spvId SPV ID
     * @return iPos 下的 SPV 库存明细查询 {@link SIPosSpvCombinedStockDetailQueryDTO} 对象
     */
    public static SIPosSpvStockDetailQueryDTO iPosSpvDetail(Long spvId) {
        return new SIPosSpvStockDetailQueryDTO().setSpvId(spvId);
    }


    /**
     * 构建 iPos 下的组合商品 SPV 库存明细查询对象
     *
     * @param spvId SPV ID
     * @return iPos 下的组合商品 SPV 库存明细查询 {@link SIPosSpvCombinedStockDetailQueryDTO} 对象
     */
    public static SIPosSpvCombinedStockDetailQueryDTO iPosSpvCombinedDetail(Long spvId) {
        return new SIPosSpvCombinedStockDetailQueryDTO().setSpvId(spvId);
    }

    /**
     * ============================= SPU/SPV 维度商品库存明细查询对象下的商品层级信息构造方法 =============================
     */




    /**
     * ============================= 库存渠道信息构造方法 =============================
     */

    /**
     * 构建渠道信息
     *
     * @param warehouseCode 渠道仓库编号
     * @return {@link SStockChannelVO} 对象
     */
    public static SStockChannelVO channel(SStockChannelTypeEnum channelType, String channelCode, String warehouseCode) {
        return new SStockChannelVO()
                .setChannelType(channelType)
                .setChannelCode(channelCode)
                .setWarehouseCode(warehouseCode);
    }


    /**
     * 构建线上渠道信息
     *
     * @param channelCode 渠道编号
     * @param warehouseCode 渠道仓库编号
     * @return {@link SStockChannelVO} 对象
     */
    public static SStockChannelVO onlineChannel(String channelCode, String warehouseCode) {
        return channel(SStockChannelTypeEnum.ONLINE, channelCode, warehouseCode);
    }

    /**
     * 构建线下渠道信息
     *
     * @param channelCode 渠道编号
     * @param warehouseCode 渠道仓库编号
     * @return {@link SStockChannelVO} 对象
     */
    public static SStockChannelVO offlineChannel(String channelCode, String warehouseCode) {
        return channel(SStockChannelTypeEnum.OFFLINE, channelCode, warehouseCode);
    }

    /**
     * ============================= 组合商品下的 SKU 组合项构造方法 =============================
     */


    /**
     * 构建组合商品 SKU 组合项
     *
     * @param quantity 组合项目中 SKU 的数量
     * @param skuCode 组合项中的 SKU 编号
     * @return 组合商品 SKU 组合项 {@link SCombinationDTO} 对象
     */
    public static SCombinationDTO combination(Integer quantity, String skuCode) {
        return new SCombinationDTO()
                .setQuantity(quantity)
                .setSkuCode(skuCode);
    }

}
