package com.thebeastshop.mbgw.sdk.util;


import com.thebeastshop.mbgw.sdk.config.MbGwSdkConfig;
import org.apache.http.client.ClientProtocolException;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.util.EntityUtils;
import org.slf4j.Logger;

import javax.activation.MimetypesFileTypeMap;
import java.io.*;
import java.net.HttpURLConnection;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.util.Map;

/**
 * 马帮ERP SDK HTTP工具类
 * 封装JSON格式POST请求和multipart/form-data格式文件上传请求的底层实现
 */
public class MbGwHttpUtil {
    // 日志对象
    private static final Logger log = MbGwLogUtil.getLogger(MbGwHttpUtil.class);
    // 连接超时时间（5秒）
    private static final int CONNECT_TIMEOUT = 5000;
    // 读取超时时间（30秒）
    private static final int READ_TIMEOUT = 30000;
    // 表单提交分隔符（固定格式，用于区分不同参数）
    private static final String FORM_BOUNDARY = "---------------------------0123456789";

    /**
     * 发送JSON格式的POST请求（用于普通接口调用）
     * @param requestJson 序列化后的请求参数JSON字符串
     * @param sign 签名字符串（HmacSHA256生成）
     * @return 接口返回的响应字符串（JSON格式）
     */
    public static String sendJsonPost(String requestJson, String sign) {
        // 从配置获取API基础地址
        String baseUrl = MbGwSdkConfig.getBaseUrl();
        log.debug("发送JSON请求到地址：{}，请求体长度：{}", baseUrl, requestJson.length());

        // 使用Apache HttpClient发送请求
        try (CloseableHttpClient httpClient = HttpClients.createDefault()) {
            HttpPost httpPost = new HttpPost(baseUrl);

            // 设置请求头
            httpPost.setHeader("Content-Type", "application/json; charset=UTF-8");
            httpPost.setHeader("X-Requested-With", "XMLHttpRequest");
            httpPost.setHeader("Authorization", sign); // 签名放在Authorization头

            // 设置请求体（JSON字符串）
            httpPost.setEntity(new StringEntity(requestJson, StandardCharsets.UTF_8));

            // 执行请求并获取响应
            try (CloseableHttpResponse response = httpClient.execute(httpPost)) {
                int statusCode = response.getStatusLine().getStatusCode();
                log.debug("JSON请求响应状态码：{}", statusCode);

                // 解析响应实体
                return EntityUtils.toString(response.getEntity(), StandardCharsets.UTF_8);
            }
        } catch (ClientProtocolException e) {
            log.error("JSON请求HTTP协议异常", e);
            throw new RuntimeException("HTTP协议错误：" + e.getMessage(), e);
        } catch (IOException e) {
            log.error("JSON请求IO异常", e);
            throw new RuntimeException("网络请求失败：" + e.getMessage(), e);
        }
    }

    /**
     * 发送multipart/form-data格式的POST请求（用于文件上传）
     * @param textParams 文本参数集合（key:参数名，value:参数值）
     * @param fileParams 文件参数集合（key:参数名，value:本地文件路径）
     * @param sign 签名字符串（HmacSHA256生成）
     * @return 接口返回的响应字符串（JSON格式）
     */
    public static String sendFormPost(Map<String, String> textParams,
                                      Map<String, String> fileParams,
                                      String sign) {
        // 从配置获取API基础地址
        String baseUrl = MbGwSdkConfig.getBaseUrl();
        log.debug("发送表单请求到地址：{}，文本参数数：{}，文件参数数：{}",
                baseUrl, textParams.size(), fileParams.size());

        HttpURLConnection conn = null;
        try {
            // 创建HTTP连接
            URL url = new URL(baseUrl);
            conn = (HttpURLConnection) url.openConnection();

            // 配置连接参数
            conn.setConnectTimeout(CONNECT_TIMEOUT);
            conn.setReadTimeout(READ_TIMEOUT);
            conn.setDoOutput(true); // 允许输出（发送请求体）
            conn.setDoInput(true);  // 允许输入（读取响应）
            conn.setUseCaches(false); // 不使用缓存
            conn.setRequestMethod("POST"); // 请求方法为POST

            // 设置请求头
            conn.setRequestProperty("Authorization", sign);
            conn.setRequestProperty("Connection", "Keep-Alive");
            conn.setRequestProperty("Content-Type",
                    "multipart/form-data; boundary=" + FORM_BOUNDARY);

            // 写入请求体（文本参数+文件参数）
            try (OutputStream out = new DataOutputStream(conn.getOutputStream())) {
                writeTextParams(out, textParams);   // 写入文本参数
                writeFileParams(out, fileParams);   // 写入文件参数
                writeFormEndMarker(out);            // 写入表单结束标记
            }

            // 读取响应结果
            int statusCode = conn.getResponseCode();
            log.debug("表单请求响应状态码：{}", statusCode);
            return readResponse(conn);

        } catch (IOException e) {
            log.error("表单请求异常", e);
            throw new RuntimeException("文件上传失败：" + e.getMessage(), e);
        } finally {
            // 关闭连接
            if (conn != null) {
                conn.disconnect();
            }
        }
    }

    /**
     * 向输出流写入文本参数（multipart/form-data格式）
     * @param out 输出流
     * @param textParams 文本参数集合
     */
    private static void writeTextParams(OutputStream out, Map<String, String> textParams) throws IOException {
        if (textParams == null || textParams.isEmpty()) {
            log.debug("无文本参数需要写入");
            return;
        }

        StringBuilder textBuilder = new StringBuilder();
        for (Map.Entry<String, String> entry : textParams.entrySet()) {
            String key = entry.getKey();
            String value = entry.getValue() == null ? "" : entry.getValue();

            // 按multipart/form-data格式拼接文本参数
            textBuilder.append("\r\n")
                    .append("--").append(FORM_BOUNDARY).append("\r\n")
                    .append("Content-Disposition: form-data; name=\"").append(key).append("\"\r\n\r\n")
                    .append(value);
        }

        // 写入流并刷新
        byte[] textBytes = textBuilder.toString().getBytes(StandardCharsets.UTF_8);
        out.write(textBytes);
        log.debug("已写入文本参数，总长度：{}字节", textBytes.length);
    }

    /**
     * 向输出流写入文件参数（multipart/form-data格式）
     * @param out 输出流
     * @param fileParams 文件参数集合（key:参数名，value:文件路径）
     */
    private static void writeFileParams(OutputStream out, Map<String, String> fileParams) throws IOException {
        if (fileParams == null || fileParams.isEmpty()) {
            log.debug("无文件参数需要写入");
            return;
        }

        // MIME类型映射器（用于获取文件ContentType）
        MimetypesFileTypeMap mimeTypeMap = new MimetypesFileTypeMap();

        for (Map.Entry<String, String> entry : fileParams.entrySet()) {
            String fieldName = entry.getKey(); // 表单字段名（如：mbFile）
            String filePath = entry.getValue(); // 文件本地路径
            File file = new File(filePath);

            // 校验文件是否存在
            if (!file.exists() || !file.isFile()) {
                throw new FileNotFoundException("文件不存在或不是有效文件：" + filePath);
            }
            log.debug("准备上传文件：{}（大小：{}KB）", filePath, file.length() / 1024);

            String fileName = file.getName(); // 文件名（如：test.jpg）
            // 获取文件ContentType（优先按后缀匹配，否则使用默认值）
            String contentType = getFileContentType(fileName, mimeTypeMap.getContentType(file));

            // 拼接文件参数头部
            StringBuilder fileHeader = new StringBuilder();
            fileHeader.append("\r\n")
                    .append("--").append(FORM_BOUNDARY).append("\r\n")
                    .append("Content-Disposition: form-data; name=\"").append(fieldName)
                    .append("\"; filename=\"").append(fileName).append("\"\r\n")
                    .append("Content-Type: ").append(contentType).append("\r\n\r\n");
            out.write(fileHeader.toString().getBytes(StandardCharsets.UTF_8));

            // 写入文件内容（按字节流读取）
            try (InputStream fileIn = new FileInputStream(file)) {
                byte[] buffer = new byte[1024]; // 1KB缓冲区
                int len;
                while ((len = fileIn.read(buffer)) != -1) {
                    out.write(buffer, 0, len);
                }
            }
            log.debug("文件{}上传完成", filePath);
        }
    }

    /**
     * 获取文件的Content-Type（MIME类型）
     * 优先按文件后缀匹配，匹配不到则使用默认值
     * @param fileName 文件名（含后缀）
     * @param defaultType 默认MIME类型
     * @return 匹配到的Content-Type
     */
    private static String getFileContentType(String fileName, String defaultType) {
        // 常见图片类型
        if (fileName.endsWith(".png")) {
            return "image/png";
        } else if (fileName.endsWith(".jpg") || fileName.endsWith(".jpeg") || fileName.endsWith(".jpe")) {
            return "image/jpeg";
        } else if (fileName.endsWith(".gif")) {
            return "image/gif";
        } else if (fileName.endsWith(".bmp")) {
            return "image/bmp";
        }
        // 文档类型
        else if (fileName.endsWith(".pdf")) {
            return "application/pdf";
        } else if (fileName.endsWith(".txt")) {
            return "text/plain";
        } else if (fileName.endsWith(".doc") || fileName.endsWith(".docx")) {
            return "application/msword";
        } else if (fileName.endsWith(".xls") || fileName.endsWith(".xlsx")) {
            return "application/vnd.ms-excel";
        }
        // 其他类型使用默认值或通用二进制流
        return (defaultType != null && !defaultType.isEmpty()) ? defaultType : "application/octet-stream";
    }

    /**
     * 向输出流写入表单结束标记
     * @param out 输出流
     */
    private static void writeFormEndMarker(OutputStream out) throws IOException {
        byte[] endMarker = ("\r\n--" + FORM_BOUNDARY + "--\r\n").getBytes(StandardCharsets.UTF_8);
        out.write(endMarker);
        out.flush(); // 确保所有数据写入
        log.debug("已写入表单结束标记");
    }

    /**
     * 从HTTP连接读取响应内容
     * @param conn HTTP连接对象
     * @return 响应字符串（JSON格式）
     */
    private static String readResponse(HttpURLConnection conn) throws IOException {
        // 根据响应状态码选择输入流（成功用InputStream，失败用ErrorStream）
        InputStream in = (conn.getResponseCode() == HttpURLConnection.HTTP_OK)
                ? conn.getInputStream()
                : conn.getErrorStream();

        try (BufferedReader reader = new BufferedReader(
                new InputStreamReader(in, StandardCharsets.UTF_8))) {
            StringBuilder responseBuilder = new StringBuilder();
            String line;
            while ((line = reader.readLine()) != null) {
                responseBuilder.append(line);
            }
            String response = responseBuilder.toString();
            log.debug("响应内容长度：{}字节", response.length());
            return response;
        }
    }
}