package com.thebeastshop.mbgw.sdk.config;

import com.thebeastshop.mbgw.sdk.util.MbGwLogUtil;
import org.slf4j.Logger;

import java.io.IOException;
import java.io.InputStream;
import java.util.Properties;

public class MbGwSdkConfig {
    private static final Logger log = MbGwLogUtil.getLogger(MbGwSdkConfig.class);
    private static final String DEFAULT_BASE_URL = "https://gwapi.mabangerp.com/api/v2";
    private static final String CONFIG_FILE = "mbgw-sdk.properties";

    private static String apiKey;
    private static String apiToken;
    private static String baseUrl = DEFAULT_BASE_URL;

    static {
        loadFromProperties();
    }

    private static void loadFromProperties() {
        try (InputStream is = MbGwSdkConfig.class.getClassLoader().getResourceAsStream(CONFIG_FILE)) {
            if (is != null) {
                Properties props = new Properties();
                props.load(is);
                apiKey = props.getProperty("mbgw.apiKey");
                apiToken = props.getProperty("mbgw.apiToken");
                String url = props.getProperty("mbgw.baseUrl");
                if (url != null && !url.isEmpty()) {
                    baseUrl = url;
                }
                log.info("从配置文件[{}]加载SDK配置完成", CONFIG_FILE);
            } else {
                log.warn("未找到配置文件[{}]，将使用代码初始化或默认配置", CONFIG_FILE);
            }
        } catch (IOException e) {
            log.error("加载配置文件[{}]失败", CONFIG_FILE, e);
        }
    }

    public static void init(String apiKey, String apiToken) {
        if (apiKey == null || apiKey.isEmpty() || apiToken == null || apiToken.isEmpty()) {
            log.error("初始化失败：API_KEY和API_TOKEN不能为空");
            throw new IllegalArgumentException("API_KEY和API_TOKEN不能为空");
        }
        MbGwSdkConfig.apiKey = apiKey;
        MbGwSdkConfig.apiToken = apiToken;
        log.info("SDK初始化成功（API_KEY：{}）", maskApiKey(apiKey)); // 脱敏显示
    }

    public static void init(String apiKey, String apiToken, String baseUrl) {
        init(apiKey, apiToken);
        if (baseUrl != null && !baseUrl.isEmpty()) {
            MbGwSdkConfig.baseUrl = baseUrl;
            log.info("SDK基础URL已设置为：{}", baseUrl);
        }
    }

    // 工具方法：API_KEY脱敏（显示前4位+后4位）
    private static String maskApiKey(String apiKey) {
        if (apiKey.length() <= 8) return "******";
        return apiKey.substring(0, 4) + "******" + apiKey.substring(apiKey.length() - 4);
    }

    // Getter方法
    public static String getApiKey() {
        checkInit();
        return apiKey;
    }

    public static String getApiToken() {
        checkInit();
        return apiToken;
    }

    public static String getBaseUrl() {
        checkInit();
        return baseUrl;
    }

    private static void checkInit() {
        if (apiKey == null || apiToken == null) {
            log.error("SDK未初始化，请先调用MbGwSdkConfig.init()");
            throw new IllegalStateException("SDK未初始化，请先调用MbGwSdkConfig.init()");
        }
    }
}