package com.thebeastshop.mbgw.sdk.enums;

import java.util.Arrays;

/**
 * 订单可发送状态枚举
 * 1.正常订单 2.异常订单 3.全部订单（默认：正常订单）
 */
public enum CanSendStatusEnum {

    /**
     * 正常订单（默认值）
     */
    NORMAL(1, "正常订单"),

    /**
     * 异常订单
     */
    ABNORMAL(2, "异常订单"),

    /**
     * 全部订单
     */
    ALL(3, "全部订单");

    /**
     * 状态编码
     */
    private final int code;

    /**
     * 状态描述
     */
    private final String desc;

    CanSendStatusEnum(int code, String desc) {
        this.code = code;
        this.desc = desc;
    }

    // ---------------------- 常用方法 ----------------------

    /**
     * 获取默认状态（正常订单）
     */
    public static CanSendStatusEnum getDefault() {
        return NORMAL;
    }

    /**
     * 根据编码获取枚举
     * 若编码无效，返回默认值（正常订单）
     */
    public static CanSendStatusEnum getByCode(int code) {
        return Arrays.stream(values())
                .filter(status -> status.code == code)
                .findFirst()
                .orElse(getDefault()); // 无效编码默认返回正常订单
    }

    /**
     * 判断是否为正常订单
     */
    public boolean isNormal() {
        return this == NORMAL;
    }

    /**
     * 判断是否为异常订单
     */
    public boolean isAbnormal() {
        return this == ABNORMAL;
    }

    /**
     * 判断是否包含全部订单
     */
    public boolean isAll() {
        return this == ALL;
    }

    // ---------------------- getter ----------------------

    public int getCode() {
        return code;
    }

    public String getDesc() {
        return desc;
    }
}
