package com.thebeastshop.mbgw.sdk.enums;

import java.util.Arrays;
import java.util.Optional;

/**
 * 订单状态枚举
 * 包含：配货中、已发货、已完成、已作废、所有未发货、所有非未发货
 */
public enum OrderStatusEnum {

    /**
     * 配货中（未发货状态）
     */
    ALLOCATING(2, "配货中", true),

    /**
     * 已发货
     */
    SHIPPED(3, "已发货", false),

    /**
     * 已完成
     */
    COMPLETED(4, "已完成", false),

    /**
     * 已作废
     */
    CANCELED(5, "已作废", false),

    /**
     * 所有未发货（聚合状态，包含配货中等未发货状态）
     */
    ALL_UNSHIPPED(6, "所有未发货", true),

    /**
     * 所有非未发货（聚合状态，包含已发货、已完成、已作废等）
     */
    ALL_SHIPPED(7, "所有非未发货", false);

    /**
     * 状态编码
     */
    private final int code;

    /**
     * 状态描述
     */
    private final String desc;

    /**
     * 是否为未发货状态（用于快速判断）
     */
    private final boolean unshipped;

    OrderStatusEnum(int code, String desc, boolean unshipped) {
        this.code = code;
        this.desc = desc;
        this.unshipped = unshipped;
    }

    // ---------------------- 常用方法 ----------------------

    /**
     * 根据编码获取枚举
     */
    public static Optional<OrderStatusEnum> getByCode(int code) {
        return Arrays.stream(values())
                .filter(status -> status.code == code)
                .findFirst();
    }

    /**
     * 判断是否为未发货状态
     */
    public boolean isUnshipped() {
        return unshipped;
    }

    /**
     * 判断是否为已发货及之后的状态
     */
    public boolean isShippedOrLater() {
        return !unshipped;
    }

    // ---------------------- getter ----------------------

    public int getCode() {
        return code;
    }

    public String getDesc() {
        return desc;
    }
}
