package com.thebeastshop.mbgw.sdk.response;


import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.thebeastshop.mbgw.sdk.util.MbGwLogUtil;
import org.slf4j.Logger;

import java.io.Serializable;

/**
 * 马帮ERP API 响应模型
 * 统一封装响应结果，支持泛型解析业务数据
 * @param <T> 业务数据类型（如：JSONObject、自定义DTO）
 */
public class MbGwResponse<T> implements Serializable {
    private static final Logger log = MbGwLogUtil.getLogger(MbGwResponse.class);
    private static final long serialVersionUID = 1L;

    // 响应码（200表示成功，其他为失败，具体含义参考接口文档）
    protected int code;
    // 响应消息（成功/失败描述）
    protected String message;
    // 业务数据（接口返回的具体数据）
    protected T data;
    // 原始响应字符串（未解析的JSON，便于调试）
    protected String rawResponse;

    // 构造方法（保护级，仅子类可调用）
    protected MbGwResponse() {}
    /**
     * 从JSON字符串解析响应对象
     * @param json 原始响应JSON
     * @param dataType 业务数据类型（如：JSONObject.class）
     * @return 解析后的响应对象
     */
    public static <T> MbGwResponse<T> parse(String json, Class<T> dataType) {
        MbGwResponse<T> response = new MbGwResponse<>();
        response.rawResponse = json;

        try {
            // 解析JSON
            JSONObject jsonObject = JSON.parseObject(json);
            response.code = jsonObject.getIntValue("code");
            response.message = jsonObject.getString("message");

            // 解析业务数据（支持null）
            if (dataType != null) {
                response.data = jsonObject.getObject("data", dataType);
            } else {
                response.data = (T) jsonObject.get("data");
            }
        } catch (Exception e) {
            log.error("响应解析失败（原始响应：{}）", json, e);
            throw new RuntimeException("API响应格式错误：" + e.getMessage(), e);
        }
        return response;
    }

    /**
     * 解析JSON为指定的响应类型（核心方法）
     * @param json 原始JSON字符串
     * @param responseClass 响应类字节码（如OrderListResponse.class）
     * @param dataClass 响应数据类字节码（如OrderListResponse.Data.class）
     * @param <R> 响应类型（必须继承MbGwResponse）
     * @param <D> 响应数据类型
     * @return 解析后的响应对象
     */
    public static <R extends MbGwResponse<D>, D> R parse(String json, Class<R> responseClass, Class<D> dataClass) {
        try {
            // 实例化响应对象（如OrderListResponse）
            R response = responseClass.newInstance();
            response.rawResponse = json;

            // 解析JSON核心字段
            JSONObject jsonObj = JSON.parseObject(json);
            response.code = jsonObj.getIntValue("code");
            response.message = jsonObj.getString("message");

            // 解析data字段为具体类型（如OrderListResponse.Data）
            if (jsonObj.containsKey("data") && jsonObj.get("data") != null) {
                response.data = JSON.toJavaObject(jsonObj.getJSONObject("data"), dataClass);
            }

            return response;
        } catch (Exception e) {
            throw new RuntimeException("解析响应失败：" + e.getMessage(), e);
        }
    }

    // Getter方法（不提供Setter，确保响应数据不可篡改）
    public int getCode() {
        return code;
    }

    public String getMessage() {
        return message;
    }

    public T getData() {
        return data;
    }

    public String getRawResponse() {
        return rawResponse;
    }

    /**
     * 判断请求是否成功（默认code=200为成功，具体以接口文档为准）
     * @return 成功返回true，失败返回false
     */
    public boolean isSuccess() {
        return code == 200;
    }

    @Override
    public String toString() {
        return "MbGwResponse{" +
                "code=" + code +
                ", message='" + message + '\'' +
                ", data=" + (data != null ? data.toString() : "null") +
                '}';
    }
}