package com.thebeastshop.mbgw.sdk.request;

import java.util.Date;

/**
 * 马帮ERP 订单列表查询请求
 * 对应接口：order-get-list（假设接口名，具体以官方文档为准）
 * 封装订单查询的常见参数（分页、时间范围、状态等）
 */
public class OrderListRequest extends MbGwRequest {

    /**
     * 构造订单列表查询请求
     */
    public OrderListRequest() {
        super("order-get-order-list-new"); // 接口名，需与马帮ERP官方文档一致
    }

    /**
     * 设置分页页码（从1开始）
     * @param page 页码
     * @return 当前请求对象
     */
    public OrderListRequest setPage(int page) {
        if (page < 0) {
            throw new IllegalArgumentException("页码必须大于等于0");
        }
        addParam("cursor", page);
        return this;
    }

    /**
     * 设置每页条数（分页查询数；默认100，上限支持1000）
     * @param pageSize 每页数量
     * @return 当前请求对象
     */
    public OrderListRequest setPageSize(int pageSize) {
        if (pageSize < 1 || pageSize > 1000) {
            throw new IllegalArgumentException("每页条数必须在1-1000之间");
        }
        addParam("maxRows", pageSize);
        return this;
    }



    public OrderListRequest setUpdateTimeStart(Date startTime) {
        addParam("updateTimeStart", formatDate(startTime));
        return this;
    }


    public OrderListRequest setUpdateTimeEnd(Date endTime) {
        addParam("updateTimeEnd", formatDate(endTime));
        return this;
    }


    /**
     * 2.配货中3.已发货 4.已完成 5.已作废 6.所有未发货 7.所有非未发货; 不维护默认查询配货中的订单
     * @param status
     * @return
     */
    public OrderListRequest setOrderStatus(String status) {
        addParam("status ", status);
        return this;
    }

    /**
     * 1.正常订单 2.异常订单 3.全部订单 默认正常订单
     * @param canSend
     * @return
     */
    public OrderListRequest setCanSend(String canSend) {
        addParam("canSend ", canSend);
        return this;
    }

    /**
     * 设置订单来源筛选（如：亚马逊、淘宝、自定义等）
     * @param source 来源标识（参考官方文档）
     * @return 当前请求对象
     */
    public OrderListRequest setOrderSource(String source) {
        addParam("order_source", source);
        return this;
    }


    /**
     * 辅助方法：将Date转换为字符串（格式：yyyy-MM-dd HH:mm:ss）
     */
    private String formatDate(Date date) {
        if (date == null) {
            return null;
        }
        java.text.SimpleDateFormat sdf = new java.text.SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        return sdf.format(date);
    }
}
