package com.thebeastshop.common.utils;

import com.thebeastshop.common.exception.UtilException;
import com.thebeastshop.common.exception.UtilExceptionErrorCode;
import jxl.NumberCell;
import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.apache.poi.ss.usermodel.*;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Scope;
import org.springframework.context.annotation.ScopedProxyMode;
import org.springframework.core.convert.ConversionService;
import org.springframework.stereotype.Component;
import org.springframework.validation.BindException;
import org.springframework.validation.FieldError;
import org.springframework.validation.ObjectError;
import org.springframework.validation.Validator;

import java.io.*;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.math.BigDecimal;
import java.text.DecimalFormat;
import java.text.NumberFormat;
import java.text.SimpleDateFormat;
import java.util.*;

@Component("poiParseExcelComponentNew")
@Scope(value = "request", proxyMode = ScopedProxyMode.TARGET_CLASS)
public class PoiParseExcelComponentNew<T> {


	private final static Logger LOGGER = LoggerFactory.getLogger(PoiParseExcelComponentNew.class);
	private final static String excel2003L =".xls";    //2003- 版本的excel
	private final static String excel2007U =".xlsx";   //2007+ 版本的excel

	private Class<T> clazz;

	private InputStream is;

	private int headerRowIndex = 0;

	private int firstRowIndex = 1;

	private String fileName;

	private List<ObjectError> errorMessages = new ArrayList<ObjectError>();

	private List<String> warnMessages = new ArrayList<String>();;


	private Set<String> warnMessagesSet = new HashSet<>();

	private Set<String> headerColumn = new HashSet<String>();

	private Map<Integer, Field> columnMap = new HashMap<Integer, Field>();

	private Map<String, String> fieldNameMap = new HashMap<String, String>();

	private List<T> availableResults = new ArrayList<T>();

	private List<T> allResults = new ArrayList<T>();

	private List<T> failureResults = new ArrayList<T>();

	private Method setErrorMsgMethod;

	private Method getErrorMsgMethod;

	private Method setColumnNumberMethod;
	@Autowired
	private Validator beanVidator;

	@Autowired
	private ConversionService myConversionService;

	public void parse(InputStream is, Class<T> clazz, String fileName, Integer headerRowIndex, Integer firstRowIndex) {
		this.is = is;
		this.clazz = clazz;
		this.fileName = fileName;
		this.headerRowIndex = headerRowIndex;
		this.firstRowIndex = firstRowIndex;

		validateExcel();
	}

	public void parse(File file, Class<T> clazz) {
		if (file == null){
			throw new IllegalArgumentException("the file is null!");
		}
		try {
			this.is = new FileInputStream(file);
			this.clazz = clazz;
			validateExcel();
		} catch (FileNotFoundException e) {
			e.printStackTrace();
			throw new UtilException(UtilExceptionErrorCode.EXCEL_PARSE_ERROR, "EXCEL解析错误，请检查格式是否规范！");
		}
	}

	public void parse(String fileName, Class<T> clazz) {
		if (fileName == null || fileName.trim().length() < 1){
			throw new IllegalArgumentException("the file path is empty");
		}
		File file = new File(fileName);
		try {
			this.is = new FileInputStream(file);
			this.clazz = clazz;
			validateExcel();
		} catch (FileNotFoundException e) {
			e.printStackTrace();
			throw new UtilException(UtilExceptionErrorCode.EXCEL_PARSE_ERROR, "EXCEL解析错误，请检查格式是否规范！");
		}
	}

	public List<T> getAvailableResults() {
		return availableResults;
	}

	public List<ObjectError> getErrorMessages() {
		return errorMessages;
	}
	public String getErrorInfo(){
		StringBuffer stringBuffer = new StringBuffer();
		if(errorMessages != null && errorMessages.size() > 0){
			for(ObjectError error :errorMessages){
				stringBuffer.append(error.getDefaultMessage().replace("must not be null", "不能为空"));
				stringBuffer.append("</br>");
			}
		}
		return stringBuffer.toString();
	}
	public List<T> getFailureResults() {
		return failureResults;
	}

	public List<T> getAllResults() {
		return allResults;
	}

	public boolean hasErrors() {
		return errorMessages.size() > 0 ? true : false;
	}

	public List<String> getWarnMessages() {
		return warnMessages;
	}

	public boolean hasWarns() {
		return warnMessages.size() > 0 ? true : false;
	}

	private void validateExcel() {
		ExcelTemplate et = clazz.getAnnotation(ExcelTemplate.class);
		if (et == null){
			throw new RuntimeException("the class is not a excel template data");
		}
		try {
			Workbook wb = getWorkbook(is, fileName);
			Sheet sheet = wb.getSheetAt(0);
			if (sheet.getPhysicalNumberOfRows() <= 1) {
				errorMessages.add(new ObjectError(clazz.getName(), "Excel文件格式不正确，请严格按照给定的格式生成excel文件!"));
				return;
			}
			Row firstRow = sheet.getRow(headerRowIndex);
			if (sheet.getLastRowNum() < 1) {
				errorMessages.add(new ObjectError(clazz.getName(), "Excel文件没有第一行，请严格按照给定的格式生成excel文件!"));
				return;
			}
			checkExcelHeader(firstRow);
			if (errorMessages.size() > 0){
				return;
			}
			if (headerColumn.size() < 1) {
				errorMessages.add(new ObjectError(clazz.getName(), "Excel文件第一行的列名与模板完全不符,请严格按照给定的格式生成excel文件!"));
				return;
			}
			// 初始化注释ErrorMessage字段的set和get方法和columnNumber字段的set方法
			initErrorMessageAndColumnNumberMehtod();
			// 把excel转换为javabean对象，然后再对每个属性进行验证
			convertToJavaBean(sheet);
			// 验证生成的javabean
			int rowTip = firstRowIndex;
			for (T t : allResults) {
				validate(t, rowTip);
				rowTip++;
			}
			// 检查注释ExcelColumn中unique为true的field
			checkUniqueConstrain();

		} catch (IOException e) {
			e.printStackTrace();
			throw new UtilException(UtilExceptionErrorCode.EXCEL_PARSE_ERROR, "EXCEL解析错误，请检查格式是否规范！");
		} catch (SecurityException e) {
			e.printStackTrace();
			throw new UtilException(UtilExceptionErrorCode.EXCEL_PARSE_ERROR, "EXCEL解析错误，请检查格式是否规范！");
		}catch (Exception e) {
			e.printStackTrace();
			throw new UtilException(UtilExceptionErrorCode.EXCEL_PARSE_ERROR, "EXCEL解析错误，请检查格式是否规范！");
		}
	}

	private void checkExcelHeader(Row firstRow) {
		Class<? super T> temp = clazz;
		List<Field> fieldsList = new ArrayList<Field>();
		while (temp != null && !temp.equals(Object.class) && temp.getAnnotation(ExcelTemplate.class) != null) {
			Field[] fs = temp.getDeclaredFields();
			for (int x = 0; x < fs.length; x++) {
				ExcelColumn ec = fs[x].getAnnotation(ExcelColumn.class);
				if (ec == null){
					continue;
				}
				fieldsList.add(fs[x]);
			}
			temp = temp.getSuperclass();
		}
		int firstCellNum = firstRow.getFirstCellNum();
		int lastCellNum = firstRow.getLastCellNum();

		for (int i = firstCellNum; i < lastCellNum; i++) {
			String content = firstRow.getCell(i).getStringCellValue();
			if (content == null || content.trim().length() < 1){
				continue;
			}
			content = content.trim();
			for (int j = 0; j < fieldsList.size(); j++) {
				ExcelColumn ec = fieldsList.get(j).getAnnotation(ExcelColumn.class);
				if (ec == null){
					continue;
				}
				String columnName = ec.name().equals("") ? fieldsList.get(j).getName() : ec.name();
				fieldNameMap.put(fieldsList.get(j).getName(), ec.name());
				if (content.equals(columnName)) {
					if (!headerColumn.add(columnName)){
						errorMessages.add(new ObjectError(clazz.getName(), "在模板中发现有重复的列名" + columnName));
					} else{
						columnMap.put(i, fieldsList.get(j));
					}
				}
			}
		}
	}

	private void convertToJavaBean(Sheet sheet) {
		try {
			int rowNum = sheet.getLastRowNum();
			for (int i = firstRowIndex; i <= rowNum; i++) {
				T obj = clazz.newInstance();
				Row row = sheet.getRow(i);
				//空行继续
				if (row == null) {
					continue;
					/*throw new UtilException(UtilExceptionErrorCode.EXCEL_PARSE_ERROR, "EXCEL解析错误，第【" + i + "】" );*/
				}
				int nullCount = 0;
				for (Integer j : columnMap.keySet()) {
					Field field = columnMap.get(j);
					if (field == null) {
						nullCount++;
						continue;
					}
					// 可能头的列数会大于下面内容的列数，所以做如下处理
					if (j < row.getLastCellNum()) {
						String chineseName = fieldNameMap.get(field.getName()).length() < 1 ? field.getName() : fieldNameMap.get(field.getName());
						Cell cell = row.getCell(j);
						if (cell == null) {
							nullCount++;
							continue;
						}
						if(Cell.CELL_TYPE_FORMULA == cell.getCellType()){
							warnMessages.add("[第" + (i+1) + "行] " + chineseName + "存在公式，无法上传");
							warnMessagesSet.add(chineseName  + (i+1));
							errorMessages.add(new
									FieldError(clazz.getName(),field.getName(),"[第"+(i+1)+"]行 "+chineseName+"存在公式，无法上传"));
							continue;
						}

						Object content = getCellValue(cell, field.getType().getName());
						if (content != null) {
							Method method = clazz.getMethod(generateSetMethod(field.getName()), field.getType());
							// 如果为string类型则不做转型
							if (!String.class.getName().equals(field.getType().getName())) {
								try {
									if (Date.class.getName().equals(field.getType().getName())) {
										TimeZone zone = TimeZone.getTimeZone("GMT");
										SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
										sdf.setTimeZone(zone);
										if (EmptyUtil.isNotEmpty(content)) {
											if (content instanceof Date) {
												method.invoke(obj, myConversionService.convert(content, field.getType()));
											} else {
												Date date = DateUtil.parseDate(content.toString(), DateUtil.DEFAULT_DATETIME_FORMAT);
												//date 日期格式化（yyyy-MM-dd）
												if (date == null) {
													date = DateUtil.parseDate(content.toString(), DateUtil.DEFAULT_DATE_FORMAT);
												}
												//date 日期格式化（yyyy/MM/dd）
												if (date == null) {
													date = DateUtil.parse(content.toString(), "yyyy/MM/dd");
												}
												method.invoke(obj, myConversionService.convert(date, field.getType()));
											}
										}

									} else {
										method.invoke(obj, myConversionService.convert(content, field.getType()));
									}
								} catch (Exception e) {
									LOGGER.error("convertToJavaBean-error", e);
									warnMessages.add("[第" + (i+1) + "行] " + chineseName + "格式不正确");
									warnMessagesSet.add(chineseName  + (i+1));
									errorMessages.add(new
											FieldError(clazz.getName(),field.getName(),"[第"+(i+1)+"行] "+chineseName+"格式不正确"));
								}
							} else {
									method.invoke(obj, content);
								// System.out.println("~~~~~~~~~~~"+method.getName());
								// Method
								// method=ReflectionUtils.findMethod(clazz,
								// field.getName());
							}
						} else {
							nullCount++;
						}
					} else {
						nullCount++;
					}
				}
				if (columnMap.size() > nullCount) {
					setExcelColumnNumber(obj, i + 1);
					allResults.add(obj);
				}
			}
		} catch (InstantiationException e) {
			e.printStackTrace();
			throw new UtilException(UtilExceptionErrorCode.EXCEL_PARSE_ERROR, "EXCEL解析错误，请检查格式是否规范！");
		} catch (IllegalAccessException e) {
			e.printStackTrace();
			throw new UtilException(UtilExceptionErrorCode.EXCEL_PARSE_ERROR, "EXCEL解析错误，请检查格式是否规范！");
		} catch (SecurityException e) {
			e.printStackTrace();
			throw new UtilException(UtilExceptionErrorCode.EXCEL_PARSE_ERROR, "EXCEL解析错误，请检查格式是否规范！");
		} catch (NoSuchMethodException e) {
			e.printStackTrace();
			throw new UtilException(UtilExceptionErrorCode.EXCEL_PARSE_ERROR, "EXCEL解析错误，请检查格式是否规范！");
		} catch (IllegalArgumentException e) {
			e.printStackTrace();
			throw new UtilException(UtilExceptionErrorCode.EXCEL_PARSE_ERROR, "EXCEL解析错误，请检查格式是否规范！");
		} catch (InvocationTargetException e) {
			e.printStackTrace();
			throw new UtilException(UtilExceptionErrorCode.EXCEL_PARSE_ERROR, "EXCEL解析错误，请检查格式是否规范！");
		} catch (Exception e) {
			e.printStackTrace();
			throw new UtilException(UtilExceptionErrorCode.EXCEL_PARSE_ERROR, "EXCEL解析错误，请检查格式是否规范！");
		}
	}

	// 验证通过添加到availableResults否则添加到failureResults
	@SuppressWarnings("unchecked")
	private void validate(T obj, int row) {
		BindException errors = new BindException(obj, clazz.getSimpleName());
		beanVidator.validate(obj, errors);
		if (errors.hasErrors()) {
			for (FieldError error : (List<FieldError>) errors.getFieldErrors()) {
				if (fieldNameMap.get(error.getField()) == null){
					continue;
				}
				String chineseName = fieldNameMap.get(error.getField()).length() < 1 ? error.getField() : fieldNameMap.get(error.getField());
				//已经校验格式有问题的单元格，无需再check是否为空
				if(!warnMessagesSet.contains(chineseName + row)) {
					errorMessages.add(new FieldError(error.getObjectName(), error.getField(), "[第" + (row) + "行] " + chineseName
																																										+ error.getDefaultMessage()));
					setErrorMessage(obj, chineseName + error.getDefaultMessage());
				}

			}
			failureResults.add(obj);
		} else {
			availableResults.add(obj);
		}
	}

	private void checkUniqueConstrain() {
		// 得到unique为true的field
		Set<Field> fields = new HashSet<Field>();
		for (int i : columnMap.keySet()) {
			Field field = columnMap.get(i);
			ExcelColumn ec = field.getAnnotation(ExcelColumn.class);
			if (ec == null){
				continue;
			}
			if (ec.unique() == true){
				fields.add(field);
			}
		}
		if (fields.size() < 1){
			return;
		}
		try {
			List<T> needToRemove = new ArrayList<T>();
			for (Field f : fields) {
				Method method = clazz.getMethod(generateGetMethod(f.getName()));
				for (int i = 0; i < availableResults.size(); i++) {
					T t = availableResults.get(i);
					Object obj = method.invoke(t);
					int rowDis = 2;
					boolean hasIt=false;
					String chineseName = fieldNameMap.get(f.getName()).length() < 1 ? f.getName() : fieldNameMap.get(f.getName());
					for (int j = 0; j < availableResults.size(); j++) {
						T sub = availableResults.get(j);
						if (j != i && !needToRemove.contains(sub)) {
							Object sobj = method.invoke(sub);
							if (obj != null && obj.equals(sobj)) {
								errorMessages.add(new FieldError(clazz.getName(), "multiple_" + f.getName(), "[第" + (i + rowDis) + "行和第"
										+ (j + rowDis) + "行] " + chineseName + "数据重复!"));
								setErrorMessage(sub, chineseName + "数据重复!");
								needToRemove.add(sub);
								hasIt=true;
							}
						}
					}
					if(hasIt){
						needToRemove.add(t);
						setErrorMessage(t, chineseName + "数据重复!");
					}
				}
				if (needToRemove.size() > 0) {
					availableResults.removeAll(needToRemove);
					failureResults.addAll(needToRemove);
				}
				needToRemove.clear();
			}
		} catch (SecurityException e) {
			e.printStackTrace();
		} catch (NoSuchMethodException e) {
			e.printStackTrace();
		} catch (IllegalArgumentException e) {
			e.printStackTrace();
		} catch (IllegalAccessException e) {
			e.printStackTrace();
		} catch (InvocationTargetException e) {
			e.printStackTrace();
		}
	}

	private void initErrorMessageAndColumnNumberMehtod() {
		Field[] fields = clazz.getDeclaredFields();
		try {
			for (int i = 0; i < fields.length; i++) {
				ErrorMessage em = fields[i].getAnnotation(ErrorMessage.class);
				ExcelColumnNumber ecn = fields[i].getAnnotation(ExcelColumnNumber.class);
				if (em != null && setErrorMsgMethod == null) {
					setErrorMsgMethod = clazz.getMethod(generateSetMethod(fields[i].getName()), fields[i].getType());
					getErrorMsgMethod = clazz.getMethod(generateGetMethod(fields[i].getName()));
				}
				if (ecn != null && setColumnNumberMethod == null) {
					setColumnNumberMethod = clazz.getMethod(generateSetMethod(fields[i].getName()), fields[i].getType());
				}
			}
		} catch (SecurityException e) {
			e.printStackTrace();
		} catch (NoSuchMethodException e) {
			e.printStackTrace();
		}
	}

	private void setExcelColumnNumber(T obj, Integer number) {
		if (setColumnNumberMethod != null) {
			try {
				setColumnNumberMethod.invoke(obj, number);
			} catch (IllegalArgumentException e) {
				e.printStackTrace();
			} catch (IllegalAccessException e) {
				e.printStackTrace();
			} catch (InvocationTargetException e) {
				e.printStackTrace();
			}
		}
	}

	private void setErrorMessage(T obj, String message) {
		if (setErrorMsgMethod != null){
			try {
				String em = getErrorMessage(obj);
				if (em != null && em.trim().length() > 0){
					setErrorMsgMethod.invoke(obj, em + ";" + message);
				} else{
					setErrorMsgMethod.invoke(obj, message);
				}
			} catch (IllegalArgumentException e) {
				e.printStackTrace();
			} catch (IllegalAccessException e) {
				e.printStackTrace();
			} catch (InvocationTargetException e) {
				e.printStackTrace();
			}
		}

	}

	private String getErrorMessage(T obj) {
		if (getErrorMsgMethod != null){
			try {
				return (String) getErrorMsgMethod.invoke(obj);
			} catch (IllegalArgumentException e) {
				e.printStackTrace();
			} catch (IllegalAccessException e) {
				e.printStackTrace();
			} catch (InvocationTargetException e) {
				e.printStackTrace();
			}
		}
		return null;
	}

	private String generateSetMethod(String fieldName) {
		if (fieldName == null || fieldName.length() < 1){
			return "";
		}
		return "set" + fieldName.toUpperCase().charAt(0) + fieldName.substring(1);
	}

	private String generateGetMethod(String fieldName) {
		if (fieldName == null || fieldName.length() < 1){
			return "";
		}
		return "get" + fieldName.toUpperCase().charAt(0) + fieldName.substring(1);
	}

	/**
	 * 描述：根据文件后缀，自适应上传文件的版本
	 * @param inStr,fileName
	 * @return
	 * @throws Exception
	 */
	public Workbook getWorkbook(InputStream inStr, String fileName) throws Exception{
		Workbook wb = null;
		String fileType = fileName.substring(fileName.lastIndexOf("."));
		if(excel2003L.equals(fileType)){
			wb = new HSSFWorkbook(inStr);  //2003-
		}else if(excel2007U.equals(fileType)){
			wb = new XSSFWorkbook(inStr);  //2007+
		}else{
			throw new Exception("解析的文件格式有误！");
		}
		return wb;
	}

	/**
	 * 描述：对表格中数值进行格式化
	 * @param cell
	 * @return
	 */
	public  Object getCellValue(Cell cell, String fieldType){
		Object value = null;
		DecimalFormat df = new DecimalFormat("0");  //格式化number String字符
		NumberFormat nf = NumberFormat.getInstance();//格式化数字
		switch (cell.getCellType()) {
			case Cell.CELL_TYPE_STRING:
				value = cell.getRichStringCellValue().getString();
				break;
			case Cell.CELL_TYPE_NUMERIC:
				if (XSSFDateUtil.isCellDateFormatted(cell)) {
					TimeZone zone = TimeZone.getTimeZone("GMT");
					SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
					sdf.setTimeZone(zone);
					System.out.print("YYYYYYYYYYYYYYYY日期" + cell.getNumericCellValue());
					//  如果是date类型则 ，获取该cell的date值
					value = cell.getDateCellValue();
					System.out.println("YYYYYYYYYYYYYYYY日期" + value);
				} else{ // 纯数字
					if ("java.math.BigDecimal".equals(fieldType)) {
            			df = new DecimalFormat("#########.####");
						value = df.format(cell.getNumericCellValue());
					} else {
						value = df.format(cell.getNumericCellValue());
					}
				}
				break;
			case Cell.CELL_TYPE_FORMULA:
				Workbook wb = cell.getSheet().getWorkbook();
				CreationHelper crateHelper = wb.getCreationHelper();
				FormulaEvaluator evaluator = crateHelper.createFormulaEvaluator();
				value = getCellValue(evaluator.evaluateInCell(cell),fieldType);
				break;
			case Cell.CELL_TYPE_BOOLEAN:
				value = cell.getBooleanCellValue();
				break;
			case Cell.CELL_TYPE_BLANK:
				value = "";
				break;
			default:
				break;
		}
		return value;
	}


	public static void main(String[] args) {

		/*
		 * File file=new File("e:/0510.xls"); ApplicationContext context = new
		 * ClassPathXmlApplicationContext( new
		 * String[]{"classpath*:META-INF/spring/applicationContext*.xml"});
		 * Validator beanValidator=(Validator)context.getBean("beanValidator");
		 * ConversionService
		 * conversionService=(ConversionService)context.getBean
		 * ("conversionService"); ParseExcelComponent<MemberImportData> pe=new
		 * ParseExcelComponent<MemberImportData>(file,MemberImportData.class);
		 * pe.setBeanVidator(beanValidator);
		 * pe.setConversionService(conversionService); pe.validateExcel();
		 *
		 * for(String e:pe.getErrorMessages()){ System.out.println(e); }
		 */
	}

	// //only for test
	// public void setConversionService(ConversionService conversionService) {
	// this.conversionService = conversionService;
	// }
	//
	// //only for test
	// public void setBeanVidator(Validator beanVidator) {
	// this.beanVidator = beanVidator;
	// }
}
