/**
 * Copyright (C) Alibaba Cloud Computing, 2012
 * All rights reserved.
 * 
 * 版权所有 （C）阿里巴巴云计算，2012
 */

package com.aliyun.oss;

import java.io.File;
import java.io.InputStream;
import java.net.URL;
import java.util.Date;
import java.util.List;

import com.aliyun.oss.common.comm.ResponseMessage;
import com.aliyun.oss.model.AbortMultipartUploadRequest;
import com.aliyun.oss.model.AccessControlList;
import com.aliyun.oss.model.Bucket;
import com.aliyun.oss.model.BucketLoggingResult;
import com.aliyun.oss.model.BucketWebsiteResult;
import com.aliyun.oss.model.CannedAccessControlList;
import com.aliyun.oss.model.CompleteMultipartUploadRequest;
import com.aliyun.oss.model.CompleteMultipartUploadResult;
import com.aliyun.oss.model.CopyObjectRequest;
import com.aliyun.oss.model.CopyObjectResult;
import com.aliyun.oss.model.CreateBucketRequest;
import com.aliyun.oss.model.GeneratePresignedUrlRequest;
import com.aliyun.oss.model.GetObjectRequest;
import com.aliyun.oss.model.InitiateMultipartUploadRequest;
import com.aliyun.oss.model.InitiateMultipartUploadResult;
import com.aliyun.oss.model.ListMultipartUploadsRequest;
import com.aliyun.oss.model.ListObjectsRequest;
import com.aliyun.oss.model.ListPartsRequest;
import com.aliyun.oss.model.MultipartUploadListing;
import com.aliyun.oss.model.OSSObject;
import com.aliyun.oss.model.ObjectListing;
import com.aliyun.oss.model.ObjectMetadata;
import com.aliyun.oss.model.OptionsRequest;
import com.aliyun.oss.model.PartListing;
import com.aliyun.oss.model.PutObjectResult;
import com.aliyun.oss.model.SetBucketCORSRequest;
import com.aliyun.oss.model.SetBucketCORSRequest.CORSRule;
import com.aliyun.oss.model.SetBucketLoggingRequest;
import com.aliyun.oss.model.SetBucketWebsiteRequest;
import com.aliyun.oss.model.UploadPartCopyRequest;
import com.aliyun.oss.model.UploadPartCopyResult;
import com.aliyun.oss.model.UploadPartRequest;
import com.aliyun.oss.model.UploadPartResult;

/**
 * 阿里云开放存储服务（Open Storage Service， OSS）的访问接口。
 * <p>
 * 阿里云存储服务（Open Storage Service，简称OSS），是阿里云对外提供的海量，安全，低成本，
 * 高可靠的云存储服务。用户可以通过简单的REST接口，在任何时间、任何地点上传和下载数据，
 * 也可以使用WEB页面对数据进行管理。<br />
 * 基于OSS，用户可以搭建出各种多媒体分享网站、网盘、个人企业数据备份等基于大规模数据的服务。
 * </p>
 */
public interface OSS {

    /**
     * 创建{@link Bucket}。
     * @param bucketName
     *          Bucket名称。
     * */
    public Bucket createBucket(String bucketName) throws OSSException,
            ClientException;
    
    /**
     * 创建{@link Bucket}。
     * @param createBucketRequest
     *          请求参数{@link CreateBucketRequest}。
     */
    public Bucket createBucket(CreateBucketRequest createBucketRequest) throws OSSException,
    ClientException;

    /**
     * 删除{@link Bucket}。
     * @param bucketName
     *          Bucket名称。
     * */
    public void deleteBucket(String bucketName) throws OSSException,
            ClientException;

    /**
     * 返回请求者拥有的所有{@link Bucket}的列表。
     * @return
     *      请求者拥有的所有{@link Bucket}的列表。
     */
    public List<Bucket> listBuckets() throws OSSException, ClientException;

    /**
     * 设置指定{@link Bucket}的Access Control List(ACL)。
     * @param bucketName
     *          Bucket名称。
     * @param acl
     *          {@link CannedAccessControlList}中列出的ACL。
     *          如果传入null，则表示使用默认值{@link CannedAccessControlList#Private}。
     * */
    public void setBucketAcl(String bucketName, CannedAccessControlList acl)
            throws OSSException, ClientException;

    /**
     * 返回给定{@link Bucket}的Access Control List(ACL)。
     * @param bucketName
     *          Bucket名称。
     * @return Access Control List(ACL) {@link AccessControlList}。
     * */
    public AccessControlList getBucketAcl(String bucketName)
            throws OSSException, ClientException;
    
    /**
     * 返回给定{@link Bucket}所在的数据中心。
     * @param bucketName
     *          Bucket名称。
     * @return Bucket所在的数据中心。
     * */ 
    public String getBucketLocation(String bucketName)
            throws OSSException, ClientException;

    /**
     * 判断给定{@link Bucket}是否存在。
     * @param bucketName
     *          Bucket名称。
     * */
    public boolean doesBucketExist(String bucketName) throws OSSException,
            ClientException;

    /**
     * 列出指定{@link Bucket}下的{@link OSSObject}。
     * @param bucketName
     *          Bucket名称。
     * @return Object列表{@link ObjectListing}
     **/
    public ObjectListing listObjects(String bucketName) throws OSSException,
            ClientException;

    /**
     * 列出指定{@link Bucket}下key以给定prefix开头的{@link OSSObject}。
     * @param bucketName
     *          Bucket名称。
     * @param prefix
     *          限定返回的Object key必须以prefix作为前缀。
     * @return Object列表{@link ObjectListing}
     * @throws OSSException
     * @throws ClientException
     */
    public ObjectListing listObjects(String bucketName, String prefix)
            throws OSSException, ClientException;

    /**
     * 列出指定{@link Bucket}下的{@link OSSObject}。
     * @param listObjectsRequest
     *          请求信息。
     * @return object列表{@link ObjectListing}
     * @throws OSSException
     * @throws ClientException
     */
    public ObjectListing listObjects(ListObjectsRequest listObjectsRequest)
            throws OSSException, ClientException;

    /**
     * 上传指定的{@link OSSObject}到OSS中指定的{@link Bucket}。
     * @param bucketName
     *          Bucket名称。
     * @param key
     *          object的key。
     * @param input
     *          输入流。
     * @param metadata
     *          object的元信息{@link ObjectMetadata}。
     * */
    public PutObjectResult putObject(String bucketName, String key,
            InputStream input, ObjectMetadata metadata) throws OSSException,
            ClientException;

    /**
     * 拷贝一个在OSS上已经存在的Object成另外一个Object。
     * @param sourceBucketName
     *          源Object所在的Bucket的名称。
     * @param sourceKey
     *          源Object的Key。
     * @param destinationBucketName
     *          目标Object所在的Bucket的名称。
     * @param destinationKey
     *          目标Object的Key。
     * @return 请求结果{@link CopyObjectResult}实例。
     * @throws OSSException
     * @throws ClientException
     */
    public CopyObjectResult copyObject(String sourceBucketName,
            String sourceKey, String destinationBucketName,
            String destinationKey) throws OSSException, ClientException;

    /**
     * 拷贝一个在OSS上已经存在的Object成另外一个Object。
     * @param copyObjectRequest
     *          请求参数{@link CopyObjectRequest}实例。
     * @return
     * @throws OSSException
     * @throws ClientException
     */
    public CopyObjectResult copyObject(CopyObjectRequest copyObjectRequest)
            throws OSSException, ClientException;

    /**
     * 从OSS指定的{@link Bucket}中导出{@link OSSObject}。
     * @param bucketName
     *          Bucket名称。
     * @param key
     *          Object Key。
     * @return 请求结果{@link OSSObject}实例。使用完之后需要手动关闭其中的ObjectContent释放请求连接。
     * */
    public OSSObject getObject(String bucketName, String key)
            throws OSSException, ClientException;

    /**
     * 从OSS指定的{@link Bucket}中导出指定的{@link OSSObject}到目标文件。
     * @param getObjectRequest
     *          请求参数{@link GetObjectRequest}。
     * @param file
     *          目标文件。
     * */
    public ObjectMetadata getObject(GetObjectRequest getObjectRequest, File file)
            throws OSSException, ClientException;

    /**
     * 从OSS指定的{@link Bucket}中导出{@link OSSObject}。
     * @param getObjectRequest
     *          请求参数{@link GetObjectRequest}。
     * @return 请求结果{@link OSSObject}实例。使用完之后需要手动关闭其中的ObjectContent释放请求连接。
     * */
    public OSSObject getObject(GetObjectRequest getObjectRequest)
            throws OSSException, ClientException;

    /**
     * 返回{@link OSSObject}的元数据。
     * @param bucketName
     *          Bucket名称。
     * @param key
     *          Object key。
     * */
    public ObjectMetadata getObjectMetadata(String bucketName, String key)
            throws OSSException, ClientException;

    /**
     * 删除指定的{@link OSSObject}。
     * @param bucketName
     *          Bucket名称。
     * @param key
     *          Object key。
     * */
    public void deleteObject(String bucketName, String key)
            throws OSSException, ClientException;

    /**
     * 生成一个用HTTP GET方法访问{@link OSSObject}的URL。
     * @param bucketName
     *          Bucket名称。
     * @param key
     *          Object key。
     * @param expiration
     *          URL的超时时间。
     * @return
     *          访问{@link OSSObject}的URL。
     * @throws ClientException 
     */
    public URL generatePresignedUrl(String bucketName, String key,
            Date expiration) throws ClientException;

    /**
     * 生成一个用指定HTTP方法访问{@link OSSObject}的URL。
     * @param bucketName
     *          Bucket名称。
     * @param key
     *          Object Key。
     * @param expiration
     *          URL的超时时间。
     * @param method
     *          HTTP方法，只支持{@link HttpMethod#GET}和{@link HttpMethod#PUT}。
     * @return
     *          访问{@link OSSObject}的URL。
     * @throws ClientException 
     */
    public URL generatePresignedUrl(String bucketName, String key,
            Date expiration, HttpMethod method) throws ClientException;

    /**
     * 生成一个包含签名信息并可以访问{@link OSSObject}的URL。
     * @param request
     *          {@link GeneratePresignedUrlRequest}对象。
     * @return 包含签名信息并可以访问{@link OSSObject}的URL。
     * @throws ClientException
     */
    public URL generatePresignedUrl(GeneratePresignedUrlRequest request)
            throws ClientException;

    /**
     * 终止一个Multipart上传事件。
     * @param request
     *          {@link AbortMultipartUploadRequest}对象。
     * @throws ClientException
     */
    public void abortMultipartUpload(AbortMultipartUploadRequest request)
            throws OSSException, ClientException;

    /**
     * 完成一个Multipart上传事件。
     * <p>
     * 在将所有数据Part 都上传完成后，可以调用 Complete Multipart Upload API
     * 来完成整个文件的 Multipart Upload。在执行该操作时，用户必须提供所有有效
     * 的数据Part的列表（包括part号码和ETAG）； OSS收到用户提交的Part列表后，
     * 会逐一验证每个数据 Part 的有效性。当所有的数据 Part 验证通过后，OSS 将把
     * 这些数据part组合成一个完整的 Object。 
     * </p>
     * 
     * <p>
     * 此方法对应的操作为非幂等操作，SDK不会对其进行重试（即使设置最大重试次数大于0也不会重试）
     * </p>
     * 
     * @param request
     *          {@link CompleteMultipartUploadRequest}对象。
     * @return  CompleteMultipartUploadResult    
     * @throws ClientException
     */
    public CompleteMultipartUploadResult completeMultipartUpload(
            CompleteMultipartUploadRequest request) throws OSSException,
            ClientException;

    /**
     * 初始化一个Multipart上传事件。
     * <p>
     * 使用Multipart模式上传数据前，必须先调用该接口来通过OSS初始化一个Multipart上传事件。
     * 该接口会返回一个OSS服务器创建的全局唯一的Upload ID，用于标识本次Multipart上传事件。
     * 用户可以根据这个ID来发起相关的操作，如中止、查询Multipart上传等。
     * </p>
     * 
     * <p>
     * 此方法对应的操作为非幂等操作，SDK不会对其进行重试（即使设置最大重试次数大于0也不会重试）
     * </p>
     * @param request
     *          {@link InitiateMultipartUploadRequest}对象。
     * @return  InitiateMultipartUploadResult    
     * @throws ClientException
     */
    public InitiateMultipartUploadResult initiateMultipartUpload(
            InitiateMultipartUploadRequest request) throws OSSException,
            ClientException;

    /**
     * 列出所有执行中的 Multipart上传事件。
     * <p>
     * 即已经被初始化的 Multipart Upload 但是未被完成或被终止的 Multipart上传事件。 
     * OSS返回的罗列结果中最多会包含1000个Multipart上传事件。
     * </p>
     * @param request
     *          {@link ListMultipartUploadsRequest}对象。
     * @return  MultipartUploadListing
     *          Multipart上传事件的列表{@link MultipartUploadListing}。
     * @throws ClientException
     */
    public MultipartUploadListing listMultipartUploads(
            ListMultipartUploadsRequest request) throws OSSException,
            ClientException;

    /**
     * 列出multipart中上传的所有part信息
     * @param request
     *          {@link ListPartsRequest}对象。
     * @return  PartListing    
     * @throws ClientException
     */
    public PartListing listParts(ListPartsRequest request) throws OSSException,
            ClientException;

    /**
     * 上传一个分块（Part）到指定的的Multipart上传事件中。
     * @param request
     *          {@link UploadPartRequest}对象。
     * @return  UploadPartResult 上传Part的返回结果{@link UploadPartResult}。
     * @throws ClientException
     */
    public UploadPartResult uploadPart(UploadPartRequest request)
            throws OSSException, ClientException;
    
    public UploadPartCopyResult uploadPartCopy(UploadPartCopyRequest request)
            throws OSSException, ClientException;
    
    /**
     * 操作将在指定的bucket上设定一个跨域资源共享(CORS)的规则，如果原规则存在则覆盖原规则
     * @param request
     * 					{@link SetBucketCORSRequest}}
     * @throws OSSException
     * @throws ClientException
     */
    public void setBucketCORS(SetBucketCORSRequest request) 
    		throws OSSException, ClientException;
    
    /**
     * 列出指定bucket
     * @param bucketName
     * @return
     * @throws OSSException
     * @throws ClientException
     */
    public List<CORSRule> getBucketCORSRules(String bucketName)
    		throws OSSException, ClientException;
    /**
     * 删除指定bucket下面的所有跨域访问规则
     * @param bucketName
     * @throws OSSException
     * @throws ClientException
     */
    public void deleteBucketCORSRules(String bucketName)
    		throws OSSException, ClientException;
    /**
     * option object
     * @param request
     * @return
     * @throws OSSException
     * @throws ClientException
     */
   public ResponseMessage optionsObject(OptionsRequest request)
		   throws OSSException, ClientException;
    /*
     * 设置{@link Bucket}的访问日志记录功能。
     * 这个功能开启后，OSS将自动记录访问这个{@link Bucket}请求的详细信息，并按照用户指定的规则，
     * 以小时为单位，将访问日志作为一个Object写入用户指定的{@link Bucket}。
     * @param request {@link PutBucketLoggingRequest}对象。
     * @return  BucketLoggingResult {@link UploadPartResult}。
     * */
    public void setBucketLogging(SetBucketLoggingRequest request)
    		throws OSSException, ClientException;
    
    /**
     * 查看{@link Bucket}的访问日志配置。
     * @param bucketName
     * @return
     * @throws OSSException
     * @throws ClientException
     */
    public BucketLoggingResult getBucketLogging(String bucketName)
    		throws OSSException, ClientException;
    
    /**
     * 关闭{@link Bucket}的访问日志记录功能。
     * @param bucketName
     * @throws OSSException
     * @throws ClientException
     */
    public void deleteBucketLogging(String bucketName) 
    		throws OSSException, ClientException;
    
    /**
     * 将一个{@link Bucket}设置成静态网站托管模式。
     * @param setBucketWebSiteRequest
     * @throws OSSException
     * @throws ClientException
     */
    public void setBucketWebsite(SetBucketWebsiteRequest setBucketWebSiteRequest)
            throws OSSException, ClientException;
    
    /**
     * 获取{@link Bucket}的静态网站托管状态。
     * @param bucketName
     * @return
     * @throws OSSException
     * @throws ClientException
     */
    public BucketWebsiteResult getBucketWebsite(String bucketName)
            throws OSSException, ClientException;
    
    /**
     * 关闭{@link Bucket}的静态网站托管模式。
     * @param bucketName
     * @throws OSSException
     * @throws ClientException
     */
    public void deleteBucketWebsite(String bucketName)
            throws OSSException, ClientException;
}