/**
 * Copyright (C), 上海布鲁爱电子商务有限公司
 */
package com.thebeastshop.pegasus.util.service.impl;

import com.thebeastshop.pegasus.util.PegasusUtilFacade;
import com.thebeastshop.pegasus.util.comm.EmptyUtil;
import com.thebeastshop.pegasus.util.cond.CommDictionaryCond;
import com.thebeastshop.pegasus.util.dao.CommCurrencyMapper;
import com.thebeastshop.pegasus.util.dao.CommDictionaryMapper;
import com.thebeastshop.pegasus.util.dao.CommTaxRateMapper;
import com.thebeastshop.pegasus.util.model.*;
import com.thebeastshop.pegasus.util.service.CommDictionaryService;
import com.thebeastshop.pegasus.util.vo.CommDictionaryVO;
import org.apache.commons.beanutils.PropertyUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.annotation.PostConstruct;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

/**
 * @author Royan
 * @version $Id: CommDictionaryServiceImpl.java, v 0.1 2015-07-03 上午11:53
 */
@Service("commDictionaryService")
public class CommDictionaryServiceImpl implements CommDictionaryService {

    private final Logger log = LoggerFactory.getLogger(CommDictionaryServiceImpl.class);

    // 币种缓存
    private List<CommCurrency> __cachedCurrencyList;

    // 税率缓存
    private List<CommTaxRate> __cachedTaxRateList;

    // 缓存字典表中的 商品状态
    private List<CommDictionary> __cachedDicCommodityStatusList;

    @Autowired
    private CommCurrencyMapper commCurrencyMapper;

    @Autowired
    private CommTaxRateMapper commTaxRateMapper;
    
    @Autowired
    private CommDictionaryMapper commDictionaryMapper;

    private static final String REFRESH_CURRENCY = "refreshCurrency";
    private static final String REFRESH_TAXRATE = "refreshTaxRate";
    private static final String REFRESH_COMMODITY_STATUS = CommDictionaryVO.DICT_TYPE_COMMODITY_STATUS ;

    @PostConstruct
    public void init() {
        refreshCurrency();
        timingRefreshCache(REFRESH_CURRENCY);

        refreshTaxRate();
        timingRefreshCache(REFRESH_TAXRATE);

        // 定时刷新 商品状态
        timingRefreshCache(REFRESH_COMMODITY_STATUS);
    }

    // 定时刷新缓存，每隔2分钟执行一次
    private void timingRefreshCache(final String refreshFlag){
        Runnable runnable = new Runnable() {
            public void run() {
                // task to run goes here
                if (REFRESH_CURRENCY.equals(refreshFlag)){
                    refreshCurrency();
                }else if(REFRESH_COMMODITY_STATUS.equals(refreshFlag)){
                    refreshDicCommodityStatus();
                }else{
                    refreshTaxRate();
                }
            }
        };
        ScheduledExecutorService service = Executors
                .newSingleThreadScheduledExecutor();
        // 第二个参数为首次执行的延时时间，第三个参数为定时执行的间隔时间
        service.scheduleAtFixedRate(runnable, 2, 6, TimeUnit.HOURS);
    }

    /**
     * 刷新币种缓存
     */
    @Override
    public void refreshCurrency() {
        log.info("CommDictionaryService.refreshCurrency start");
        CommCurrencyExample commCurrencyExample = new CommCurrencyExample();
        commCurrencyExample.setOrderByClause("id");
        __cachedCurrencyList = commCurrencyMapper.selectByExample(commCurrencyExample);
        log.info("CommDictionaryService.refreshCurrency finish");
    }

    /**
     * 刷新税率缓存
     */
    @Override
    public void refreshTaxRate() {
        log.info("CommDictionaryService.refreshTaxRate start");
        CommTaxRateExample commTaxRateExample = new CommTaxRateExample();
        commTaxRateExample.setOrderByClause("id");
        __cachedTaxRateList = commTaxRateMapper.selectByExample(commTaxRateExample);
        log.info("CommDictionaryService.refreshTaxRate finish");
    }

    /**
     * 查找所有币种
     *
     * @return 币种列表
     * */
    @Override
    public List<CommCurrency> findAllCurrency() {
        List<CommCurrency> result = new ArrayList<CommCurrency>();
        try {
            for (CommCurrency commCurrency : __cachedCurrencyList) {
                CommCurrency _currency = new CommCurrency();
                PropertyUtils.copyProperties(_currency, commCurrency);
                result.add(_currency);
            }
        } catch (Exception e) {
            log.warn("", e);
        }
        return result;
    }

    /**
     * 查找所有税率
     *
     * @return 税率列表
     * */
    @Override
    public List<CommTaxRate> findAllTaxRate() {
        List<CommTaxRate> result = new ArrayList<CommTaxRate>();
        try {
            for (CommTaxRate commTaxRate : __cachedTaxRateList) {
                // 原来所有的税率，即为当前rateType为0的适应税率列表
                if (commTaxRate.getRateType() == 0){
                    CommTaxRate _taxRate = new CommTaxRate();
                    PropertyUtils.copyProperties(_taxRate, commTaxRate);
                    result.add(_taxRate);
                }
            }
            Collections.sort(result, new Comparator<CommTaxRate>() {
                @Override
                public int compare(CommTaxRate o1, CommTaxRate o2) {
                    if (o1.getOrderIndex() > o2.getOrderIndex()) {
                        return 1;
                    }
                    return -1;
                }
            });
        } catch (Exception e) {
            log.warn("", e);
        }
        return result;
    }



    /**
     * 通过适应税率类型查找税率
     *
     * @return 税率列表
     * */
    @Override
    public List<CommTaxRate> findTaxRateByType(Integer type) {
        List<CommTaxRate> result = new ArrayList<CommTaxRate>();
        try {
            for (CommTaxRate commTaxRate : __cachedTaxRateList) {
                if (commTaxRate.getRateType()==type){
                    CommTaxRate _taxRate = new CommTaxRate();
                    PropertyUtils.copyProperties(_taxRate, commTaxRate);
                    result.add(_taxRate);
                }
            }
        } catch (Exception e) {
            log.warn("", e);
        }
        return result;
    }

    private void refreshDicCommodityStatus() {
        log.info("CommDictionaryService.refreshDicCommodityStatus start");
        CommDictionaryCond cond = new CommDictionaryCond();
        cond.setDictType(CommDictionaryVO.DICT_TYPE_COMMODITY_STATUS);
        cond.setOrderBy(" DICT_ORDER ASC ");
        __cachedDicCommodityStatusList = listDictionaryByCond(cond);
        log.info("CommDictionaryService.refreshDicCommodityStatus finish");
    }

    /**
     * 查找所有币种
     *
     * @return 币种列表
     * */
    @Override
    public List<CommDictionary> findDicCommodityStatus() {
        log.warn("findDicCommodityStatus...");
        if (EmptyUtil.isNotEmpty(__cachedDicCommodityStatusList)){
            return __cachedDicCommodityStatusList;
        }
        refreshDicCommodityStatus();
        return __cachedDicCommodityStatusList;
    }

	@Override
	public List<CommDictionary> findDictionaryByType(String dictType) {
		return commDictionaryMapper.findDictionaryByType(dictType);
	}

	@Override
	public List<CommDictionary> findDictionaryByCond(CommDictionaryCond cond) {
		return commDictionaryMapper.findDictionaryByCond(cond);
	}

    @Override
    public List<CommDictionary> listDictionaryByCond(CommDictionaryCond cond) {
        CommDictionaryExample example = buildCommDictionaryExampleByCond(cond);
        return commDictionaryMapper.selectByExample(example);
    }

    private CommDictionaryExample buildCommDictionaryExampleByCond(CommDictionaryCond cond){
        CommDictionaryExample example = new CommDictionaryExample();
        CommDictionaryExample.Criteria criteria = example.createCriteria();
        if (EmptyUtil.isNotEmpty(cond.getDictValue())){
            criteria.andDictValueEqualTo(cond.getDictValue());
        }
        if (EmptyUtil.isNotEmpty(cond.getDictType())){
            criteria.andDictTypeEqualTo(cond.getDictType());
        }
        if (EmptyUtil.isNotEmpty(cond.getOrderBy())){
            example.setOrderByClause(cond.getOrderBy());
        }
        return example;
    }

	@Override
	public String findExpressCodeByExpressName(String ExpressName){
		return commDictionaryMapper.findExpressCodeByExpressName(ExpressName);
	}

    @Override
    public Map<String, String> getCommDictionaryMap(String dictType) {
      Map<String , String> dictMap = new HashMap<String, String>();
      List<CommDictionary> commDictionaryList = findDictionaryByType(dictType);
      int dictTotal = commDictionaryList.size();
      for (int i = 0; i < dictTotal; i ++) {
        dictMap.put(commDictionaryList.get(i).getDictValue(), commDictionaryList.get(i).getDictName());
      }
      return dictMap;
    }

    @Override
    public int batchInsert(List<CommDictionary> commDictionaries){
		return commDictionaryMapper.batchInsert(commDictionaries);
	}
}