package com.thebeastshop.pegasus.util.route;

import org.apache.ibatis.logging.Log;
import org.apache.ibatis.logging.LogFactory;
import org.mybatis.spring.transaction.SpringManagedTransaction;
import org.springframework.jdbc.datasource.DataSourceUtils;

import javax.sql.DataSource;
import java.sql.Connection;
import java.sql.SQLException;

import static org.springframework.jdbc.datasource.DataSourceUtils.isConnectionTransactional;
import static org.springframework.jdbc.datasource.DataSourceUtils.releaseConnection;
import static org.springframework.util.Assert.notNull;

/**
 * Created by roy on 16-1-6.
 */
public class ScmSpringManagedTransaction extends SpringManagedTransaction {

    private final Log logger = LogFactory.getLog(getClass());

    private final DataSource slaveDataSouce;

    private final DataSource msterDataSouce;

    private Connection connection;

    private Connection slaveConnection;

    private boolean isConnectionTransactional;

    private boolean autoCommit;

    public ScmSpringManagedTransaction(DataSource dataSource, DataSource slaveDataSouce) {
        super(dataSource);
        this.msterDataSouce = dataSource;
        this.slaveDataSouce = slaveDataSouce;
        notNull(dataSource, "No DataSource specified");
    }

    /**
     * {@inheritDoc}
     */
    public Connection getConnection() throws SQLException {
        if (DBType.SLAVE.equals(DBTypeHolder.getDBType())) {
            if (slaveConnection == null) {
                openSlaveConnection();
            }
            return slaveConnection;
        }
        if (this.connection == null) {
            openConnection();
        }
        return this.connection;
    }

    /**
     * 打开只读库的连接
     * @throws SQLException
     */
    private void openSlaveConnection() throws SQLException {
        this.slaveConnection = DataSourceUtils.getConnection(msterDataSouce);
    }

    /**
     * Gets a connection from Spring transaction manager and discovers if this
     * {@code Transaction} should manage connection or let it to Spring.
     * <p>
     * It also reads autocommit setting because when using Spring Transaction MyBatis
     * thinks that autocommit is always false and will always call commit/rollback
     * so we need to no-op that calls.
     */
    private void openConnection() throws SQLException {
        this.connection = DataSourceUtils.getConnection(this.msterDataSouce);
        this.autoCommit = this.connection.getAutoCommit();
        this.isConnectionTransactional = isConnectionTransactional(this.connection, this.msterDataSouce);

        if (this.logger.isDebugEnabled()) {
            this.logger.debug(
                    "JDBC Connection ["
                            + this.connection
                            + "] will"
                            + (this.isConnectionTransactional ? " " : " not ")
                            + "be managed by Spring");
        }
    }

    /**
     * {@inheritDoc}
     */
    public void commit() throws SQLException {
        if (this.connection != null && !this.isConnectionTransactional && !this.autoCommit) {
            if (this.logger.isDebugEnabled()) {
                this.logger.debug("Committing JDBC Connection [" + this.connection + "]");
            }
            this.connection.commit();
        }
    }

    /**
     * {@inheritDoc}
     */
    public void rollback() throws SQLException {
        if (this.connection != null && !this.isConnectionTransactional && !this.autoCommit) {
            if (this.logger.isDebugEnabled()) {
                this.logger.debug("Rolling back JDBC Connection [" + this.connection + "]");
            }
            this.connection.rollback();
        }
    }

    /**
     * {@inheritDoc}
     */
    public void close() throws SQLException {
        releaseConnection(this.connection, this.msterDataSouce);
        releaseConnection(this.slaveConnection, this.slaveDataSouce);
    }


}
