package com.thebeastshop.pegasus.util.comm;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;

import org.apache.commons.beanutils.PropertyUtils;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.io.IOUtils;

import org.apache.commons.lang.ArrayUtils;
import org.apache.poi.hssf.usermodel.HSSFCell;
import org.apache.poi.hssf.usermodel.HSSFRichTextString;
import org.apache.poi.hssf.usermodel.HSSFRow;
import org.apache.poi.hssf.usermodel.HSSFSheet;
import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;

/**
 * <p>导出xls文件</p>
 * 
 * <pre>
 *	new ExcelExportBuilder()
 *		.title("商品id", "商品名", ...)
 *		.property("prodId", "prodName", ...)
 *		.fileName("下载文件名").data(list集合)
 *		.builder();
 * </pre>
 * @author donghui
 */
public class ExcelExportBuilder {
	private static final Logger log = LoggerFactory.getLogger(ExcelExportBuilder.class);

    /**
     * sheet最大记数数
     */
    private static final int MAX_SHEET_RECORD_NUM = 50000;

    /**
     * 下载文件名
     */
    private String fileName = "demo"; 
    
    /**
     * 数据记录集
     */
    private List data = new ArrayList(); 
    
    /**
     * 标题行
     */
    private List<String> titleList = new ArrayList<String>();
    
    /**
     * 数据属性集合
     */
    private List<String> propertyNameList = new ArrayList<String>();
    
    /**
     * 去重的属性名记录集
     */
    private List<String> uniqFieldNameList = new ArrayList<>();
    
    
    /**
     * 设置文件名
     */
    public ExcelExportBuilder fileName(String fileName){
    	this.fileName = fileName;
    	return this;
    }
    
    /**
     * 设置数据集
     */
    public ExcelExportBuilder data(List data){
    	this.data = data;
    	return this;
    }
    
    /**
     * 增加标题
     */
    public ExcelExportBuilder title(String... titleName) {
    	if (ArrayUtils.isNotEmpty(titleName)) {
    		for (String name : titleName) {
    			titleList.add(name);
    		}
    	}
    	return this;
    }
    
   /**
    * 设置属性名
    */
    public ExcelExportBuilder property(String... propertyName) {
    	if (ArrayUtils.isNotEmpty(propertyName)) {
    		for (String name : propertyName) {
    			propertyNameList.add(name);
    		}
		}
        return this;
    }
    
    /**
     * 设置去重的属性名
     */
    public ExcelExportBuilder uniqFieldName(String fieldName){
    	uniqFieldNameList.add(fieldName);
    	return this;
    }
    
    /**
     * 执行导出
     */
    public ResponseEntity<byte[]> builder() {
    	byte[] bytes = this.getExcelBytes();
        return new FileDownLoadBuilder().name(fileName + ".xls").file(bytes).builder();
    }

    /**
     * 导出字节流
     */
    private byte[] getExcelBytes(){
    	int dataListSize = data.size();
    	HSSFWorkbook workBook = new HSSFWorkbook();
    	 for (int i = 0; i <= dataListSize / MAX_SHEET_RECORD_NUM; i++) {
             int fromIndex = i * MAX_SHEET_RECORD_NUM;
             int toIndex = fromIndex + MAX_SHEET_RECORD_NUM;
             if (toIndex > dataListSize) {
                 toIndex = dataListSize;
             }
             List subList = data.subList(fromIndex, toIndex);
             // 新建一个工作页
             HSSFSheet sheet = workBook.createSheet();
             // 标题行
             if (CollectionUtils.isNotEmpty(titleList)) {
                 HSSFRow titleRow = sheet.createRow(0); // 下标为0的行开始
                 HSSFCell[] firstColCell = new HSSFCell[titleList.size()];
                 for (int j = 0; j < this.titleList.size(); j++) {
                	firstColCell[j] = titleRow.createCell(j);
                    firstColCell[j].setCellValue(new HSSFRichTextString(titleList.get(j)));
 				}
             }
             //数据行
             for (int j = 0; j < subList.size(); j++) {
                 HSSFRow row = sheet.createRow(j + 1); // 创建一行
                 Object record = subList.get(j);
                 for (int index = 0; index < this.propertyNameList.size(); index++) {
                 	HSSFCell cell = row.createCell(index);
                     Object value = ExcelExportBuilder.getPropertyValue(record, propertyNameList.get(index));
                     cell.setCellValue(Objects.toString(value, ""));
                 }
             }
         }
     	 ByteArrayOutputStream out = new ByteArrayOutputStream();
         try {
             workBook.write(out);
         } catch (IOException e) {
             log.error(null, e);
         } finally {
        	 IOUtils.closeQuietly(workBook);
         }
     	return out.toByteArray();
    }

    private static Object getPropertyValue(Object bean, String propertyName) {
        Object value = new Object();
        try {
            if (bean instanceof Map) {
                Map map = (Map) bean;
                value = MapUtils.getObject(map, propertyName, "");
            } else {
                value = PropertyUtils.getSimpleProperty(bean, propertyName);
            }
        } catch (Exception e) {
            log.warn(null, e);
            return "";
        }
        return value;
    }

}