package com.thebeastshop.pegasus.util.service.impl;

import com.google.common.collect.Lists;
import com.jd.open.api.sdk.DefaultJdClient;
import com.jd.open.api.sdk.JdClient;
import com.jd.open.api.sdk.JdException;
import com.jd.open.api.sdk.domain.ware.SkuReadService.Sku;
import com.jd.open.api.sdk.domain.ware.StockReadService.SkuStock;
import com.jd.open.api.sdk.request.ware.SkuReadSearchSkuListRequest;
import com.jd.open.api.sdk.request.ware.StockReadFindSkuStockRequest;
import com.jd.open.api.sdk.request.ware.StockWriteUpdateSkuStockRequest;
import com.jd.open.api.sdk.response.ware.SkuReadSearchSkuListResponse;
import com.jd.open.api.sdk.response.ware.StockReadFindSkuStockResponse;
import com.jd.open.api.sdk.response.ware.StockWriteUpdateSkuStockResponse;
import com.thebeastshop.pegasus.util.comm.EmptyUtil;
import com.thebeastshop.pegasus.util.comm.NullUtil;
import com.thebeastshop.pegasus.util.exception.CommExceptionErrorCode;
import com.thebeastshop.pegasus.util.exception.UtilException;
import com.thebeastshop.pegasus.util.service.CommJdService;
import com.thebeastshop.pegasus.util.vo.JdAppCfgVO;
import com.thebeastshop.pegasus.util.vo.JdStockSyncResult;
import org.apache.commons.lang.StringUtils;
import org.springframework.stereotype.Service;

import java.util.*;

/**
 * Created by Administrator on 2018/9/3.
 */
@Service("commJdService")
public class CommJdServiceImpl implements CommJdService {

    private final static String SERVER_URL = "https://api.jd.com/routerjson";

    private final static String OK = "0";

    private final static String JD_WH = "WH020600010195";

    private final static String JD_HOME_WH = "WH000332910332";

    private final static Map<String,String> whChannelMap;

    static {
        whChannelMap = new HashMap<>();
        whChannelMap.put(JD_WH,"CHN2042");
        whChannelMap.put(JD_HOME_WH,"CHN2079");
    }

    @Override
    public boolean needSyncStock(String warehouseCode) {
        return JD_WH.equals(warehouseCode) || JD_HOME_WH.equals(warehouseCode);
    }

    @Override
    public Map<String,JdStockSyncResult> syncSkuStock(String warehouseCode, Map<String, Integer> skuQuantityMap) {
        Map<String,JdStockSyncResult> resultMap = new HashMap<>();
        if(!needSyncStock(warehouseCode) ||skuQuantityMap.isEmpty()){
            return resultMap;
        }
        JdAppCfgVO appCfg = getAppCfg(warehouseCode);
        String channelCode = getChannelCode(warehouseCode);
        JdClient jdClient = getJdClient(appCfg);
        //查询jdskuId
        Map<String,Sku> jdSkuMap = searchJdSkuList(jdClient,skuQuantityMap.keySet());
        for(Map.Entry<String,Integer> entry : skuQuantityMap.entrySet()){
            Sku jdSku = jdSkuMap.get(entry.getKey());
            JdStockSyncResult syncResult = new JdStockSyncResult();
            syncResult.setSkuCode(entry.getKey());
            syncResult.setQuantity(entry.getValue());
            syncResult.setChannelCode(channelCode);
            resultMap.put(entry.getKey(),syncResult);
            if(NullUtil.isNotNull(jdSku)){
                if(entry.getValue() == 0){
                    //无需同步
                    syncResult.setSuccess(true);
                    continue;
                }
                //获取jdsku 库存
                syncResult.setJdSkuId(jdSku.getSkuId());
                Long jdStockNum = getJdSkuStock(jdClient,jdSku);
                if(NullUtil.isNotNull(jdStockNum)){
                    jdSku.setStockNum(jdStockNum+entry.getValue());
                    //更新jdsku库存
                    String errorMsg = updateJdSkuStock(jdClient,jdSku);
                    if(errorMsg != null){
                        syncResult.setErrorMsg(errorMsg);
                    }else{
                        syncResult.setSuccess(true);
                    }
                }else{
                    syncResult.setErrorMsg("获取SKU库存失败");
                }
            }else{
                syncResult.setErrorMsg("京东不存在该SKU");
            }
        }
        return resultMap;
    }

    private String updateJdSkuStock(JdClient client,Sku jdSku){
        StockWriteUpdateSkuStockRequest request = new StockWriteUpdateSkuStockRequest();
        request.setSkuId(jdSku.getSkuId());
        request.setStockNum(jdSku.getStockNum());
        if(jdSku.getStockNum() < 0){
            return "库存不足";
        }
        try {
            StockWriteUpdateSkuStockResponse response = client.execute(request);
            if(OK.equals(response.getCode()) && response.getSuccess()){
                return null;
            }else{
                return response.getZhDesc();
            }
        }catch (JdException e){
            throw new UtilException(CommExceptionErrorCode.ILLEGAL_PARAM,e.getMessage());
        }
    }

    private Long getJdSkuStock(JdClient client,Sku jdSku){
        StockReadFindSkuStockRequest request=new StockReadFindSkuStockRequest();
        request.setSkuId(jdSku.getSkuId());
        try {
            StockReadFindSkuStockResponse response = client.execute(request);
            if(OK.equals(response.getCode())){
                List<SkuStock> skuStockList = response.getSkuStocks();
                if(EmptyUtil.isNotEmpty(skuStockList)){
                    return skuStockList.get(0).getStockNum();
                }
                return null;
            }else{
                throw new UtilException(CommExceptionErrorCode.ILLEGAL_PARAM,"获取库存失败!");
            }
        }catch (JdException e){
            throw new UtilException(CommExceptionErrorCode.ILLEGAL_PARAM,e.getMessage());
        }
    }

    public Map<String,Sku> searchJdSkuList(JdClient client, Set<String> skuCodes){
        if(EmptyUtil.isEmpty(skuCodes)){
            throw new UtilException(CommExceptionErrorCode.ILLEGAL_PARAM,"SKU为空");
        }
        List<String> codesList = Arrays.asList(skuCodes.toArray(new String[skuCodes.size()]));
        Map<String,Sku> skuMap = new HashMap<>();
        List<List<String>> parts = Lists.partition(codesList,19);//不可超过20个
        for(List<String> part : parts){
            Map<String,Sku> tmpMap = searchJdSkuListPage(client,part);
            skuMap.putAll(tmpMap);
        }
        return skuMap;
    }

    public Map<String,Sku> searchJdSkuListPage(JdClient client, List<String> skuCodes) {
        if(EmptyUtil.isEmpty(skuCodes)){
            throw new UtilException(CommExceptionErrorCode.ILLEGAL_PARAM,"SKU为空");
        }
        int pageNo = 1;
        Map<String,Sku> skuMap = new HashMap<>();
        List<Sku> rtList = new ArrayList<>();
        while(true){
            SkuReadSearchSkuListRequest request = new SkuReadSearchSkuListRequest();
            request.setSkuStatuValue("1,2");
            String outId = StringUtils.join(skuCodes,",");
            request.setOutId(outId);
            request.setPageNo(pageNo);
            request.setPageSize(100);
            try {
                SkuReadSearchSkuListResponse response = client.execute(request);
                if(OK.equals(response.getCode())){
                    List<Sku> jdSkuList = response.getPage().getData();
                    if(EmptyUtil.isNotEmpty(jdSkuList)){
                        rtList.addAll(jdSkuList);
                    }
                    if(rtList.size() == response.getPage().getTotalItem()){
                        break;
                    }
                    pageNo = response.getPage().getPageNo();
                    pageNo++;
                }else{
                    throw new UtilException(CommExceptionErrorCode.STATUS_NOT_EXPECTED,response.getZhDesc());
                }
            }catch (JdException e){
                throw new UtilException(CommExceptionErrorCode.ILLEGAL_PARAM,e.getMessage());
            }
        }
        Collections.sort(rtList, new Comparator<Sku>() {
            @Override
            public int compare(Sku o1, Sku o2) {
                return o1.getStatus().compareTo(o2.getStatus());
            }
        });
        for(Sku jdSku : rtList){
            Sku tmp = skuMap.get(jdSku.getOuterId());
            if(NullUtil.isNull(tmp)){
                skuMap.put(jdSku.getOuterId(),jdSku);
            }
        }
        return skuMap;
    }

    private JdClient getJdClient(JdAppCfgVO appCfg){
        return new DefaultJdClient(
                SERVER_URL
                ,appCfg.getAccessToken()
                ,appCfg.getAppKey()
                ,appCfg.getAppSecret());
    }

    private JdAppCfgVO getAppCfg(String warehouseCode){
        JdAppCfgVO appCfg = new JdAppCfgVO();
        if(JD_WH.equals(warehouseCode)){
            appCfg.setAppKey("CB79069CD881B39EC25CB569053A6B30");
            appCfg.setAppSecret("8e16059ec7264e6ca0a5ea0f81d4326b");
            appCfg.setAccessToken("714e1379-97a8-437b-9fb7-6e1b130b808c");
            appCfg.setRefreshToken("1df4e597-e6dc-4cf0-9466-da72d68046ae");
        }else if(JD_HOME_WH.equals(warehouseCode)){
            appCfg.setAppKey("BC87A4010848855465B46C723EDA177C");
            appCfg.setAppSecret("8a4c65bac1e94a859ff3b6531cfd8989");
            appCfg.setAccessToken("74734c15-8250-4eb3-a8ad-efd06db81f4d");
            appCfg.setRefreshToken("10cf7285-7bd2-4aa4-a45e-16f0e71bec3f");
        }else{
            throw new UtilException("非京东仓无需同步");
        }
        return appCfg;
    }

    private String getChannelCode(String warehouseCode){
        return whChannelMap.get(warehouseCode);
    }

//京东授权流程
//http://jos.jd.com/doc/channel.htm?id=152
// app授权码，时间关系，暂不刷新access_token
//    {
//        "access_token": "74734c15-8250-4eb3-a8ad-efd06db81f4d",
//            "code": 0,
//            "expires_in": 31103999,
//            "refresh_token": "1df4e597-e6dc-4cf0-9466-da72d68046ae",
//            "time": "1535539928586",
//            "token_type": "bearer",
//            "uid": "1427401372",
//            "user_nick": "野兽派家居旗舰店"
//    }
//
//    {
//        "access_token": "714e1379-97a8-437b-9fb7-6e1b130b808c",
//            "code": 0,
//            "expires_in": 31103999,
//            "refresh_token": "10cf7285-7bd2-4aa4-a45e-16f0e71bec3f",
//            "time": "1535970113449",
//            "token_type": "bearer",
//            "uid": "9719083190",
//            "user_nick": "野兽派官方旗舰店"
//    }


}
