/*
 * Copyright (C), 上海布鲁爱电子商务有限公司
 */
package com.thebeastshop.pegasus.util.service.impl;

import com.thebeastshop.pegasus.util.PegasusConstants;
import com.thebeastshop.pegasus.util.comm.*;
import com.thebeastshop.pegasus.util.dao.CommFileMapper;
import com.thebeastshop.pegasus.util.exception.UtilException;
import com.thebeastshop.pegasus.util.exception.UtilExceptionErrorCode;
import com.thebeastshop.pegasus.util.model.CommFile;
import com.thebeastshop.pegasus.util.model.CommFileExample;
import com.thebeastshop.pegasus.util.service.CommFileService;
import com.thebeastshop.pegasus.util.vo.FileRequestVO;
import com.thebeastshop.pegasus.util.vo.FileResponseVO;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.beans.BeanUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.PostConstruct;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

/**
 * @author Eric.Lou
 * @version $Id: CommFileServiceImpl.java, v 0.1 2015-07-29 07:55
 */
@Service("commFileServiceImpl")
public class CommFileServiceImpl implements CommFileService, InitializingBean {

    private final Logger log = LoggerFactory.getLogger(CommFileServiceImpl.class);

    @Autowired
    private CommFileMapper commFileMapper;

    @Value("${STORE_BASE_PATH}")
    private String __storePath;

    @Value("${STORE_PATH_SEPARATOR}")
    private String __pathSeparator;

    /**
     * spring扫描service时创建一些参数
     */
    @PostConstruct
    public void init() {
        log.info("[Service] CommFileServiceImpl is initialing");
        log.info("[Service] CommFileServiceImpl __storePath is " + __storePath);
        log.info("[Service] CommFileServiceImpl __pathSeparator is " + __pathSeparator);
        log.info("[Service] CommFileServiceImpl is initilalied");
    }

    @Override
    public String getStorePath() {
        if(EmptyUtil.isEmpty(__storePath)){
            throw new UtilException(UtilExceptionErrorCode.FILE_COMMONS_ERROR, "__storePath is null");
        }
        return __storePath;
    }

    @Override
    public String getPathSeparator() {
        if(EmptyUtil.isEmpty(__pathSeparator)){
            throw new UtilException(UtilExceptionErrorCode.FILE_COMMONS_ERROR, "__pathSeparator is null");
        }
        return __pathSeparator;
    }

    @Override
    public void afterPropertiesSet() throws Exception {
        createFileDir(__storePath);
    }

    /**
     * 存储文件
     *
     * @param fileRequestVO
     * @return fileResponseVO
     */
    @Override
    @Transactional
    public FileResponseVO store(FileRequestVO fileRequestVO) {
        if (EmptyUtil.isNotEmpty(fileRequestVO.getInputStream())
                && EmptyUtil.isNotEmpty(fileRequestVO.getOriginalFilename())
                && EmptyUtil.isNotEmpty(fileRequestVO.getExtName())) {
            Boolean isCreat = false;
            String finalPath = getFinalPath(fileRequestVO.getExtName()); //文件最终保存路径
            try {
                isCreat = FileUtil.InputStreamTOFile(fileRequestVO.getInputStream(), finalPath);
            } catch (IOException e) {
                log.error("[Service] CommFileServiceImpl ", e);
                // roll back data
                deleteFile(finalPath);
            }
            if (isCreat) {
                log.info("[Service] CommFileServiceImpl store File,filePath:" + finalPath);
                try {
                    CommFile record = create(fileRequestVO.getOriginalFilename(), finalPath);
                    if (NullUtil.isNotNull(record)) {
                        FileResponseVO fileResponseVO = new FileResponseVO();
                        BeanUtils.copyProperties(record, fileResponseVO);
                        return fileResponseVO;
                    } else {
                        deleteFile(finalPath);
                    }
                } catch (Exception e) {
                    log.error("[Service] CommFileServiceImpl ", e);
                    // roll back data
                    deleteFile(finalPath);
                }
            }
        }
        return null;
    }

    @Override
    @Transactional
    public FileResponseVO storeAfterDelete(FileRequestVO fileRequestVO, long id) {
        deleteFileById(id);
        return store(fileRequestVO);
    }

    @Override
    public FileResponseVO storeImage(FileRequestVO fileRequestVO, Integer width, Integer height) {
        if (EmptyUtil.isNotEmpty(fileRequestVO.getInputStream())
                && EmptyUtil.isNotEmpty(fileRequestVO.getOriginalFilename())
                && EmptyUtil.isNotEmpty(fileRequestVO.getExtName())) {
            Boolean isCreat = false;
            String finalPath = getFinalPath(fileRequestVO.getExtName()); //文件最终保存路径
            try {
                if (EmptyUtil.isNotEmpty(width) && EmptyUtil.isNotEmpty(height)) {
                    isCreat = FileUtil.InputStreamTOFile(fileRequestVO.getInputStream(), finalPath);
                    if (isCreat) {
                        FileUtil.resize(width, height, finalPath, true);
                    }
                } else {
                    isCreat = FileUtil.InputStreamTOFile(fileRequestVO.getInputStream(), finalPath);
                }
            } catch (IOException e) {
                log.error("[Service] CommFileServiceImpl ", e);
                // roll back data
                deleteFile(finalPath);
            }
            if (isCreat) {
                log.info("[Service] CommFileServiceImpl store File,filePath:" + finalPath);
                try {
                    CommFile record = create(fileRequestVO.getOriginalFilename(), finalPath);
                    if (NullUtil.isNotNull(record)) {
                        FileResponseVO fileResponseVO = new FileResponseVO();
                        BeanUtils.copyProperties(record, fileResponseVO);
                        return fileResponseVO;
                    } else {
                        deleteFile(finalPath);
                    }
                } catch (Exception e) {
                    log.error("[Service] CommFileServiceImpl ", e);
                    // roll back data
                    deleteFile(finalPath);
                }
            }
        }
        return null;
    }

    @Override
    public FileResponseVO getFile(long id, boolean isread) {
        if (id > 0) {
            CommFile record = commFileMapper.selectByPrimaryKey(id);
            return buildFileResponseVO(record, isread);
        }
        return null;
    }

    @Override
    public List<FileResponseVO> getFile(List<Long> ids, boolean isread) {
        CommFileExample example = new CommFileExample();
        example.createCriteria().andIdIn(ids);
        List<CommFile> commFiles = commFileMapper.selectByExample(example);
        if (commFiles == null || commFiles.size() == 0) {
           return  null;
        }
        List<FileResponseVO> files = new ArrayList<FileResponseVO>();
        for (CommFile commFile : commFiles) {
          FileResponseVO fileResponseVO = buildFileResponseVO(commFile, isread);
          if (fileResponseVO != null) {
            files.add(fileResponseVO);
          }
        }

        return files;
    }

    @Override
    @Transactional
    public Boolean deleteFileById(long id) {
        if (EmptyUtil.isEmpty(id)) {
            throw new UtilException(UtilExceptionErrorCode.FILE_DELETE_ERROR, "file id is null");
        }
        CommFile commFile = commFileMapper.selectByPrimaryKey(id);
        if (NullUtil.isNull(commFile)) {
            throw new UtilException(UtilExceptionErrorCode.FILE_DELETE_ERROR, "commFile is null");
        }
        if (commFile.getIsDelete().equals(PegasusConstants.YES)) {
            throw new UtilException(UtilExceptionErrorCode.FILE_DELETE_ERROR, "commFile update failed");
        }
        if (EmptyUtil.isEmpty(commFile.getFilePath())) {
            throw new UtilException(UtilExceptionErrorCode.FILE_DELETE_ERROR, "filePath is null");
        }
        commFile.setIsDelete(PegasusConstants.YES);
        if (commFileMapper.updateByPrimaryKey(commFile) == 0) {
            throw new UtilException(UtilExceptionErrorCode.FILE_DELETE_ERROR, "commFile update failed");
        }
        deleteFile(commFile.getFilePath());
        return true;
    }

    @Override
    public Boolean modifyFileName(long id, String fileName) {
        if (EmptyUtil.isEmpty(id)) {
            throw new UtilException(UtilExceptionErrorCode.FILE_DELETE_ERROR, "file id is null");
        }
        CommFile commFile = commFileMapper.selectByPrimaryKey(id);
        if (NullUtil.isNull(commFile)) {
            throw new UtilException(UtilExceptionErrorCode.FILE_DELETE_ERROR, "commFile is null");
        }
        if (EmptyUtil.isEmpty(commFile.getFilePath())) {
            throw new UtilException(UtilExceptionErrorCode.FILE_DELETE_ERROR, "filePath is null");
        }
        if (!commFile.getFileName().equals(fileName)) {
            commFile.setFileName(fileName);
            if (commFileMapper.updateByPrimaryKey(commFile) == 0) {
                throw new UtilException(UtilExceptionErrorCode.FILE_DELETE_ERROR, "commFile update failed");
            }
        }
        return true;
    }

    @Override
    public FileResponseVO saveOssInfoToDb(FileRequestVO fileRequestVO) {
        FileResponseVO fileResponseVO = new FileResponseVO();
        CommFile commFile = this.create(fileRequestVO.getOriginalFilename(), fileRequestVO.getOssKey());
        fileResponseVO.setId(EmptyUtil.isNotEmpty(commFile) ? commFile.getId() : null);
        fileResponseVO.setFileName(fileRequestVO.getOriginalFilename());
        return fileResponseVO;
    }

    /**
     * 文件最终保存的路径
     *
     * @param extName 文件扩展名
     * @return
     */
    private synchronized String getFinalPath(String extName) {
        String fileDir = __storePath + __pathSeparator + DateUtil.format(new Date(), "YYYYMMDD");
        createFileDir(fileDir);
        return fileDir + __pathSeparator + ObjectId.get() + "." + extName;
    }

    /**
     * 删除文件
     *
     * @param filePath
     */
    private void deleteFile(String filePath) {
        File file = new File(filePath);
        if (file.exists() && file.canRead()) {
            file.delete();
            log.info("[Service] CommFileServiceImpl filePath[" + filePath + "] is exists and deleted");
        } else {
            log.info("[Service] CommFileServiceImpl filePath[" + filePath + "] is not exists and not deleted");
        }
    }

    /**
     * @param originalFilename 文件原始名称
     * @param finalPath        文件最终保存的路径
     * @return
     */
    @Transactional
    private CommFile create(String originalFilename, String finalPath) {
        CommFile record = new CommFile();
        record.setCreateTime(new Date());
        record.setFileName(originalFilename);
        record.setFilePath(finalPath);
        record.setIsDelete(PegasusConstants.NO);
        commFileMapper.insertSelective(record);
        return record;
    }

    private FileResponseVO  buildFileResponseVO(CommFile record, boolean isread) {
        FileResponseVO fileResponseVO = new FileResponseVO();
        if (NullUtil.isNotNull(record)) {
            BeanUtils.copyProperties(record, fileResponseVO);
            if (isread) {
                try {
                    fileResponseVO.setFileData(FileUtil.getByteFileData(fileResponseVO.getFilePath()));
                } catch (IOException e) {
                    log.error("", e);
                }
            }
            return fileResponseVO;
        }
        return null;
    }
    /**
     * 创建目录
     *
     * @param dir
     */
    private void createFileDir(String dir) {
        String filePath = dir;
        File fileDir = new File(filePath);
        if (!fileDir.exists() && !fileDir.isDirectory()) {
            fileDir.mkdir();
            log.info("[Service] CommFileServiceImpl mkdir " + dir);
        }
    }

}
