/**
 * Copyright (C), 上海布鲁爱电子商务有限公司
 */
package com.thebeastshop.pegasus.util.service.impl;


import com.thebeastshop.pegasus.util.comm.EmptyUtil;
import com.thebeastshop.pegasus.util.comm.NullUtil;
import com.thebeastshop.pegasus.util.dao.CommCityMapper;
import com.thebeastshop.pegasus.util.dao.CommCountryMapper;
import com.thebeastshop.pegasus.util.dao.CommDistrictMapper;
import com.thebeastshop.pegasus.util.dao.CommProvinceMapper;
import com.thebeastshop.pegasus.util.model.*;
import com.thebeastshop.pegasus.util.service.CommAddressService;
import com.thebeastshop.pegasus.util.service.CommThirdPartyService;
import com.thebeastshop.pegasus.util.vo.CommAreaMatchVO;
import com.thebeastshop.pegasus.util.vo.Region;

import org.apache.commons.beanutils.PropertyUtils;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.annotation.PostConstruct;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

/**
 * @author Royan
 * @version $Id: CommAddressServiceImpl.java, v 0.1 2015-07-03 上午10:06
 */
@Service("commAddressService")
public class CommAddressServiceImpl implements CommAddressService {

    private final Logger           log = LoggerFactory.getLogger(CommAddressServiceImpl.class);

    // 缓存内容
    private Map<Long, CommCountry> __cachedCountryMap;

    @Autowired
    private CommCountryMapper      commCountryMapper;

    @Autowired
    private CommProvinceMapper     commProvinceMapper;

    @Autowired
    private CommCityMapper         commCityMapper;

    @Autowired
    private CommDistrictMapper     commDistrictMapper;

    @Autowired
    private CommThirdPartyService commThirdPartyService;
    @PostConstruct
    public void init() {
        refreshCache();
        timingRefreshCache();
    }

    // 定时刷新缓存，每隔两分钟执行一次
    private void timingRefreshCache(){
        Runnable runnable = new Runnable() {
            public void run() {
                // task to run goes here
                refreshCache();
            }
        };
        ScheduledExecutorService service = Executors
                .newSingleThreadScheduledExecutor();
        // 第二个参数为首次执行的延时时间，第三个参数为定时执行的间隔时间
        service.scheduleAtFixedRate(runnable, 2, 6, TimeUnit.HOURS);
    }

    /**
     * 刷新缓存
     */
    @Override
    public void refreshCache() {
        log.info("CommAddressService.refreshCache start");

        // all countries
        CommCountryExample countryExample = new CommCountryExample();
        countryExample.setOrderByClause("id");
        List<CommCountry> allCountryList = commCountryMapper.selectByExample(countryExample);
        log.debug("allCountryList.size=" + allCountryList.size());

        // all provinces
        CommProvinceExample commProvinceExample = new CommProvinceExample();
        commProvinceExample.setOrderByClause("id");
        List<CommProvince> allProvinceList = commProvinceMapper
                .selectByExample(commProvinceExample);
        log.debug("allProvinceList.size=" + allProvinceList.size());

        // all cities
        CommCityExample commCityExample = new CommCityExample();
        commCityExample.setOrderByClause("id");
        List<CommCity> allCitylist = commCityMapper.selectByExample(commCityExample);
        log.debug("allCitylist.size=" + allCitylist.size());

        // all districts
        CommDistrictExample districtExample = new CommDistrictExample();
        districtExample.setOrderByClause("id");
        List<CommDistrict> allDistrictList = commDistrictMapper
                .selectByExample(districtExample);
        log.debug("allDistrictList.size=" + allDistrictList.size());

        Map<Long, CommCountry> tmp = new ConcurrentHashMap<Long, CommCountry>();
        // combine address
        for (CommCountry commCountry : allCountryList) {
            tmp.put(commCountry.getId(), commCountry);
            List<CommProvince> provinceList = new ArrayList<CommProvince>();
            commCountry.setProvinceList(provinceList);

            for (CommProvince commProvince : allProvinceList) {
                if (commCountry.getId().equals(commProvince.getCountryId())) {
                    provinceList.add(commProvince);
                    List<CommCity> cityList = new ArrayList<CommCity>();
                    commProvince.setCityList(cityList);

                    for (CommCity commCity : allCitylist) {
                        if (commProvince.getId().equals(commCity.getProvinceId())) {
                            cityList.add(commCity);
                            List<CommDistrict> districtList = new ArrayList<CommDistrict>();
                            commCity.setDistrictList(districtList);

                            for (CommDistrict commDistrict : allDistrictList) {
                                if (commCity.getId().equals(commDistrict.getCityId())) {
                                    districtList.add(commDistrict);
                                }
                            }
                        }
                    }
                }
            }
        }
        __cachedCountryMap = tmp;

        log.info("CommAddressService.refreshCache finish");
    }

    /**
     * 查找所有国家
     *
     * @param cascade 是否级联;
     * @return 国家列表
     */
    @Override
    public List<CommCountry> findAllCountry(boolean cascade) {
        List<CommCountry> result = new ArrayList<CommCountry>();
        try {
            for (CommCountry commCountry : __cachedCountryMap.values()) {
                CommCountry _country = new CommCountry();
                PropertyUtils.copyProperties(_country, commCountry);
                result.add(_country);

                if (cascade) {
                    List<CommProvince> _provinceList = new ArrayList<CommProvince>();
                    _country.setProvinceList(_provinceList);

                    for (CommProvince commProvince : commCountry.getProvinceList()) {
                        CommProvince _province = new CommProvince();
                        PropertyUtils.copyProperties(_province, commProvince);
                        _provinceList.add(_province);

                        List<CommCity> _cityList = new ArrayList<CommCity>();
                        _province.setCityList(_cityList);
                        for (CommCity commCity : commProvince.getCityList()) {
                            CommCity _city = new CommCity();
                            PropertyUtils.copyProperties(_city, commCity);
                            _cityList.add(_city);

                            List<CommDistrict> _districtList = new ArrayList<CommDistrict>();
                            _city.setDistrictList(_districtList);
                            for (CommDistrict commDistrict : commCity.getDistrictList()) {
                                CommDistrict _district = new CommDistrict();
                                PropertyUtils.copyProperties(_district, commDistrict);
                                _districtList.add(_district);
                            }
                        }
                    }
                } else {
                    _country.setProvinceList(new ArrayList<CommProvince>());
                }
            }
        } catch (Exception e) {
            log.warn("", e);
        }
        return result;
    }

    /**
     * 根据ID查找国家
     *
     * @param id 国家ID;
     * @param cascade 是否级联;
     * @return 国家
     */
    @Override
    public CommCountry findCountryById(Long id, boolean cascade) {
        try {
            for (CommCountry commCountry : __cachedCountryMap.values()) {
                if (!commCountry.getId().equals(id)) {
                    continue;
                }
                CommCountry result = new CommCountry();
                PropertyUtils.copyProperties(result, commCountry);

                List<CommProvince> _provinceList = new ArrayList<CommProvince>();
                result.setProvinceList(_provinceList);

                if (cascade) {
                    for (CommProvince commProvince : commCountry.getProvinceList()) {
                        CommProvince _province = new CommProvince();
                        PropertyUtils.copyProperties(_province, commProvince);
                        _provinceList.add(_province);

                        List<CommCity> _cityList = new ArrayList<CommCity>();
                        _province.setCityList(_cityList);
                        for (CommCity commCity : commProvince.getCityList()) {
                            CommCity _city = new CommCity();
                            PropertyUtils.copyProperties(_city, commCity);
                            _cityList.add(_city);

                            List<CommDistrict> _districtList = new ArrayList<CommDistrict>();
                            _city.setDistrictList(_districtList);
                            for (CommDistrict commDistrict : commCity.getDistrictList()) {
                                CommDistrict _district = new CommDistrict();
                                PropertyUtils.copyProperties(_district, commDistrict);
                                _districtList.add(_district);
                            }
                        }
                    }
                }
                return result;
            }
        } catch (Exception e) {
            log.warn("", e);
        }
        return null;
    }

    /**
     *
     * 根据名称查找国家,精确匹配
     * @param name
     * @return 国家
     */
    public CommCountry findCountryByName(String name) {
        for (CommCountry commCountry : __cachedCountryMap.values()) {
            if (commCountry.getName().equals(name)) {
                return commCountry;
            }
        }
        return null;
    }

    /**
     * 根据国家ID查找省份
     *
     * @param countryId 国家ID;
     * @param cascade 是否级联;
     * @return 省份列表
     */
    @Override
    public List<CommProvince> findProvinceByCountryId(Long countryId, boolean cascade) {
        List<CommProvince> result = new ArrayList<CommProvince>();
        CommCountry commCountry = findCountryById(countryId, true);
        if (commCountry != null) {
            for (CommProvince commProvince : commCountry.getProvinceList()) {
                result.add(commProvince);
                if (!cascade) {
                    commProvince.setCityList(new ArrayList<CommCity>());
                }
            }
        }

        return result;
    }

    /**
     * 根据ID查找省份
     *
     * @param id 省份ID;
     * @param cascade 是否级联;
     * @return 省份
     */
    @Override
    public CommProvince findProvinceById(Long id, boolean cascade) {
        try {
            for (CommCountry commCountry : __cachedCountryMap.values()) {
                for (CommProvince commProvince : commCountry.getProvinceList()) {
                    if (!commProvince.getId().equals(id)) {
                        continue;
                    }
                    CommProvince result = new CommProvince();
                    PropertyUtils.copyProperties(result, commProvince);
                    List<CommCity> _cityList = new ArrayList<CommCity>();
                    result.setCityList(_cityList);
                    if (cascade) {
                        for (CommCity commCity : commProvince.getCityList()) {
                            CommCity _city = new CommCity();
                            PropertyUtils.copyProperties(_city, commCity);
                            _cityList.add(_city);

                            List<CommDistrict> _districtList = new ArrayList<CommDistrict>();
                            _city.setDistrictList(_districtList);
                            for (CommDistrict commDistrict : commCity.getDistrictList()) {
                                CommDistrict _district = new CommDistrict();
                                PropertyUtils.copyProperties(_district, commDistrict);
                                _districtList.add(_district);
                            }
                        }
                    }
                    return result;
                }
            }
        } catch (Exception e) {
            log.warn("", e);
        }
        return null;
    }

    @Override
    public List<CommProvince> findProvinceByCond(String cond) {
        CommProvinceExample example = new CommProvinceExample();
        example.createCriteria().andFullNameLike("%" + cond + "%");
        return commProvinceMapper.selectByExample(example);
    }
    
    @Override
    public CommProvince findProvinceBydistrictIdId(Long districtId) {
        return commProvinceMapper.findProvinceBydistrictIdId(districtId);
    }

    /**
     * 根据省份ID查找城市
     *
     * @param provinceId 省份ID;
     * @param cascade 是否级联;
     * @return 城市列表
     */
    @Override
    public List<CommCity> findCityByProvinceId(Long provinceId, boolean cascade) {
        List<CommCity> result = new ArrayList<CommCity>();
        CommProvince commProvince = findProvinceById(provinceId, true);
        if (commProvince != null) {
            for (CommCity commCity : commProvince.getCityList()) {
                result.add(commCity);
                if (!cascade) {
                    commCity.setDistrictList(new ArrayList<CommDistrict>());
                }
            }
        }

        return result;
    }

    @Override
    public List<CommCity> findCityByProvinceIdList(List<Long> provinceIdList, boolean cascade) {
        List<CommCity> cityList = new ArrayList<>();
        for (int i = 0; i < provinceIdList.size(); i++) {
            Long id = provinceIdList.get(i);
            List<CommCity> list = findCityByProvinceId(id, cascade);
            cityList.addAll(list);
        }
        return cityList;
    }

    /**
     * 根据ID查找城市
     *
     * @param id      城市ID;
     * @param cascade 是否级联;
     * @return 城市
     */
    @Override
    public CommCity findCityById(Long id, boolean cascade) {
        try {
            for (CommCountry commCountry : __cachedCountryMap.values()) {
                for (CommProvince commProvince : commCountry.getProvinceList()) {
                    for (CommCity commCity : commProvince.getCityList()) {
                        if (!commCity.getId().equals(id)) {
                            continue;
                        }
                        CommCity result = new CommCity();
                        PropertyUtils.copyProperties(result, commCity);
                        List<CommDistrict> _districtList = new ArrayList<CommDistrict>();
                        result.setDistrictList(_districtList);
                        if (cascade) {
                            for (CommDistrict commDistrict : commCity.getDistrictList()) {
                                CommDistrict _district = new CommDistrict();
                                PropertyUtils.copyProperties(_district, commDistrict);
                                _districtList.add(_district);
                            }
                        }
                        return result;
                    }

                }
            }
        } catch (Exception e) {
            log.warn("", e);
        }
        return null;
    }

    @Override
    public List<CommCity> findCityByIds(List<Long> ids, boolean cascade) {
        List<CommCity> list = new ArrayList<>();
        if(EmptyUtil.isNotEmpty(ids)){
            for(Long id : ids){
                CommCity city = findCityById(id,cascade);
                if(NullUtil.isNotNull(city)){
                    list.add(city);
                }
            }
        }
        return list;
    }

    /**
     * 根据城市ID查找区县
     *
     * @param cityId 城市ID;
     * @return 区县列表
     */
    @Override
    public List<CommDistrict> findDistrictByCityId(Long cityId) {
        List<CommDistrict> result = new ArrayList<CommDistrict>();
        CommCity commCity = findCityById(cityId, true);
        if (commCity != null) {
            for (CommDistrict commDistrict : commCity.getDistrictList()) {
                result.add(commDistrict);
            }
        }

        return result;
    }

    /**
     * 根据ID查找区县
     *
     * @param id 区县ID;
     * @return 区县
     */
    @Override
    public CommDistrict findDistrictById(Long id) {
        try {
            for (CommCountry commCountry : __cachedCountryMap.values()) {
                for (CommProvince commProvince : commCountry.getProvinceList()) {
                    for (CommCity commCity : commProvince.getCityList()) {
                        for (CommDistrict commDistrict : commCity.getDistrictList()) {
                            if (!commDistrict.getId().equals(id)) {
                                continue;
                            }
                            CommDistrict result = new CommDistrict();
                            PropertyUtils.copyProperties(result, commDistrict);
                            return result;
                        }
                    }
                }
            }
        } catch (Exception e) {
            log.warn("", e);
        }
        return null;
    }

    @Override
    public List<CommDistrict> findDistrictByIds(List<Long> ids) {
        List<CommDistrict> list = new ArrayList<>();
        if(EmptyUtil.isNotEmpty(ids)){
            for(Long id : ids){
                CommDistrict district = findDistrictById(id);
                if(NullUtil.isNotNull(district)){
                    list.add(district);
                }
            }
        }
        return list;
    }

    /**
     * 根据条件查找城市
     *
     * @param cond 条件;
     * @return 城市列表
     */
    @Override
    public List<CommCity> findCityByCond(String cond) {
        CommCityExample example = new CommCityExample();
        example.createCriteria().andFullNameLike("%" + cond + "%");
        return commCityMapper.selectByExample(example);
    }

    @Override
    public List<CommCity> findCityByProvinceAndName(Long provinceId, String cityName) {
        CommCityExample example = new CommCityExample();
        CommCityExample.Criteria criteria = example.createCriteria();
        criteria.andNameLike("%" + cityName + "%");
        criteria.andProvinceIdEqualTo(provinceId);
        return commCityMapper.selectByExample(example);
    }

    @Override
    public List<CommDistrict> findDistrictByCond(String cond) {
        CommDistrictExample example = new CommDistrictExample();
        example.createCriteria().andFullNameLike("%" + cond + "%");
        return commDistrictMapper.selectByExample(example);
    }

    @Override
    public List<CommDistrict> findDistrictByCityAndDistrictName(Long cityId, String districtName) {
        CommDistrictExample example = new CommDistrictExample();
        CommDistrictExample.Criteria criteria = example.createCriteria();
        criteria.andCityIdEqualTo(cityId);
        criteria.andNameLike("%" + districtName + "%");
        return commDistrictMapper.selectByExample(example);
    }

    @Override
    public List<CommDistrict> findDistrictByProvinceId(long id) {
        List<CommCity> cityList = findCityByProvinceId(id, true);
        List<CommDistrict> districtList = new LinkedList<CommDistrict>();
        for (CommCity commCity : cityList) {
            if(CollectionUtils.isNotEmpty(commCity.getDistrictList())){
                districtList.addAll(commCity.getDistrictList());
            }
        }
        if(CollectionUtils.isEmpty(districtList)){
            return Collections.emptyList();
        }else{
            return districtList;
        }
    }

    @Override
    public List<Long> findDistrictIdByProvinceId(long id) {
        List<CommDistrict> districtList = findDistrictByProvinceId(id);
        List<Long> idList = new ArrayList<Long>();
        if(CollectionUtils.isNotEmpty(districtList)){
            for (CommDistrict commDistrict : districtList) {
                idList.add(commDistrict.getId());
            }
        }
        if(CollectionUtils.isEmpty(idList)){
            return Collections.emptyList();
        }else{
            return idList;
        }
    }

	@Override
	public List<CommCountry> findForeignCountry(String countryName, boolean cascade) {
		return commCountryMapper.findForeignCountryByCond(countryName);
	}

	@Override
	public CommDistrict getDistrictIdByRegion(Region region) {
		
		CommProvince province = this.commProvinceMapper.selectByName(region.getProvince());
		if(EmptyUtil.isEmpty(province)) {
        CommAreaMatchVO commAreaMatchVO = commThirdPartyService.findMatchAreaByNameAndLevel(region.getProvince(), (short)2, 0);
        //从匹配表拿到对应的省
        if (EmptyUtil.isNotEmpty(commAreaMatchVO)) {
            province = this.commProvinceMapper.selectByPrimaryKey(commAreaMatchVO.getAreaId().longValue());
        } else {
            return null;
        }
		}
		Map<String, Object> parameter = new HashMap<String, Object>();
		parameter.put("id", province.getId());
		parameter.put("name", region.getCity());
		CommCity commCity = this.commCityMapper.selectByCityName(parameter);
		if(EmptyUtil.isEmpty(commCity)) {
        CommAreaMatchVO commAreaMatchVO = commThirdPartyService.findMatchAreaByNameAndLevel(region.getCity(), (short)3, province.getId().intValue());
        //从匹配表拿到对应的市
        if (EmptyUtil.isNotEmpty(commAreaMatchVO)) {
            commCity = commCityMapper.selectByPrimaryKey(commAreaMatchVO.getAreaId().longValue());
        } else {
            return null;
        }
		}
		parameter.put("id", commCity.getId());
		parameter.put("name", region.getDistrict());
		
		CommDistrict district = this.commDistrictMapper.selectByDistrictName(parameter);
		if(district == null) {
      CommAreaMatchVO commAreaMatchVO = commThirdPartyService.findMatchAreaByNameAndLevel(region.getDistrict(), (short)4, commCity.getId().intValue());
      if (EmptyUtil.isNotEmpty(commAreaMatchVO)) {
          district = commDistrictMapper.selectByPrimaryKey(commAreaMatchVO.getAreaId().longValue());
          return  district;
      }
			parameter.put("name", "其它区");
			district = this.commDistrictMapper.selectByDistrictName(parameter);
			if(district == null) {
				return district;
			}
			district.setMatch(false);
		}
		return district;
	}

    @Override
    public List<CommCity> getShopCity(List list) {
        return commCityMapper.getShopCity(list);
    }
}