/*
 * Copyright (C), 上海布鲁爱电子商务有限公司
 */
package com.thebeastshop.pegasus.util.comm;

import java.awt.*;
import java.awt.geom.AffineTransform;
import java.awt.image.AffineTransformOp;
import java.awt.image.BufferedImage;
import java.io.*;
import java.util.*;

import java.net.HttpURLConnection;
import java.net.URL;

import com.github.junrar.Archive;
import com.github.junrar.rarfile.FileHeader;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.ArrayUtils;
import org.apache.tools.zip.ZipEntry;
import org.apache.tools.zip.ZipFile;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.apache.tools.ant.Project;
import org.apache.tools.ant.taskdefs.Zip;
import org.apache.tools.ant.types.FileSet;
import org.apache.tools.zip.ZipEntry;
import org.apache.tools.zip.ZipOutputStream;

import javax.imageio.ImageIO;


/**
 * @author Eric.Lou
 * @version $Id: FileUtil.java, v 0.1 2015-07-28 22:21
 */
public class FileUtil {

    private static final int BUFFER = 1024;
    private static final Logger log = LoggerFactory.getLogger(FileUtil.class);

    private static final String[] IMAGE_NAMES = {
            "bmp"
            , "dib"
            , "gif"
            , "jfif"
            , "jpe"
            , "jpeg"
            , "jpg"
            , "png"
            , "tif"
            , "tiff"
            , "ico"};



    private static enum FILES {
    	PDF("pdf"),HTML("html"),EXCEL("excel"),DOC("doc"),ZIP("zip"),RAR("rar");
    	private String value;

		private FILES(String value) {
			this.value = value;
		}
    }
    /**
     * 根据文件路径判断其是否存在
     *
     * @param filePath
     * @return
     */
    public static Boolean exists(String filePath) {
        Boolean result = false;
        if (EmptyUtil.isNotEmpty(filePath)) {
            File file = new File(filePath);
            if (file.exists() && file.canRead()) {
                result = true;
            }
        }
        return result;
    }

    /**
     * 根据文件路径获取数据
     *
     * @param filePath
     * @return byte[]
     * @throws IOException
     */
    public static byte[] getByteFileData(String filePath) throws IOException {
        if (FileUtil.exists(filePath)) {
            File file = new File(filePath);
            return FileUtils.readFileToByteArray(file);
        } else {
            throw new IOException("file is not exist");
        }
    }

    /**
     * 把一个输入流转成文件
     *
     * @param in
     * @param filePath
     * @throws IOException
     */
    public static Boolean InputStreamTOFile(InputStream in, String filePath) throws IOException {
        if (EmptyUtil.isNotEmpty(in) && EmptyUtil.isNotEmpty(filePath)) {
            File file = new File(filePath);
            byte[] b = IOUtils.toByteArray(in);
            FileUtils.writeByteArrayToFile(file, b);
            IOUtils.closeQuietly(in);
            return true;
        }
        return false;
    }

    /**
     * 获取文件扩展名
     *
     * @param fileName 文件名
     * @return
     */
    public static String getExtName(String fileName) {
        if (EmptyUtil.isEmpty(fileName)) {
            return "";
        }
        if (!fileName.contains(".")) {
            return "";
        }
        return fileName.substring(fileName.lastIndexOf(".") + 1);
    }

    /**
     * 判断一个文件扩展名是否是图片格式
     *
     * @param extName 文件扩展名
     * @return
     */
    public static boolean isImage(String extName) {
        boolean result = false;
        if (EmptyUtil.isNotEmpty(extName)) {
            if (ArrayUtils.contains(IMAGE_NAMES, extName.toLowerCase()))
                result = true;
        }
        return result;
    }
    
    /**
     * 判断一个文件扩展名是否是图片格式
     *
     * @param extName 文件扩展名
     * @return
     */
    public static boolean isPdf(String extName) {
        boolean result = false;
        if (EmptyUtil.isNotEmpty(extName)) {
            if (extName.toLowerCase().equals(FILES.PDF.value))
                result = true;
        }
        return result;
    }

    /**
     * 判断一个文件扩展名是否是压缩格式(zip/rar)
     *
     * @param extName 文件扩展名
     * @return
     */
    public static boolean isCompressed(String extName) {
        boolean result = false;
        if (EmptyUtil.isNotEmpty(extName)) {
            if (extName.toLowerCase().equals(FILES.ZIP.value) ||
                    extName.toLowerCase().equals(FILES.RAR.value))
                result = true;
        }
        return result;
    }

    /**
     * 判断一个文件扩展名是否是RAR压缩格式
     *
     * @param extName 文件扩展名
     * @return
     */
    public static boolean isRAR(String extName) {
        boolean result = false;
        if (EmptyUtil.isNotEmpty(extName)) {
            if (extName.toLowerCase().equals(FILES.RAR.value))
                result = true;
        }
        return result;
    }

    /**
     * 判断一个文件扩展名是否是ZIP压缩格式
     *
     * @param extName 文件扩展名
     * @return
     */
    public static boolean isZIP(String extName) {
        boolean result = false;
        if (EmptyUtil.isNotEmpty(extName)) {
            if (extName.toLowerCase().equals(FILES.ZIP.value))
                result = true;
        }
        return result;
    }

    /**
     * 重置图片大小
     *
     * @param width
     * @param height
     * @param filePath
     * @return
     * @throws IOException
     */
    public static Boolean resize(int width, int height, String filePath) throws IOException {
        width = Math.max(width, 1);
        height = Math.max(height, 1);
        Image targetImage = ImageIO.read(new File(filePath));
        BufferedImage image = new BufferedImage(width, height,
                BufferedImage.TYPE_INT_RGB);
        image.getGraphics().drawImage(targetImage, 0, 0, width, height, Color.WHITE, null);
        String formatName = filePath.substring(filePath.lastIndexOf(".") + 1);
        return ImageIO.write(image, formatName, new File(filePath));
    }


    /**
     * 重置图片大小并补白
     * @param width
     * @param height
     * @param filePath
     * @param fillWhite
     */
    public static void resize(int width, int height, String filePath, boolean fillWhite) {
        try {
            double ratio = 0.0; //缩放比例
            File file = new File(filePath);
            BufferedImage tempBufferedImage = ImageIO.read(file);
            Image targetImage = tempBufferedImage.getScaledInstance(width, height, tempBufferedImage.SCALE_SMOOTH);
            //计算比例
            if ((tempBufferedImage.getHeight() > height) || (tempBufferedImage.getWidth() > width)) {
                if (tempBufferedImage.getHeight() > tempBufferedImage.getWidth()) {
                    ratio = (new Integer(height)).doubleValue() / tempBufferedImage.getHeight();
                } else {
                    ratio = (new Integer(width)).doubleValue() / tempBufferedImage.getWidth();
                }
                AffineTransformOp op = new AffineTransformOp(AffineTransform.getScaleInstance(ratio, ratio), null);
                targetImage = op.filter(tempBufferedImage, null);
            }
            //是否补白
            if (fillWhite) {
                BufferedImage image = new BufferedImage(width, height, BufferedImage.TYPE_INT_RGB);
                Graphics2D graphics2D = image.createGraphics();
                graphics2D.setColor(Color.white);
                graphics2D.fillRect(0, 0, width, height);
                if (width == targetImage.getWidth(null)) {
                    graphics2D.drawImage(targetImage, 0, (height - targetImage.getHeight(null)) / 2, targetImage.getWidth(null), targetImage.getHeight(null), Color.white, null);
                } else {
                    graphics2D.drawImage(targetImage, (width - targetImage.getWidth(null)) / 2, 0, targetImage.getWidth(null), targetImage.getHeight(null), Color.white, null);
                }
                graphics2D.dispose();
                targetImage = image;
            }
            ImageIO.write((BufferedImage) targetImage, "jpg", file);
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    public static java.util.List<File> unCompressedFile(File originFile,String parentPath) {
        if (!originFile.exists()){
            return null;
        }
        if (isZIP(getExtName(originFile.getName()))){
            return unZipFile(originFile,parentPath);
        }else if(isRAR(getExtName(originFile.getName()))){
            return unRarFile(originFile,parentPath);
        }
        return null;
    }
    /**
     * 压缩一个目录
     * @param dir
     * @param out
     * @param basedir
     */
    private void compressDirectory(File dir, ZipOutputStream out, String basedir) {
        if (!dir.exists()){
            return;
        }
        File[] files = dir.listFiles();
        for (int i = 0; i < files.length; i++) {
            /* 递归 */
            compressByType(files[i], out, basedir + dir.getName() + "/");
        }
    }

    /**
     * 压缩一个文件
     * @param file
     * @param out
     * @param basedir
     */
    private void compressFile(File file, ZipOutputStream out, String basedir) {
        if (!file.exists()) {
            return;
        }
        try {
            BufferedInputStream bis = new BufferedInputStream(new FileInputStream(file));
            ZipEntry entry = new ZipEntry(basedir + file.getName());
            out.putNextEntry(entry);
            int count;
            byte data[] = new byte[BUFFER];
            while ((count = bis.read(data, 0, BUFFER)) != -1) {
                out.write(data, 0, count);
            }
            bis.close();
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * 判断是目录还是文件，根据类型（文件/文件夹）执行不同的压缩方法
     * @param file
     * @param out
     * @param basedir
     */
    private void compressByType(File file, ZipOutputStream out, String basedir) {
        /* 判断是目录还是文件 */
        if (file.isDirectory()) {
            this.compressDirectory(file, out, basedir);
        } else {
            this.compressFile(file, out, basedir);
        }
    }

    /**
     * 执行压缩操作
     * @param srcPathName  需要被压缩的文件/文件夹
     * @param zipFile  生成的压缩文件
     */
    public static boolean compressZip(String srcPathName, File zipFile) {
        File srcdir = new File(srcPathName);
        try{
            if (!srcdir.exists()){
                throw new RuntimeException(srcPathName + "不存在！");
            }
            Project prj = new Project();
            Zip zip = new Zip();
            zip.setProject(prj);
            zip.setDestFile(zipFile);
            FileSet fileSet = new FileSet();
            fileSet.setProject(prj);
            fileSet.setDir(srcdir);
            //fileSet.setIncludes("**/*.java"); //包括哪些文件或文件夹 eg:zip.setIncludes("*.java");
            //fileSet.setExcludes(...); //排除哪些文件或文件夹
            zip.addFileset(fileSet);
            zip.execute();
            return true;
        }catch (Exception e){
            return false;
        }
    }

    public static void deleteCurrFile(String file) {
        deleteFile(new File(file));
    }

    //递归删除文件夹
    public static void deleteFile(File file) {
        if (file.exists()) {//判断文件是否存在
            if (file.isFile()) {//判断是否是文件
                file.delete();//删除文件
            } else if (file.isDirectory()) {//否则如果它是一个目录
                File[] files = file.listFiles();//声明目录下所有的文件 files[];
                for (int i = 0;i < files.length;i ++) {//遍历目录下所有的文件
                    deleteFile(files[i]);//把每个文件用这个方法进行迭代
                }
                file.delete();//删除文件夹
            }
        }
    }

    public static void downloadPic(String picUrl,String parentPath,String fileName){
        File rootFile = new File(parentPath);
        if (!rootFile.exists()){
            rootFile.mkdirs();
        }
        downloadPic(picUrl,parentPath+fileName);
    }
    /**
     * @param picUrl
     */
    public static void downloadPic(String picUrl,String destPath){
        FileOutputStream outStream = null;
        try{
            URL url = new URL(picUrl);
            //打开链接
            HttpURLConnection conn = (HttpURLConnection)url.openConnection();
            //设置请求方式为"GET"
            conn.setRequestMethod("GET");
            //超时响应时间为5秒
            conn.setConnectTimeout(5 * 1000);
            //通过输入流获取图片数据
            InputStream inStream = conn.getInputStream();
            //得到图片的二进制数据，以二进制封装得到数据，具有通用性
            byte[] data = readInputStream(inStream);
            //new一个文件对象用来保存图片，默认保存当前工程根目录
            File imageFile = new File(destPath);
            //创建输出流
            outStream = new FileOutputStream(imageFile);
            //写入数据
            outStream.write(data);
            //关闭输出流
            outStream.close();
        }catch(Exception e){
            e.printStackTrace();
        }finally {
            if (outStream != null){
                try {
                    outStream.close();
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
        }
    }

    public static byte[] readInputStream(InputStream inStream) throws Exception{
        ByteArrayOutputStream outStream = new ByteArrayOutputStream();
        //创建一个Buffer字符串
        byte[] buffer = new byte[1024];
        //每次读取的字符串长度，如果为-1，代表全部读取完毕
        int len = 0;
        //使用一个输入流从buffer里把数据读取出来
        while( (len=inStream.read(buffer)) != -1 ){
            //用输出流往buffer里写入数据，中间参数代表从哪个位置开始读，len代表读取的长度
            outStream.write(buffer, 0, len);
        }
        //关闭输入流
        inStream.close();
        //把outStream里的数据写入内存
        return outStream.toByteArray();
    }


    /**
     * 根据原始rar路径，解压到指定文件夹下.
     * @param originfile 原始rar路径
     */
    public static synchronized java.util.List<File> unRarFile(File originfile,String parentPath) {
        if (!originfile.exists()){
            return null;
        }
        java.util.List<File> files = new ArrayList<File>();
        Archive archive = null;
        try {
            archive = new Archive(originfile);
            if (archive != null) {
                FileHeader fh = archive.nextFileHeader();
                while (fh != null) {
                    if (fh.isDirectory()) { // 文件夹
                        fh = archive.nextFileHeader();
                        continue;
                    } else { // 文件
                        String fileName=fh.getFileNameW();// 中文名称
                        if (EmptyUtil.isEmpty(fileName)){
                            fileName = fh.getFileNameString();
                        }
                        // 过滤掉mac压缩文件时，产生的临时文件
                        if (fileName.contains("__MACOSX")){
                            continue;
                        }
                        File unRarFile = new File(parentPath,fileName);
                        if (!new File(unRarFile.getParent()).exists()){
                            new File(unRarFile.getParent()).mkdirs();
                        }
                        try {// 之所以这么写try，是因为万一这里面有了异常，不影响继续解压.
                            FileOutputStream os = new FileOutputStream(unRarFile);
                            archive.extractFile(fh, os);
                            os.close();
                        } catch (Exception ex) {
                            ex.printStackTrace();
                        }
                        files.add(unRarFile);
                        fh = archive.nextFileHeader();
                    }
                }
            }else{
                return null;
            }
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            if (archive != null){
                try {
                    archive.close();
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
        }
        return files;
    }

    private static void ergodicZipFile(File originfile,String encoding,java.util.List<File> files,String parentPath){
        ZipFile zipFile = null;
        BufferedOutputStream bout=null;
        try {
            log.info("encodeing>>"+encoding);
            if (encoding != null){
                zipFile = new ZipFile(originfile,encoding);
            }else{
                zipFile = new ZipFile(originfile);
            }
            for (Enumeration<? extends ZipEntry> e = zipFile.getEntries(); e.hasMoreElements();) {
                ZipEntry entry = e.nextElement();
                if (entry.isDirectory()) {
                    continue;
                }
                // 过滤掉mac压缩文件时，产生的临时文件
                if (entry.getName().toUpperCase().contains("__MACOSX")) {
                    continue;
                }
                File unZipFile = new File(parentPath,entry.getName());
                if (!new File(unZipFile.getParent()).exists()) {
                    new File(unZipFile.getParent()).mkdirs();
                }
                bout = new BufferedOutputStream(new FileOutputStream(unZipFile));
                byte[] buf1 = new byte[1024];
                int len;
                InputStream in = zipFile.getInputStream(entry);
                while ((len = in.read(buf1)) > 0) {
                    bout.write(buf1, 0, len);
                }
                bout.close();
                files.add(unZipFile);
            }
        }catch (Exception e1) {
                //e1.printStackTrace();
            files.clear();
            ergodicZipFile(originfile,"GBK",files,parentPath);
        }finally {
            if(bout != null){
                try {
                    bout.close();
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
        }
    }

    public static synchronized java.util.List<File> unZipFile(File originfile,String parentPath){
        String encodeing = System.getProperty("sun.jnu.encoding");
        System.setProperty("sun.zip.encoding", encodeing);
        if (!originfile.exists()){
            return null;
        }
        java.util.List<File> files = new ArrayList<File>();
        // 默认按GBK解压
        ergodicZipFile(originfile,null,files,parentPath);
        return files;
    }

    /**
     * 删除单个文件
     * @param   sPath    被删除文件的文件名
     * @return 单个文件删除成功返回true，否则返回false
     */
    public static boolean deleteFile(String sPath) {
        boolean flag = false;
        File file = new File(sPath);
        // 路径为文件且不为空则进行删除
        if (file.isFile() && file.exists()) {
            file.delete();
            flag = true;
        }
        return flag;
    }

    /**
     * 删除目录（文件夹）以及目录下的文件
     * @param   sPath 被删除目录的文件路径
     * @return  目录删除成功返回true，否则返回false
     */
    public static boolean deleteDirectory(String sPath) {
        //如果sPath不以文件分隔符结尾，自动添加文件分隔符
        if (!sPath.endsWith(File.separator)) {
            sPath = sPath + File.separator;
        }
        File dirFile = new File(sPath);
        //如果dir对应的文件不存在，或者不是一个目录，则退出
        if (!dirFile.exists() || !dirFile.isDirectory()) {
            return false;
        }
        boolean flag = true;
        //删除文件夹下的所有文件(包括子目录)
        File[] files = dirFile.listFiles();
        for (int i = 0; i < files.length; i++) {
            //删除子文件
            if (files[i].isFile()) {
                flag = deleteFile(files[i].getAbsolutePath());
                if (!flag) break;
            } //删除子目录
            else {
                flag = deleteDirectory(files[i].getAbsolutePath());
                if (!flag) break;
            }
        }
        if (!flag) return false;
        //删除当前目录
        if (dirFile.delete()) {
            return true;
        } else {
            return false;
        }
    }

    public static void main(String[] args) {
        String parentPath = System.getProperty("java.io.tmpdir")+System.currentTimeMillis()+File.separator;
        //java.util.List<File> files = unCompressedFile(new File("C:\\上传图片\\归档(1).zip"),parentPath);
        java.util.List<File> files = unCompressedFile(new File("C:\\测试.zip"),parentPath);
        if (files != null){
            boolean needDel = true;
            if (needDel) {
                System.out.println(parentPath);
                boolean ok = deleteDirectory(parentPath);
                System.out.println(ok);
            }
        }

        //downloadPic("http://beasttest.oss-cn-hangzhou.aliyuncs.com/image/测试.jpg","D:/Documents/Pictures/test/abc.jpg");
    }
}
