package com.thebeastshop.support.util;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Objects;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.thebeastshop.support.mark.HasIdGetter;
import com.thebeastshop.support.mark.HasIdGetter.HasParent;

/**
 * 关于HasId系列接口的工具类。
 * 
 * @author Paul-xiong
 *
 */
public class IdUtil {
	/**
	 * id升序比较器。
	 */
	private static Comparator<HasIdGetter<?>> ID_ASC = new Comparator<HasIdGetter<?>>() {
		@Override
		public int compare(com.thebeastshop.support.mark.HasIdGetter<?> o1,
				com.thebeastshop.support.mark.HasIdGetter<?> o2) {
			return Long.compare(Long.valueOf(o1.getId().toString()), Long.valueOf(o2.getId().toString()));
		};
	};
	/**
	 * id降序比较器。
	 */
	private static Comparator<HasIdGetter<?>> ID_DESC = new Comparator<HasIdGetter<?>>() {
		@Override
		public int compare(com.thebeastshop.support.mark.HasIdGetter<?> o1,
				com.thebeastshop.support.mark.HasIdGetter<?> o2) {
			return Long.compare(Long.valueOf(o2.getId().toString()), Long.valueOf(o1.getId().toString()));
		};
	};

	/**
	 * 将一个${@link HasIdGetter}的集合转换成它们的id的集合。
	 *
	 * @param source
	 * @param <I>
	 * @return
	 */
	public static <I extends Comparable<I>> List<I> toIds(final Collection<? extends HasIdGetter<I>> source) {
		List<I> list = Lists.newArrayList();
		for (HasIdGetter<I> i : source) {
			list.add(i.getId());
		}
		return list;
	}

	/**
	 * 在一个id列表中检查是否包含指定实体的id。
	 *
	 * @param entity
	 * @param ids
	 * @param <I>
	 * @return
	 */
	@SafeVarargs
	public static <I extends Comparable<I>> boolean contains(final HasIdGetter<I> entity, final I... ids) {
		return contains(entity, Arrays.asList(ids));
	}

	/**
	 * 在一个id列表中检查是否包含指定实体的id。
	 *
	 * @param entity
	 * @param ids
	 * @param <I>
	 * @return
	 */
	public static <I extends Comparable<I>> boolean contains(final HasIdGetter<I> entity, final Collection<I> ids) {
		return ids.contains(entity.getId());
	}

	/**
	 * 在一个实体列表中检查是否包含指定id的实体
	 *
	 * @param id
	 * @param entities
	 * @param <I>
	 * @return
	 */
	@SafeVarargs
	public static <I extends Comparable<I>> boolean contains(final I id, final HasIdGetter<I>... entities) {
		return contains(id, Arrays.asList(entities));
	}

	/**
	 * 在一个实体列表中检查是否包含指定id的实体
	 *
	 * @param id
	 * @param entities
	 * @param <I>
	 * @return
	 */
	public static <I extends Comparable<I>> boolean contains(final I id,
			final Collection<? extends HasIdGetter<I>> entities) {
		return toIds(entities).contains(id);
	}

	/**
	 * 将一个{@link HasParent}的集合转换成它们的父对象id的集合。
	 *
	 * @param source
	 * @param <I>
	 * @return
	 */
	public static <I extends Comparable<I>> Collection<I> toParentIds(final Collection<? extends HasParent<I>> source) {
		List<I> list = Lists.newArrayList();
		for (HasParent<I> i : source) {
			list.add(i.getParentId());
		}
		return list;
	}

	/**
	 * 将一个${@link HasIdGetter}的集合转换成一个以id为键，对象本身为值的${@link Map}。
	 *
	 * @param source
	 * @param <I>
	 * @param <T>
	 * @return
	 */
	public static <I extends Comparable<I>, T extends HasIdGetter<I>> Map<I, T> map(final Iterable<T> source) {
		final Map<I, T> map = Maps.newLinkedHashMap();
		for (final T t : source) {
			map.put(t.getId(), t);
		}
		return map;
	}

	public static int hashCode(final HasIdGetter<?> hasIdGetter) {
		return hasIdGetter == null || hasIdGetter.getId() == null ? 0 : hasIdGetter.getId().hashCode();
	}

	public static boolean equalsUsingId(final HasIdGetter<?> hasIdGetter1, final HasIdGetter<?> hasIdGetter2) {
		return equalsUsingId(hasIdGetter1, hasIdGetter2, false);
	}

	public static boolean equalsUsingId(final HasIdGetter<?> hasIdGetter1, final HasIdGetter<?> hasIdGetter2,
			final boolean nullEquals) {
		if (hasIdGetter1 == null) {
			if (hasIdGetter2 == null) {
				return nullEquals;
			} else {
				return false;
			}
		}
		if (hasIdGetter2 == null) {// hasIdGetter1 != null
			return false;
		}

		final Object id1 = hasIdGetter1.getId();
		final Object id2 = hasIdGetter2.getId();
		if (id1 == null) {
			if (id2 == null) {
				return nullEquals;
			} else {
				return false;
			}
		}
		if (id2 == null) {// id1 != null
			return false;
		}
		return id1.equals(id2);
	}

	/**
	 * 对一个${@link HasIdGetter}的集合根据id属性进行升序排序。
	 *
	 * @param source
	 * @param <T>
	 * @return
	 */
	public static <T extends HasIdGetter<?>> List<T> idAsc(final Collection<T> source) {
		return sort(source, ID_ASC);
	}

	/**
	 * 对一个${@link HasIdGetter}的集合根据id属性进行降序排序。
	 *
	 * @param source
	 * @param <T>
	 * @return
	 */
	public static <T extends HasIdGetter<?>> List<T> idDesc(final Collection<T> source) {
		return sort(source, ID_DESC);
	}

	public static <T extends HasIdGetter<?>> T findById(final Object id, final Collection<T> source) {
		for (final T t : source) {
			if (Objects.equals(id, t.getId())) {
				return t;
			}
		}
		return null;
	}

	/**
	 * 对一个${@link HasIdGetter}的集合根据指定的排序器进行排序。
	 *
	 * @param source
	 * @param comparator
	 * @param <T>
	 * @return
	 */
	private static <T extends HasIdGetter<?>> List<T> sort(final Collection<T> source,
			final Comparator<HasIdGetter<?>> comparator) {
		final List<T> list;
		if (source instanceof List) {
			list = (List<T>) source;
		} else {
			list = Lists.newArrayList(source);
		}
		Collections.sort(list, comparator);
		return list;
	}

}
