package com.thebeastshop.support.coll;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import com.google.common.base.Function;
import com.google.common.collect.Collections2;
import com.google.common.collect.Maps;
import com.thebeastshop.support.exception.WrongArgException;
import com.thebeastshop.support.mark.Unmodifiable;

/**
 * @author Liang Wenjian
 */
public class CollectionUtil {

	public static final Function<Object, String> TO_STRING = new Function<Object, String>() {
		@Override
		public String apply(Object input) {
			return input.toString();
		};
	};

	public static Collection<String> toString(final Collection<?> source) {
		return Collections2.transform(source, TO_STRING);
	}

	public static boolean isUnmodifiable(final Object obj) {
		if (obj == null) {
			return true;
		}
		if (obj instanceof Unmodifiable) {
			return true;
		}
		for (final Class<?> knownUnmodifiableClass : KNOWN_UNMODIFIABLE_CLASSES) {
			if (knownUnmodifiableClass.isAssignableFrom(obj.getClass())) {
				return true;
			}
		}
		return false;
	}

	private static final Collection<Class<?>> KNOWN_UNMODIFIABLE_CLASSES = Arrays.asList(//
			Collections.unmodifiableCollection(Collections.emptyList()).getClass(), //
			Collections.emptyList().getClass(), //
			Collections.emptySet().getClass(), //
			Collections.emptyMap().getClass(), //
			Collections.emptyEnumeration().getClass()//
	);

	private CollectionUtil() {
		super();
	}

	@SuppressWarnings("unchecked")
	public static <T> List<T> avoidNull(final List<T> list) {
		return (List<T>) (list == null ? Collections.emptyList() : list);
	}

	public static <K, V> Map<K, V> map(final List<K> keys, final List<V> values) {
		final int size = keys.size();
		if (size != values.size()) {
			throw new WrongArgException("size", size);
		}
		final Map<K, V> rt = Maps.newLinkedHashMap();
		for (int i = 0; i < size; i++) {
			rt.put(keys.get(i), values.get(i));
		}
		return rt;
	}
}
