package com.thebeastshop.support.util.json.jackson;

import com.fasterxml.jackson.core.JsonFactory;
import com.fasterxml.jackson.core.JsonParser.Feature;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.JavaType;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.thebeastshop.support.exception.SerializationException;

import java.io.IOException;
import java.lang.reflect.Type;
import java.nio.charset.Charset;
import java.util.List;
import java.util.Map;

final public class Jacksons {
    private static final JsonFactory JSON_FACTORY = new JsonFactory();
    private static final Jacksons DEFAULT = new Jacksons(null);

    private final MapperConfigurator mapperConfigurator;

    private Jacksons(final MapperConfigurator mapperConfigurator) {
        this.mapperConfigurator = mapperConfigurator;
    }

    public static Jacksons defaultJacksons() {
        return DEFAULT;
    }

    public static Jacksons create(final MapperConfigurator mapperConfigurator) {
        return new Jacksons(mapperConfigurator);
    }

    private static ObjectMapper newMapper() {
        final ObjectMapper mapper = new ObjectMapper(JSON_FACTORY);
        mapper.configure(Feature.ALLOW_SINGLE_QUOTES, true);
        mapper.configure(Feature.ALLOW_UNQUOTED_CONTROL_CHARS, true);
//        mapper.configure(JsonParser.Feature.INTERN_FIELD_NAMES, false);
//        mapper.configure(JsonParser.Feature.CANONICALIZE_FIELD_NAMES, false);
        mapper.configure(DeserializationFeature.ACCEPT_SINGLE_VALUE_AS_ARRAY, true);
        mapper.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
        mapper.getTypeFactory().constructArrayType(List.class);
        return mapper;
    }

    private ObjectMapper getMapper() {
        final ObjectMapper mapper = newMapper();
        if (mapperConfigurator != null) {
            mapperConfigurator.configureMapper(mapper);
        }
        return mapper;
    }

    public <T> T toObject(final String jsonString, final Class<?> parametrized, final Class<?>... parameterClasses) {
        final ObjectMapper mapper = getMapper();
        try {
/*          jackson 2.7.2在mapper.getTypeFactory().constructParametricType这里有个bug:
            当调用objectMapper.getTypeFactory().constructParametricType(parametrized,parameterClasses);
            Exception in thread "main" java.lang.NullPointerException
            at com.fasterxml.jackson.databind.type.TypeFactory._fromVariable(TypeFactory.java:1353)
            at com.fasterxml.jackson.databind.type.TypeFactory._fromAny(TypeFactory.java:1120)
            at com.fasterxml.jackson.databind.type.TypeFactory._fromParamType(TypeFactory.java:1336)
            at com.fasterxml.jackson.databind.type.TypeFactory._fromAny(TypeFactory.java:1110)
            at com.fasterxml.jackson.databind.type.TypeFactory._resolveSuperInterfaces(TypeFactory.java:1254)
            at com.fasterxml.jackson.databind.type.TypeFactory._fromClass(TypeFactory.java:1198)
            at com.fasterxml.jackson.databind.type.TypeFactory.constructParametricType(TypeFactory.java:841)

            TypeFactory.java:841是这样的
            pt[i] = _fromClass(null, parameterClasses[i], null);
            给_fromClass方法的第三个参数（TypeBindings bindings）传递的是null，之后一直传递到TypeFactory._fromVariable（TypeFactory.java:1353），没有对bindings参数进行非null检查，就直接用了：
            JavaType type = bindings.findBoundType(name);
            导致空指针异常
*/
            final JavaType javaType = mapper.getTypeFactory().constructParametricType(parametrized, parameterClasses);
            return mapper.readValue(jsonString, javaType);
        } catch (final IOException e) {
            throw new SerializationException(e, "jsonString", jsonString);
        }
    }

    public <T> T toObject(final String jsonString, final Type type) {
        final ObjectMapper mapper = getMapper();
        try {
            final JavaType javaType = mapper.getTypeFactory().constructType(type);
            return mapper.readValue(jsonString, javaType);
        } catch (final IOException e) {
            throw new SerializationException(e, "jsonString", jsonString);
        }
    }

    public <T> T toObject(final String jsonString, final TypeConstructor typeConstructor) {
        final ObjectMapper mapper = getMapper();
        try {
            return mapper.readValue(jsonString, typeConstructor.constructType(mapper));
        } catch (final IOException e) {
            throw new SerializationException(e, "jsonString", jsonString);
        }
    }

    public <T> T toObject(final byte[] jsonBytes, final Class<?> parametrized, final Class<?>... parameterClasses) {
        final ObjectMapper mapper = getMapper();
        try {
            final JavaType javaType = mapper.getTypeFactory().constructParametricType(parametrized, parameterClasses);
            return mapper.readValue(jsonBytes, javaType);
        } catch (final IOException e) {
            throw new SerializationException(e, "jsonBytes", jsonBytes);
        }
    }

    public <T> T toObject(final byte[] jsonBytes, final TypeConstructor typeConstructor) {
        final ObjectMapper mapper = getMapper();
        try {
            return mapper.readValue(jsonBytes, typeConstructor.constructType(mapper));
        } catch (final IOException e) {
            throw new SerializationException(e, "jsonBytes", jsonBytes);
        }
    }

    public <K, V> List<Map<K, V>> toMaps(final String jsonString) {
        return toObject(jsonString, List.class, Map.class);
    }

    public <K, V> List<Map<K, V>> toMaps(final byte[] jsonBytes) {
        return toObject(jsonBytes, List.class, Map.class);
    }

    @SuppressWarnings("unchecked")
    public <K, V> Map<K, V> toMap(final Object object) {
        final ObjectMapper mapper = getMapper();
        try {
            return mapper.convertValue(object, Map.class);
        } catch (final Exception e) {
            throw new SerializationException(e, "object", object);
        }
    }

    public <K, V> Map<K, V> toMap(final String jsonString) {
        return toObject(jsonString, Map.class, String.class, Object.class);
    }

    public <K, V> Map<K, V> toMap(final byte[] jsonBytes) {
        return toObject(jsonBytes, Map.class, String.class, Object.class);
    }

    public <T> List<T> toBeans(final String jsonString, final Class<T> clazz) {
        return toObject(jsonString, List.class, clazz);
    }

    public <T> List<T> toBeans(final byte[] jsonBytes, final Class<T> clazz) {
        return toObject(jsonBytes, List.class, clazz);
    }

    public <T> T toBean(final String jsonStr, final Class<T> clazz) {
        return toObject(jsonStr, clazz);
    }

    public <T> T toBean(final byte[] jsonBytes, final Class<T> clazz) {
        return toObject(jsonBytes, clazz);
    }

    public String toString(final Object object) {
        final ObjectMapper mapper = getMapper();
        try {
            return mapper.writeValueAsString(object);
        } catch (final IOException e) {
            throw new SerializationException(e, "object", object);
        }
    }

    public String toString(final JsonNode jsonNode) {
        final ObjectMapper mapper = getMapper();
        try {
            return mapper.writeValueAsString(jsonNode);
        } catch (final IOException e) {
            throw new SerializationException(e, "jsonNode", jsonNode);
        }
    }

    public byte[] toBytes(final Object object) {
        final ObjectMapper mapper = getMapper();
        try {
            return mapper.writeValueAsBytes(object);
        } catch (final IOException e) {
            throw new SerializationException(e, "object", object);
        }
    }

    public byte[] toBytes(final JsonNode jsonNode) {
        return toString(jsonNode).getBytes(Charset.forName("UTF-8"));
    }

    public JsonNode toJsonNode(final Object object) {
        final ObjectMapper mapper = getMapper();
        try {
            final byte[] jsonBytes = mapper.writeValueAsBytes(object);
            return mapper.readTree(jsonBytes);
        } catch (final IOException e) {
            throw new SerializationException(e, "object", object);
        }
    }

    public JsonNode toJsonNode(final byte[] jsonBytes) {
        final ObjectMapper mapper = getMapper();
        try {
            return mapper.readTree(jsonBytes);
        } catch (final IOException e) {
            throw new SerializationException(e, "jsonBytes", jsonBytes);
        }
    }

    public JsonNode toJsonNode(final String jsonString) {
        final ObjectMapper mapper = getMapper();
        try {
            return mapper.readTree(jsonString);
        } catch (final IOException e) {
            throw new SerializationException(e, "jsonString", jsonString);
        }
    }
}
