package com.thebeastshop.support.util;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.collections.CollectionUtils;
import org.slf4j.LoggerFactory;

import com.google.common.collect.Maps;
import com.thebeastshop.support.exception.UnknownException;
import com.thebeastshop.support.mark.HasPrice;

/**
 * 价格计算工具
 * 
 * @author Paul-xiong
 *
 */
public class PriceUtil {

	/**
	 * 保留到分，厘位四舍五入
	 *
	 * @return
	 */
	public static BigDecimal keepToCent(final BigDecimal d) {
		return d.setScale(2, RoundingMode.HALF_UP);
	}

	public static BigDecimal keepToCent(final HasPrice hasPrice) {
		return keepToCent(hasPrice.getPrice());
	}

	/**
	 * 计算总价，四舍五入保留到分
	 *
	 * @param price
	 * @param count
	 * @return
	 */
	public static BigDecimal multiPrice(final BigDecimal price, final int count) {
		return keepToCent(price.multiply(BigDecimal.valueOf(count)));
	}

	/**
	 * 计算总价
	 *
	 * @param source
	 * @return
	 */
	public static BigDecimal sumPrice(final Collection<? extends HasPrice> source) {
		BigDecimal rt = BigDecimal.ZERO;
		for (final HasPrice t : source) {
			rt = rt.add(t.getPrice());
		}
		return keepToCent(rt);
	}

	/**
	 * 分摊价格
	 * 
	 * @param source
	 * @param price
	 * @return
	 */
	public static Map<HasPrice, BigDecimal> sharePrice(List<? extends HasPrice> source, BigDecimal price) {
		if (CollectionUtils.isNotEmpty(source)) {
			Map<HasPrice, BigDecimal> origin = Maps.newHashMap();
			for (HasPrice t : source) {
				origin.put(t, t.getPrice());
			}
			return sharePrice(origin, price);
		}
		return null;
	}

	/**
	 * 分摊价格
	 * 
	 * @param source
	 * @param price
	 * @return
	 */
	public static <T> Map<T, BigDecimal> sharePrice(Map<T, BigDecimal> source, BigDecimal price) {
		Map<T, BigDecimal> rt = Maps.newHashMap(source);
		BigDecimal allPrice = BigDecimal.ZERO;
		BigDecimal balancePrice = price;
		Set<T> keySet = source.keySet();
		if (CollectionUtils.isNotEmpty(keySet)) {
			for (Object key : keySet) {
				if (key == null) {
					LoggerFactory.getLogger(PriceUtil.class).error("谁把null放进来了:" + source);
					throw new UnknownException();
				}
				allPrice = allPrice.add(source.get(key));
			}
			Iterator<T> iterator = keySet.iterator();
			while (iterator.hasNext()) {
				T t = iterator.next();
				BigDecimal originPrice = source.get(t);
				if (iterator.hasNext()) {
					BigDecimal sharedPrice = keepToCent(
							originPrice.multiply(price).divide(allPrice, 10, RoundingMode.HALF_UP));
					balancePrice = balancePrice.subtract(sharedPrice);
					rt.put(t, sharedPrice);
				} else {
					rt.put(t, balancePrice);
				}
			}
		}
		return rt;
	}

	/**
	 * 分摊折扣
	 * 
	 * @param source
	 * @param discount
	 * @return
	 */
	public static Map<HasPrice, BigDecimal> shareDiscount(List<? extends HasPrice> source, BigDecimal discount) {
		if (CollectionUtils.isNotEmpty(source)) {
			Map<HasPrice, BigDecimal> origin = Maps.newHashMap();
			for (HasPrice t : source) {
				origin.put(t, t.getPrice());
			}
			return shareDiscount(origin, discount);
		}
		return null;
	}

	/**
	 * 分摊折扣
	 * 
	 * @param source
	 * @param dicount
	 * @return
	 */
	public static <T> Map<T, BigDecimal> shareDiscount(Map<T, BigDecimal> source, BigDecimal dicount) {
		Map<T, BigDecimal> rt = Maps.newHashMap(source);
		BigDecimal allPrice = BigDecimal.ZERO;
		BigDecimal balancePrice = dicount;
		Set<T> keySet = source.keySet();
		if (CollectionUtils.isNotEmpty(keySet)) {
			for (Object key : keySet) {
				if (key == null) {
					LoggerFactory.getLogger(PriceUtil.class).error("谁把null放进来了:" + source);
					throw new UnknownException();
				}
				allPrice = allPrice.add(source.get(key));
			}
			Iterator<T> iterator = keySet.iterator();
			while (iterator.hasNext()) {
				T t = iterator.next();
				BigDecimal originPrice = source.get(t);
				if (iterator.hasNext()) {
					BigDecimal sharedPrice = keepToCent(
							originPrice.multiply(dicount).divide(allPrice, 10, RoundingMode.HALF_UP));
					balancePrice = balancePrice.subtract(sharedPrice).compareTo(BigDecimal.ZERO) > 0
							? balancePrice.subtract(sharedPrice) : BigDecimal.ZERO;
					BigDecimal price = originPrice.subtract(sharedPrice).compareTo(BigDecimal.ZERO) > 0
							? keepToCent(originPrice.subtract(sharedPrice)) : BigDecimal.ZERO;
					rt.put(t, price);
				} else {
					BigDecimal price = originPrice.subtract(balancePrice).compareTo(BigDecimal.ZERO) > 0
							? keepToCent(originPrice.subtract(balancePrice)) : BigDecimal.ZERO;
					rt.put(t, price);
				}
			}
		}
		return rt;
	}

}
