package com.thebeastshop.support.util;

import java.util.EnumSet;
import java.util.Objects;

import com.thebeastshop.support.mark.HasIdGetter;

/**
 * @author Liang Wenjian
 */
public class EnumUtil {

    /**
     * 根据id获取某枚举类的枚举值。
     *
     * @param id
     * @param clazz 实现了HasIdGetter接口的枚举类
     * @param <I>
     * @param <E>
     * @return
     */
    public static <I extends Comparable<I>, E extends Enum<E> & HasIdGetter<I>> E valueOf(final I id,
                                                                                          final Class<E> clazz) {
        return valueOf(id, clazz.getEnumConstants());
    }

    /**
     * 从某枚举类的枚举值列表中获取第一个id等于指定id的枚举值
     * 例子：AccessWay a=EnumUtil.valueOf(1,AccessWay.values());
     *
     * @param id
     * @param source
     * @param <I>
     * @param <E>
     * @return
     */
    @SafeVarargs
    public static <I extends Comparable<I>, E extends Enum<E> & HasIdGetter<I>> E valueOf(final I id,
                                                                                          final E... source) {
        for (final E e : source) {
            if (Objects.equals(e.getId(), id)) {
                return e;
            }
        }
        return null;
    }

    /**
     * 指定一个二进制数字，获取某枚举类的一些枚举值，以它们的id为下标（右起），在该二进制数字中，该位的值是1。
     *
     * @param bits
     * @param clazz
     * @param <I>
     * @param <E>
     * @return
     */
    public static <I extends Number & Comparable<I>, E extends Enum<E> & HasIdGetter<I>> EnumSet<E> bitsToEnums(
            final long bits, final Class<E> clazz) {
        final EnumSet<E> rt = EnumSet.noneOf(clazz);
        if (bits == 0) {
            return rt;
        }
        for (final E e : clazz.getEnumConstants()) {
            final I id = (I) e.getId();
            final long bitIndex = 1L << id.longValue();
            if ((bits & bitIndex) != 0) {
                rt.add(e);
            }
        }
        return rt;
    }

}
