package com.thebeastshop.support.mark;

import com.google.common.base.Function;
import com.thebeastshop.support.mark.HasIdGetter.HasParent;

/**
 * 实现本接口意味着拥有id属性。
 *
 * @param <I>
 *            id属性的类型
 * @author Liang Wenjian
 */
public interface HasIdGetter<I extends Comparable<I>> extends IdCons {
	String PROPERTY_NAME = IdCons.PROPERTY_NAME;

	/**
	 * 获取id属性的值。
	 *
	 * @return
	 */
	I getId();

	/**
	 * 实现本接口意味着拥有层级关系。子对象拥有一个叫parentId的属性指向父对象的id属性（即蕴含了本类拥有id属性的事实）。
	 *
	 * @param <I>
	 */
	interface HasParent<I extends Comparable<I>> extends HasIdGetter<I> {
		String PARENT_ID_PROPERTY_NAME = "parentId";
		String PARENT_PROPERTY_NAME = "parent";

		/**
		 * 获取父对象的id属性的值。
		 *
		 * @return
		 */
		I getParentId();
	}

	/**
	 * 拥有int/Integer型的id属性。
	 */
	interface HasIntIdGetter extends HasIdGetter<Integer> {
		@Override
		Integer getId();

		/**
		 * 拥有父对象，其id是int型的id.
		 */
		interface HasIntIdParent extends HasIntIdGetter, HasParent<Integer> {
			@Override
			Integer getParentId();
		}
	}

	/**
	 * 拥有long/Long型的id属性。
	 */
	interface HasLongIdGetter extends HasIdGetter<Long> {
		@Override
		Long getId();

		/**
		 * 拥有父对象，其id是long/Long型的id。
		 */
		interface HasLongIdParent extends HasLongIdGetter, HasParent<Long> {
			@Override
			Long getParentId();
		}
	}

	/**
	 * 拥有String型的id属性。
	 */
	interface HasStrIdGetter extends HasIdGetter<String> {
		@Override
		String getId();

		/**
		 * 拥有父对象，其id是String型的。
		 */
		interface HasStrIdParent extends HasStrIdGetter, HasParent<String> {
			@Override
			String getParentId();
		}
	}

	/**
	 * 拥有可读写的id属性。
	 *
	 * @param <I>
	 *            id属性的类型。
	 */
	interface HasId<I extends Comparable<I>> extends HasIdGetter<I> {
		void setId(I id);
	}

	/**
	 * 拥有可读写的int/Integer型的id属性。
	 */
	interface HasIntId extends HasId<Integer>, HasIntIdGetter {
	}

	/**
	 * 拥有可读写的long/Long型的id属性。
	 */
	interface HasLongId extends HasId<Long>, HasLongIdGetter {
	}

	/**
	 * 拥有可读写的String型的id属性。
	 */
	interface HasStrId extends HasId<String>, HasStrIdGetter {
	}

	class IdOnly<T extends Comparable<T>> implements HasId<T> {
		private T id;

		@Override
		public T getId() {
			return id;
		}

		@Override
		public void setId(final T id) {
			this.id = id;
		}
	}

	class LongIdOnly extends IdOnly<Long> implements HasLongId {
	}

	class IntIdOnly extends IdOnly<Integer> implements HasIntId {
	}

	class StrIdOnly extends IdOnly<String> implements HasStrId {
	}

}

interface IdCons {
	String PROPERTY_NAME = "id";

	/**
	 * 将拥有父对象的对象转换为父对象id的转换器。
	 */
	Function<HasParent, Comparable> TO_PARENT_ID = new Function<HasIdGetter.HasParent, Comparable>() {
		@Override
		public Comparable apply(HasParent input) {
			return input.getParentId();
		};
	};

}