package com.thebeastshop.support.mark;

import com.google.common.base.Function;
import com.google.common.collect.Collections2;
import com.google.common.collect.Maps;

import java.util.Collection;
import java.util.Comparator;
import java.util.Map;

/**
 * 实现本接口意味着拥有名字（name）属性。
 */
@FunctionalInterface
public interface HasName extends HasNameCons {
    String PROPERTY_NAME = HasNameCons.PROPERTY_NAME;

    /**
     * 获取名字属性的值。
     *
     * @return
     */
    String getName();

    /**
     * 关于${@link HasName}的工具类
     */
    class NameUtil implements HasNameCons {

        /**
         * 将一个${@link HasName}的集合转换成它们的名字的集合。
         *
         * @param source
         * @param <T>
         * @return
         */
        public static <T extends HasName> Collection<String> toNames(final Collection<T> source) {
            return Collections2.transform(source, TO_NAME);
        }

        /**
         * 将一个${@link HasName}的集合转换成一个以它们的名字为键，对象本身为值的${@link Map}。
         * 注：如果集合中有名字相同的元素，在遍历次序中靠后的将会覆盖靠前的。
         *
         * @param source
         * @param <T>
         * @return
         */
        public static <T extends HasName> Map<String, T> map(final Iterable<T> source) {
            final Map<String, T> map = Maps.newHashMap();
            for (final T t : source) {
                map.put(t.getName(), t);
            }
            return map;
        }

        public static <T extends HasName> T findByName(final T target, final Collection<? extends T> source) {
            return findByName(target.getName(), source);
        }

        public static <T extends HasName> T findByName(final String name, final Collection<? extends T> source) {
            for (final T t : source) {
                if (t.getName().equals(name)) {
                    return t;
                }
            }
            return null;
        }

        private NameUtil() {
            super();
        }
    }
}

interface HasNameCons {

    String PROPERTY_NAME = "name";
    /**
     * 将${@link HasName}对象转换成它的名字的转换器
     */
    Function<HasName, String> TO_NAME = HasName::getName;

    Comparator<HasName> NAME_ASC = Comparator.nullsLast(
            (o1, o2) -> Comparator.nullsLast(Comparator.<String>naturalOrder()).compare(o1.getName(), o2.getName()));

    Comparator<HasName> NAME_DESC = Comparator.nullsLast(
            (o1, o2) -> Comparator.nullsLast(Comparator.<String>naturalOrder()).compare(o2.getName(), o1.getName()));
}