package com.thebeastshop.support.mark;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

import com.google.common.base.Function;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.thebeastshop.support.mark.HasIdGetter.HasParent;

/**
 * 实现本接口意味着拥有id属性。
 *
 * @param <I> id属性的类型
 * @author Liang Wenjian
 */
@FunctionalInterface
public interface HasIdGetter<I extends Comparable<I>> extends IdCons {
    String PROPERTY_NAME = IdCons.PROPERTY_NAME;

    /**
     * 获取id属性的值。
     *
     * @return
     */
    I getId();

    /**
     * 实现本接口意味着拥有层级关系。子对象拥有一个叫parentId的属性指向父对象的id属性（即蕴含了本类拥有id属性的事实）。
     *
     * @param <I>
     */
    interface HasParent<I extends Comparable<I>> extends HasIdGetter<I> {
        String PARENT_ID_PROPERTY_NAME = "parentId";
        String PARENT_PROPERTY_NAME = "parent";

        /**
         * 获取父对象的id属性的值。
         *
         * @return
         */
        I getParentId();
    }

    /**
     * 拥有int/Integer型的id属性。
     */
    @FunctionalInterface
    interface HasIntIdGetter extends HasIdGetter<Integer> {
        @Override
        Integer getId();

        /**
         * 拥有父对象，其id是int型的id.
         */
        interface HasIntIdParent extends HasIntIdGetter, HasParent<Integer> {
            @Override
            Integer getParentId();
        }
    }

    /**
     * 拥有long/Long型的id属性。
     */
    @FunctionalInterface
    interface HasLongIdGetter extends HasIdGetter<Long> {
        @Override
        Long getId();

        /**
         * 拥有父对象，其id是long/Long型的id。
         */
        interface HasLongIdParent extends HasLongIdGetter, HasParent<Long> {
            @Override
            Long getParentId();
        }
    }

    /**
     * 拥有String型的id属性。
     */
    @FunctionalInterface
    interface HasStrIdGetter extends HasIdGetter<String> {
        @Override
        String getId();

        /**
         * 拥有父对象，其id是String型的。
         */
        interface HasStrIdParent extends HasStrIdGetter, HasParent<String> {
            @Override
            String getParentId();
        }
    }

    /**
     * 拥有可读写的id属性。
     *
     * @param <I> id属性的类型。
     */
    interface HasId<I extends Comparable<I>> extends HasIdGetter<I> {
        void setId(I id);
    }

    /**
     * 拥有可读写的int/Integer型的id属性。
     */
    interface HasIntId extends HasId<Integer>, HasIntIdGetter {
    }

    /**
     * 拥有可读写的long/Long型的id属性。
     */
    interface HasLongId extends HasId<Long>, HasLongIdGetter {
    }

    /**
     * 拥有可读写的String型的id属性。
     */
    interface HasStrId extends HasId<String>, HasStrIdGetter {
    }

    class IdOnly<T extends Comparable<T>> implements HasId<T> {
        private T id;

        @Override
        public T getId() {
            return id;
        }

        @Override
        public void setId(final T id) {
            this.id = id;
        }
    }

    class LongIdOnly extends IdOnly<Long> implements HasLongId {
    }

    class IntIdOnly extends IdOnly<Integer> implements HasIntId {
    }

    class StrIdOnly extends IdOnly<String> implements HasStrId {
    }

    /**
     * 关于HasId系列接口的工具类。
     */
    class IdUtil implements IdCons {
        /**
         * 将一个${@link HasIdGetter}的集合转换成它们的id的集合。
         *
         * @param source
         * @param <I>
         * @return
         */
        @SuppressWarnings("unchecked")
        public static <I extends Comparable<I>> List<I> toIds(final Collection<? extends HasIdGetter<I>> source) {
            return source.stream().map(HasIdGetter::getId).collect(Collectors.toList());
        }

        /**
         * 在一个id列表中检查是否包含指定实体的id。
         *
         * @param entity
         * @param ids
         * @param <I>
         * @return
         */
        @SafeVarargs
        public static <I extends Comparable<I>> boolean contains(final HasIdGetter<I> entity, final I... ids) {
            return contains(entity, Arrays.asList(ids));
        }

        /**
         * 在一个id列表中检查是否包含指定实体的id。
         *
         * @param entity
         * @param ids
         * @param <I>
         * @return
         */
        public static <I extends Comparable<I>> boolean contains(final HasIdGetter<I> entity, final Collection<I> ids) {
            return ids.contains(entity.getId());
        }

        /**
         * 在一个实体列表中检查是否包含指定id的实体
         *
         * @param id
         * @param entities
         * @param <I>
         * @return
         */
        @SafeVarargs
        public static <I extends Comparable<I>> boolean contains(final I id, final HasIdGetter<I>... entities) {
            return contains(id, Arrays.asList(entities));
        }

        /**
         * 在一个实体列表中检查是否包含指定id的实体
         *
         * @param id
         * @param entities
         * @param <I>
         * @return
         */
        public static <I extends Comparable<I>> boolean contains(final I id,
                                                                 final Collection<? extends HasIdGetter<I>> entities) {
            return toIds(entities).contains(id);
        }

        /**
         * 将一个{@link HasParent}的集合转换成它们的父对象id的集合。
         *
         * @param source
         * @param <I>
         * @return
         */
        @SuppressWarnings("unchecked")
        public static <I extends Comparable<I>> Collection<I> toParentIds(
                final Collection<? extends HasParent<I>> source) {
            return source.stream().map(HasParent::getParentId).collect(Collectors.toList());
        }

        /**
         * 将一个${@link HasIdGetter}的集合转换成一个以id为键，对象本身为值的${@link Map}。
         *
         * @param source
         * @param <I>
         * @param <T>
         * @return
         */
        public static <I extends Comparable<I>, T extends HasIdGetter<I>> Map<I, T> map(final Iterable<T> source) {
            final Map<I, T> map = Maps.newLinkedHashMap();
            for (final T t : source) {
                map.put(t.getId(), t);
            }
            return map;
        }

        public static int hashCode(final HasIdGetter<?> hasIdGetter) {
            return hasIdGetter == null || hasIdGetter.getId() == null ? 0 : hasIdGetter.getId().hashCode();
        }

        public static boolean equalsUsingId(final HasIdGetter<?> hasIdGetter1, final HasIdGetter<?> hasIdGetter2) {
            return equalsUsingId(hasIdGetter1, hasIdGetter2, false);
        }

        public static boolean equalsUsingId(final HasIdGetter<?> hasIdGetter1, final HasIdGetter<?> hasIdGetter2,
                                            final boolean nullEquals) {
            if (hasIdGetter1 == null) {
                if (hasIdGetter2 == null) {
                    return nullEquals;
                } else {
                    return false;
                }
            }
            if (hasIdGetter2 == null) {//hasIdGetter1 != null
                return false;
            }

            final Object id1 = hasIdGetter1.getId();
            final Object id2 = hasIdGetter2.getId();
            if (id1 == null) {
                if (id2 == null) {
                    return nullEquals;
                } else {
                    return false;
                }
            }
            if (id2 == null) {//id1 != null
                return false;
            }
            return id1.equals(id2);
        }

        /**
         * 对一个${@link HasIdGetter}的集合根据id属性进行升序排序。
         *
         * @param source
         * @param <T>
         * @return
         */
        public static <T extends HasIdGetter<?>> List<T> idAsc(final Collection<T> source) {
            return sort(source, ID_ASC);
        }

        /**
         * 对一个${@link HasIdGetter}的集合根据id属性进行降序排序。
         *
         * @param source
         * @param <T>
         * @return
         */
        public static <T extends HasIdGetter<?>> List<T> idDesc(final Collection<T> source) {
            return sort(source, ID_DESC);
        }

        public static <T extends HasIdGetter<?>> T findById(final Object id, final Collection<T> source) {
            for (final T t : source) {
                if (Objects.equals(id, t.getId())) {
                    return t;
                }
            }
            return null;
        }

        /**
         * 对一个${@link HasIdGetter}的集合根据指定的排序器进行排序。
         *
         * @param source
         * @param comparator
         * @param <T>
         * @return
         */
        private static <T extends HasIdGetter<?>> List<T> sort(final Collection<T> source,
                                                               final Comparator<HasIdGetter<?>> comparator) {
            final List<T> list;
            if (source instanceof List) {
                list = (List<T>) source;
            } else {
                list = Lists.newArrayList(source);
            }
            Collections.sort(list, comparator);
            return list;
        }

        private IdUtil() {
            super();
        }
    }//IdUtil
}

interface IdCons {
    String PROPERTY_NAME = "id";
    /**
     * id升序比较器。
     */
    @SuppressWarnings("unchecked")
    Comparator<HasIdGetter<?>> ID_ASC = Comparator.nullsLast(
            (o1, o2) -> Comparator.nullsLast(Comparator.<Comparable>naturalOrder()).compare(o1.getId(), o2.getId()));

    /**
     * id降序比较器。
     */
    Comparator<HasIdGetter<?>> ID_DESC = Comparator.nullsLast(
            (o1, o2) -> Comparator.nullsLast(Comparator.<Comparable>naturalOrder()).compare(o2.getId(), o1.getId()));

    /**
     * 将拥有id属性的对象转换为id的转换器。
     */
    Function<HasIdGetter, Comparable> TO_ID = HasIdGetter::getId;

    /**
     * 将拥有父对象的对象转换为父对象id的转换器。
     */
    Function<HasParent, Comparable> TO_PARENT_ID = HasParent::getParentId;

}