/**
 * Copyright (C), 上海布鲁爱电子商务有限公司
 */
package com.thebeastshop.support.encode;

import java.io.ByteArrayOutputStream;
import java.security.KeyFactory;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.X509EncodedKeySpec;

import javax.crypto.Cipher;

import org.apache.commons.lang.StringUtils;


/**
 * RSA工具
 * 
 * @author Paul-xiong
 *
 */
public class RSAUtil {

	/** RSA最大加密明文大小 */
	private static final int MAX_ENCRYPT_BLOCK = 117;

	/** RSA最大解密密文大小 */
	private static final int MAX_DECRYPT_BLOCK = 128;

	private static final String SIGN_TYPE_RSA = "RSA";

	/**
	 * 公钥加密
	 * 
	 * @param content
	 *            待加密内容
	 * @param publicKey
	 *            公钥
	 * @param charset
	 *            字符集，如UTF-8, GBK, GB2312
	 * @return 密文内容
	 */
	public static String rsaEncrypt(String content, String publicKey, String charset) throws Exception {
		PublicKey pubKey = getPublicKeyFromX509(SIGN_TYPE_RSA, publicKey);
		Cipher cipher = Cipher.getInstance(SIGN_TYPE_RSA);
		cipher.init(Cipher.ENCRYPT_MODE, pubKey);
		byte[] data = StringUtils.isEmpty(charset) ? content.getBytes() : content.getBytes(charset);
		int inputLen = data.length;
		ByteArrayOutputStream out = new ByteArrayOutputStream();
		int offSet = 0;
		byte[] cache;
		int i = 0;
		// 对数据分段加密
		while (inputLen - offSet > 0) {
			if (inputLen - offSet > MAX_ENCRYPT_BLOCK) {
				cache = cipher.doFinal(data, offSet, MAX_ENCRYPT_BLOCK);
			} else {
				cache = cipher.doFinal(data, offSet, inputLen - offSet);
			}
			out.write(cache, 0, cache.length);
			i++;
			offSet = i * MAX_ENCRYPT_BLOCK;
		}
		String encryptedData = Base64Util.encode(out.toByteArray());
		out.close();

		return encryptedData;
	}

	/**
	 * 私钥解密
	 * 
	 * @param content
	 *            待解密内容
	 * @param privateKey
	 *            私钥
	 * @param charset
	 *            字符集，如UTF-8, GBK, GB2312
	 * @return 明文内容
	 * @throws Exception
	 */
	public static String rsaDecrypt(String content, String privateKey, String charset) throws Exception {
		PrivateKey priKey = getPrivateKeyFromPKCS8(SIGN_TYPE_RSA, privateKey);
		Cipher cipher = Cipher.getInstance(SIGN_TYPE_RSA);
		cipher.init(Cipher.DECRYPT_MODE, priKey);
		byte[] encryptedData = StringUtils.isEmpty(charset) ? Base64Util.decode(new String(content.getBytes()))
				: Base64Util.decode(new String(content.getBytes(charset)));
		int inputLen = encryptedData.length;
		ByteArrayOutputStream out = new ByteArrayOutputStream();
		int offSet = 0;
		byte[] cache;
		int i = 0;
		// 对数据分段解密
		while (inputLen - offSet > 0) {
			if (inputLen - offSet > MAX_DECRYPT_BLOCK) {
				cache = cipher.doFinal(encryptedData, offSet, MAX_DECRYPT_BLOCK);
			} else {
				cache = cipher.doFinal(encryptedData, offSet, inputLen - offSet);
			}
			out.write(cache, 0, cache.length);
			i++;
			offSet = i * MAX_DECRYPT_BLOCK;
		}
		byte[] decryptedData = out.toByteArray();
		out.close();

		return StringUtils.isEmpty(charset) ? new String(decryptedData) : new String(decryptedData, charset);
	}

	public static PublicKey getPublicKeyFromX509(String algorithm, String privateKey) throws Exception {
		if (StringUtils.isEmpty(algorithm) || StringUtils.isEmpty(privateKey)) {
			return null;
		}

		KeyFactory keyFactory = KeyFactory.getInstance(algorithm);

		return keyFactory.generatePublic(new X509EncodedKeySpec(Base64Util.decode(privateKey)));
	}

	public static PrivateKey getPrivateKeyFromPKCS8(String algorithm, String privateKey) throws Exception {
		if (StringUtils.isEmpty(algorithm) || StringUtils.isEmpty(privateKey)) {
			return null;
		}

		KeyFactory keyFactory = KeyFactory.getInstance(algorithm);

		return keyFactory.generatePrivate(new PKCS8EncodedKeySpec(Base64Util.decode(privateKey)));
	}
}
