package com.thebeastshop.pegasus.service.warehouse.util;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.TimeUnit;

import javax.annotation.Resource;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;
import redis.clients.jedis.Pipeline;
import redis.clients.jedis.exceptions.JedisConnectionException;

/**
 * @author bryan.zhang
 * @date 2015-1-21
 * @description 基于Redis分布式锁的实现
 */
//@Component
@Deprecated
public class RedisLockHandler {

	private final Log log = LogFactory.getLog(this.getClass());

	private final int DEFAULT_SINGLE_EXPIRE_TIME = 3;
	
	private final int DEFAULT_BATCH_EXPIRE_TIME = 6;

	@Resource
	private JedisPool jedisPool;
	
	/**
	 * 获取锁  如果锁可用   立即返回true，  否则返回false
	 */
	public boolean tryLock(String key) {
		return tryLock(key, 0L, null);
	}

	/**
	 * 锁在给定的等待时间内空闲，则获取锁成功 返回true， 否则返回false
	 */
	public boolean tryLock(String key, long timeout, TimeUnit unit) {
		Jedis jedis = null;
		try {
			jedis = getResource();
			long nano = System.nanoTime();
			do {
				log.debug("try lock key: " + key);
				Long i = jedis.setnx(key, key);
				if (i == 1) { 
					jedis.expire(key, DEFAULT_SINGLE_EXPIRE_TIME);
					log.debug("get lock, key: " + key + " , expire in " + DEFAULT_SINGLE_EXPIRE_TIME + " seconds.");
					return Boolean.TRUE;
				} else { // 存在锁
					if (log.isDebugEnabled()) {
						String desc = jedis.get(key);
						log.debug("key: " + key + " locked by another business：" + desc);
					}
				}
				if (timeout == 0) {
					break;
				}
				Thread.sleep(300);
			} while ((System.nanoTime() - nano) < unit.toNanos(timeout));
			return Boolean.FALSE;
		} catch (JedisConnectionException je) {
			log.error(je.getMessage(), je);
			returnBrokenResource(jedis);
		} catch (Exception e) {
			log.error(e.getMessage(), e);
		} finally {
			returnResource(jedis);
		}
		return Boolean.FALSE;
	}

	/**
	 * 如果锁空闲立即返回   获取失败 一直等待
	 */
	public void lock(String key) {
		Jedis jedis = null;
		try {
			jedis = getResource();
			do {
				log.debug("lock key: " + key);
				Long i = jedis.setnx(key, key);
				if (i == 1) { 
					jedis.expire(key, DEFAULT_SINGLE_EXPIRE_TIME);
					log.debug("get lock, key: " + key + " , expire in " + DEFAULT_SINGLE_EXPIRE_TIME + " seconds.");
					return;
				} else {
					if (log.isDebugEnabled()) {
						String desc = jedis.get(key);
						log.debug("key: " + key + " locked by another business：" + desc);
					}
				}
				Thread.sleep(300); 
			} while (true);
		} catch (JedisConnectionException je) {
			log.error(je.getMessage(), je);
			returnBrokenResource(jedis);
		} catch (Exception e) {
			log.error(e.getMessage(), e);
		} finally {
			returnResource(jedis);
		}
	}

	/**
	 * 释放锁
	 */
	public void unLock(String key) {
		List<String> list = new ArrayList<String>();
		list.add(key);
		unLock(list);
	}

	/**
	 * 批量获取锁  如果全部获取   立即返回true, 部分获取失败 返回false
	 */
	public boolean tryLock(List<String> keyList) {
		return tryLock(keyList, 0L, null);
	}
	
	/**
	 * 锁在给定的等待时间内空闲，则获取锁成功 返回true， 否则返回false
	 */
	public boolean tryLock(List<String> keyList, long timeout, TimeUnit unit) {
		Jedis jedis = null;
		try {
			List<String> needLocking = new CopyOnWriteArrayList<String>();	
			List<String> locked = new CopyOnWriteArrayList<String>();	
			jedis = getResource();
			long nano = System.nanoTime();
			do {
				// 构建pipeline，批量提交
				Pipeline pipeline = jedis.pipelined();
				for (String key : keyList) {
					needLocking.add(key);
					pipeline.setnx(key, key);
				}
				log.debug("try lock keys: " + needLocking);
				// 提交redis执行计数
				List<Object> results = pipeline.syncAndReturnAll();
				for (int i = 0; i < results.size(); ++i) {
					Long result = (Long) results.get(i);
					String key = needLocking.get(i);
					if (result == 1) {	// setnx成功，获得锁
						jedis.expire(key, DEFAULT_BATCH_EXPIRE_TIME);
						locked.add(key);
					} 
				}
				needLocking.removeAll(locked);	// 已锁定资源去除
				
				if (CollectionUtils.isEmpty(needLocking)) {
					return true;
				} else {	
					// 部分资源未能锁住
					log.debug("keys: " + needLocking + " locked by another business：");
				}
				
				if (timeout == 0) {	
					break;
				}
				Thread.sleep(500);	
			} while ((System.nanoTime() - nano) < unit.toNanos(timeout));

			// 得不到锁，释放锁定的部分对象，并返回失败
			if (!CollectionUtils.isEmpty(locked)) {
				jedis.del(locked.toArray(new String[0]));
			}
			return false;
		} catch (JedisConnectionException je) {
			log.error(je.getMessage(), je);
			returnBrokenResource(jedis);
		} catch (Exception e) {
			log.error(e.getMessage(), e);
		} finally {
			returnResource(jedis);
		}
		return true;
	}

	/**
	 * 批量释放锁
	 */
	public void unLock(List<String> keyList) {
		List<String> keys = new CopyOnWriteArrayList<String>();
		for (String key : keyList) {
			keys.add(key);
		}
		Jedis jedis = null;
		try {
			jedis = getResource();
			jedis.del(keys.toArray(new String[0]));
			log.debug("release lock, keys :" + keys);
		} catch (JedisConnectionException je) {
			log.error(je.getMessage(), je);
			returnBrokenResource(jedis);
		} catch (Exception e) {
			log.error(e.getMessage(), e);
		} finally {
			returnResource(jedis);
		}
	}
	
	private Jedis getResource() {
		return jedisPool.getResource();
	}
	
	/**
	 * 销毁连接
	 */
	private void returnBrokenResource(Jedis jedis) {
		if (jedis == null) {
			return;
		}
		try {
			//容错
			jedisPool.returnBrokenResource(jedis);
		} catch (Exception e) {
			log.error(e.getMessage(), e);
		}
	}
	
	private void returnResource(Jedis jedis) {
		if (jedis == null) {
			return;
		}
		try {
			jedisPool.returnResource(jedis);
		} catch (Exception e) {
			log.error(e.getMessage(), e);
		}
	}
}