package com.thebeastshop.pegasus.service.warehouse.service.impl;


import com.thebeastshop.common.validation.Validation;
import com.thebeastshop.pegasus.service.warehouse.cond.WhWmsShelvesSkuInfoCond;
import com.thebeastshop.pegasus.service.warehouse.cond.WhWmsSkuPickShelvesCond;
import com.thebeastshop.pegasus.service.warehouse.dao.WhWmsShelvesSkuInfoMapper;
import com.thebeastshop.pegasus.service.warehouse.dao.WhWmsSkuPickShelvesMapper;
import com.thebeastshop.pegasus.service.warehouse.exception.WarehouseException;
import com.thebeastshop.pegasus.service.warehouse.model.*;
import com.thebeastshop.pegasus.service.warehouse.service.WhWmsShelvesSkuInfoService;
import com.thebeastshop.pegasus.service.warehouse.vo.WhWmsShelvesSkuInfoVO;
import com.thebeastshop.pegasus.service.warehouse.vo.WhWmsSkuPickShelvesVO;
import com.thebeastshop.pegasus.util.comm.*;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.List;

/**
 * Created by devin.li on 2016/3/17.
 */
@Service("whWmsShelvesSkuInfoService")
public class WhWmsShelvesSkuInfoServiceImpl implements WhWmsShelvesSkuInfoService {

    @Autowired
    private WhWmsShelvesSkuInfoMapper whShelvesSkuInfoMapper;

    @Autowired
    private WhWmsSkuPickShelvesMapper whWmsSkuPickShelvesMapper;

    @Override
    public List<WhWmsShelvesSkuInfoVO> getSkuInfoByCond(WhWmsShelvesSkuInfoCond cond){
        List<WhWmsShelvesSkuInfoVO> shelvesSkuInfoVOs = whShelvesSkuInfoMapper.getSkuInfoByCond(cond);
        // 获取sku的默认拣货位
        if (CollectionUtils.isNotEmpty(shelvesSkuInfoVOs) && cond.isFetchSkuPickShelvesFlag()){
            for (WhWmsShelvesSkuInfoVO skuInfoVO : shelvesSkuInfoVOs){
                skuInfoVO.setWhWmsSkuPickShelves(listSkuPickShelvesBySkuAndPhyCode(skuInfoVO.getSkuCode(),cond.getPhysicalWarehouseCode()));
            }
        }
        return shelvesSkuInfoVOs;
    }

    private List<WhWmsSkuPickShelves> listSkuPickShelvesBySkuAndPhyCode(String skuCode, String physicalWarehouseCode) {
        WhWmsSkuPickShelvesCond cond = new WhWmsSkuPickShelvesCond();
        cond.setSkuCode(skuCode);
        cond.setPhysicalWarehouseCode(physicalWarehouseCode);
        return listSkuPickShelvesByCond(cond);
    }

    @Override
    public List<WhWmsSkuPickShelves> listSkuPickShelvesByCond(WhWmsSkuPickShelvesCond cond) {
        WhWmsSkuPickShelvesExample example = buildWhWmsSkuPickShelvesExample(cond);
        return whWmsSkuPickShelvesMapper.selectByExample(example);
    }

    @Override
    public List<WhWmsSkuPickShelves> listSkuPickShelvesBySkuCode(String skuCode) {
        WhWmsSkuPickShelvesCond cond = new WhWmsSkuPickShelvesCond();
        Validation.paramNotNull(skuCode, "skuCode为空!");
        cond.setSkuCode(skuCode);
        return listSkuPickShelvesByCond(cond);
    }

    @Override
    public List<WhWmsSkuPickShelvesVO> listSkuPickShelvesVOBySkuCode(String skuCode) {
        WhWmsSkuPickShelvesCond cond = new WhWmsSkuPickShelvesCond();
        cond.setSkuCode(skuCode);
        return listSkuPickShelvesVOByCond(cond);
    }

    @Override
    public List<WhWmsSkuPickShelvesVO> listSkuPickShelvesVOByCond(WhWmsSkuPickShelvesCond cond) {
        return whWmsSkuPickShelvesMapper.listSkuPickShelvesVOByCond(cond);
    }

    private WhWmsSkuPickShelvesExample buildWhWmsSkuPickShelvesExample(WhWmsSkuPickShelvesCond cond) {
        WhWmsSkuPickShelvesExample example = new WhWmsSkuPickShelvesExample();
        WhWmsSkuPickShelvesExample.Criteria criteria = example.createCriteria();
        if (EmptyUtil.isNotEmpty(cond.getId())){
            criteria.andIdEqualTo(cond.getId());
        }else if(CollectionUtils.isNotEmpty(cond.getIds())){
            criteria.andIdIn(cond.getIds());
        }

        if (EmptyUtil.isNotEmpty(cond.getSkuCode())){
            criteria.andSkuCodeEqualTo(cond.getSkuCode());
        }else if(CollectionUtils.isNotEmpty(cond.getSkuCodes())){
            criteria.andSkuCodeIn(cond.getSkuCodes());
        }

        if (EmptyUtil.isNotEmpty(cond.getPhysicalWarehouseCode())){
            criteria.andPhysicalWarehouseCodeEqualTo(cond.getPhysicalWarehouseCode());
        }

        if (EmptyUtil.isNotEmpty(cond.getShelvesCode())){
            criteria.andShelvesCodeEqualTo(cond.getShelvesCode());
        }else if(EmptyUtil.isNotEmpty(cond.getShelvesCodeLike())){
            criteria.andShelvesCodeLike(SQLUtils.allLike(cond.getShelvesCodeLike()));
        }else if(CollectionUtils.isNotEmpty(cond.getShelvesCodes())){
            criteria.andShelvesCodeIn(cond.getShelvesCodes());
        }

        if (EmptyUtil.isNotEmpty(cond.getCreateTimeStart())){
            criteria.andCreateTimeGreaterThanOrEqualTo(cond.getCreateTimeStart());
        }
        if (EmptyUtil.isNotEmpty(cond.getCreateTimeEnd())){
            criteria.andCreateTimeLessThanOrEqualTo(cond.getCreateTimeEnd());
        }

        if (EmptyUtil.isNotEmpty(cond.getOrderBy())){
            example.setOrderByClause(cond.getOrderBy());
        }
        return example;
    }

    @Override
    @Transactional
    public Boolean newSkuInfo(WhWmsShelvesSkuInfo skuInfo){
        WhWmsShelvesSkuInfo sku = getSkuInfoByCode(skuInfo.getSkuCode());
        if (EmptyUtil.isNotEmpty(sku)) throw new WarehouseException("","["+skuInfo.getSkuCode()+"]the sku is already exist");
        int result = whShelvesSkuInfoMapper.insert(skuInfo);
        if (result<=0) throw new WarehouseException("","insert data error!");
        return true;
    }

    @Override
    @Transactional
    public Boolean newWhShelvesSkuInfoVO(WhWmsShelvesSkuInfoVO skuInfoVO){
        WhWmsShelvesSkuInfo skuInfo = BeanUtil.buildFrom(skuInfoVO,WhWmsShelvesSkuInfo.class);
        boolean result = newSkuInfo(skuInfo);
        if (result){
            // 返回前台
            skuInfoVO.setId(skuInfo.getId());
            if (CollectionUtils.isNotEmpty(skuInfoVO.getWhWmsSkuPickShelves())){
                batchCreateSkuPickShelves(skuInfoVO.getWhWmsSkuPickShelves(),skuInfo.getSkuCode());
            }
        }
        return result;
    }

    @Override
    public Boolean delSkuPickShelvesBySkuCode(String skuCode) {
        WhWmsSkuPickShelvesCond cond= new WhWmsSkuPickShelvesCond();
        cond.setSkuCode(skuCode);
        return delSkuPickShelvesByCond(cond);
    }

    @Override
    @Transactional
    public Boolean delSkuPickShelvesByCond(WhWmsSkuPickShelvesCond cond) {
        WhWmsSkuPickShelvesExample example = buildWhWmsSkuPickShelvesExample(cond);
        return whWmsSkuPickShelvesMapper.deleteByExample(example)>=0;
    }

    @Override
    @Transactional
    public Boolean createSkuPickShelves(WhWmsSkuPickShelves skuPickShelves) {
        return whWmsSkuPickShelvesMapper.insert(skuPickShelves)>0;
    }

    @Override
    @Transactional
    public Boolean batchCreateSkuPickShelves(List<WhWmsSkuPickShelves> skuPickShelvesList,String skuCode) {
        for (WhWmsSkuPickShelves skuPickShelve : skuPickShelvesList){
            skuPickShelve.setSkuCode(skuCode);
            skuPickShelve.setCreateTime(DateUtil.getNow());
            if (EmptyUtil.isEmpty(skuPickShelve.getSort())){
                skuPickShelve.setSort(1);
            }
            createSkuPickShelves(skuPickShelve);
        }
        return true;
    }

    @Override
    @Transactional
    public Boolean updateSkuInfo(WhWmsShelvesSkuInfo skuInfo){
        if (NumberUtil.isNullOrZero(skuInfo.getId()) ||EmptyUtil.isEmpty(skuInfo.getSkuCode()))
            throw new WarehouseException("","null param!");
        if (skuInfo.getSkuCode()=="") skuInfo.setSkuCode(null);
        int result = whShelvesSkuInfoMapper.updateByPrimaryKey(skuInfo);
        if (result<=0) throw new WarehouseException("","["+skuInfo.getSkuCode()+"]update data error!");
        return true;
    }

    @Override
    public Boolean updateWhShelvesSkuInfoVO(WhWmsShelvesSkuInfoVO skuInfoVO) {
        boolean result = updateSkuInfo(BeanUtil.buildFrom(skuInfoVO,WhWmsShelvesSkuInfo.class));
        if (result && CollectionUtils.isNotEmpty(skuInfoVO.getWhWmsSkuPickShelves())){
            if (delSkuPickShelvesBySkuCode(skuInfoVO.getSkuCode())){
                batchCreateSkuPickShelves(skuInfoVO.getWhWmsSkuPickShelves(),skuInfoVO.getSkuCode());
            }
        }
        return true;
    }

    @Override
    public Boolean saveOrUpdateWhWmsSkuPickShelves(List<WhWmsSkuPickShelves> whWmsSkuPickShelves) {
        String skuCode = whWmsSkuPickShelves.get(0).getSkuCode();
        if (delSkuPickShelvesBySkuCode(skuCode)){
            batchCreateSkuPickShelves(whWmsSkuPickShelves,skuCode);
        }
        return true;
    }

    @Override
    public WhWmsShelvesSkuInfo getSkuInfoByCode(String code) {
        WhWmsShelvesSkuInfoExample example = new WhWmsShelvesSkuInfoExample();
        example.createCriteria().andSkuCodeEqualTo(code);
        List<WhWmsShelvesSkuInfo> list = whShelvesSkuInfoMapper.selectByExample(example);
        if (CollectionUtils.isEmpty(list)) return null;
        return  list.get(0);
    }

    @Override
    public WhWmsShelvesSkuInfoVO getSkuInfoVOBySkuCode(String skuCode,boolean fetchPickShelve) {
        List<WhWmsShelvesSkuInfoVO> shelvesSkuInfoVOs = whShelvesSkuInfoMapper.getSkuInfoVOBySkuCode(skuCode);
        if (CollectionUtils.isEmpty(shelvesSkuInfoVOs)){
            return null;
        }
        WhWmsShelvesSkuInfoVO shelvesSkuInfoVO = shelvesSkuInfoVOs.get(0);
        if (fetchPickShelve){
            shelvesSkuInfoVO.setWhWmsSkuPickShelvesVO(listSkuPickShelvesVOBySkuCode(shelvesSkuInfoVO.getSkuCode()));
        }
        return shelvesSkuInfoVO;
    }

    @Override
    public List<WhWmsShelvesSkuInfo> getSkuInfoLikeCode(String codeLike) {
        return whShelvesSkuInfoMapper.getSkuInfoLikeCode(codeLike);
    }


}
