package com.thebeastshop.pegasus.service.warehouse.service.impl;

import com.thebeastshop.pegasus.service.warehouse.cond.WhWmsReplenishInfoCond;
import com.thebeastshop.pegasus.service.warehouse.dao.WhWmsReplenishInfoMapper;
import com.thebeastshop.pegasus.service.warehouse.dao.custom.WhWmsReplenishInfoCustomMapper;
import com.thebeastshop.pegasus.service.warehouse.exception.WarehouseException;
import com.thebeastshop.pegasus.service.warehouse.exception.WarehouseExceptionErrorCode;
import com.thebeastshop.pegasus.service.warehouse.model.*;
import com.thebeastshop.pegasus.service.warehouse.service.*;
import com.thebeastshop.pegasus.service.warehouse.vo.*;
import com.thebeastshop.pegasus.util.comm.*;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import page.Pagination;

import java.util.*;

/**
 * @author xw
 * @create 2018-08-14 16:25
 */
@Service("whWmsReplenishInfoService")
public class WhWmsReplenishInfoServiceImpl implements WhWmsReplenishInfoService {

    @Autowired
    private WhWmsReplenishInfoMapper whWmsReplenishInfoMapper;

    @Autowired
    private WhWmsReplenishInfoCustomMapper whWmsReplenishInfoCustomMapper;

    @Autowired
    private WhWmsMoveStockService whWmsMoveStockService;

    @Autowired
    private WhWmsSkuBarcodeService whWmsSkuBarcodeService;

    @Autowired
    private WhWmsSkuStockService whWmsSkuStockService;

    @Autowired
    private WhWmsHouseShelvesService whWmsHouseShelvesService;

    @Autowired
    private WhWmsWarehouseAreaService whWmsWarehouseAreaService;

    @Autowired
    private WhWmsWaitPutawayService whWmsWaitPutawayService;

    @Override
    public Pagination<WhWmsReplenishInfoVO> pageWhWmsReplenishInfoVOByCond(WhWmsReplenishInfoCond cond) {
        Pagination<WhWmsReplenishInfoVO> page = new Pagination<>(cond.getCurrpage(), cond.getPagenum());
        int count = countWhWmsReplenishInfo(cond);
        page.setRecord(count);
        if (NumberUtil.isNullOrZero(count))
            return page;
        List<WhWmsReplenishInfoVO> replenishRuleVOs = whWmsReplenishInfoCustomMapper.listWhWmsReplenishInfoVOByCond(cond);
        page.setResultList(replenishRuleVOs);
        return page;
    }

    @Override
    public List<WhWmsReplenishInfoVO> listWhWmsReplenishInfoVOByCond(WhWmsReplenishInfoCond cond) {
        return whWmsReplenishInfoCustomMapper.listWhWmsReplenishInfoVOByCond(cond);
    }

    @Override
    public List<WhWmsReplenishInfo> listWhWmsReplenishInfoByCond(WhWmsReplenishInfoCond cond) {
        cond.setCurrpage(null);
        WhWmsReplenishInfoExample example = buildWhWmsReplenishInfoExampleByCond(cond);
        return whWmsReplenishInfoMapper.selectByExample(example);
    }

    @Override
    public WhWmsReplenishInfo findWhWmsReplenishInfoById(Long id) {
        return whWmsReplenishInfoMapper.selectByPrimaryKey(id);
    }

    @Override
    public WhWmsReplenishInfoVO findWhWmsReplenishInfoVOById(Long id) {
        return whWmsReplenishInfoCustomMapper.findVOById(id);
    }

    @Override
    public WhWmsReplenishInfoVO saveOrUpdateWhWmsReplenishInfoVO(WhWmsReplenishInfoVO replenishRuleVO) {
        if (EmptyUtil.isNotEmpty(replenishRuleVO.getId())){
            updateWhWmsReplenishInfoVO(replenishRuleVO);
        }else{
            saveWhWmsReplenishInfoVO(replenishRuleVO);
        }
        return replenishRuleVO;
    }

    @Override
    public boolean batchSaveWhWmsReplenishInfo(List<WhWmsReplenishInfo> replenishInfos) {
        return whWmsReplenishInfoCustomMapper.batchInsertWhWmsReplenishInfo(replenishInfos)>0;
    }

    @Override
    @Transactional
    public String batchSaveWhWmsReplenishInfoVO(WhWmsBatchReplenishInfoVO batchReplenishInfo) throws WarehouseException{
        WhWmsReplenishInfo replenishInfo = null;
        List<WhWmsReplenishInfoVO> replenishInfoVOs = batchReplenishInfo.getReplenishInfoList();
        StringBuilder warnMsg = new StringBuilder();
        Set<String> skuCodeSet = new HashSet<>();
        for (WhWmsReplenishInfo replenishInfoTmp : replenishInfoVOs){
            skuCodeSet.add(replenishInfoTmp.getSkuCode());
        }
        Map<String, Boolean> skuBooleanMap = usedSupplierAndEffectiveGoods(Arrays.asList(skuCodeSet.toArray(new String[skuCodeSet.size()])));

        for (WhWmsReplenishInfoVO replenishInfoVO : replenishInfoVOs){
            try{
                replenishInfo = BeanUtil.buildFrom(replenishInfoVO,WhWmsReplenishInfo.class);
                replenishInfo.setCreateTime(DateUtil.getNow());
                // 补货信息唯一性 校验
                if (replenishInfoVO.isLessSafeMinStock() || batchReplenishInfo.isAutoReplenish()){
                    if (!checkReplenishInfoUnique(replenishInfo,warnMsg)){
                        continue;
                    }
                }
                whWmsReplenishInfoMapper.insert(replenishInfo);
                // 可用库存 小于安全库存下限 则直接创建移库单
                if (replenishInfoVO.isLessSafeMinStock() || batchReplenishInfo.isAutoReplenish()){
                    // 创建补货移库单
                    WhWmsReplenishInfoCond cond = new WhWmsReplenishInfoCond();
                    cond.setId(replenishInfo.getId());
                    cond.setCreateUserId(replenishInfoVOs.get(0).getCreateUserId());
                    cond.setSourceConnectStart(batchReplenishInfo.isSourceConnectStart());
                    String result = buildAndCreateMoveStock(replenishInfo, skuBooleanMap, cond);
                    warnMsg.append(result);
                    if (EmptyUtil.isNotEmpty(result)){
                        // 补货移库创建失败，回滚单条补货数据
                        whWmsReplenishInfoMapper.deleteByPrimaryKey(replenishInfo.getId());
                    }
                }
            }catch (Exception e){}
        }
        return warnMsg.toString();
    }

    private boolean checkReplenishInfoUnique(WhWmsReplenishInfo replenishInfo,StringBuilder warnMsg){
        WhWmsReplenishInfoCond replenishInfoCond = new WhWmsReplenishInfoCond();
        replenishInfoCond.setPhysicalWarehouseCode(replenishInfo.getPhysicalWarehouseCode());
        replenishInfoCond.setSkuCode(replenishInfo.getSkuCode());
        replenishInfoCond.setShelvesCode(replenishInfo.getShelvesCode());
        replenishInfoCond.setReplenishStates(WhWmsReplenishInfoVO.STATES_REPLENISHING);
        if (CollectionUtils.isNotEmpty(listWhWmsReplenishInfoByCond(replenishInfoCond))){
            warnMsg.append("同一:"+replenishInfoCond.toCustomString()+" 只能存在一条！<br/>");
            return false;
        }
        return true;
    }

    @Override
    @Transactional
    public String createReplenishMoveStock(WhWmsReplenishInfoCond cond) throws Exception{
        List<WhWmsReplenishInfo> replenishInfos = listWhWmsReplenishInfoByCond(cond);
        String warnMsg = "";
        if (CollectionUtils.isNotEmpty(replenishInfos)){
            Set<String> skuCodeSet = new HashSet<>();
            for (WhWmsReplenishInfo replenishInfo : replenishInfos){
                skuCodeSet.add(replenishInfo.getSkuCode());
            }
            Map<String, Boolean> skuBooleanMap = usedSupplierAndEffectiveGoods(Arrays.asList(skuCodeSet.toArray(new String[skuCodeSet.size()])));
            for (WhWmsReplenishInfo replenishInfo : replenishInfos){
                try{
                    warnMsg += buildAndCreateMoveStock(replenishInfo, skuBooleanMap, cond);
                }catch (Exception e){
                }
            }
        }
        return warnMsg;
    }

    @Override
    public Map<String, Boolean> usedSupplierAndEffectiveGoods(List<String> skuCodes){
        WhWmsWaitPutawayVO cond = new WhWmsWaitPutawayVO();
        cond.setCerSkuCodeList(skuCodes);
        List<WhWmsWaitPutawayVO> waitPutawayVOs = whWmsWaitPutawayService.listPutawayListBySkuCode(cond);
        return buildUseSupplierCodeByWaitPutaway(waitPutawayVOs);
    }

    /**
     * 使用供应商条码管理商品的条件
     * @param whWmsWaitPutawayVOs
     * @return
     */
    public Map<String, Boolean> buildUseSupplierCodeByWaitPutaway(List<WhWmsWaitPutawayVO> whWmsWaitPutawayVOs) {
        Map<String, Boolean> skuBooleanMap = new HashMap<>();
        for (WhWmsWaitPutawayVO whWmsWaitPutawayVO : whWmsWaitPutawayVOs){
            if (getBooleanByCheckIntValue(whWmsWaitPutawayVO.getIsHasTag(), 1) // 已关联供应商标签
                    && getBooleanByCheckIntValue(whWmsWaitPutawayVO.getCerType(), 2) // 是简易标签
                    && !getBooleanByCheckIntValue(whWmsWaitPutawayVO.getPrintMpCode(), 1) // 无小程序码
                    && getBooleanByCheckIntValue(whWmsWaitPutawayVO.getSkuType(), 3)){ // sku是成品
                // 是效期商品
                if (EmptyUtil.isNotEmpty(whWmsWaitPutawayVO.getShelfLife())
                        && whWmsWaitPutawayVO.getShelfLife() > 0){
                    if (!skuBooleanMap.keySet().contains(whWmsWaitPutawayVO.getSkuCode())){
                        skuBooleanMap.put(whWmsWaitPutawayVO.getSkuCode(), true);
                    }
                }
            }
        }
        return skuBooleanMap;
    }

    public boolean getBooleanByCheckIntValue(Integer value, int flag){
        if (EmptyUtil.isNotEmpty(value)
                && flag == value){
            return true;
        }
        return false;
    }

    @Override
    public boolean saveWhWmsReplenishInfoVO(WhWmsReplenishInfoVO replenishRuleVO) {
        WhWmsReplenishInfo replenishRule = BeanUtil.buildFrom(replenishRuleVO,WhWmsReplenishInfo.class);
        replenishRule.setCreateTime(DateUtil.getNow());
        return whWmsReplenishInfoMapper.insert(replenishRule)>0;
    }

    @Override
    public boolean updateWhWmsReplenishInfoVO(WhWmsReplenishInfoVO replenishRuleVO) {
        WhWmsReplenishInfo replenishRule = BeanUtil.buildFrom(replenishRuleVO,WhWmsReplenishInfo.class);
        return whWmsReplenishInfoMapper.updateByPrimaryKeySelective(replenishRule)>0;
    }

    @Override
    public boolean updateWhWmsReplenishInfoByCond(WhWmsReplenishInfo replenishInfo,WhWmsReplenishInfoCond cond){
        WhWmsReplenishInfoExample example = buildWhWmsReplenishInfoExampleByCond(cond);
        return whWmsReplenishInfoMapper.updateByExampleSelective(replenishInfo,example)>0;
    }

    @Override
    public boolean updateWhWmsReplenishInfoStatesByRefCode(String refCode, Integer refType, Integer newReplenishStates) {
        WhWmsReplenishInfo replenishInfo = new WhWmsReplenishInfo();
        replenishInfo.setReplenishStates(newReplenishStates);
        WhWmsReplenishInfoCond cond = new WhWmsReplenishInfoCond();
        cond.setRefCode(refCode);
        cond.setRefType(refType);
        return updateWhWmsReplenishInfoByCond(replenishInfo,cond);
    }

    private int countWhWmsReplenishInfo(WhWmsReplenishInfoCond cond){
        return whWmsReplenishInfoCustomMapper.countWhWmsReplenishInfoByCond(cond);
    }

    private List<WhWmsSkuStockVO> listWmsSkuStockByCond(WhWmsSkuStockVO stockCond, WhWmsReplenishInfo replenishInfo){
        WhWmsSkuStockVO cond = new WhWmsSkuStockVO();
        cond.setPhysicalWarehouseCode(stockCond.getPhysicalWarehouseCode());
        cond.setShelvesCode(replenishInfo.getShelvesCode());
        cond.setSkuCode(replenishInfo.getSkuCode());
        cond.setSkuStatus(WhWarehouseVO.COMMODITY_STATUS_FOR_NONDEFECTIVE);
        cond.setGtZero(true);
        return whWmsSkuStockService.getSkuStocksByShelvesCondForReport(cond);
    }

    /**
     * 构建并创建补货移库单
     * @param replenishInfo
     * @param cond
     */
    @Override
    public String buildAndCreateMoveStock(WhWmsReplenishInfo replenishInfo, Map<String, Boolean> skuBooleanMap, WhWmsReplenishInfoCond cond) throws WarehouseException{
        if (EmptyUtil.isNotEmpty(replenishInfo)) {
            if (EmptyUtil.isEmpty(cond.getPhysicalWarehouseCode())){
                cond.setPhysicalWarehouseCode(replenishInfo.getPhysicalWarehouseCode());
            }
            List<String> barCodes = new ArrayList<>();
            Map<String,WhWmsSkuBarcode> barcodeMap = new HashMap<>();
            List<WhWmsSkuBarcode> skuBarcodeList  = whWmsSkuBarcodeService.findBarCodesForPickSkuStock(replenishInfo.getSkuCode());
            for(WhWmsSkuBarcode barcode : skuBarcodeList){
                barCodes.add(barcode.getBarCode());
                barcodeMap.put(barcode.getBarCode(),barcode);
            }
            //获取普通区
            Map<String,WhWmsWarehouseArea> areaMap = whWmsWarehouseAreaService.getAreaHouseMapByType(WhWmsWarehouseAreaVO.TYPE_NORMAL);
            List<String> houseTypes = Arrays.asList(areaMap.keySet().toArray(new String[areaMap.size()]));
            List<WhWmsSkuStockVO> stockList = null;
            if(EmptyUtil.isNotEmpty(barCodes)){
                WhWmsSkuStockVO stockCond = new WhWmsSkuStockVO();
                stockCond.setPhysicalWarehouseCode(cond.getPhysicalWarehouseCode());
                stockCond.setSkuStatus(WhWarehouseVO.COMMODITY_STATUS_FOR_NONDEFECTIVE);
                stockCond.setBarCodeList(barCodes);
                stockCond.setHouseTypes(houseTypes);
                List<Integer> shelvesTypeList = new ArrayList<>();
                shelvesTypeList.add(WhWmsHouseShelvesVO.SHELVES_TYPE_HOLDING);//保管库位
                stockCond.setShelvesTypeList(shelvesTypeList);
                if (cond.isSourceConnectStart()){
                    // 波次启动的 查询 良品可用-预占用的库存
                    stockList = whWmsSkuStockService.findWmsSkuAvailableAccountWithPreOccupy(stockCond, true);
                }else{
                    stockList = whWmsSkuStockService.getSkuStocksByShelvesCondForReport(stockCond);
                }

                if (CollectionUtils.isEmpty(stockList)){
                    // throw new WarehouseException(WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED, "仓库:"+stockCond.getPhysicalWarehouseCode()+",SKU："+replenishInfo.getSkuCode()+",保管库位没有可用良品库存，无法生成补货数据!");
                    return "仓库:"+stockCond.getPhysicalWarehouseCode()+"，SKU："+replenishInfo.getSkuCode()+"，保管库位没有可用良品库存，无法创建补货移库单!</br>";
                }

                for (WhWmsSkuStockVO skuStockVO : stockList){
                    if (EmptyUtil.isNotEmpty(skuStockVO) && skuStockVO.getAvailableAmount()>0){
                        if (EmptyUtil.isEmpty(skuStockVO.getHouseType())
                                && EmptyUtil.isNotEmpty(skuStockVO.getShelvesCode())){
                            WhWmsHouseShelvesVO shelvesVO = whWmsHouseShelvesService.getHouseShelvesByCode(skuStockVO.getShelvesCode());
                            if (EmptyUtil.isNotEmpty(shelvesVO)){
                                skuStockVO.setHouseType(shelvesVO.getHouseType());
                            }
                        }
                    }
                }
                sortStock(stockList,barcodeMap);
                Map<String,List<WhWmsSkuStockVO>> skuStockMap = organizeStock(stockList);
                return buildAndCreateMoveStock(replenishInfo, skuStockMap, skuBooleanMap, stockCond, cond);
            }else{
                return "SKU："+replenishInfo.getSkuCode()+"，没有对应的批次，无法创建补货移库单!</br>";
            }
        }
        return "";
    }

    private String buildAndCreateMoveStock(WhWmsReplenishInfo replenishInfo,
                                           Map<String, List<WhWmsSkuStockVO>> skuStockMap,
                                           Map<String, Boolean> skuBooleanMap,
                                           WhWmsSkuStockVO stockCond,
                                           WhWmsReplenishInfoCond cond) {
        Boolean skuBooleanFlag = skuBooleanMap.get(replenishInfo.getSkuCode());
        List<WhWmsSkuStockVO> targetStockList = null;
        if (skuBooleanFlag != null && skuBooleanFlag){
            // 目标库位 良品库存
            targetStockList = listWmsSkuStockByCond(stockCond, replenishInfo);
            if (CollectionUtils.isNotEmpty(targetStockList) && targetStockList.size() > 1){
                return "效期商品:["+replenishInfo.getSkuCode()+"]，仓库:["+stockCond.getPhysicalWarehouseCode()+"]，目标库位：["+replenishInfo.getShelvesCode()+"]存在多个批次，无法创建补货移库单!</br>";
            }
        }
        List<WhWmsSkuStockVO> skuStockList = skuStockMap.get(replenishInfo.getSkuCode());
        List<WhWmsMoveSkuVO> moveSkuList = new ArrayList<>();
        if(EmptyUtil.isNotEmpty(skuStockList)){
            WhWmsMoveSkuVO moveSkuVO = new WhWmsMoveSkuVO();
            moveSkuVO.setPhysicalWarehouseCode(replenishInfo.getPhysicalWarehouseCode());
            moveSkuVO.setCreateTime(DateUtil.getNow());
            Iterator<WhWmsSkuStockVO> it = skuStockList.iterator();
            while(it.hasNext()){
                WhWmsSkuStockVO stockVO = it.next();
                moveSkuVO.setBarCode(stockVO.getBarCode());
                moveSkuVO.setAmount(stockVO.getAvailableAmount());

                // 当前行库存使用过，则移除
                it.remove();
                if(moveSkuVO.getAmount()<=0){
                    continue;
                }

                moveSkuVO.setOriginalShelvesCode(stockVO.getShelvesCode());
                // 源 目标库位不能一致
                if (moveSkuVO.getOriginalShelvesCode().equals(replenishInfo.getShelvesCode())){
                    moveSkuVO.setTargetShelvesCode(replenishInfo.getShelvesCode());
                    continue;
                }else{
                    moveSkuVO.setTargetShelvesCode("");
                }
                moveSkuVO.setOriginalHouseType(stockVO.getHouseType());

                // 如果是 [效期且使用供应商条码管理]的商品，则源库位和目标库位的批次必须相同
                if (skuBooleanFlag != null && skuBooleanFlag){
                    if (CollectionUtils.isNotEmpty(targetStockList)){
                        WhWmsSkuStockVO targetWmsSkuStockVO = targetStockList.get(0);
                        if (!moveSkuVO.getBarCode().equals(targetWmsSkuStockVO.getBarCode())){
                            // 批次不相等，则采用下一个批次进行移库操作
                            moveSkuVO.setTargetBarcode(targetWmsSkuStockVO.getBarCode());
                            continue;
                        }else{
                            moveSkuVO.setTargetBarcode("");
                        }
                    }
                }
                // 找到符合条件，则跳出
                break;
            }
            if(NumberUtil.isNullOrZero(moveSkuVO.getAmount())){
                return "仓库:"+stockCond.getPhysicalWarehouseCode()+"，SKU："+replenishInfo.getSkuCode()+"，保管库位没有可用良品库存，无法创建补货移库单!</br>";
            }

            if (EmptyUtil.isNotEmpty(moveSkuVO.getTargetShelvesCode())){
                return "SKU:["+replenishInfo.getSkuCode()+"]，仓库:["+stockCond.getPhysicalWarehouseCode()+"]，源库位,目标库位不能相同:["+moveSkuVO.getTargetShelvesCode()+"]!</br>";
            }

            if (EmptyUtil.isNotEmpty(moveSkuVO.getTargetBarcode())){
                return "效期商品:["+replenishInfo.getSkuCode()+"]，仓库:["+stockCond.getPhysicalWarehouseCode()+"]，源库位批次:["+moveSkuVO.getBarCode()+"]，目标库位批次:["+moveSkuVO.getTargetBarcode()+"]不相等，无法创建补货移库单!</br>";
            }
            // 排序的第一个 即为：先进先出规则的批次
            moveSkuVO.setTargetShelvesCode(replenishInfo.getShelvesCode());
            moveSkuVO.setTargetHouseType(getHouseTypeByCode(moveSkuVO.getTargetShelvesCode()));
            moveSkuVO.setSkuCode(replenishInfo.getSkuCode());
            moveSkuVO.setSkuStatus(WhWarehouseVO.COMMODITY_STATUS_FOR_NONDEFECTIVE);
            moveSkuVO.setStatus(WhWmsMoveSku.MOVE_SKU_STATUS_WAIT_MOVE);
            moveSkuList.add(moveSkuVO);
        }
        if (EmptyUtil.isNotEmpty(moveSkuList)) {
            WhWmsMoveStockVO moveStockVO = new WhWmsMoveStockVO();
            moveStockVO.setPhysicalWarehouseCode(replenishInfo.getPhysicalWarehouseCode());
            moveStockVO.setCreateUserId(cond.getCreateUserId());
            // 移库类型为：补货移库
            moveStockVO.setMoveType(WhWmsMoveStockVO.MOVE_TYPE_SUPPLEMENT);
            // moveStockVO.setReferenceCode("");
            moveStockVO.setMoveSkuList(moveSkuList);

            // 校验数据是否惟一，存在多条，则报错回滚
            WhWmsReplenishInfoCond replenishInfoCond = new WhWmsReplenishInfoCond();
            replenishInfoCond.setPhysicalWarehouseCode(moveStockVO.getPhysicalWarehouseCode());
            replenishInfoCond.setReplenishStates(WhWmsReplenishInfoVO.STATES_REPLENISHING);
            String warnMsg = checkWhWmsReplenishInfoPrimary(replenishInfoCond,cond.getIds());
            if (EmptyUtil.isNotEmpty(warnMsg)){
                return warnMsg;
            }

            boolean result;
            try{
                result = whWmsMoveStockService.newMoveStock(moveStockVO);
            }catch(WarehouseException e){
                return e.getMessage();
            }
            // 移库单创建后，将补货数据状态置为“补货中”
            if (result){
                WhWmsReplenishInfo record = new WhWmsReplenishInfo();
                record.setReplenishStates(WhWmsReplenishInfoVO.STATES_REPLENISHING);
                record.setRefCode(moveStockVO.getCode());
                record.setRefType(WhWmsReplenishInfoVO.REF_TYPE_MOVE_STOCK);
                // 更新时，不需此条件
                cond.setCreateUserId(null);
                updateWhWmsReplenishInfoByCond(record,cond);
            }
        }else{
            return "仓库:"+stockCond.getPhysicalWarehouseCode()+"，SKU："+replenishInfo.getSkuCode()+"，保管库位没有可用良品库存，无法创建补货移库单!</br>";
        }
        return "";
    }

    private String checkWhWmsReplenishInfoPrimary(WhWmsReplenishInfoCond replenishInfoCond,List<Long> replenishInfoIds){
        List<WhWmsReplenishInfo> oldReplenishInfos = listWhWmsReplenishInfoByCond(replenishInfoCond);
        StringBuilder warnMsg = new StringBuilder();
        if (CollectionUtils.isNotEmpty(oldReplenishInfos)){
            List<String> primaryKeys = new ArrayList<>();
            for (WhWmsReplenishInfo oldReplenishInfo : oldReplenishInfos){
                primaryKeys.add(oldReplenishInfo.getPhysicalWarehouseCode()+oldReplenishInfo.getSkuCode()+oldReplenishInfo.getShelvesCode());
            }

            if (CollectionUtils.isNotEmpty(replenishInfoIds)){
                replenishInfoCond = new WhWmsReplenishInfoCond();
                replenishInfoCond.setIds(replenishInfoIds);
                List<WhWmsReplenishInfo> replenishInfos = listWhWmsReplenishInfoByCond(replenishInfoCond);
                if (CollectionUtils.isNotEmpty(replenishInfos)){
                    String key = "";
                    for (WhWmsReplenishInfo replenishInfo : replenishInfos){
                        key = replenishInfo.getPhysicalWarehouseCode()+replenishInfo.getSkuCode()+replenishInfo.getShelvesCode();
                        if (primaryKeys.contains(key)){
                            warnMsg.append("同一:"+new WhWmsReplenishInfoCond(replenishInfo.getPhysicalWarehouseCode(),replenishInfo.getSkuCode(),replenishInfo.getShelvesCode(),WhWmsReplenishInfoVO.STATES_REPLENISHING).toCustomString()).append("只能存在一条！<br/>");
                        }
                    }
                }
                /*if (warnMsg.length() > 0){
                    throw new WarehouseException(WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED, warnMsg.toString());
                }*/
            }
        }
        return warnMsg.toString();
    }

    private Map<String,List<WhWmsSkuStockVO>> organizeStock(List<WhWmsSkuStockVO> stockList){
        Map<String,List<WhWmsSkuStockVO>> skuStockMap = new HashMap<>();
        for(WhWmsSkuStockVO skuStockVO : stockList){
            List<WhWmsSkuStockVO> stockInfoList= skuStockMap.get(skuStockVO.getSkuCode());
            if(NullUtil.isNull(stockInfoList)){
                stockInfoList = new ArrayList<>();
                skuStockMap.put(skuStockVO.getSkuCode(),stockInfoList);
            }
            stockInfoList.add(skuStockVO);
        }
        return skuStockMap;
    }

    //效期->数量少的库位 排序
    private void sortStock(List<WhWmsSkuStockVO> stockList
            ,final Map<String,WhWmsSkuBarcode> barcodeMap){
        Collections.sort(stockList, new Comparator<WhWmsSkuStockVO>() {
            @Override
            public int compare(WhWmsSkuStockVO s1, WhWmsSkuStockVO s2) {
                //比较批次码
                int result = compareBarCode(barcodeMap.get(s1.getBarCode()),barcodeMap.get(s2.getBarCode()));
                if(result == 0){
                    //数量
                    result = s1.getAvailableAccount()-s2.getAvailableAccount();
                }
                return result;
            }
        });
    }

    private String getHouseTypeByCode(String shelvesCode){
        if (EmptyUtil.isEmpty(shelvesCode)){
            return "";
        }
        WhWmsHouseShelvesVO houseShelvesVO = whWmsHouseShelvesService.getHouseShelvesByCode(shelvesCode);
        if (EmptyUtil.isNotEmpty(houseShelvesVO)){
            return houseShelvesVO.getHouseType();
        }
        return "";
    }

    // 根据先进先出规则获取批次
    private void getAndSortBarcodeBySku(List<String> skuCodes,Map<String,List<WhWmsSkuBarcode>> skuBarcodeMap){
        List<WhWmsSkuBarcode> skuBarcodeList  = whWmsSkuBarcodeService.findBarCodesForPickSkuStock(skuCodes);
        for (WhWmsSkuBarcode barcode : skuBarcodeList){
            List<WhWmsSkuBarcode> barcodes = skuBarcodeMap.get(barcode.getSkuCode());
            if (CollectionUtils.isEmpty(barcodes)){
                barcodes = new ArrayList<>();
                skuBarcodeMap.put(barcode.getSkuCode(),barcodes);
            }
            barcodes.add(barcode);
        }
        for(Map.Entry<String,List<WhWmsSkuBarcode>> entry : skuBarcodeMap.entrySet()){
            Collections.sort(entry.getValue(), new Comparator<WhWmsSkuBarcode>() {
                @Override
                public int compare(WhWmsSkuBarcode s1, WhWmsSkuBarcode s2) {
                    //比较批次码
                    return compareBarCode(s1,s2);
                    /*if(result == 0){
                        //库区
                        result = compareShelvesArea(areaMap.get(s1.getHouseType()),areaMap.get(s2.getHouseType()));
                        if(result == 0){
                            //库位类型顺序
                            result = compareShelvesType(shelvesMap.get(s1.getShelvesCode()),shelvesMap.get(s2.getShelvesCode()));
                            if(result == 0){
                                //数量
                                result = s1.getAvailableAccount()-s2.getAvailableAccount();
                            }
                        }
                    }*/
                    // return result;
                }
            });
        }
    }

    // 可用库存少的 先用
    private void sortSkuStock(List<WhWmsSkuStockVO> stockList){
        if (CollectionUtils.isNotEmpty(stockList)){
            Collections.sort(stockList, new Comparator<WhWmsSkuStockVO>() {
                @Override
                public int compare(WhWmsSkuStockVO s1, WhWmsSkuStockVO s2) {
                    return s1.getAvailableAmount()-s2.getAvailableAmount();
                }
            });
        }
    }

    private int compareShelvesType(WhWmsHouseShelves s1,WhWmsHouseShelves s2){
        if(NullUtil.isNull(s1.getShelvesType()) && NullUtil.isNull(s2.getShelvesType())){
            return 0;
        }else if(NullUtil.isNull(s1.getShelvesType())){
            return 1;
        }else if(NullUtil.isNull(s2.getShelvesType())){
            return -1;
        }else {
            return s1.getShelvesType().compareTo(s2.getShelvesType());
        }
    }

    private int compareShelvesSortOrder(WhWmsHouseShelves s1,WhWmsHouseShelves s2){
        if(NullUtil.isNull(s1.getSortOrder()) && NullUtil.isNull(s1.getSortOrder())){
            return 0;
        }else if(NullUtil.isNull(s1.getSortOrder())){
            return 1;
        }else if(NullUtil.isNull(s1.getSortOrder())){
            return -1;
        }else {
            return s1.getSortOrder().compareTo(s1.getSortOrder());
        }
    }

    private int compareShelvesArea(WhWmsWarehouseArea a1,WhWmsWarehouseArea a2){
        if(NullUtil.isNull(a1.getSortOrder()) && NullUtil.isNull(a2.getSortOrder())){
            return 0;
        }else if(NullUtil.isNull(a1.getSortOrder())){
            return 1;
        }else if(NullUtil.isNull(a2.getSortOrder())){
            return -1;
        }else {
            return a1.getSortOrder().compareTo(a2.getSortOrder());
        }
    }

    //先进先出:优先出库最早过期的,其次生产日期最早的,再次入库时间最早的
    public int compareBarCode(WhWmsSkuBarcode barcode1,WhWmsSkuBarcode barcode2){
        return whWmsSkuBarcodeService.compareBarCode(barcode1,barcode2);
    }

    private Integer buildMoveStockAmount(WhWmsMoveSkuVO moveSkuVO){
        WhWmsSkuStockVO stockCond = new WhWmsSkuStockVO();
        stockCond.setPhysicalWarehouseCode(moveSkuVO.getPhysicalWarehouseCode());
        stockCond.setShelvesCode(moveSkuVO.getOriginalShelvesCode());
        stockCond.setBarCode(moveSkuVO.getBarCode());
        stockCond.setSkuStatus(WhWarehouseVO.COMMODITY_STATUS_FOR_NONDEFECTIVE);
        List<WhWmsSkuStockVO> stockList = whWmsSkuStockService.getSkuStocksByShelvesCondForReport(stockCond);
        if (CollectionUtils.isNotEmpty(stockList)){
            return stockList.get(0).getAvailableAmount();
        }
        return 0;
    }

    private WhWmsSkuStockVO generateSourceShelvesCodeByRule(WhWmsMoveSkuVO moveSkuVO) throws Exception{
        WhWmsSkuStockVO stockCond = new WhWmsSkuStockVO();
        stockCond.setPhysicalWarehouseCode(moveSkuVO.getPhysicalWarehouseCode());
        stockCond.setSkuStatus(WhWarehouseVO.COMMODITY_STATUS_FOR_NONDEFECTIVE);
        stockCond.setBarCode(moveSkuVO.getBarCode());
        //stockCond.setHouseTypes(houseTypes);
        List<Integer> shelvesTypeList = new ArrayList<>();
        // shelvesTypeList.add(WhWmsHouseShelvesVO.SHELVES_TYPE_PICKING);//拣货库位
        // 查找此 sku 有良品库存的保管位
        shelvesTypeList.add(WhWmsHouseShelvesVO.SHELVES_TYPE_HOLDING);//保管库位
        stockCond.setShelvesTypeList(shelvesTypeList);
        List<WhWmsSkuStockVO> stockList = whWmsSkuStockService.getSkuStocksByShelvesCondForReport(stockCond);
        sortSkuStock(stockList);
        if (CollectionUtils.isNotEmpty(stockList)){
            for (WhWmsSkuStockVO skuStockVO : stockList){
                if (EmptyUtil.isNotEmpty(skuStockVO) && skuStockVO.getAvailableAmount()>0){
                    WhWmsHouseShelvesVO shelvesVO = whWmsHouseShelvesService.getHouseShelvesByCode(skuStockVO.getShelvesCode());
                    if (EmptyUtil.isNotEmpty(shelvesVO)){
                        skuStockVO.setHouseType(shelvesVO.getHouseType());
                    }
                    return skuStockVO;
                }
            }
        }
        throw new WarehouseException(WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED, "仓库:"+stockCond.getPhysicalWarehouseCode()+" 批次："+stockCond.getBarCode()+",没有保管库位良品库存!");
    }

    private WhWmsReplenishInfoExample buildWhWmsReplenishInfoExampleByCond(WhWmsReplenishInfoCond cond){
        WhWmsReplenishInfoExample example = new WhWmsReplenishInfoExample();
        WhWmsReplenishInfoExample.Criteria criteria = example.createCriteria();

        if (EmptyUtil.isNotEmpty(cond.getId())){
            criteria.andIdEqualTo(cond.getId());
        }else if(CollectionUtils.isNotEmpty(cond.getIds())){
            criteria.andIdIn(cond.getIds());
        }

        if (EmptyUtil.isNotEmpty(cond.getPhysicalWarehouseCode())){
            criteria.andPhysicalWarehouseCodeEqualTo(cond.getPhysicalWarehouseCode());
        }

        if (EmptyUtil.isNotEmpty(cond.getSkuCode())){
            criteria.andSkuCodeEqualTo(cond.getSkuCode());
        }

        if (EmptyUtil.isNotEmpty(cond.getHouseType())){
            criteria.andHouseTypeEqualTo(cond.getHouseType());
        }

        if (EmptyUtil.isNotEmpty(cond.getShelvesCode())){
            criteria.andShelvesCodeEqualTo(cond.getShelvesCode());
        }else if(EmptyUtil.isNotEmpty(cond.getShelvesCodeLike())){
            criteria.andShelvesCodeLike(SQLUtils.allLike(cond.getShelvesCodeLike()));
        }

        if (EmptyUtil.isNotEmpty(cond.getCreateTimeStart())){
            criteria.andCreateTimeGreaterThanOrEqualTo(cond.getCreateTimeStart());
        }
        if (EmptyUtil.isNotEmpty(cond.getCreateTimeEnd())){
            criteria.andCreateTimeLessThanOrEqualTo(cond.getCreateTimeEnd());
        }

        if (EmptyUtil.isNotEmpty(cond.getRefCode())){
            criteria.andRefCodeEqualTo(cond.getRefCode());
        }
        if (EmptyUtil.isNotEmpty(cond.getRefType())){
            criteria.andRefTypeEqualTo(cond.getRefType());
        }

        if (EmptyUtil.isNotEmpty(cond.getReplenishStates())){
            criteria.andReplenishStatesEqualTo(cond.getReplenishStates());
        }else if(CollectionUtils.isNotEmpty(cond.getReplenishStatesList())){
            criteria.andReplenishStatesIn(cond.getReplenishStatesList());
        }
        return example;
    }
}
