package com.thebeastshop.pegasus.service.warehouse.service.impl;

import com.thebeastshop.common.utils.BeanUtil;
import com.thebeastshop.pegasus.service.warehouse.exception.WarehouseException;
import com.thebeastshop.pegasus.service.warehouse.exception.WarehouseExceptionErrorCode;
import com.thebeastshop.pegasus.service.warehouse.model.*;
import com.thebeastshop.pegasus.service.warehouse.service.*;
import com.thebeastshop.pegasus.service.warehouse.vo.*;
import com.thebeastshop.pegasus.util.comm.DateUtil;
import com.thebeastshop.pegasus.util.comm.EmptyUtil;
import com.thebeastshop.pegasus.util.comm.NullUtil;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.*;

import static java.util.stream.Collectors.*;

@Service("whWmsMoveWarehouseService")
public class WhWmsMoveWarehouseServiceImpl implements WhWmsMoveWarehouseService {

    @Autowired
    private WhWmsHouseShelvesService whWmsHouseShelvesService;

    @Autowired
    private WhWmsSkuStockService whWmsSkuStockService;

    @Autowired
    private WhInfoService whInfoService;

    @Autowired
    private WhWmsMoveWarehouseProcessService whWmsMoveWarehouseProcessService;

    private static final String newPhyWh = "WH10580083";

    private static final String oldPhyWh = "WH02060001";

    @Override
    public boolean moveStock(List<WhWmsMoveHouseShelvesExcelVO> excelList,Long operatorId) {
        if(NullUtil.isNull(operatorId)){
            operatorId = 1L;
        }
        convertCode(excelList);
        List<String> sourceShelvesCodes = excelList.stream()
                .map(WhWmsMoveHouseShelvesExcelVO::getSourceShelves)
                .collect(toList());
        Map<String,WhWmsHouseShelves> sourceShelvesMap = whWmsHouseShelvesService.getHouseShelvesMapByCode(sourceShelvesCodes);
        String sourceMsg = phyWhShelvesCheck(sourceShelvesCodes,oldPhyWh,sourceShelvesMap);
        List<String> targetShelvesCodes = excelList.stream()
                .map(WhWmsMoveHouseShelvesExcelVO::getTargetShelves)
                .collect(toList());
        Map<String,WhWmsHouseShelves> targetShelvesMap = whWmsHouseShelvesService.getHouseShelvesMapByCode(targetShelvesCodes);
        String targetMsg = phyWhShelvesCheck(targetShelvesCodes,newPhyWh,targetShelvesMap);
        if(EmptyUtil.isNotEmpty(sourceMsg) || EmptyUtil.isNotEmpty(targetMsg)){
            throw new WarehouseException(WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED,sourceMsg +"<br/>"+ targetMsg);
        }
        Map<String,WhWmsHouseShelves> shelvesMap = new HashMap<>(sourceShelvesMap);
        shelvesMap.putAll(targetShelvesMap);
        //构建数据
        WhWmsMoveHouseStockVO moveHouseStock = buildWhWmsMoveStoreStockInfo(excelList,shelvesMap);
        moveHouseStock.setOperatorId(operatorId);
        //构建调拨记录
        buildAllotRcd(moveHouseStock);
        //构建出入数据
        buildPhyWhSkuStockRcd(moveHouseStock);
        //处理
        whWmsMoveWarehouseProcessService.process(moveHouseStock);
        return true;
    }

    private void buildAllotRcd(WhWmsMoveHouseStockVO moveHouseStock){
        Map<Integer,Map<String,Integer>> skuStatusMap = moveHouseStock.getSkuStockList().stream()
                .collect(groupingBy(
                        WhWmsMoveHouseSkuStockVO::getSkuStatus
                        ,groupingBy(WhWmsMoveHouseSkuStockVO::getSkuCode
                                ,summingInt(WhWmsMoveHouseSkuStockVO::getQuantity))));
        Map<Integer,String> defalutInWhMap = findPhyWhDefaultInWarehouseCode(moveHouseStock.getSourcePhyWhCode());
        List<WhAllotRcd> altRcdList = new ArrayList<>();
        skuStatusMap.forEach((skuStatus,skuMap)->{
            String whCode = defalutInWhMap.get(skuStatus);
            if(NullUtil.isNull(whCode)){
                throw new WarehouseException(WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED,String.format("商品状态[%s]无默认入库仓",skuStatus));
            }
            //调拨数据
            WhAllotRcd whAllotRcd = new WhAllotRcd();
            whAllotRcd.setSourcePhysicalWarehouseCode(moveHouseStock.getSourcePhyWhCode());
            whAllotRcd.setSourceWarehouseCode(whCode);
            whAllotRcd.setTargetPhysicalWarehouseCode(moveHouseStock.getTargetPhyWhCode());
            whAllotRcd.setTargetWarehouseCode(whCode);
            whAllotRcd.setCreateUserId(moveHouseStock.getOperatorId().intValue());
            whAllotRcd.setCreateTime(DateUtil.getNow());
            whAllotRcd.setEstimatedAllocationDate(DateUtil.getNow());
            whAllotRcd.setFinishTime(DateUtil.getNow());
            whAllotRcd.setAllotType(WhAllotRcd.TYPE_NORMAL);
            whAllotRcd.setAllotStatus(WhAllotRcd.STATUS_FINISHED);
            whAllotRcd.setRemark("搬仓调拨单，系统自动生成");
            List<WhAllotRcdSku> rcdSkuList = new ArrayList<>();
            skuMap.forEach((skuCode,quantity)->{
                WhAllotRcdSku rcdSku = new WhAllotRcdSku();
                rcdSku.setSkuCode(skuCode);
                rcdSku.setQuantity(quantity);
                rcdSkuList.add(rcdSku);
            });
            whAllotRcd.setWhAllotRcdSkuList(rcdSkuList);
            altRcdList.add(whAllotRcd);
        });
        moveHouseStock.setAltRcdList(altRcdList);
    }

    private void buildPhyWhSkuStockRcd(WhWmsMoveHouseStockVO moveHouseStock){
        Date now = DateUtil.getNow();
        String receiptNo = "MVWH_"+DateUtil.format(now,DateUtil.DATE_STR_FORMAT);
        List<WhWmsSkuStockRecord> skuStockRcdList = moveHouseStock.getSkuStockList().stream()
                .map(stock->{
                    List<WhWmsSkuStockRecord> stockRcdList = new ArrayList<>();
                    //出库记录数据
                    WhWmsSkuStockRecord skuOutStockRcd = BeanUtil.buildFrom(stock,WhWmsSkuStockRecord.class);
                    skuOutStockRcd.setPhysicalWarehouseCode(stock.getSourcePhyWhCode());
                    skuOutStockRcd.setHouseType(stock.getSourceShelvesHouseType());
                    skuOutStockRcd.setShelvesCode(stock.getSourceShelvesCode());
                    skuOutStockRcd.setQuantity(-stock.getQuantity());
                    skuOutStockRcd.setCreateUserId(moveHouseStock.getOperatorId());
                    skuOutStockRcd.setCreateTime(now);
                    skuOutStockRcd.setReceiptNo(receiptNo);
                    skuOutStockRcd.setMemo("搬仓出库");
                    skuOutStockRcd.setInOutType(WhCommand.TYPE_ALLOT_OUT);
                    skuOutStockRcd.setIsUpdateScm(1);
                    stockRcdList.add(skuOutStockRcd);
                    //入库记录数据
                    WhWmsSkuStockRecord skuInStockRcd = BeanUtil.buildFrom(skuOutStockRcd,WhWmsSkuStockRecord.class);
                    skuInStockRcd.setPhysicalWarehouseCode(stock.getTargetPhyWhCode());
                    skuInStockRcd.setHouseType(stock.getTargetShelvesHouseType());
                    skuInStockRcd.setShelvesCode(stock.getTargetShelvesCode());
                    skuInStockRcd.setQuantity(stock.getQuantity());
                    skuInStockRcd.setInOutType(WhCommand.TYPE_ALLOT_IN);
                    skuInStockRcd.setMemo("搬仓入库");
                    stockRcdList.add(skuInStockRcd);
                    return stockRcdList;
                })
                .flatMap(List::stream)
                .collect(toList());
        moveHouseStock.setSkuStockRcdList(skuStockRcdList);
    }

    private Map<Integer,String> findPhyWhDefaultInWarehouseCode(String phyWhCode){
        WhPhysicalWarehouseRelateWarehouseVO cond = new WhPhysicalWarehouseRelateWarehouseVO();
        cond.setPhysicalWarehouseCode(phyWhCode);
        cond.setType(WhWarehouseVO.RELATE_IN);
        List<WhPhysicalWarehouseRelateWarehouseVO> list = whInfoService.findPhysicalWarehouseRelateWarehouseByCond(cond);
        return list.stream().filter(relate->WhWarehouseVO.RELATE_IN.equals(relate.getType()))
                .collect(toMap(WhPhysicalWarehouseRelateWarehouseVO::getSkuStatus,WhPhysicalWarehouseRelateWarehouseVO::getWarehouseCode,(r1,r2)->r1));
    }


    private WhWmsMoveHouseStockVO buildWhWmsMoveStoreStockInfo(List<WhWmsMoveHouseShelvesExcelVO> excelList,Map<String,WhWmsHouseShelves> shelvesMap){
        List<String> sourceShelvesCodes = excelList.stream()
                .map(WhWmsMoveHouseShelvesExcelVO::getSourceShelves)
                .collect(toList());
        WhWmsSkuStockVO cond = new WhWmsSkuStockVO();
        cond.setSkuStatus(WhWarehouseVO.COMMODITY_STATUS_FOR_NONDEFECTIVE);
        cond.setShelvesCodeList(sourceShelvesCodes);
        List<WhWmsSkuStock> skuStockList = whWmsSkuStockService.getStockByCond(cond);
        if(EmptyUtil.isEmpty(skuStockList)){
            skuStockList = new ArrayList<>();
        }
        Map<String,List<WhWmsSkuStock>> stockMap = skuStockList.stream()
                .collect(groupingBy((skuStock)->buildMapKey(skuStock.getShelvesCode(),skuStock.getSkuCode())));
        List<WhWmsMoveHouseSkuStockVO> moveSkuStockList = excelList.stream()
                .map(excelVo->{
                    String key = buildMapKey(excelVo.getSourceShelves(),excelVo.getSkuCode());
                    List<WhWmsSkuStock> stockList = stockMap.get(key);
                    if(NullUtil.isNull(stockList)){
                        stockList = new ArrayList<>();
                    }
                    WhWmsHouseShelves targetShelves = shelvesMap.get(excelVo.getTargetShelves());
                    return stockList.stream().filter(stock->stock.getAmount()>0)
                            .map(skuStock->{
                                WhWmsMoveHouseSkuStockVO moveSkuStock = BeanUtil.buildFrom(skuStock,WhWmsMoveHouseSkuStockVO.class);
                                moveSkuStock.setSourcePhyWhCode(skuStock.getPhysicalWarehouseCode());
                                moveSkuStock.setSourceShelvesCode(skuStock.getShelvesCode());
                                moveSkuStock.setSourceShelvesHouseType(skuStock.getHouseType());
                                moveSkuStock.setTargetPhyWhCode(targetShelves.getPhysicalWarehouseCode());
                                moveSkuStock.setTargetShelvesCode(targetShelves.getCode());
                                moveSkuStock.setTargetShelvesHouseType(targetShelves.getHouseType());
                                moveSkuStock.setQuantity(skuStock.getAmount());
                                return moveSkuStock;
                            }).collect(toList());
                })
                .flatMap(List::stream)
                .collect(toList());
        WhWmsMoveHouseStockVO moveHouseStock = new WhWmsMoveHouseStockVO();
        moveHouseStock.setSourcePhyWhCode(oldPhyWh);
        moveHouseStock.setTargetPhyWhCode(newPhyWh);
        moveHouseStock.setSkuStockList(moveSkuStockList);
        return moveHouseStock;
    }

    private String buildMapKey(String shelveCode,String skuCode){
        return shelveCode + "_" + skuCode;
    }

    private void convertCode(List<WhWmsMoveHouseShelvesExcelVO> excelList){
        excelList.forEach(vo->{
            vo.setSkuCode(vo.getSkuCode().toUpperCase());
            vo.setSourceShelves(vo.getSourceShelves().toUpperCase());
            vo.setTargetShelves(vo.getTargetShelves().toUpperCase());
        });
    }

    private String phyWhShelvesCheck(List<String> sourceShelvesCodes,String phyWhCode,Map<String,WhWmsHouseShelves> sourceShelvesMap){
        StringBuilder errorBuffer = new StringBuilder();
        int rowNum = 1;
        for(String sourceShelvesCode : sourceShelvesCodes){
            rowNum++;
            WhWmsHouseShelves shelves = sourceShelvesMap.get(sourceShelvesCode);
            if(NullUtil.isNull(shelves)){
                errorBuffer.append(String.format("行[%s]库位[%s]不存在！<br/>",rowNum,sourceShelvesCode));
                continue;
            }
            if(!phyWhCode.equals(shelves.getPhysicalWarehouseCode())){
                errorBuffer.append(String.format("行[%s]库位[%s]物理仓非[%s]！<br/>",rowNum,sourceShelvesCode,phyWhCode));
            }
        }
        return errorBuffer.toString();
    }


}
