package com.thebeastshop.pegasus.service.warehouse.service.impl;

import com.thebeastshop.common.exception.BusinessException;
import com.thebeastshop.common.lock.RedisDistributLock;
import com.thebeastshop.pegasus.service.warehouse.WMSConstants;
import com.thebeastshop.pegasus.service.warehouse.cond.WhWmsMoveStorehouseBarcodeCond;
import com.thebeastshop.pegasus.service.warehouse.cond.WhWmsMoveStorehouseCond;
import com.thebeastshop.pegasus.service.warehouse.dao.WhWmsMoveStorehouseBarcodeMapper;
import com.thebeastshop.pegasus.service.warehouse.dao.WhWmsMoveStorehouseMapper;
import com.thebeastshop.pegasus.service.warehouse.enums.WhWmsMoveStorehouseStatusEnum;
import com.thebeastshop.pegasus.service.warehouse.exception.WarehouseException;
import com.thebeastshop.pegasus.service.warehouse.exception.WarehouseExceptionErrorCode;
import com.thebeastshop.pegasus.service.warehouse.model.*;
import com.thebeastshop.pegasus.service.warehouse.service.*;
import com.thebeastshop.pegasus.service.warehouse.vo.*;
import com.thebeastshop.pegasus.util.PegasusConstants;
import com.thebeastshop.pegasus.util.comm.BeanUtil;
import com.thebeastshop.pegasus.util.comm.DateUtil;
import com.thebeastshop.pegasus.util.comm.EmptyUtil;
import com.thebeastshop.pegasus.util.comm.NullUtil;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.*;
import java.util.concurrent.TimeUnit;
import static java.util.stream.Collectors.*;

/**
 * @author xw
 * @create 2019-04-03 11:16
 */
@Service("whWmsMoveStorehouseService")
public class WhWmsMoveStorehouseServiceImpl implements WhWmsMoveStorehouseService {

    private final static Logger log = LoggerFactory.getLogger(WhWmsMoveStorehouseServiceImpl.class);

    @Autowired
    private WhWmsMoveStorehouseMapper whWmsMoveStorehouseMapper;

    @Autowired
    private WhWmsMoveStorehouseBarcodeMapper whWmsMoveStorehouseBarcodeMapper;

    @Autowired
    private RedisDistributLock redisDistributLock;

    @Autowired
    private WhWmsOccupyService whWmsOccupyService;

    @Autowired
    private WhWmsHouseShelvesService whWmsHouseShelvesService;

    @Autowired
    private WhWmsSkuStockService whWmsSkuStockService;

    @Autowired
    private WhInfoService whInfoService;

    @Autowired
    private WhWmsMoveStockService whWmsMoveStockService;

    @Autowired
    private WhAllotService whAllotService;

    @Override
    public List<WhWmsMoveStorehouseVO> listWhWmsMoveStorehouseVOsByCond(WhWmsMoveStorehouseCond cond) {
        return whWmsMoveStorehouseMapper.listWhWmsMoveStorehouseVOsByCond(cond);
    }

    @Override
    public List<WhWmsMoveStorehouse> listWhWmsMoveStorehouseByCond(WhWmsMoveStorehouseCond cond) {
        WhWmsMoveStorehouseExample example = buildWhWmsMoveStorehouseExampleByCond(cond);
        return whWmsMoveStorehouseMapper.selectByExample(example);
    }

    @Override
    public void updateWhWmsMoveStorehouse(WhWmsMoveStorehouse record, WhWmsMoveStorehouseExample example) {
        whWmsMoveStorehouseMapper.updateByExampleSelective(record,example);
    }

    @Override
    public WhWmsMoveStorehouse findWhWmsMoveStorehouseById(Long id) {
        if (EmptyUtil.isEmpty(id)){
            return null;
        }
        return whWmsMoveStorehouseMapper.selectByPrimaryKey(id);
    }

    @Override
    public List<WhWmsMoveStorehouseBarcodeVO> listWhWmsMoveStorehouseBarcodeVOsByCond(WhWmsMoveStorehouseBarcodeCond cond) {
        return whWmsMoveStorehouseBarcodeMapper.listWhWmsMoveStorehouseBarcodeVOsByCond(cond);
    }

    @Override
    public List<WhWmsMoveStorehouseBarcode> listWhWmsMoveStorehouseBarcodeByCond(WhWmsMoveStorehouseBarcodeCond cond) {
        WhWmsMoveStorehouseBarcodeExample example = buildWhWmsMoveStorehouseBarcodeExampleByCond(cond);
        return whWmsMoveStorehouseBarcodeMapper.selectByExample(example);
    }

    @Override
    @Transactional
    public void addOrEditWhWmsMoveStorehouseVO(WhWmsMoveStorehouseVO moveStorehouseVO) {
        if (EmptyUtil.isNotEmpty(moveStorehouseVO.getId())){
            // 只新增搬仓行信息
            WhWmsMoveStorehouseVO info = findWhWmsMoveStorehouseById(moveStorehouseVO.getId(),false);
            moveStorehouseVO.setCode(info.getCode());
            moveStorehouseVO.setSourcePhyCode(info.getSourcePhyCode());
            addMoveStorehouseBarcodes(moveStorehouseVO.getId(), moveStorehouseVO.getCode(), moveStorehouseVO.getMoveStorehouseBarcodes());
        }else{
            // 新增搬仓及行信息
            addMoveStorehouseAndBarcodes(moveStorehouseVO);
        }
        // WMS占用库存
        wmsOccupy(moveStorehouseVO.getMoveStorehouseBarcodes(), moveStorehouseVO);
    }

    public void wmsOccupy(List<WhWmsMoveStorehouseBarcode> moveStorehouseBarcodes, WhWmsMoveStorehouseVO vo){
        if(EmptyUtil.isEmpty(vo.getSourcePhyCode()) || EmptyUtil.isEmpty(vo.getCode())){
            throw new WarehouseException(WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED,"参数异常");
        }
        //获取分布式锁
        String lockKey = "lock:whWmsMoveStorehouse:wmsOccupy:"+vo.getCode();
        try {
            Boolean getLock = redisDistributLock.tryLock(lockKey, 3L, TimeUnit.SECONDS);
            if (!getLock) {
                log.error("获取锁失败   false  lockKey="+lockKey);
                throw new WarehouseException(
                        WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED,
                        "请稍后重试!");
            }

            // 获取库位类型
            Map<String,WhWmsHouseShelves> shelvesMap = getHouseTypeMapByBarcodes(moveStorehouseBarcodes);

            List<WhWmsOccupyVO> occupyList = new ArrayList<>();
            for (WhWmsMoveStorehouseBarcode bc : moveStorehouseBarcodes) {
                if (EmptyUtil.isNotEmpty(shelvesMap.get(bc.getShelvesCode()))){
                    //创建原库位占用
                    WhWmsOccupyVO occupyVO = new WhWmsOccupyVO();
                    occupyVO.setSkuCode(bc.getSkuCode());
                    occupyVO.setBarCode(bc.getBarCode());
                    occupyVO.setSkuStatus(bc.getSkuStatus());
                    occupyVO.setAmount(-bc.getQuantity());
                    occupyVO.setHouseType(shelvesMap.get(bc.getShelvesCode()).getHouseType());
                    occupyVO.setPhysicalWarehouseCode(vo.getSourcePhyCode());
                    occupyVO.setReceiptsNo(vo.getCode()+"_"+bc.getId());
                    occupyVO.setOriShelvesCode(bc.getShelvesCode());
                    occupyVO.setType(WhWmsOccupyVO.TYPE_MOVE_STOREHOUSE);
                    occupyVO.setStatus(WhWmsOccupyVO.OCCUPIED);
                    occupyList.add(occupyVO);
                }
            }
            whWmsOccupyService.wmsBatchOccupy(occupyList);
        }finally {
            redisDistributLock.unLock(lockKey);
        }
    }

    private Map<String,WhWmsHouseShelves> getHouseTypeMapByBarcodes(List<WhWmsMoveStorehouseBarcode> moveStorehouseBarcodes){
        Set<String> shelvesCodes = new HashSet<>();
        for (WhWmsMoveStorehouseBarcode barcode : moveStorehouseBarcodes){
            shelvesCodes.add(barcode.getShelvesCode());
        }
        List<WhWmsHouseShelves> whWmsHouseShelves = whWmsHouseShelvesService.getHouseShelvesByCode(new ArrayList<>(shelvesCodes));
        Map<String,WhWmsHouseShelves> shelvesMap = new HashMap<>();
        for (WhWmsHouseShelves houseShelves : whWmsHouseShelves){
            shelvesMap.put(houseShelves.getCode(),houseShelves);
        }
        return shelvesMap;
    }

    private void addMoveStorehouseAndBarcodes(WhWmsMoveStorehouseVO moveStorehouseVO) {
        WhWmsMoveStorehouse moveStorehouse = addWhWmsMoveStorehouse(BeanUtil.buildFrom(moveStorehouseVO, WhWmsMoveStorehouse.class));
        moveStorehouseVO.setId(moveStorehouse.getId());
        addMoveStorehouseBarcodes(moveStorehouse.getId(), moveStorehouse.getCode(), moveStorehouseVO.getMoveStorehouseBarcodes());
    }

    // 保存搬仓
    private WhWmsMoveStorehouse addWhWmsMoveStorehouse(WhWmsMoveStorehouse moveStorehouse) {
        moveStorehouse.setCreateTime(DateUtil.getNow());
        whWmsMoveStorehouseMapper.insert(moveStorehouse);
        return moveStorehouse;
    }

    // 保存搬仓行
    private void addMoveStorehouseBarcodes(Long refId, String refCode, List<WhWmsMoveStorehouseBarcode> moveStorehouseBarcodes) {
        if (CollectionUtils.isNotEmpty(moveStorehouseBarcodes)){
            moveStorehouseBarcodes.forEach(bc -> {
                bc.setRefId(refId);
                bc.setRefCode(refCode);
                bc.setCreateTime(DateUtil.getNow());
                whWmsMoveStorehouseBarcodeMapper.insert(bc);
            });
        }
    }

    private WhWmsMoveStorehouseVO findWaitPutAwayMoveStorehouseByCode(String code){
        WhWmsMoveStorehouseExample example = new WhWmsMoveStorehouseExample();
        example.createCriteria().andCodeEqualTo(code)
                .andStatusEqualTo(WhWmsMoveStorehouseStatusEnum.truck_loaded.getValue());//已装车
        List<WhWmsMoveStorehouse> list = whWmsMoveStorehouseMapper.selectByExample(example);
        if(EmptyUtil.isEmpty(list)){
            throw new WarehouseException(WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED,String.format("[%s]无待上架任务！",code));
        }
        if(list.size() > 1){
            throw new WarehouseException(WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED,String.format("[%s]上架任务数据异常！",code));
        }
        WhWmsMoveStorehouseVO vo = BeanUtil.buildFrom(list.get(0),WhWmsMoveStorehouseVO.class);
        WhWmsMoveStorehouseBarcodeCond bcCond = new WhWmsMoveStorehouseBarcodeCond();
        bcCond.setRefId(vo.getId());
        List<WhWmsMoveStorehouseBarcodeVO> bcList = listWhWmsMoveStorehouseBarcodeVOsByCond(bcCond);
        vo.setMoveStorehouseBarcodeVOs(bcList);
        return vo;
    }

    @Override
    public WhWmsMoveStorehouseVO findWhWmsMoveStorehouseById(Long id,boolean cascade){
        WhWmsMoveStorehouse moveStorehouse = whWmsMoveStorehouseMapper.selectByPrimaryKey(id);
        WhWmsMoveStorehouseVO vo = BeanUtil.buildFrom(moveStorehouse,WhWmsMoveStorehouseVO.class);
        if(NullUtil.isNotNull(vo) && cascade){
            WhWmsMoveStorehouseBarcodeCond bcCond = new WhWmsMoveStorehouseBarcodeCond();
            bcCond.setRefId(vo.getId());
            List<WhWmsMoveStorehouseBarcodeVO> bcList = listWhWmsMoveStorehouseBarcodeVOsByCond(bcCond);
            vo.setMoveStorehouseBarcodeVOs(bcList);
        }
        return vo;
    }


    @Override
    @Transactional
    public boolean moveStorehousePutAway(WhWmsMoveStorehouseVO vo){
        if(WhWmsMoveStorehouseStatusEnum.in_storged.getValue().equals(vo.getStatus())){
            throw new WarehouseException(WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED
                    ,String.format("[%s]当前状态[%s]",vo.getCode(),WhWmsMoveStorehouseStatusEnum.getDescByValue(vo.getStatus())));
        }
        WhPhysicalWarehouse targetPhyWh = whInfoService.findPhysicalWarehouseByCode(vo.getTargetPhyCode());
        if(NullUtil.isNull(targetPhyWh)){
            throw new WarehouseException(WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED
                    ,String.format("[%s]物理仓不存在！",vo.getTargetPhyCode()));
        }
        WhWmsHouseShelvesVO targetShelves = whWmsHouseShelvesService.getHouseShelvesByCode(vo.getTargetShelvesCode());
        if(NullUtil.isNull(targetShelves) || !targetShelves.getPhysicalWarehouseCode().equals(vo.getTargetPhyCode())){
            throw new WarehouseException(WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED
                    ,String.format("[%s]%s库位不存在！",vo.getTargetPhyCode(),vo.getTargetShelvesCode()));
        }
        //更新搬仓数据
        updateMoveStorehouseFinish(vo);
        //释放wms占用
        releaseWmsOccupy(vo);
        if(vo.getSourcePhyCode().equals(vo.getTargetPhyCode())){
            //同一物理仓-移库
            return createMoveStockThenFinish(vo);
        }else{
            //生成搬仓数据
            //生成调拨记录
            generateWhAltRcd(vo);
            //处理物理仓库存
            processPhyWhSkuStock(vo);
            return true;
        }
    }

    private void updateMoveStorehouseFinish(WhWmsMoveStorehouseVO vo){
        if(NullUtil.isNull(vo.getFinishTime())){
            vo.setFinishTime(DateUtil.getNow());
        }
        if(NullUtil.isNull(vo.getFinishUserId())){
            vo.setFinishUserId(1L);
        }
        vo.setStatus(WhWmsMoveStorehouseStatusEnum.in_storged.getValue());
        WhWmsMoveStorehouseExample example = new WhWmsMoveStorehouseExample();
        example.createCriteria()
                .andIdEqualTo(vo.getId());
        boolean success = whWmsMoveStorehouseMapper.updateByExampleSelective(vo,example) == 1;
        if(!success){
            WhWmsMoveStorehouse tmp = findWhWmsMoveStorehouseById(vo.getId());
            throw new WarehouseException(WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED
                    ,String.format("当前状态[%s]",getMoveStorehouseStatusName(tmp.getStatus())));
        }
    }

    private String getMoveStorehouseStatusName(Integer status){
        WhWmsMoveStorehouseStatusEnum statusEnum = WhWmsMoveStorehouseStatusEnum.getEnumByValue(status);
        if(NullUtil.isNull(statusEnum)){
            return "";
        }else{
            return statusEnum.getDesc();
        }
    }

    private boolean releaseWmsOccupy(WhWmsMoveStorehouseVO vo){
        List<String> receiptsNoList = vo.getMoveStorehouseBarcodeVOs().stream()
                .map(bc->bc.getRefCode()+'_'+bc.getId())
                .collect(toList());
        WhWmsOccupyVO cond = new WhWmsOccupyVO();
        cond.setReceiptsNoList(receiptsNoList);
        cond.setType(WhWmsOccupyVO.TYPE_MOVE_STOREHOUSE);
        cond.setStatus(WhWmsOccupyVO.OCCUPIED);
        return whWmsOccupyService.releaseWmsOccupyByCond(cond);
    }

    private boolean generateWhAltRcd(WhWmsMoveStorehouseVO vo){
        Map<Integer,Map<String,Integer>> skuStatusMap = vo.getMoveStorehouseBarcodeVOs().stream()
                .collect(groupingBy(
                                WhWmsMoveStorehouseBarcodeVO::getSkuStatus
                                ,groupingBy(WhWmsMoveStorehouseBarcodeVO::getSkuCode
                                        ,summingInt(WhWmsMoveStorehouseBarcodeVO::getQuantity))));
        List<WhAllotRcd> altRcdList = new ArrayList<>();
        skuStatusMap.forEach((skuStatus,skuMap)->{
            WhWarehouse wh = whInfoService.findDefaultInOrOutWarehouseByPhyWhCode(vo.getSourcePhyCode(),WhWarehouseVO.RELATE_IN,skuStatus);
            //调拨数据
            WhAllotRcd whAllotRcd = new WhAllotRcd();
            whAllotRcd.setSourcePhysicalWarehouseCode(vo.getSourcePhyCode());
            whAllotRcd.setSourceWarehouseCode(wh.getCode());
            whAllotRcd.setTargetPhysicalWarehouseCode(vo.getTargetPhyCode());
            whAllotRcd.setTargetWarehouseCode(wh.getCode());
            whAllotRcd.setCreateUserId(vo.getFinishUserId().intValue());
            whAllotRcd.setCreateTime(DateUtil.getNow());
            whAllotRcd.setEstimatedAllocationDate(DateUtil.getNow());
            whAllotRcd.setFinishTime(DateUtil.getNow());
            whAllotRcd.setAllotType(WhAllotRcd.TYPE_NORMAL);
            whAllotRcd.setAllotStatus(WhAllotRcd.STATUS_FINISHED);
            whAllotRcd.setRemark("搬仓-自动创建");
            List<WhAllotRcdSku> rcdSkuList = new ArrayList<>();
            skuMap.forEach((skuCode,quantity)->{
                WhAllotRcdSku rcdSku = new WhAllotRcdSku();
                rcdSku.setSkuCode(skuCode);
                rcdSku.setQuantity(quantity);
                rcdSkuList.add(rcdSku);
            });
            whAllotRcd.setWhAllotRcdSkuList(rcdSkuList);
            altRcdList.add(whAllotRcd);
            //波次拣货数据-暂时跳过
        });
        whAllotService.recordAllotRcd(altRcdList);
        return true;
    }

    private boolean processPhyWhSkuStock(WhWmsMoveStorehouseVO vo){
        List<String> shelvesCodes = vo.getMoveStorehouseBarcodeVOs().stream()
                .map(WhWmsMoveStorehouseBarcodeVO::getShelvesCode)
                .distinct().collect(toList());
        List<WhWmsHouseShelves> sourceShelvesList = whWmsHouseShelvesService.getHouseShelvesByCode(shelvesCodes);
        Map<String,WhWmsHouseShelves> sourceShelvesMap = sourceShelvesList.stream()
                .collect(toMap(WhWmsHouseShelves::getCode,s->s,(s1,s2)->s1));
        WhWmsHouseShelvesVO targetShelves = whWmsHouseShelvesService.getHouseShelvesByCode(vo.getTargetShelvesCode());
        if(NullUtil.isNull(targetShelves)){
            throw new WarehouseException(WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED,String.format("[%s]库存不存在！",vo.getTargetShelvesCode()));
        }
        List<WhWmsSkuStockRecord> skuStockRcdList = vo.getMoveStorehouseBarcodeVOs().stream()
                .map(bc->{
                    List<WhWmsSkuStockRecord> stockRcdList = new ArrayList<>();
                    //出库记录数据
                    WhWmsHouseShelves sourceShelves = sourceShelvesMap.get(bc.getShelvesCode());
                    WhWmsSkuStockRecord skuOutStockRcd = BeanUtil.buildFrom(bc,WhWmsSkuStockRecord.class);
                    skuOutStockRcd.setPhysicalWarehouseCode(vo.getSourcePhyCode());
                    skuOutStockRcd.setHouseType(sourceShelves.getHouseType());
                    skuOutStockRcd.setShelvesCode(sourceShelves.getCode());
                    skuOutStockRcd.setQuantity(-bc.getQuantity());
                    skuOutStockRcd.setCreateUserId(vo.getFinishUserId());
                    skuOutStockRcd.setCreateTime(DateUtil.getNow());
                    skuOutStockRcd.setReceiptNo(vo.getId()+"_"+bc.getId());
                    skuOutStockRcd.setMemo("搬仓出库");
                    skuOutStockRcd.setInOutType(WhCommand.TYPE_ALLOT_OUT);
                    skuOutStockRcd.setIsUpdateScm(1);
                    stockRcdList.add(skuOutStockRcd);
                    //入库记录数据
                    WhWmsSkuStockRecord skuInStockRcd = BeanUtil.buildFrom(skuOutStockRcd,WhWmsSkuStockRecord.class);
                    skuInStockRcd.setPhysicalWarehouseCode(vo.getTargetPhyCode());
                    skuInStockRcd.setHouseType(targetShelves.getHouseType());
                    skuInStockRcd.setShelvesCode(targetShelves.getCode());
                    skuInStockRcd.setQuantity(bc.getQuantity());
                    skuInStockRcd.setInOutType(WhCommand.TYPE_ALLOT_IN);
                    skuInStockRcd.setMemo("搬仓入库");
                    stockRcdList.add(skuInStockRcd);
                    return stockRcdList;
                })
                .flatMap(List::stream)
                .collect(toList());
        whWmsSkuStockService.batchUpdateWhSkuStock(skuStockRcdList);
        return true;
    }

//    private boolean generateWhWmsConnectInfo(WhWmsMoveStorehouseVO vo){
//        WhWmsConnectInfoVO connect = new WhWmsConnectInfoVO();
//        connect.setPhysicalWarehouseCode(vo.getSourcePhyCode());
//        connect.setConnectDate(DateUtil.getNow());
//        connect.setSkuStatus(skuStatus);
//        connect.setConnectType(WMSConstants.ConnectType.MULTI_PRODUCT_MULTI_PIECE);
//        connect.setConnectStatus(WMSConstants.ConnectStatus.FINISHED);
//        connect.setInOutType(WhCommand.TYPE_ALLOT_OUT);
//        connect.setDistributeStatus(PegasusConstants.YES);
//    }



    private boolean createMoveStockThenFinish(WhWmsMoveStorehouseVO vo){
        //移库头信息
        WhWmsMoveStockVO moveStockInfo = new WhWmsMoveStockVO();
        moveStockInfo.setPhysicalWarehouseCode(vo.getSourcePhyCode());
        moveStockInfo.setMoveType(WhWmsMoveStockVO.MOVE_TYPE_COMMON);
        moveStockInfo.setReferenceCode(vo.getId().toString());
        moveStockInfo.setCreateUserId(vo.getFinishUserId());
        moveStockInfo.setCreateTime(DateUtil.getNow());
        moveStockInfo.setOperatorId(vo.getFinishUserId());
        moveStockInfo.setOperateTime(DateUtil.getNow());
        //库位数据
        List<String> shelvesCodes = vo.getMoveStorehouseBarcodeVOs().stream()
                .map(WhWmsMoveStorehouseBarcodeVO::getShelvesCode)
                .distinct().collect(toList());
        List<WhWmsHouseShelves> sourceShelvesList = whWmsHouseShelvesService.getHouseShelvesByCode(shelvesCodes);
        Map<String,WhWmsHouseShelves> sourceShelvesMap = sourceShelvesList.stream()
                .collect(toMap(WhWmsHouseShelves::getCode,s->s,(s1,s2)->s1));
        WhWmsHouseShelvesVO targetShelves = whWmsHouseShelvesService.getHouseShelvesByCode(vo.getTargetShelvesCode());
        if(NullUtil.isNull(targetShelves)){
            throw new WarehouseException(WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED,String.format("[%s]库存不存在！",vo.getTargetShelvesCode()));
        }
        //按商品状态分组
        Map<Integer,List<WhWmsMoveStorehouseBarcodeVO>> skuStatusMap = vo.getMoveStorehouseBarcodeVOs().stream()
                .collect(groupingBy(WhWmsMoveStorehouseBarcodeVO::getSkuStatus,toList()));
        //构建移库数据
        List<WhWmsMoveStockVO> moveStockList = new ArrayList<>();
        skuStatusMap.forEach((skuStatus,bcList)->{
            WhWmsMoveStockVO moveStock = BeanUtil.buildFrom(moveStockInfo,WhWmsMoveStockVO.class);//clone
            List<WhWmsMoveSkuVO> moveSkuList = bcList.stream().map(bc->{
                WhWmsHouseShelves sourceShelves = sourceShelvesMap.get(bc.getShelvesCode());
                WhWmsMoveSkuVO moveSku = BeanUtil.buildFrom(bc,WhWmsMoveSkuVO.class);
                moveSku.setOriginalHouseType(sourceShelves.getHouseType());
                moveSku.setOriginalShelvesCode(sourceShelves.getCode());
                moveSku.setTargetHouseType(targetShelves.getHouseType());
                moveSku.setTargetShelvesCode(targetShelves.getCode());
                moveSku.setAmount(bc.getQuantity());
                moveSku.setSkuStatus(skuStatus);
                return moveSku;
            }).collect(toList());
            moveStock.setMoveSkuList(moveSkuList);
            moveStockList.add(moveStock);
        });
        moveStockList.forEach(moveStock->{
            whWmsMoveStockService.rebuildMoveStock(moveStock);
            whWmsMoveStockService.executeMoveStock(moveStock);
        });
        return true;
    }


    private WhWmsMoveStorehouseExample buildWhWmsMoveStorehouseExampleByCond(WhWmsMoveStorehouseCond cond) {
        WhWmsMoveStorehouseExample example = new WhWmsMoveStorehouseExample();
        WhWmsMoveStorehouseExample.Criteria criteria = example.createCriteria();
        if (EmptyUtil.isNotEmpty(cond.getId())){
            criteria.andIdEqualTo(cond.getId());
        }else if(CollectionUtils.isNotEmpty(cond.getIds())){
            criteria.andIdIn(cond.getIds());
        }

        if (EmptyUtil.isNotEmpty(cond.getCode())){
            criteria.andCodeEqualTo(cond.getCode());
        }else if(CollectionUtils.isNotEmpty(cond.getCodes())){
            criteria.andCodeIn(cond.getCodes());
        }

        if (EmptyUtil.isNotEmpty(cond.getStatus())){
            criteria.andStatusEqualTo(cond.getStatus());
        }else if(CollectionUtils.isNotEmpty(cond.getStatusList())){
            criteria.andStatusIn(cond.getStatusList());
        }

        if (EmptyUtil.isNotEmpty(cond.getSourcePhyCode())){
            criteria.andSourcePhyCodeEqualTo(cond.getSourcePhyCode());
        }

        if (EmptyUtil.isNotEmpty(cond.getTargetPhyCode())){
            criteria.andTargetPhyCodeEqualTo(cond.getTargetPhyCode());
        }

        return example;
    }

    private WhWmsMoveStorehouseBarcodeExample buildWhWmsMoveStorehouseBarcodeExampleByCond(WhWmsMoveStorehouseBarcodeCond cond) {
        WhWmsMoveStorehouseBarcodeExample example = new WhWmsMoveStorehouseBarcodeExample();
        WhWmsMoveStorehouseBarcodeExample.Criteria criteria = example.createCriteria();
        if (EmptyUtil.isNotEmpty(cond.getId())){
            criteria.andIdEqualTo(cond.getId());
        }else if(CollectionUtils.isNotEmpty(cond.getIds())){
            criteria.andIdIn(cond.getIds());
        }

        if (EmptyUtil.isNotEmpty(cond.getRefId())){
            criteria.andRefIdEqualTo(cond.getRefId());
        }else if(CollectionUtils.isNotEmpty(cond.getRefIds())){
            criteria.andRefIdIn(cond.getRefIds());
        }

        if (EmptyUtil.isNotEmpty(cond.getRefCode())){
            criteria.andRefCodeEqualTo(cond.getRefCode());
        }else if(CollectionUtils.isNotEmpty(cond.getRefCodes())){
            criteria.andRefCodeIn(cond.getRefCodes());
        }

        if (EmptyUtil.isNotEmpty(cond.getSkuCode())){
            criteria.andSkuCodeEqualTo(cond.getSkuCode());
        }else if(CollectionUtils.isNotEmpty(cond.getSkuCodes())){
            criteria.andSkuCodeIn(cond.getSkuCodes());
        }

        if (EmptyUtil.isNotEmpty(cond.getBarCode())){
            criteria.andBarCodeEqualTo(cond.getBarCode());
        }else if(CollectionUtils.isNotEmpty(cond.getBarCodes())){
            criteria.andBarCodeIn(cond.getBarCodes());
        }

        if (EmptyUtil.isNotEmpty(cond.getShelvesCode())){
            criteria.andShelvesCodeEqualTo(cond.getShelvesCode());
        }else if(CollectionUtils.isNotEmpty(cond.getShelvesCodes())){
            criteria.andShelvesCodeIn(cond.getShelvesCodes());
        }

        if (EmptyUtil.isNotEmpty(cond.getSkuStatus())){
            criteria.andSkuStatusEqualTo(cond.getSkuStatus());
        }else if(CollectionUtils.isNotEmpty(cond.getSkuStatusList())){
            criteria.andSkuStatusIn(cond.getSkuStatusList());
        }else if(CollectionUtils.isNotEmpty(cond.getNotSkuStatusList())){
            criteria.andSkuStatusNotIn(cond.getNotSkuStatusList());
        }

        return example;
    }
}
