package com.thebeastshop.pegasus.service.warehouse.service.impl;

import com.thebeastshop.pegasus.service.warehouse.cond.WhWmsMoveStockCond;
import com.thebeastshop.pegasus.service.warehouse.cond.WhWmsReplenishInfoCond;
import com.thebeastshop.pegasus.service.warehouse.dao.WhWmsMoveSkuMapper;
import com.thebeastshop.pegasus.service.warehouse.dao.WhWmsMoveStockMapper;
import com.thebeastshop.pegasus.service.warehouse.exception.WarehouseException;
import com.thebeastshop.pegasus.service.warehouse.exception.WarehouseExceptionErrorCode;
import com.thebeastshop.pegasus.service.warehouse.model.*;
import com.thebeastshop.pegasus.service.warehouse.service.*;
import com.thebeastshop.pegasus.service.warehouse.vo.*;
import com.thebeastshop.pegasus.util.comm.*;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import page.Pagination;

import java.util.*;
import java.util.stream.Collectors;

/**
 * Created by devin.li on 2016/3/27.
 */
@Service("whWmsMoveStockService")
public class WhWmsMoveStockServiceImpl implements WhWmsMoveStockService {
    private static final Logger log = LoggerFactory.getLogger(WhWmsMoveStockServiceImpl.class);
    @Autowired
    private WhWmsMoveStockMapper            whWmsMoveStockMapper;

    @Autowired
    private WhWmsSkuStockService            whWmsSkuStockService;

    @Autowired
    private WhWmsOccupyService              whWmsOccupyService;

    @Autowired
    private WhWmsMoveSkuService             whWmsMoveSkuService;

    @Autowired
    private WhWmsHouseShelvesService        whWmsHouseShelvesService;
    @Autowired
    private WhWmsMoveSkuMapper whWmsMoveSkuMapper;

    @Autowired
    private WhWmsTaskAssignService whWmsTaskAssignService;

    @Autowired
    private WhWmsReplenishInfoService whWmsReplenishInfoService;

    @Override
    public Pagination<WhWmsMoveStockVO> getMoveStockByCond(WhWmsMoveStockCond cond) {
        Pagination<WhWmsMoveStockVO> page = new Pagination<>(cond.getCurrpage(),cond.getPagenum());
        List<WhWmsMoveStockVO> list = whWmsMoveStockMapper.getMoveStockByCond(cond);
        Integer counts = whWmsMoveStockMapper.countMoveStockByCond(cond);

        page.setResultList(list);
        page.setRecord(counts);
        return page;
    }

    @Override
    public List<WhWmsMoveStockVO> getMoveStockByCodes(List<String> moveCodes, boolean cascade) {
        if(EmptyUtil.isEmpty(moveCodes)){
            return new ArrayList<>();
        }
        WhWmsMoveStockExample example = new WhWmsMoveStockExample();
        example.createCriteria().andCodeIn(moveCodes);
        List<WhWmsMoveStock> list = whWmsMoveStockMapper.selectByExample(example);
        List<WhWmsMoveStockVO> rtList = BeanUtil.buildListFrom(list,WhWmsMoveStockVO.class);
        if(cascade){
            fullMoveStockSku(rtList);
        }
        return rtList;
    }

    private void fullMoveStockSku(List<WhWmsMoveStockVO> moveStockList){
        if(EmptyUtil.isEmpty(moveStockList)){
            return;
        }
        List<String> codeList = moveStockList.stream().map(WhWmsMoveStockVO::getCode)
                .collect(Collectors.toList());
        WhWmsMoveSkuExample example = new WhWmsMoveSkuExample();
        example.createCriteria().andMoveCodeIn(codeList);
        List<WhWmsMoveSku> moveSkuList = whWmsMoveSkuMapper.selectByExample(example);
        List<WhWmsMoveSkuVO> voList = BeanUtil.buildListFrom(moveSkuList,WhWmsMoveSkuVO.class);
        Map<String,List<WhWmsMoveSkuVO>> groupMapByMoveCode = voList.stream()
                .collect(Collectors.groupingBy(WhWmsMoveSkuVO::getMoveCode));
        moveStockList.forEach(moveStock->{
            List<WhWmsMoveSkuVO> list = groupMapByMoveCode.get(moveStock.getCode());
            moveStock.setMoveSkuList(list);
        });
    }



    @Override
    public List<WhWmsMoveSkuVO> findMoveStockSkuRf(String moveCode) {
        return whWmsMoveSkuMapper.findMoveStockSkuRf(moveCode);
    }

    @Override
    public WhWmsMoveStock getMoveStockById(Long id) {
        return whWmsMoveStockMapper.selectByPrimaryKey(new Long(id));
    }


    @Override
    public WhWmsMoveStockVO getMoveStockByCode(String code) {
        WhWmsMoveStockExample example = new WhWmsMoveStockExample();
        example.createCriteria().andCodeEqualTo(code);
        List<WhWmsMoveStock> moveStock = whWmsMoveStockMapper.selectByExample(example);
        if (CollectionUtils.isEmpty(moveStock)) return null;
        return BeanUtil.buildListFrom(moveStock,WhWmsMoveStockVO.class).get(0);
    }



    /**
     * 生成一个唯一的移库单号
     * @return
     */
    private String createNewStockCode(){
        String newCode = "MS"+DateUtil.format(new Date(),"yyyyMMddHHmmssSSS")+getCodeRandom();
        if (EmptyUtil.isNotEmpty(getMoveStockByCode(newCode))){
            newCode = createNewStockCode();
        }
        return newCode;
    }

    private int getCodeRandom(){
        return (int)Math.floor(Math.random()*10);
    }

    @Override
    @Transactional
    public Boolean newMoveStock(WhWmsMoveStockVO moveStockVO) {
        //重构移库信息
        rebuildMoveStock(moveStockVO);
        //检查库存
        moveStockCheck(moveStockVO);
        //创建移库单
        int newMoveStock = whWmsMoveStockMapper.insert(moveStockVO);
        if (newMoveStock<=0) {
            throw new WarehouseException(WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED, "移库创建失败！");
        }
        //创建移库行
        whWmsMoveSkuService.batchCreateMoveSku(moveStockVO.getMoveSkuList());
        //构造占用数据
        List<WhWmsOccupyVO> occupyList = buildMoveStockOccupy(moveStockVO);
        whWmsOccupyService.wmsBatchOccupy(occupyList);
        return true;
    }

    private List<WhWmsMoveSku> findMoveSkuListByMoveCode(String moveCode){
        WhWmsMoveSkuExample example = new WhWmsMoveSkuExample();
        example.createCriteria().andMoveCodeEqualTo(moveCode);
        return whWmsMoveSkuMapper.selectByExample(example);
    }

    @Override
    public boolean buildMoveStock(WhWmsMoveStockVO moveStockVO) {
        //重构移库信息
        rebuildMoveStock(moveStockVO);
        //检查库存
        moveStockCheck(moveStockVO);
        List<WhWmsMoveSkuVO> skuList = moveStockVO.getMoveSkuList();
        List<WhWmsOccupyVO> occupyList = new ArrayList<>();
        for (WhWmsMoveSkuVO moveSkuVO :skuList) {
            //创建原库位占用
            WhWmsOccupyVO occupyOri = new WhWmsOccupyVO();
            occupyOri.setSkuCode(moveSkuVO.getSkuCode());
            occupyOri.setBarCode(moveSkuVO.getBarCode());
            occupyOri.setSkuStatus(moveSkuVO.getSkuStatus());
            occupyOri.setAmount(-moveSkuVO.getAmount());
            occupyOri.setHouseType(moveSkuVO.getOriginalHouseType());
            occupyOri.setPhysicalWarehouseCode(moveStockVO.getPhysicalWarehouseCode());
            occupyOri.setReceiptsNo(moveStockVO.getCode());
            occupyOri.setOriShelvesCode(moveSkuVO.getOriginalShelvesCode());
            if (EmptyUtil.isEmpty(moveSkuVO.getMoveType())){
                occupyOri.setType(WhWmsOccupyVO.TYPE_MOVE);
            }
            occupyOri.setStatus(WhWmsOccupyVO.OCCUPIED);
            occupyList.add(occupyOri);
        }
        moveStockVO.setOccupyList(occupyList);
        return true;
    }

    @Override
    @Transactional
    public String createMoveStock(WhWmsMoveStockVO moveStockVO) {
        //创建移库单
        int newMoveStock = whWmsMoveStockMapper.insert(moveStockVO);
        if (newMoveStock<=0) {
            throw new WarehouseException(WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED, "移库创建失败！");
        }
        whWmsMoveSkuService.batchCreateMoveSku(moveStockVO.getMoveSkuList());
        //占用构造许移库行id
        List<WhWmsOccupyVO> occupyList = buildMoveStockOccupy(moveStockVO);
        whWmsOccupyService.wmsBatchOccupy(occupyList);
        return moveStockVO.getCode();
    }

    //构造移库占用数据-原库位+目标库位+批次号+商品状态唯一
    private List<WhWmsOccupyVO> buildMoveStockOccupy(WhWmsMoveStockVO moveStockVO){
        //构造占用数据
        List<WhWmsMoveSku> moveSkuList = findMoveSkuListByMoveCode(moveStockVO.getCode());
        List<WhWmsOccupyVO> occupyList = new ArrayList<>();
        for (WhWmsMoveSku moveSkuVO :moveSkuList) {
            //创建原库位占用
            WhWmsOccupyVO occupyOri = new WhWmsOccupyVO();
            occupyOri.setSkuCode(moveSkuVO.getSkuCode());
            occupyOri.setBarCode(moveSkuVO.getBarCode());
            occupyOri.setSkuStatus(moveSkuVO.getSkuStatus());
            occupyOri.setAmount(-moveSkuVO.getAmount());
            occupyOri.setHouseType(moveSkuVO.getOriginalHouseType());
            occupyOri.setPhysicalWarehouseCode(moveStockVO.getPhysicalWarehouseCode());
            occupyOri.setReceiptsNo(moveSkuVO.getId().toString());
            occupyOri.setOriShelvesCode(moveSkuVO.getOriginalShelvesCode());
            if (EmptyUtil.isEmpty(moveStockVO.getMoveType())){
                occupyOri.setType(WhWmsOccupyVO.TYPE_MOVE);
            }
            occupyOri.setStatus(WhWmsOccupyVO.OCCUPIED);
            occupyList.add(occupyOri);
        }
        return occupyList;
    }

    @Override
    public String executeMoveStock(WhWmsMoveStockVO moveStockVO) {
        moveStockVO.setOperateTime(DateUtil.getNow());
        moveStockVO.setOperatorId(moveStockVO.getCreateUserId());
        moveStockVO.setStatus(WhWmsMoveStockVO.STATUS_FINISHED);
        int newMoveStock = whWmsMoveStockMapper.insert(moveStockVO);
        if (newMoveStock<=0) {
            throw new WarehouseException(WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED, "移库创建失败！");
        }
        //创建移库行
        whWmsMoveSkuService.batchCreateMoveSku(moveStockVO.getMoveSkuList());
        //更新库存
        for(WhWmsMoveSkuVO moveSkuVO : moveStockVO.getMoveSkuList()){
            //对原库位库存进行库存扣减
            whWmsSkuStockService.updateStockByCond(
                    -moveSkuVO.getAmount(), moveStockVO.getPhysicalWarehouseCode(),
                    moveSkuVO.getOriginalHouseType(), moveSkuVO.getBarCode(), moveSkuVO.getOriginalShelvesCode(),
                    moveSkuVO.getSkuCode(), moveSkuVO.getSkuStatus(),WhInvRcd.TYPE_MOVE_OUT,moveSkuVO.getMoveCode(),moveStockVO.getOperatorId());
            //在目标库位库存基础上增加库存
            whWmsSkuStockService.updateStockByCond(
                    moveSkuVO.getAmount(), moveStockVO.getPhysicalWarehouseCode(),
                    moveSkuVO.getTargetHouseType(), moveSkuVO.getBarCode(), moveSkuVO.getTargetShelvesCode(),
                    moveSkuVO.getSkuCode(), moveSkuVO.getSkuStatus(),WhInvRcd.TYPE_MOVE_IN,moveSkuVO.getMoveCode(),moveStockVO.getOperatorId());
        }
        return moveStockVO.getCode();
    }

    @Override
    @Transactional
    public String quickMoveStock(WhWmsMoveStockVO moveStockVO){
        rebuildMoveStock(moveStockVO);
        moveStockCheck(moveStockVO);
        //创建移库单直接完成完成
        moveStockVO.setOperateTime(DateUtil.getNow());
        moveStockVO.setOperatorId(moveStockVO.getCreateUserId());
        moveStockVO.setStatus(WhWmsMoveStockVO.STATUS_FINISHED);
        int newMoveStock = whWmsMoveStockMapper.insert(moveStockVO);
        if (newMoveStock<=0) {
            throw new WarehouseException(WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED, "移库创建失败！");
        }
        //创建移库行
        whWmsMoveSkuService.batchCreateMoveSku(moveStockVO.getMoveSkuList());
        //更新库存
        for(WhWmsMoveSkuVO moveSkuVO : moveStockVO.getMoveSkuList()){
            //对原库位库存进行库存扣减
            whWmsSkuStockService.updateStockByCond(
                    -moveSkuVO.getAmount(), moveStockVO.getPhysicalWarehouseCode(),
                    moveSkuVO.getOriginalHouseType(), moveSkuVO.getBarCode(), moveSkuVO.getOriginalShelvesCode(),
                    moveSkuVO.getSkuCode(), moveSkuVO.getSkuStatus(),WhInvRcd.TYPE_MOVE_OUT,moveSkuVO.getMoveCode(),moveStockVO.getOperatorId());
            //在目标库位库存基础上增加库存
            whWmsSkuStockService.updateStockByCond(
                    moveSkuVO.getAmount(), moveStockVO.getPhysicalWarehouseCode(),
                    moveSkuVO.getTargetHouseType(), moveSkuVO.getBarCode(), moveSkuVO.getTargetShelvesCode(),
                    moveSkuVO.getSkuCode(), moveSkuVO.getSkuStatus(),WhInvRcd.TYPE_MOVE_IN,moveSkuVO.getMoveCode(),moveStockVO.getOperatorId());
        }

        return moveStockVO.getCode();
    }

    //重构移库信息
    public void rebuildMoveStock(final WhWmsMoveStockVO moveStockVO){
        if(EmptyUtil.isEmpty(moveStockVO)){
            throw new WarehouseException(WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED,"参数为空");
        }
        if(EmptyUtil.isEmpty(moveStockVO.getMoveSkuList())){
            throw new WarehouseException(WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED,"移库行信息为空");
        }
        //设置移库单号
        moveStockVO.setCode(createNewStockCode());
        List<WhWmsMoveSkuVO> moveSkuVOs = moveStockVO.getMoveSkuList();
        Map<String,WhWmsMoveSkuVO> map = new HashMap<>();
        for(WhWmsMoveSkuVO vo:moveSkuVOs){
            StringBuffer sb = new StringBuffer();
            sb.append(vo.getOriginalShelvesCode());
            sb.append("_");
            sb.append(vo.getSkuCode());
            sb.append("_");
            sb.append(vo.getBarCode());
            sb.append("_");
            sb.append(vo.getSkuStatus());
            sb.append("_")
                    .append(vo.getTargetShelvesCode());
            if(vo.getAmount() > 0){
                if(map.get(sb.toString()) == null){
                    map.put(sb.toString(),vo);
                }else{
                    WhWmsMoveSkuVO whWmsMoveSkuVO = map.get(sb.toString());
                    whWmsMoveSkuVO.setAmount(whWmsMoveSkuVO.getAmount() + vo.getAmount());
                }
            }
        }
        List<WhWmsMoveSkuVO> moveSkuVOList = new ArrayList<>();
        for(WhWmsMoveSkuVO vo:map.values()){
            vo.setMoveCode(moveStockVO.getCode());
            moveSkuVOList.add(vo);
        }
        moveStockVO.setMoveSkuList(moveSkuVOList);
        moveStockVO.setStatus(WhWmsMoveStockVO.STATUS_WAITTINT_OUT);
        moveStockVO.setCreateTime(new Date());
    }

    //移库检查
    public void moveStockCheck(WhWmsMoveStockVO moveStockVO){
        WhWmsSkuStockVO cond = new WhWmsSkuStockVO();
        for (WhWmsMoveSkuVO moveSkuVO:moveStockVO.getMoveSkuList()){
            if(     EmptyUtil.isEmpty(moveSkuVO.getPhysicalWarehouseCode()) ||
                    EmptyUtil.isEmpty(moveSkuVO.getOriginalHouseType()) ||
                    EmptyUtil.isEmpty(moveSkuVO.getOriginalShelvesCode()) ||
                    EmptyUtil.isEmpty(moveSkuVO.getTargetShelvesCode()) ||
                    EmptyUtil.isEmpty(moveSkuVO.getTargetHouseType()) ||
                    EmptyUtil.isEmpty(moveSkuVO.getBarCode())||
                    moveSkuVO.getAmount()<=0){
                throw new WarehouseException(WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED,"移库行参数信息为空</br>");
            }
            if(EmptyUtil.isEmpty(whWmsHouseShelvesService.getHouseShelvesByCode(moveSkuVO.getOriginalShelvesCode()))){
                throw new WarehouseException(WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED,
                        "源库位["+moveSkuVO.getOriginalShelvesCode()+"]不存在</br>");
            }
            if(EmptyUtil.isEmpty(whWmsHouseShelvesService.getHouseShelvesByCode(moveSkuVO.getTargetShelvesCode()))){
                throw new WarehouseException(WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED,
                        "目标库位["+moveSkuVO.getTargetShelvesCode()+"]不存在</br>");
            }
            if(NumberUtil.isNullOrZero(moveSkuVO.getAmount())){
                throw new WarehouseException(WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED,"移库数量为空或零</br>");
            }
            if(moveSkuVO.getOriginalShelvesCode().equals(moveSkuVO.getTargetShelvesCode())){
                throw new WarehouseException(WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED,"源库位和目标库位一致</br>");
            }
            cond.setSkuStatus(moveSkuVO.getSkuStatus());
            cond.setBarCode(moveSkuVO.getBarCode());
            cond.setPhysicalWarehouseCode(moveStockVO.getPhysicalWarehouseCode());
            cond.setHouseType(moveSkuVO.getOriginalHouseType());
            cond.setShelvesCode(moveSkuVO.getOriginalShelvesCode());
            cond.setSkuCode(moveSkuVO.getSkuCode());
//            List<WhWmsSkuStockVO> stock = whWmsSkuStockService.findWmsSkuAvailableAccount(cond);
//            if(EmptyUtil.isEmpty(stock)){//库存检查
//                throw new WarehouseException("","["+moveSkuVO.getBarCode()+"]" +
//                        "["+moveSkuVO.getOriginalShelvesCode()+"],[0]<["+moveSkuVO.getAmount()+"]");
//            }
//            if (CollectionUtils.isNotEmpty(stock)){
//                if (stock.get(0).getAvailableAccount()<moveSkuVO.getAmount())
//                    throw new WarehouseException("","["+moveSkuVO.getBarCode()+"]" +
//                            "["+moveSkuVO.getOriginalShelvesCode()+"],["+stock.get(0).getAvailableAccount()+"]<["+moveSkuVO.getAmount()+"]");
//            }
        }
    }

    @Override
    @Transactional
    public Boolean createMoveStockThenFinish(WhWmsMoveStockVO moveStockVO) {
        //return newMoveStock(moveStockVO) && doMoveStockAndReleaseOccupyAndFinish(moveStockVO.getCode(),moveStockVO.getCreateUserId());
        quickMoveStock(moveStockVO);
        return true;
    }

    @Override
    @Transactional
    public Boolean moveStockFinishByMoveSku(String moveCode, Long operatorId, Long moveSkuId) {
        int result = 0;
        Boolean update = false;
        WhWmsMoveStockVO moveStockVO =  getMoveStockByCode(moveCode);
        if (EmptyUtil.isEmpty(moveStockVO)) throw new WarehouseException("","error,not fund["+moveCode+"]");
        List<WhWmsMoveSkuVO> moveSkuVOs = moveStockVO.getMoveSkuVOs();
        boolean isFinished = true;
        //循环同一移库单号移库任务移库操作,移库后释放占用,并更新库存
        if (WhWmsMoveStockVO.STATUS_WAITTINT_OUT.equals(moveStockVO.getStatus())
                || WhWmsMoveStockVO.STATUS_PROCESSING.equals(moveStockVO.getStatus())){
            for (WhWmsMoveSkuVO moveSkuVO :moveSkuVOs ) {
                if(moveSkuVO.getId().equals(moveSkuId)){
                    if(moveSkuVO.getStatus() == null || moveSkuVO.getStatus().equals(WhWmsMoveSku.MOVE_SKU_STATUS_WAIT_MOVE)){
                        Boolean success = doMoveStock(moveSkuVO, moveStockVO,operatorId);
                    }
                }
                if(moveSkuVO.getStatus() == null || moveSkuVO.getStatus().equals(WhWmsMoveSku.MOVE_SKU_STATUS_WAIT_MOVE)){
                    isFinished = false;
                }
            }
            //移库单任务行全部成功后，更新移库单状态
            if (isFinished){
                update = updateMoveStockStatusToFinish(moveStockVO);
            }
            //更新状态为部分完成
            if(!isFinished && WhWmsMoveStockVO.STATUS_WAITTINT_OUT.equals(moveStockVO.getStatus())){
                updateMoveStockStatusToProcessing(moveStockVO.getId());
            }
            return update;
        }
        else if(moveStockVO.getStatus()==WhWmsMoveStockVO.STATUS_WAITTING_IN){
            throw new WarehouseException("","["+moveCode+"]状态不合法");

        }else if(moveStockVO.getStatus()==WhWmsMoveStockVO.STATUS_FINISHED){
            throw new WarehouseException("","移库单已完成["+moveCode+"]");
        }
        return false;
    }

    @Override
    @Transactional
    public Boolean moveStockFinishByMoveSku(String moveCode, Long operatorId, Long moveSkuId, WhWmsTaskAssignVO taskAssignVO) {
        boolean result = moveStockFinishByMoveSku(moveCode,operatorId,moveSkuId);
        {
            WhWmsMoveStockVO moveStockVO =  getMoveStockByCode(moveCode);
            if(WhWmsMoveStockVO.STATUS_FINISHED.equals(moveStockVO.getStatus())){
                whWmsTaskAssignService.finishTaskAssigned(taskAssignVO.getId(),taskAssignVO.getOperatorId());
            }
        }
        return result;
    }

    @Override
    @Transactional
    public Boolean moveStockFinishByMoveSku(WhWmsMoveSkuVO whWmsMoveSkuVO, WhWmsTaskAssignVO taskAssignVO) {
        WhWmsMoveSkuVO oriMoveSkuVO = whWmsMoveSkuService.getMoveSkuById(whWmsMoveSkuVO.getId());
        if(NullUtil.isNull(oriMoveSkuVO)){
            throw new WarehouseException(WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED,String.format("[%s]移库行不存在！",whWmsMoveSkuVO.getId()));
        }
        if(!oriMoveSkuVO.getTargetShelvesCode().equals(whWmsMoveSkuVO.getTargetShelvesCode())){
            //目标库位变更
            WhWmsMoveSku update = new WhWmsMoveSku();
            update.setId(oriMoveSkuVO.getId());
            update.setRecommendTargetShelvesCode(oriMoveSkuVO.getTargetShelvesCode());
            update.setRecommendTargetHouseType(oriMoveSkuVO.getTargetHouseType());
            update.setTargetShelvesCode(whWmsMoveSkuVO.getTargetShelvesCode());
            update.setTargetHouseType(whWmsMoveSkuVO.getTargetHouseType());
            whWmsMoveSkuMapper.updateByPrimaryKeySelective(update);
        }
        if(WhWmsMoveStockVO.STATUS_FINISHED.equals(oriMoveSkuVO.getStatus())){
            throw new WarehouseException(WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED,String.format("[%s]移库行已完成",whWmsMoveSkuVO.getId()));
        }
        if(WhWmsMoveStockVO.STATUS_CANCEL.equals(oriMoveSkuVO.getStatus())){
            throw new WarehouseException(WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED,String.format("[%s]移库行已取消",whWmsMoveSkuVO.getId()));
        }
        boolean result = moveStockFinishByMoveSku(oriMoveSkuVO.getMoveCode(),taskAssignVO.getOperatorId(),oriMoveSkuVO.getId(),taskAssignVO);
        if(result && !oriMoveSkuVO.getTargetShelvesCode().equals(whWmsMoveSkuVO.getTargetShelvesCode())){
            WhWmsMoveSku update = new WhWmsMoveSku();
            update.setId(oriMoveSkuVO.getId());
            update.setTargetShelvesCode(whWmsMoveSkuVO.getTargetShelvesCode());
            whWmsMoveSkuMapper.updateByPrimaryKeySelective(update);
        }
        return result;
    }

    @Override
    public List<WhWmsMoveStock> findWmsMoveStockByPrdCode(String prdCode) {
        return whWmsMoveStockMapper.findWmsMoveStockByPrdCode(prdCode);
    }

    @Override
    public Integer cancelWmsMoveStockByIdList(List<Long> idList) {
        return whWmsMoveStockMapper.cancelWmsMoveStockByIdList(idList);
    }

    @Override
    @Transactional
    public void cancelMoveStock(List<WhWmsMoveStockVO> list,Long operatorId) {
        if(list != null && list.size() > 0){
            for(WhWmsMoveStockVO vo:list){
                vo.getMoveSkuList()
                        .forEach(moveSku->{
                            WhWmsOccupyVO occupyVO = new WhWmsOccupyVO();
                            occupyVO.setReceiptsNo(moveSku.getId().toString());
                            occupyVO.setType(WhWmsOccupyVO.TYPE_MOVE);
                            whWmsOccupyService.releaseWmsOccupyByMove(occupyVO);
                        });
                WhWmsMoveStock whWmsMoveStock = new WhWmsMoveStock();
                whWmsMoveStock.setId(vo.getId());
                whWmsMoveStock.setStatus(WhWmsMoveStockVO.STATUS_CANCEL);
                whWmsMoveStock.setOperatorId(operatorId);
                whWmsMoveStock.setOperateTime(Calendar.getInstance().getTime());
                whWmsMoveStockMapper.updateByPrimaryKeySelective(whWmsMoveStock);
            }
        }
    }

    @Override
    @Transactional
    public Boolean doMoveStockAndReleaseOccupyAndFinish(String moveCode,Long operatorId) {
        int result = 0;
        Boolean update = false;
        WhWmsMoveStockVO moveStockVO =  getMoveStockByCode(moveCode);
        if (EmptyUtil.isEmpty(moveStockVO)){
            throw new WarehouseException(WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED,"移库单["+moveCode+"]不存在");
        }
        List<WhWmsMoveSkuVO> moveSkuVOs = moveStockVO.getMoveSkuVOs();
        //循环同一移库单号移库任务移库操作,移库后释放占用,并更新库存
        if (WhWmsMoveStockVO.STATUS_WAITTINT_OUT.equals(moveStockVO.getStatus())){
            for (WhWmsMoveSkuVO moveSkuVO :moveSkuVOs ) {
                Boolean success = doMoveStock(moveSkuVO, moveStockVO,operatorId);
                if (success) result += 1;
            }
            //移库单任务行全部成功后，更新移库单状态
            if (result == moveSkuVOs.size()) {
                moveStockVO.setOperatorId(operatorId);
                moveStockVO.setOperateTime(DateUtil.getNow());
                update = updateMoveStockStatusToFinish(moveStockVO);
            }
            return update;
        }
        else if(moveStockVO.getStatus()==WhWmsMoveStockVO.STATUS_WAITTING_IN){
            throw new WarehouseException("","["+moveCode+"]not open");

        }else if(WhWmsMoveStockVO.STATUS_FINISHED.equals(moveStockVO.getStatus())){
            throw new WarehouseException(WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED,"移库单["+moveCode+"]已完成");
        }
       return false;
    }


    /**
     * 操作一条移库行任务
     * @param moveSkuVO
     * @return
     */
    @Transactional
    private Boolean doMoveStock(WhWmsMoveSkuVO moveSkuVO,WhWmsMoveStockVO moveStockVO,Long operatorId){
        if(EmptyUtil.isEmpty(moveSkuVO.getPhysicalWarehouseCode()) ||
                EmptyUtil.isEmpty(moveSkuVO.getAmount()) ||
                EmptyUtil.isEmpty(moveSkuVO.getOriginalHouseType()) ||
                EmptyUtil.isEmpty(moveSkuVO.getOriginalShelvesCode()) ||
                EmptyUtil.isEmpty(moveSkuVO.getOriginalHouseType()) ||
                EmptyUtil.isEmpty(moveSkuVO.getTargetShelvesCode()) ||
                EmptyUtil.isEmpty(moveSkuVO.getTargetHouseType()) ||
                EmptyUtil.isEmpty(moveSkuVO.getBarCode())){
            throw new WarehouseException("","null param!");
        }
        long time1 = Calendar.getInstance().getTimeInMillis();
        //查询该条移库任务的源库位占用记录
        WhWmsOccupyVO oriOccupyCond = new WhWmsOccupyVO();
        oriOccupyCond.setBarCode(moveSkuVO.getBarCode());
        oriOccupyCond.setOriShelvesCode(moveSkuVO.getOriginalShelvesCode());
        oriOccupyCond.setAmount(-moveSkuVO.getAmount());
        oriOccupyCond.setReceiptsNo(moveSkuVO.getId().toString());
        oriOccupyCond.setSkuStatus(moveSkuVO.getSkuStatus());
        oriOccupyCond.setStatus(WhWmsOccupyVO.OCCUPIED);
        oriOccupyCond.setSkuCode(moveSkuVO.getSkuCode());
        oriOccupyCond.setPhysicalWarehouseCode(moveStockVO.getPhysicalWarehouseCode());
        oriOccupyCond.setHouseType(moveSkuVO.getOriginalHouseType());
        List<WhWmsOccupyVO> occupyVOList = whWmsOccupyService.getWmsOccupyByCond(oriOccupyCond);
        if(occupyVOList.size()>1){
            throw new WarehouseException("","the occupy is not unique["+moveStockVO.getCode()+"]["+moveSkuVO.getBarCode()+"]");
        }
        if (CollectionUtils.isEmpty(occupyVOList))
            throw new WarehouseException("","the occupy is not exist["+moveStockVO.getCode()+"]["+moveSkuVO.getBarCode()+"]");
        long time2 = Calendar.getInstance().getTimeInMillis();
        log.warn(moveStockVO.getCode()+"移库单执行第1步耗费时间：" + (time2-time1) + ";skuCode：" + moveSkuVO.getSkuCode());


//        if (CollectionUtils.isEmpty(whWmsOccupyService.getWmsOccupyByCond(tarOccupyCond)))
//            throw new WarehouseException("","error,the occupy is not exist["+moveStockVO.getCode()+"]["+moveSkuVO.getBarCode()+"]");
        //对原库位库存进行库存扣减
        whWmsSkuStockService.updateStockByCond(
                -moveSkuVO.getAmount(), moveStockVO.getPhysicalWarehouseCode(),
                moveSkuVO.getOriginalHouseType(), moveSkuVO.getBarCode(), moveSkuVO.getOriginalShelvesCode(),
                moveSkuVO.getSkuCode(), moveSkuVO.getSkuStatus(),WhInvRcd.TYPE_MOVE_OUT,moveSkuVO.getMoveCode(),operatorId);
        long time3 = Calendar.getInstance().getTimeInMillis();
        log.warn(moveStockVO.getCode()+"移库单执行第2步耗费时间：" + (time3-time2) + ";skuCode：" + moveSkuVO.getSkuCode());
        //释放源库位库存占用
        boolean success = whWmsOccupyService.releaseWmsOccupyByMove(oriOccupyCond);
        if(!success){
            throw new WarehouseException(WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED,"移库释放占用失败，请稍后重试！");
        }
        long time4 = Calendar.getInstance().getTimeInMillis();
        log.warn(moveStockVO.getCode()+"移库单执行第3步耗费时间：" + (time4-time3) + ";skuCode：" + moveSkuVO.getSkuCode());
        //在目标库位库存基础上增加库存
        whWmsSkuStockService.updateStockByCond(
                moveSkuVO.getAmount(), moveStockVO.getPhysicalWarehouseCode(),
                moveSkuVO.getTargetHouseType(), moveSkuVO.getBarCode(), moveSkuVO.getTargetShelvesCode(),
                moveSkuVO.getSkuCode(), moveSkuVO.getSkuStatus(),WhInvRcd.TYPE_MOVE_IN,moveSkuVO.getMoveCode(),operatorId);
        long time5 = Calendar.getInstance().getTimeInMillis();
        log.warn(moveStockVO.getCode()+"移库单执行第4步耗费时间：" + (time5-time4) + ";skuCode：" + moveSkuVO.getSkuCode());
        WhWmsMoveSku whWmsMoveSku = whWmsMoveSkuMapper.selectByPrimaryKey(moveSkuVO.getId());
        whWmsMoveSku.setStatus(WhWmsMoveSku.MOVE_SKU_STATUS_FINISH);
        whWmsMoveSkuMapper.updateByPrimaryKey(whWmsMoveSku);
        moveSkuVO.setStatus(WhWmsMoveSku.MOVE_SKU_STATUS_FINISH);
        return true;
    }

    public Boolean updateMoveStockStatusToFinish(WhWmsMoveStockVO moveStockVO){
        moveStockVO.setStatus(WhWmsMoveStockVO.STATUS_FINISHED);
        int result =  whWmsMoveStockMapper.updateByPrimaryKeySelective(moveStockVO);
        if (result<=0){
            throw new WarehouseException("","更新失败!");
        }

        // 移库单完成后，如果是补货数据创建的移库单，则将补货数据状态置为“已完成”
        updateReplenishStatusAfterMoveStockFinished(moveStockVO.getCode());
        return true;
    }

    private void updateReplenishStatusAfterMoveStockFinished(String moveStockCode){
        // 移库单完成后，如果是补货数据创建的移库单，则将补货数据状态置为“已完成”
        WhWmsReplenishInfoCond cond = new WhWmsReplenishInfoCond();
        cond.setRefCode(moveStockCode);
        cond.setRefType(WhWmsReplenishInfoVO.REF_TYPE_MOVE_STOCK);
        List<WhWmsReplenishInfo> replenishInfos = whWmsReplenishInfoService.listWhWmsReplenishInfoByCond(cond);
        if (CollectionUtils.isNotEmpty(replenishInfos)){
            whWmsReplenishInfoService.updateWhWmsReplenishInfoStatesByRefCode(cond.getRefCode(),cond.getRefType(),WhWmsReplenishInfoVO.STATES_REPLENISH_FINISHED);
        }
    }

    public boolean updateMoveStockStatusToProcessing(Long moveStockId){
        WhWmsMoveStock update = new WhWmsMoveStock();
        update.setId(moveStockId);
        update.setStatus(WhWmsMoveStockVO.STATUS_PROCESSING);
        int result =  whWmsMoveStockMapper.updateByPrimaryKeySelective(update);
        if (result<=0) throw new WarehouseException("","更新失败!");
        return true;
    }
}
