package com.thebeastshop.pegasus.service.warehouse.service.impl;

import com.alibaba.fastjson.JSONObject;
import com.thebeastshop.common.exception.BusinessException;
import com.thebeastshop.common.lock.RedisDistributLock;
import com.thebeastshop.pegasus.service.warehouse.cond.*;
import com.thebeastshop.pegasus.service.warehouse.dao.*;
import com.thebeastshop.pegasus.service.warehouse.dao.custom.WhGradeOfQualityCustomMapper;
import com.thebeastshop.pegasus.service.warehouse.dao.custom.WhWmsGradeOfQualityCustomMapper;
import com.thebeastshop.pegasus.service.warehouse.exception.WarehouseException;
import com.thebeastshop.pegasus.service.warehouse.exception.WarehouseExceptionErrorCode;
import com.thebeastshop.pegasus.service.warehouse.model.*;
import com.thebeastshop.pegasus.service.warehouse.service.*;
import com.thebeastshop.pegasus.service.warehouse.vo.*;
import com.thebeastshop.pegasus.util.PegasusUtilFacade;
import com.thebeastshop.pegasus.util.comm.*;
import com.thebeastshop.pegasus.util.model.CommGlobalConfig;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import page.Pagination;

import java.io.PipedReader;
import java.util.*;
import java.util.concurrent.TimeUnit;

/**
 * @author xw
 * @create 2018-04-23 15:26
 */
@Service("whWmsGradeOfQualityService")
public class WhWmsGradeOfQualityServiceImpl implements WhWmsGradeOfQualityService {

    private final static Logger log = LoggerFactory.getLogger(WhWmsGradeOfQualityServiceImpl.class);

    @Autowired
    private WhWmsGradeOfQualityMapper whWmsGradeOfQualityMapper;
    @Autowired
    private WhWmsGradeOfQualitySkuMapper whWmsGradeOfQualitySkuMapper;
    @Autowired
    private WhWmsGradeOfQualityBarcodeMapper whWmsGradeOfQualityBarcodeMapper;
    @Autowired
    private WhWmsGradeOfQualityCustomMapper whWmsGradeOfQualityCustomMapper;

    @Autowired
    private WhWmsOccupyService whWmsOccupyService;

    @Autowired
    private WhWmsHouseShelvesService whWmsHouseShelvesService;

    @Autowired
    private WhWmsWaitPutawayService whWmsWaitPutawayService;

    @Autowired
    private WhGradeOfQualityService whGradeOfQualityService;

    @Autowired
    private WhGradeOfQualityCustomMapper whGradeOfQualityCustomMapper;

    @Autowired
    private WhInfoService whInfoService;

    @Autowired
    private WhInvService whInvService;

    @Autowired
    private WhCommandService whCommandService;

    @Autowired
    private WhWmsSkuStockService whWmsSkuStockService;

    @Autowired
    private WhWmsMoveStockService whWmsMoveStockService;

    @Autowired
    private WhWmsWarehouseAreaService whWmsWarehouseAreaService;

    @Autowired
    private RedisDistributLock redisDistributLock;

    @Override
    public Pagination<WhWmsGradeOfQualityVO> listWhWmsGradeOfQualityVOByCond(WhWmsGradeOfQualityCond cond) {
        Pagination<WhWmsGradeOfQualityVO> page = new Pagination<>(cond.getCurrpage(), cond.getPagenum());
        int count = countWhWmsGradeOfQuality(cond);
        page.setRecord(count);
        if (NumberUtil.isNullOrZero(count))
            return page;
        List<WhWmsGradeOfQualityVO> gradeOfQualityVOs = whWmsGradeOfQualityCustomMapper.listWhWmsGradeOfQualityVOByCond(cond);
        page.setResultList(gradeOfQualityVOs);
        return page;
    }

    private int countWhWmsGradeOfQuality(WhWmsGradeOfQualityCond cond){
        return whWmsGradeOfQualityCustomMapper.countWhWmsGradeOfQualityByCond(cond);
    }

    @Override
    public WhWmsGradeOfQuality findWhWmsGradeOfQualityById(Long id){
        return whWmsGradeOfQualityMapper.selectByPrimaryKey(id);
    }

    @Override
    public WhWmsGradeOfQualityVO findWhWmsGradeOfQualityVOById(Long id, boolean fetchGradeSku, boolean fetchGradeBarcode) {
        WhWmsGradeOfQualityVO vo = whWmsGradeOfQualityCustomMapper.findVOById(id);
        if (EmptyUtil.isNotEmpty(vo)){
            fetchGradeDetail(vo,fetchGradeSku,fetchGradeBarcode);
        }
        return vo;
    }

    @Override
    public List<WhWmsGradeOfQualitySku> listWhWmsGradeOfQualitySkuByCond(WhWmsGradeOfQualitySkuCond skuCond){
        WhWmsGradeOfQualitySkuExample example = buildWhWmsGradeOfQualitySkuExampleByCond(skuCond);
        return whWmsGradeOfQualitySkuMapper.selectByExample(example);
    }

    @Override
    public List<WhWmsGradeOfQualityBarcode> listWhWmsGradeOfQualityBarcodeByCond(WhWmsGradeOfQualityBarcodeCond barcodeCond){
        WhWmsGradeOfQualityBarcodeExample example = buildWhWmsGradeOfQualityBarcodeExampleByCond(barcodeCond);
        return whWmsGradeOfQualityBarcodeMapper.selectByExample(example);
    }

    @Override
    public List<WhWmsGradeOfQualityBarcode> listWhWmsGradeOfQualityBarcodeByWmsGradeSkuId(Long wmsGradeSkuId){
        WhWmsGradeOfQualityBarcodeCond barcodeCond = new WhWmsGradeOfQualityBarcodeCond();
        barcodeCond.setWmsGradeSkuId(wmsGradeSkuId);
        return listWhWmsGradeOfQualityBarcodeByCond(barcodeCond);
    }

    public List<WhWmsGradeOfQualityBarcode> listWhWmsGradeOfQualityBarcodesByWmsGradeId(Long wmsGradeId){
        WhWmsGradeOfQualityBarcodeCond barcodeCond = new WhWmsGradeOfQualityBarcodeCond();
        barcodeCond.setWmsGradeId(wmsGradeId);
        return listWhWmsGradeOfQualityBarcodeByCond(barcodeCond);
    }

    @Override
    public List<WhWmsGradeOfQualitySkuVO> listWhWmsGradeOfQualitySkuVOsByCond(WhWmsGradeOfQualitySkuCond skuCond){
        List<WhWmsGradeOfQualitySkuVO> wmsGradeSkuVOs = BeanUtil.buildListFrom(listWhWmsGradeOfQualitySkuByCond(skuCond),WhWmsGradeOfQualitySkuVO.class);
        if (CollectionUtils.isNotEmpty(wmsGradeSkuVOs) && skuCond.isFetchGradeBarcode()){
            for (WhWmsGradeOfQualitySkuVO skuVO : wmsGradeSkuVOs){
                skuVO.setGradeBarcodes(listWhWmsGradeOfQualityBarcodeByWmsGradeSkuId(skuVO.getId()));
            }
        }
        return wmsGradeSkuVOs;
    }

    @Override
    public List<WhWmsGradeOfQualitySkuVO> listWhWmsGradeOfQualitySkuVOsByWmsGradeId(Long wmsGradeId,boolean fetchGradeBarcode){
        WhWmsGradeOfQualitySkuCond skuCond = new WhWmsGradeOfQualitySkuCond();
        skuCond.setWmsGradeId(wmsGradeId);
        skuCond.setFetchGradeBarcode(fetchGradeBarcode);
        return listWhWmsGradeOfQualitySkuVOsByCond(skuCond);
    }

    private void fetchGradeDetail(WhWmsGradeOfQualityVO vo,boolean fetchGradeSku,boolean fetchGradeBarcode){
        if (fetchGradeSku){
            WhWmsGradeOfQualitySkuCond skuCond = new WhWmsGradeOfQualitySkuCond();
            skuCond.setWmsGradeId(vo.getId());
            List<WhWmsGradeOfQualitySkuVO> gradeSkuVOs = whWmsGradeOfQualityCustomMapper.listWhWmsGradeOfQualitySkuVOByCond(skuCond);
            vo.setGradeSkuVOs(gradeSkuVOs);
        }
        if (fetchGradeBarcode){
            WhWmsGradeOfQualityBarcodeCond barcodeCond = new WhWmsGradeOfQualityBarcodeCond();
            barcodeCond.setWmsGradeId(vo.getId());
            List<WhWmsGradeOfQualityBarcodeVO> gradeBarcodeVOs = whWmsGradeOfQualityCustomMapper.listWhWmsGradeOfQualityBarcodeVOByCond(barcodeCond);
            vo.setGradeBarcodeVOs(gradeBarcodeVOs);
        }
    }

    private void fetchGradeSku(WhWmsGradeOfQualityVO vo,boolean fetchGradeBarcode){
        WhWmsGradeOfQualitySkuCond skuCond = new WhWmsGradeOfQualitySkuCond();
        skuCond.setWmsGradeId(vo.getId());
        List<WhWmsGradeOfQualitySkuVO> gradeSkuVOs = whWmsGradeOfQualityCustomMapper.listWhWmsGradeOfQualitySkuVOByCond(skuCond);
        if (CollectionUtils.isNotEmpty(gradeSkuVOs) && fetchGradeBarcode){
            for (WhWmsGradeOfQualitySkuVO skuVO : gradeSkuVOs){
                List<WhWmsGradeOfQualityBarcode> gradeBarcodes = listWhWmsGradeOfQualityBarcodeByWmsGradeSkuId(skuVO.getId());
                skuVO.setGradeBarcodes(gradeBarcodes);
            }
            vo.setGradeSkuVOs(gradeSkuVOs);
        }
    }

    @Override
    public WhWmsGradeOfQualityVO findWhWmsGradeOfQualityVOByCond(WhWmsGradeOfQualityCond cond) {
        WhWmsGradeOfQualityExample example = buildWhWmsGradeOfQualityExampleByCond(cond);
        List<WhWmsGradeOfQuality> gradeOfQualities = whWmsGradeOfQualityMapper.selectByExample(example);
        if (CollectionUtils.isNotEmpty(gradeOfQualities)){
            WhWmsGradeOfQualityVO vo = BeanUtil.buildFrom(gradeOfQualities.get(0),WhWmsGradeOfQualityVO.class);
            fetchGradeDetail(vo,cond.isFetchGradeSku(),cond.isFetchGradeBarcode());
            return vo;
        }
        return null;
    }

    private WhWmsGradeOfQualityExample buildWhWmsGradeOfQualityExampleByCond(WhWmsGradeOfQualityCond cond){
        WhWmsGradeOfQualityExample example = new WhWmsGradeOfQualityExample();
        WhWmsGradeOfQualityExample.Criteria criteria = example.createCriteria();

        if (EmptyUtil.isNotEmpty(cond.getId())){
            criteria.andIdEqualTo(cond.getId());
        }else if(CollectionUtils.isNotEmpty(cond.getIds())){
            criteria.andIdIn(cond.getIds());
        }

        if (EmptyUtil.isNotEmpty(cond.getCode())){
            criteria.andCodeEqualTo(cond.getCode());
        }

        if (EmptyUtil.isNotEmpty(cond.getWhGradeId())){
            criteria.andWhGradeIdEqualTo(cond.getWhGradeId());
        }else if (CollectionUtils.isNotEmpty(cond.getWhGradeIds())){
            criteria.andWhGradeIdIn(cond.getWhGradeIds());
        }

        if (EmptyUtil.isNotEmpty(cond.getPhysicalWarehouseCode())){
            criteria.andPhysicalWarehouseCodeEqualTo(cond.getPhysicalWarehouseCode());
        }

        if (EmptyUtil.isNotEmpty(cond.getSourceGrade())){
            criteria.andSourceGradeEqualTo(cond.getSourceGrade());
        }

        if (EmptyUtil.isNotEmpty(cond.getTargetGrade())){
            criteria.andTargetGradeEqualTo(cond.getTargetGrade());
        }

        if (EmptyUtil.isNotEmpty(cond.getApproveStatus())){
            criteria.andApproveStatusEqualTo(cond.getApproveStatus());
        }

        if (EmptyUtil.isNotEmpty(cond.getCreateTimeStart())){
            criteria.andCreateTimeGreaterThanOrEqualTo(DateUtil.parse(cond.getCreateTimeStart(),DateUtil.DEFAULT_DATE_FORMAT));
        }

        if (EmptyUtil.isNotEmpty(cond.getCreateTimeEnd())){
            criteria.andCreateTimeLessThanOrEqualTo(DateUtil.parse(cond.getCreateTimeEnd(),DateUtil.DEFAULT_DATE_FORMAT));
        }

        if (EmptyUtil.isNotEmpty(cond.getSubmitTimeStart())){
            criteria.andSubmitTimeGreaterThanOrEqualTo(DateUtil.parse(cond.getSubmitTimeStart(),DateUtil.DEFAULT_DATE_FORMAT));
        }

        if (EmptyUtil.isNotEmpty(cond.getSubmitTimeEnd())){
            criteria.andSubmitTimeLessThanOrEqualTo(DateUtil.parse(cond.getSubmitTimeEnd(),DateUtil.DEFAULT_DATE_FORMAT));
        }

        if (EmptyUtil.isNotEmpty(cond.getCreateUserId())){
            criteria.andCreateUserIdEqualTo(cond.getCreateUserId());
        }

        if (EmptyUtil.isNotEmpty(cond.getSubmitUserId())){
            criteria.andSubmitUserIdEqualTo(cond.getSubmitUserId());
        }

        return example;
    }

    private WhWmsGradeOfQualitySkuExample buildWhWmsGradeOfQualitySkuExampleByCond(WhWmsGradeOfQualitySkuCond cond){
        WhWmsGradeOfQualitySkuExample example = new WhWmsGradeOfQualitySkuExample();
        WhWmsGradeOfQualitySkuExample.Criteria criteria = example.createCriteria();

        if (EmptyUtil.isNotEmpty(cond.getId())){
            criteria.andIdEqualTo(cond.getId());
        }else if(CollectionUtils.isNotEmpty(cond.getIds())){
            criteria.andIdIn(cond.getIds());
        }

        if (EmptyUtil.isNotEmpty(cond.getWmsGradeId())){
            criteria.andWmsGradeIdEqualTo(cond.getWmsGradeId());
        }else if (CollectionUtils.isNotEmpty(cond.getWmsGradeIds())){
            criteria.andWmsGradeIdIn(cond.getWmsGradeIds());
        }

        if (EmptyUtil.isNotEmpty(cond.getSkuCode())){
            criteria.andSkuCodeEqualTo(cond.getSkuCode());
        }

        if (EmptyUtil.isNotEmpty(cond.getAdjustType())){
            criteria.andAdjustTypeEqualTo(cond.getAdjustType());
        }
        return example;
    }

    private WhWmsGradeOfQualityBarcodeExample buildWhWmsGradeOfQualityBarcodeExampleByCond(WhWmsGradeOfQualityBarcodeCond cond){
        WhWmsGradeOfQualityBarcodeExample example = new WhWmsGradeOfQualityBarcodeExample();
        WhWmsGradeOfQualityBarcodeExample.Criteria criteria = example.createCriteria();

        if (EmptyUtil.isNotEmpty(cond.getId())){
            criteria.andIdEqualTo(cond.getId());
        }else if(CollectionUtils.isNotEmpty(cond.getIds())){
            criteria.andIdIn(cond.getIds());
        }

        if (EmptyUtil.isNotEmpty(cond.getWmsGradeId())){
            criteria.andWmsGradeIdEqualTo(cond.getWmsGradeId());
        }else if (CollectionUtils.isNotEmpty(cond.getWmsGradeIds())){
            criteria.andWmsGradeIdIn(cond.getWmsGradeIds());
        }

        if (EmptyUtil.isNotEmpty(cond.getWmsGradeSkuId())){
            criteria.andWmsGradeSkuIdEqualTo(cond.getWmsGradeSkuId());
        }else if (CollectionUtils.isNotEmpty(cond.getWmsGradeSkuIds())){
            criteria.andWmsGradeSkuIdIn(cond.getWmsGradeSkuIds());
        }

        if (EmptyUtil.isNotEmpty(cond.getSkuCode())){
            criteria.andSkuCodeEqualTo(cond.getSkuCode());
        }

        if (EmptyUtil.isNotEmpty(cond.getBarcode())){
            criteria.andBarcodeEqualTo(cond.getBarcode());
        }

        if (EmptyUtil.isNotEmpty(cond.getShelvesCode())){
            criteria.andSkuCodeEqualTo(cond.getShelvesCode());
        }

        return example;
    }

    @Override
    public List<WhWmsGradeOfQuality> findWhWmsGradeOfQualityByCond(WhWmsGradeOfQualityCond cond) {
        WhWmsGradeOfQualityExample example = buildWhWmsGradeOfQualityExampleByCond(cond);
        return whWmsGradeOfQualityMapper.selectByExample(example);
    }

    @Override
    @Transactional
    public boolean create(WhWmsGradeOfQualityVO vo) throws Exception{
        WhWmsGradeOfQuality gradeOfQuality = BeanUtil.buildFrom(vo,WhWmsGradeOfQuality.class);
        gradeOfQuality.setCreateTime(DateUtil.getNow());
        int result = whWmsGradeOfQualityMapper.insert(gradeOfQuality);
        vo.setId(gradeOfQuality.getId());
        if (result > 0){
            insertGradeSkuAndBarcodes(vo);

            // 生成CODE
            Map<String, Object> params = new HashMap<>();
            params.put("createTime", gradeOfQuality.getCreateTime());
            params.put("sourceGrade", gradeOfQuality.getSourceGrade());
            params.put("id", gradeOfQuality.getId());
            String code = CodeGenerator.getInstance().generate("WMS_GA_CODE", params);
            gradeOfQuality.setCode(code);
            vo.setCode(code);
            whWmsGradeOfQualityMapper.updateByPrimaryKeySelective(gradeOfQuality);

            // 如果是提交审批 则占用对应wms库存 并生成scm品级调整审批单
            if (WhWmsGradeOfQualityVO.APPROVE_STATUS_WAIT == vo.getApproveStatus()){
                submitWmsApprove(vo);
            }
        }
        return result > 0;
    }

    private void submitWmsApprove(WhWmsGradeOfQualityVO vo) throws Exception{
        // vo.setApproveStatus(WhWmsGradeOfQualityVO.APPROVE_STATUS_WAIT);
        vo.setSubmitUserId(vo.getCreateUserId());
        // 占用wms库存
        wmsOccupy(vo);
        // 生成scm品级调整审批单
        createWhGrade(vo);

        // 是否手动审批
        boolean manualApprove = manualApprovalFlag(vo);
        if (manualApprove){
            // 人工审批，邮件通知
            sendGradeAdjustApplyMail(vo);
        }else{
            // 自动审批
            doAutoApprove(vo);
        }
    }

    private void sendGradeAdjustApplyMail(WhWmsGradeOfQualityVO vo){
        WhGradeOfQualityCond whGradeCond = new WhGradeOfQualityCond();
        whGradeCond.setRefId(vo.getId());
        List<WhGradeOfQualityVO> gradeOfQualityVOs = whGradeOfQualityCustomMapper.listWhGradeOfQualityVOByCond(whGradeCond);
        whGradeOfQualityService.gradeAdjustApplySendMail(gradeOfQualityVOs);
    }

    public boolean doAutoApprove(WhWmsGradeOfQualityVO vo) throws Exception{
        WhGradeOfQualityCond cond = new WhGradeOfQualityCond();
        cond.setRefId(vo.getId());
        List<WhGradeOfQualityVO> whGradeOfQualityVOs = whGradeOfQualityService.listWhGradeOfQualityVOByCond(cond);
        if (CollectionUtils.isNotEmpty(whGradeOfQualityVOs)){
            for (WhGradeOfQualityVO whGradeVO : whGradeOfQualityVOs){
                // 自动审批，直接通过
                whGradeVO.setOptPass(true);
                if (EmptyUtil.isNotEmpty(vo.getSubmitUserId())){
                    whGradeVO.setApproveUserId(vo.getSubmitUserId());
                }else{
                    whGradeVO.setApproveUserId(whGradeVO.getCreateUserId());
                }
                boolean result = whGradeOfQualityService.auditWhGradeOfQuality(whGradeVO);
                if (result){
                    vo.setAutoApproveSuccess(true);
                }
            }
        }
        return true;
    }

    public boolean manualApprovalFlag(WhWmsGradeOfQualityVO vo){
        // 系统调用的 一律自动审批
        if (vo.isSystemAutoPass()){
            return false;
        }
        final CommGlobalConfig config = PegasusUtilFacade.getInstance().findConfigByKey(WhGradeOfQualityVO.NEED_MANUAL_APPROVE_GRADE_STATUS);
        if(EmptyUtil.isNotEmpty(config)){
            String configValue = config.getConfigValue();
            if (EmptyUtil.isNotEmpty(configValue)){
                Set<Integer> statusSet = new HashSet<>();
                for (String status : configValue.split(",")){
                    statusSet.add(Integer.parseInt(status));
                }
                return statusSet.contains(vo.getTargetGrade());
            }
        }
        return false;
    }

    @Override
    @Transactional
    public boolean approveWhWmsGradeOfQuality(WhWmsGradeOfQualityVO vo) throws Exception {
        // 占用wms库存
        wmsOccupy(vo);
        // 创建scm品级单 并占用scm库存
        createWhGrade(vo);
        // 更新状态
        WhWmsGradeOfQuality gradeOfQuality = new WhWmsGradeOfQuality();
        gradeOfQuality.setId(vo.getId());
        gradeOfQuality.setApproveStatus(WhWmsGradeOfQualityVO.APPROVE_STATUS_WAIT);
        gradeOfQuality.setSubmitTime(DateUtil.getNow());
        gradeOfQuality.setSubmitUserId(vo.getSubmitUserId());
        boolean result = whWmsGradeOfQualityMapper.updateByPrimaryKeySelective(gradeOfQuality)>0;

        // 如果自动审批，则进行审批操作
        boolean manualApprove = manualApprovalFlag(vo);
        if (result){
            // 如果需要手动审批，则邮件提醒
            if (manualApprove){
                sendGradeAdjustApplyMail(vo);
            }else{
                result = doAutoApprove(vo);
            }
        }
        return result;
    }

    @Override
    public List<WhGradeSalesOrderVO> listSalesOrderByWmsOccupyRefCodes(WhWmsGradeOfQualityCond cond) throws Exception {
        return whWmsGradeOfQualityCustomMapper.listSalesOrderByWmsOccupyRefCodes(cond);
    }

    @Override
    @Transactional
    public void createWhGrade(WhWmsGradeOfQualityVO vo) throws Exception{
        createWhGrades(vo);

        // 生成scm品级调整单
        //List<WhGradeOfQualityVO> whGradeVOS = new ArrayList<>();
        //buildeWhGradeAndCreate(vo,whGradeVOS);
        // 便于记录日志
        //vo.setWhGradeVOS(whGradeVOS);
    }

    @Override
    @Transactional
    public boolean createOrUpdate(WhWmsGradeOfQualityVO vo) throws Exception{
        boolean result = false;
        if (EmptyUtil.isNotEmpty(vo.getId()) && EmptyUtil.isNotEmpty(whWmsGradeOfQualityMapper.selectByPrimaryKey(vo.getId()))){
            result = update(vo);
        }else{
            result = create(vo);
        }
        return result;
    }

    @Override
    public boolean cancelWhWmsGradeOfQuality(WhWmsGradeOfQuality record) throws Exception {
        WhWmsGradeOfQualityExample example = new WhWmsGradeOfQualityExample();
        WhWmsGradeOfQualityExample.Criteria criteria = example.createCriteria();
        criteria.andIdEqualTo(record.getId());
        criteria.andApproveStatusEqualTo(record.getApproveStatus());

        WhWmsGradeOfQuality waitUpdateGrade = new WhWmsGradeOfQuality();
        waitUpdateGrade.setApproveStatus(WhWmsGradeOfQualityVO.APPROVE_STATUS_CANCEL);
        boolean result = whWmsGradeOfQualityMapper.updateByExampleSelective(waitUpdateGrade,example) > 0;
        // 取消行sku
        WhWmsGradeOfQualitySkuCond gradeSkuCond = new WhWmsGradeOfQualitySkuCond();
        gradeSkuCond.setWmsGradeId(record.getId());
        if (result && CollectionUtils.isNotEmpty(whWmsGradeOfQualityCustomMapper.listWhWmsGradeOfQualitySkuVOByCond(gradeSkuCond))){
            result = cancelWhWmsGradeOfQualitySkuByCond(gradeSkuCond);
        }
        return result;
    }

    @Override
    public boolean cancelWhWmsGradeOfQualitySku(WhWmsGradeOfQualitySku record) throws Exception {
        WhWmsGradeOfQualitySku waitUpdate = new WhWmsGradeOfQualitySku();
        waitUpdate.setId(record.getId());
        waitUpdate.setStatus(WhWmsGradeOfQualitySkuVO.APPROVE_STATUS_CANCEL);
        return whWmsGradeOfQualitySkuMapper.updateByPrimaryKey(waitUpdate) > 0;
    }

    public boolean cancelWhWmsGradeOfQualitySkuByCond(WhWmsGradeOfQualitySkuCond cond) throws Exception {
        WhWmsGradeOfQualitySkuExample example = new WhWmsGradeOfQualitySkuExample();
        WhWmsGradeOfQualitySkuExample.Criteria criteria = example.createCriteria();
        if (CollectionUtils.isNotEmpty(cond.getIds())){
            criteria.andIdIn(cond.getIds());
        }
        if (EmptyUtil.isNotEmpty(cond.getWmsGradeId())){
            criteria.andWmsGradeIdEqualTo(cond.getWmsGradeId());
        }else if (CollectionUtils.isNotEmpty(cond.getWmsGradeIds())){
            criteria.andWmsGradeIdIn(cond.getWmsGradeIds());
        }

        WhWmsGradeOfQualitySku waitUpdate = new WhWmsGradeOfQualitySku();
        waitUpdate.setStatus(WhWmsGradeOfQualitySkuVO.APPROVE_STATUS_CANCEL);
        return whWmsGradeOfQualitySkuMapper.updateByExampleSelective(waitUpdate,example) > 0;
    }

    private void createWhGrades(WhWmsGradeOfQualityVO vo) throws Exception {
        if (CollectionUtils.isEmpty(vo.getGradeSkuVOs())){
            throw new WarehouseException(WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED,"品级调整审批信息不能为空，无法构建SCM调整单!");
        }
        List<WhGradeOfQualityVO> whGradeVOs = new ArrayList<>();
        for (WhWmsGradeOfQualitySkuVO gradeSkuVO : vo.getGradeSkuVOs()){
            if (EmptyUtil.isNotEmpty(gradeSkuVO.getWhGradeVO())){
                WhGradeOfQualityVO whGradeVO = gradeSkuVO.getWhGradeVO();
                whGradeVO.setRefId(vo.getId());
                whGradeVO.setRefCode(vo.getCode());
                whGradeVO.setWmsGradeSkuId(gradeSkuVO.getId());
                whGradeVOs.add(whGradeVO);
            }
        }
        vo.setWhGradeVOS(whGradeVOs);
        whGradeOfQualityService.batchCreateWhGradeAndFetch(vo.getWhGradeVOS());
    }

    @Deprecated
    private void buildeWhGradeAndCreate(WhWmsGradeOfQualityVO vo,List<WhGradeOfQualityVO> whGradeVOS) throws Exception{
        if (CollectionUtils.isEmpty(vo.getGradeSkuVOs())){
            throw new WarehouseException(WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED,"品级调整单 行SKU不能为空，无法构建SCM调整单!");
        }
        for (WhWmsGradeOfQualitySku gradeSku : vo.getGradeSkuVOs()){
            WhGradeOfQualityVO whGradeVO = new WhGradeOfQualityVO();
            whGradeVO.setRefId(vo.getId());
            whGradeVO.setRefCode(vo.getCode());
            whGradeVO.setSkuCode(gradeSku.getSkuCode());
            whGradeVO.setAdjustAmount(gradeSku.getAdjustAmount());
            whGradeVO.setSourceGrade(vo.getSourceGrade());
            whGradeVO.setTargetGrade(vo.getTargetGrade());
            whGradeVO.setPhysicalWarehouseCode(vo.getPhysicalWarehouseCode());
            whGradeVO.setApproveStatus(WhGradeOfQualityVO.APPROVE_STATUS_WAIT);
            whGradeVO.setCreateUserId(vo.getCreateUserId());
            whGradeVO.setAdjustReason(gradeSku.getAdjustReason());
            whGradeVO.setSkuImage(gradeSku.getSkuImage());
            whGradeVO.setOriginType(WhGradeOfQualityVO.ORIGIN_TYPE_WMS);


            WhWarehouse targetWhWarehouse = findGradeWarehouseByCond(vo.getPhysicalWarehouseCode(), WhWarehouseVO.RELATE_IN,vo.getTargetGrade());
            // 计算占用逻辑仓
            // 物理仓——》逻辑仓分组 当前商品状态下所有的逻辑仓列表
            List<WhWarehouse> whWarehouses = whInfoService.findAllWarehousesByPhyCodeAndStatus(vo.getPhysicalWarehouseCode(),vo.getSourceGrade(),WhWarehouseVO.RELATE_OUT);
            List<String> warehouseCodes = new ArrayList<>();
            for (WhWarehouse warehouse : whWarehouses){
                warehouseCodes.add(warehouse.getCode());
            }
            // 当前sku，相应逻辑仓的库存
            List<WhWarehouseSkuInvVO> whWarehouseSkuInvVOs = whInvService.findWarehousesSkuInvByCond(warehouseCodes,gradeSku.getSkuCode());

            Map<String, WhWarehouse> warehouseMap = whInvService.getWarehouseMap(warehouseCodes);
            // 按warehouse sort升序
            sortWhSkuInvList(whWarehouseSkuInvVOs,warehouseMap);

            int adjustAmount = gradeSku.getAdjustAmount();

            int canUseInv = 0;
            for (WhWarehouseSkuInvVO skuInvVO : whWarehouseSkuInvVOs){
                canUseInv += skuInvVO.getCanUseInv();
            }
            List<WhGradeOfQualityDetail> whGradeDetails = new ArrayList<>();
            // 如果逻辑仓可用库存够用,则占用相应逻辑仓库存
            //if (canUseInv >= adjustAmount){
            for (WhWarehouseSkuInvVO skuInvVO : whWarehouseSkuInvVOs){
                if (skuInvVO.getCanUseInv() >= adjustAmount){
                    adjustAmount = 0;
                    buildWhGradeOfQualityDetails(vo,gradeSku,adjustAmount,skuInvVO.getWarehouseCode(),targetWhWarehouse.getCode(),whGradeDetails);
                    break;
                }else{
                    adjustAmount -= skuInvVO.getCanUseInv();
                    buildWhGradeOfQualityDetails(vo,gradeSku,skuInvVO.getCanUseInv(),skuInvVO.getWarehouseCode(),targetWhWarehouse.getCode(),whGradeDetails);
                }
            }
            // 所有逻辑仓可用库存不足，则继续根据占用订单的逻辑仓占用
            if (adjustAmount > 0){
                /*for (String warehouseCode : warehouseCodes){
                    WhInvRcdCond cond = new WhInvRcdCond();
                    cond.setWarehouseCode(warehouseCode);
                    cond.setSkuCode(gradeSku.getSkuCode());
                    invOccupyList.addAll(whInvService.getInvOccupyByCond(cond));
                }*/
                WhInvRcdCond cond = new WhInvRcdCond();
                cond.setWarehouseCodes(warehouseCodes);
                cond.setSkuCode(gradeSku.getSkuCode());
                List<Integer> inOutTypes = new ArrayList<Integer>(){{
                    add(WhInvRcd.TYPE_SALES_OUT);
                    add(WhInvRcd.TYPE_CHANGE_OUT);
                }};
                cond.setInOutTypes(inOutTypes);
                List<WhInvOccupy> invOccupyList = whInvService.getInvOccupyByCond(cond);
                List<WhInvOccupy> soInvOccupyList = new ArrayList<>();
                List<WhInvOccupy> cmdInvOccupyList = new ArrayList<>();
                List<String> commandSkuCodes = new ArrayList<>();
                for (WhInvOccupy occupy : invOccupyList){
                    if (occupy.getReferenceCode().startsWith("SO")){
                        soInvOccupyList.add(occupy);
                    }else if(occupy.getReferenceCode().startsWith("CMD")){
                        cmdInvOccupyList.add(occupy);
                        commandSkuCodes.add(occupy.getReferenceCode());
                    }
                }
                List<WhCommandSku> commandSkus = whCommandService.findCommandSkuByCommandSkuCodes(commandSkuCodes);

            }
            //}

            // quantityInOccupy  canUseInv
            /*WhInvRcdCond cond = new WhInvRcdCond();
            if (EmptyUtil.isNotEmpty("")) {
                cond.setWarehouseCode("");
            }
            if (EmptyUtil.isNotEmpty("")) {
                cond.setSkuCode("");
            }
            List<WhInvOccupy> invOccupyList = whInvService.getInvOccupyByCond(cond);*/


            // gradeDetail
            /*WhGradeOfQualityDetail gradeDetail = new WhGradeOfQualityDetail();
            gradeDetail.setWmsGradeId(vo.getId());
            gradeDetail.setWmsGradeSkuId(gradeSku.getId());
            gradeDetail.setSkuCode(gradeSku.getSkuCode());
            gradeDetail.setAdjustAmount(gradeSku.getAdjustAmount());
            gradeDetail.setSourceWarehouseCode(vo.getSourceWarehouseCode());
            gradeDetail.setApproveStatus(WhGradeOfQualityDetailVO.APPROVE_STATUS_WAIT);
            // 目标入库逻辑仓
            WhWarehouse whWarehouse = findGradeWarehouseByCond(vo.getPhysicalWarehouseCode(), WhWarehouseVO.RELATE_IN,vo.getTargetGrade());
            gradeDetail.setTargetWarehouseCode(whWarehouse.getCode());*/
            whGradeVO.setWhGradeDetails(whGradeDetails);
            whGradeVOS.add(whGradeVO);
        }
        whGradeOfQualityService.batchCreateWhGradeAndFetch(whGradeVOS);
    }

    private void buildWhGradeOfQualityDetails(WhWmsGradeOfQualityVO vo,WhWmsGradeOfQualitySku gradeSku,Integer adjustAmount,String sourceWarehouseCode,String targetWarehouseCode,List<WhGradeOfQualityDetail> whGradeDetails){
        WhGradeOfQualityDetail gradeDetail = new WhGradeOfQualityDetail();
        gradeDetail.setWmsGradeId(vo.getId());
        gradeDetail.setWmsGradeSkuId(gradeSku.getId());
        gradeDetail.setSkuCode(gradeSku.getSkuCode());
        gradeDetail.setAdjustAmount(adjustAmount);
        gradeDetail.setSourceWarehouseCode(sourceWarehouseCode);
        gradeDetail.setApproveStatus(WhGradeOfQualityDetailVO.APPROVE_STATUS_WAIT);
        gradeDetail.setTargetWarehouseCode(targetWarehouseCode);
        whGradeDetails.add(gradeDetail);
    }

    //按sort升序排列 若为空则排到最后面
    private void sortWhSkuInvList(List<WhWarehouseSkuInvVO> whSkuInvList ,final Map<String, WhWarehouse> warehouseMap){
        if(EmptyUtil.isEmpty(whSkuInvList) || whSkuInvList.size()==1){
            return;
        }
        Collections.sort(whSkuInvList, new Comparator<WhWarehouseSkuInvVO>() {
            @Override
            public int compare(WhWarehouseSkuInvVO o1, WhWarehouseSkuInvVO o2) {
                WhWarehouse warehouse1 = warehouseMap.get(o1.getWarehouseCode());
                WhWarehouse warehouse2 = warehouseMap.get(o2.getWarehouseCode());
                if (EmptyUtil.isEmpty(warehouse1.getSort()) && EmptyUtil.isEmpty(warehouse2.getSort())){
                    return -2;
                }
                if (EmptyUtil.isEmpty(warehouse1.getSort())){
                    return 1;
                }
                if (EmptyUtil.isEmpty(warehouse2.getSort())){
                    return -1;
                }
                return warehouse1.getSort().compareTo(warehouse2.getSort());
            }
        });
    }

    public boolean wmsReleaseOccupy(WhWmsGradeOfQualityVO vo,Long wmsGradeSkuId){
        String receiptNo = vo.getCode()+"_"+wmsGradeSkuId;
        //获取分布式锁
        String lockKey = "lock:whWmsGrade:wmsReleaseOccupy:"+receiptNo;
        try {
            Boolean getLock = redisDistributLock.tryLock(lockKey, 3L, TimeUnit.SECONDS);
            if (!getLock) {
                log.error("获取锁失败   false  lockKey="+lockKey);
                throw new WarehouseException(
                        WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED,
                        "请稍后重试!");
            }
            WhWmsOccupyVO cond = new WhWmsOccupyVO();
            // 释放当前调整单的，当前SKU的占用
            cond.setReceiptsNo(receiptNo);
            cond.setType(WhWmsOccupyVO.TYPE_GRADE_ADJUST);
            cond.setStatus(WhWmsOccupyVO.OCCUPIED);
            log.info("wmsReleaseOccupy start ["+receiptNo+"] Thread=" +Thread.currentThread().getName()+"  time="+System.currentTimeMillis());
            return whWmsOccupyService.releaseWmsOccupyById(cond);
        }finally {
            redisDistributLock.unLock(lockKey);
        }
    }

    @Override
    @Transactional
    public boolean passWmsGradeOfQuality(WhWmsGradeOfQualityVO vo, Long wmsGradeSkuId) {
        WhWmsGradeOfQualitySku gradeSku = whWmsGradeOfQualitySkuMapper.selectByPrimaryKey(wmsGradeSkuId);
        if (EmptyUtil.isEmpty(gradeSku)){
            throw new WarehouseException(WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED,
                    "wmsGradeSkuId["+wmsGradeSkuId+"]不存在!");
        }
        boolean flag = updateSkuStockByCond(vo,gradeSku.getSkuCode(),wmsGradeSkuId);
        if(flag){
            wmsReleaseOccupy(vo,wmsGradeSkuId);

            WhWmsGradeOfQualitySku gradeOfQualitySku = new WhWmsGradeOfQualitySku();
            gradeOfQualitySku.setId(wmsGradeSkuId);
            gradeOfQualitySku.setStatus(WhWmsGradeOfQualitySkuVO.APPROVE_STATUS_PASS);
            whWmsGradeOfQualitySkuMapper.updateByPrimaryKeySelective(gradeOfQualitySku);

            // 获取分布式锁
            String lockKey = "lock:whWmsGrade:passWmsGradeOfQuality:"+vo.getId();
            try{
                Boolean getLock = redisDistributLock.tryLock(lockKey,2L, TimeUnit.MINUTES);
                log.info("tryGetLock start passWmsGradeOfQuality [wmsGradeId]="+vo.getId()+"  [wmsGradeSkuId]="+wmsGradeSkuId+"  == "+System.currentTimeMillis());
                if (!getLock){
                    log.error("获取锁失败   false  lockKey="+lockKey);
                    throw new WarehouseException(
                            WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED,
                            "请稍后重试!");
                }
                log.info("tryGetLock end passWmsGradeOfQuality [wmsGradeId]="+vo.getId()+"  [wmsGradeSkuId]="+wmsGradeSkuId+"  == "+System.currentTimeMillis());
                WhWmsGradeOfQualityVO wmsGradeOfQualityVO = findWhWmsGradeOfQualityVOById(vo.getId(),true,false);
                Set<Integer> statusSet = new HashSet<>();
                for (WhWmsGradeOfQualitySkuVO skuVO : wmsGradeOfQualityVO.getGradeSkuVOs()){
                    statusSet.add(skuVO.getStatus());
                }
                int approveStatus = 0;
                if (statusSet.contains(WhWmsGradeOfQualitySkuVO.APPROVE_STATUS_INIT) &&
                        (statusSet.contains(WhWmsGradeOfQualitySkuVO.APPROVE_STATUS_PASS)
                                || statusSet.contains(WhWmsGradeOfQualitySkuVO.APPROVE_STATUS_REJECT))){
                    approveStatus = WhWmsGradeOfQualityVO.APPROVE_STATUS_APPROVING;
                }else if(!statusSet.contains(WhWmsGradeOfQualitySkuVO.APPROVE_STATUS_INIT)
                        && !statusSet.contains(WhWmsGradeOfQualitySkuVO.APPROVE_STATUS_REJECT)){
                    approveStatus = WhWmsGradeOfQualityVO.APPROVE_STATUS_COMPLETED;
                }else if(!statusSet.contains(WhWmsGradeOfQualitySkuVO.APPROVE_STATUS_INIT)
                        && statusSet.contains(WhWmsGradeOfQualitySkuVO.APPROVE_STATUS_PASS)
                        && statusSet.contains(WhWmsGradeOfQualitySkuVO.APPROVE_STATUS_REJECT)){
                    approveStatus = WhWmsGradeOfQualityVO.APPROVE_STATUS_PARTLY_COMPLETED;
                }
                WhWmsGradeOfQuality gradeOfQuality = new WhWmsGradeOfQuality();
                gradeOfQuality.setId(vo.getId());
                gradeOfQuality.setApproveStatus(approveStatus);
                gradeOfQuality.setSubmitUserId(vo.getSubmitUserId());
                gradeOfQuality.setSubmitTime(DateUtil.getNow());
                int result = whWmsGradeOfQualityMapper.updateByPrimaryKeySelective(gradeOfQuality);
                if (result > 0){
                    // 满足条件 则生成一张"待出库"的移库单
                    buildAndCreateMoveStock(gradeOfQuality);
                }
                return result>0;
            }finally {
                redisDistributLock.unLock(lockKey);
            }
        }else{
            throw new WarehouseException(WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED,"WMS品级申请通过时,更新物理仓库存失败!");
        }
    }

    private String getHouseTypeByShelvesCode(Map<String,String> shelvesMap,String shelvesCode){
        return EmptyUtil.isNotEmpty(shelvesMap.get(shelvesCode))?shelvesMap.get(shelvesCode):"";
    }

    private void buildAndCreateMoveStock(WhWmsGradeOfQuality gradeOfQuality){
        if (gradeOfQuality.getApproveStatus() == WhWmsGradeOfQualityVO.APPROVE_STATUS_PARTLY_COMPLETED
                || gradeOfQuality.getApproveStatus() == WhWmsGradeOfQualityVO.APPROVE_STATUS_COMPLETED) {
            // 生成一张"待出库"的移库单
            WhWmsGradeOfQualityVO wmsGradeVO = whWmsGradeOfQualityCustomMapper.findVOById(gradeOfQuality.getId());
            if (EmptyUtil.isNotEmpty(wmsGradeVO)) {
                fetchGradeSku(wmsGradeVO,true);
                // 只有通过的批次才会生成移库单
                List<WhWmsGradeOfQualityBarcode> gradeBarcodes = new ArrayList<>();
                if (CollectionUtils.isNotEmpty(wmsGradeVO.getGradeSkuVOs())){
                    for (WhWmsGradeOfQualitySkuVO gradeSkuVO : wmsGradeVO.getGradeSkuVOs()){
                        if (WhWmsGradeOfQualitySkuVO.APPROVE_STATUS_PASS == gradeSkuVO.getStatus()
                                && CollectionUtils.isNotEmpty(gradeSkuVO.getGradeBarcodes())){
                            gradeBarcodes.addAll(gradeSkuVO.getGradeBarcodes());
                        }
                    }
                }
                // 聚合
                gradeBarcodes = aggregateAdjustAmountByBarcodes(gradeBarcodes);

                /*
                // 依照上架开始的逻辑创建 移库单
                createMoveByPutawayLogic(wmsGrade,(List)aggregateBarcode.values());*/

                if (CollectionUtils.isNotEmpty(gradeBarcodes)) {
                    Map<String, WhWmsHouseShelves> shelvesMap = getHouseTypeMapByBarcodes(gradeBarcodes);
                    List<WhWmsMoveSkuVO> moveSkuList = new ArrayList<>();
                    //Map<String,String> targetShelvesMap = getTargetShelvesCodeByConfig();
                    for (WhWmsGradeOfQualityBarcode barcode : gradeBarcodes) {
                        if (EmptyUtil.isNotEmpty(shelvesMap.get(barcode.getShelvesCode()))) {
                            WhWmsMoveSkuVO moveSkuVO = new WhWmsMoveSkuVO();
                            moveSkuVO.setPhysicalWarehouseCode(wmsGradeVO.getPhysicalWarehouseCode());
                            moveSkuVO.setCreateTime(DateUtil.getNow());
                            moveSkuVO.setOriginalHouseType(shelvesMap.get(barcode.getShelvesCode()).getHouseType());
                            moveSkuVO.setAmount(barcode.getAdjustAmount());
                            moveSkuVO.setOriginalShelvesCode(barcode.getShelvesCode());
                            moveSkuVO.setTargetShelvesCode(generateTargetByRule(shelvesMap.get(barcode.getShelvesCode()), wmsGradeVO.getTargetGrade(),wmsGradeVO.getPhysicalWarehouseCode()));
                            moveSkuVO.setTargetHouseType(getHouseTypeByCode(moveSkuVO.getTargetShelvesCode()));
                            moveSkuVO.setBarCode(barcode.getBarcode());
                            moveSkuVO.setSkuCode(barcode.getSkuCode());
                            moveSkuVO.setSkuStatus(wmsGradeVO.getTargetGrade());
                            moveSkuVO.setStatus(WhWmsMoveSku.MOVE_SKU_STATUS_WAIT_MOVE);
                            if (EmptyUtil.isNotEmpty(moveSkuVO.getTargetHouseType())) {
                                // 目标库位不为空，才使用
                                moveSkuList.add(moveSkuVO);
                            }
                        }
                    }
                    if (EmptyUtil.isNotEmpty(moveSkuList)) {
                        WhWmsMoveStockVO moveStockVO = new WhWmsMoveStockVO();
                        moveStockVO.setPhysicalWarehouseCode(wmsGradeVO.getPhysicalWarehouseCode());
                        moveStockVO.setCreateUserId(wmsGradeVO.getSubmitUserId());
                        /*if (EmptyUtil.isNotEmpty(moveStockVO.getCreateUserId())
                                && moveStockVO.getCreateUserId() == 1){
                            moveStockVO.setMoveType(WhWmsMoveStockVO.MOVE_TYPE_SKU_TRANSLATE);
                        }else{
                            moveStockVO.setMoveType(WhWmsMoveStockVO.MOVE_TYPE_COMMON);
                        }*/
                        // 品级调整单创建的移库类型为：状态变更移库
                        moveStockVO.setMoveType(WhWmsMoveStockVO.MOVE_TYPE_SKU_TRANSLATE);
                        moveStockVO.setReferenceCode(wmsGradeVO.getCode());
                        moveStockVO.setMoveSkuList(moveSkuList);
                        whWmsMoveStockService.newMoveStock(moveStockVO);
                    }
                }
            }
        }
    }

    private String generateTargetByRule(WhWmsHouseShelves whWmsHouseShelves,Integer skuStatus,String phyCode){
        if (WhPhysicalWarehouseVO.BJ_CENTRAL_PHYSICAL_WAREHOUSE_CODE.equals(phyCode)){
            // rd #7115
            if (WhWarehouseVO.COMMODITY_STATUS_FOR_NONDEFECTIVE.equals(skuStatus)){
                // 北京总仓物理仓时，目标状态为良品，则移库单目标库位为
                return "02-B-01-01-01-01";
            }else{
                return "02-D-01-01-01-01";
            }
        }else{
            if (whWmsHouseShelves.getCode().contains("H")){
                return "01-H-10-02-01-01";
            }else if(whWmsHouseShelves.getCode().contains("G")){
                return "01-G-98-04-04-04";
            }else if(WhWarehouseVO.isDamagedCommodity(skuStatus)){
                return "01-D-06-02-01-04";
            }else if(WhWarehouseVO.COMMODITY_STATUS_FOR_NONDEFECTIVE.equals(skuStatus)){
                // 良品库位
                return "01-L-01-01-01-01";
            }else if(WhWarehouseVO.COMMODITY_STATUS_FOR_SAMPLE.equals(skuStatus)
                    || WhWarehouseVO.COMMODITY_STATUS_FOR_DISPLAY.equals(skuStatus)){
                // 样品库位
                return "01-R-01-01-01-01";
            }else if(WhWarehouseVO.COMMODITY_STATUS_FOR_WASTED.equals(skuStatus)){
                // 废品库位
                return "01-P-01-01-01-01";
            }
        }
        return "";
    }

    private String getHouseTypeByCode(String shelvesCode){
        if (EmptyUtil.isEmpty(shelvesCode)){
            return "";
        }
        WhWmsHouseShelvesVO houseShelvesVO = whWmsHouseShelvesService.getHouseShelvesByCode(shelvesCode);
        if (EmptyUtil.isNotEmpty(houseShelvesVO)){
            return houseShelvesVO.getHouseType();
        }
        return "";
    }

    private Map<String,String> getTargetShelvesCodeByConfig(){
        // String params = "{A:'01-A-01-01-01-01',B:'01-B-01-01-01-01',C:'01-C-15-03-01-02'}";
        List<WhWmsWarehouseArea> whWmsWarehouseAreas = whWmsWarehouseAreaService.getAllArea();
        final CommGlobalConfig config = PegasusUtilFacade.getInstance().findConfigByKey(WhGradeOfQualityVO.GRADE_ADJUST_MOVE_TARGET_SHELVES_CODE);
        Map<String,String> targetShelvesMap = new HashMap<>();
        if(EmptyUtil.isNotEmpty(config) && EmptyUtil.isNotEmpty(config.getConfigValue())){
            JSONObject json = JSONObject.parseObject(config.getConfigValue());
            for (WhWmsWarehouseArea whWmsWarehouseArea : whWmsWarehouseAreas){
                json.get(whWmsWarehouseArea.getCode());
                if (EmptyUtil.isEmpty(json.get(whWmsWarehouseArea.getCode()))){
                    targetShelvesMap.put(whWmsWarehouseArea.getHouseType(),json.get(whWmsWarehouseArea.getCode()).toString());
                }
            }
        }
        return targetShelvesMap;
    }

    //组织上架数据
    private boolean createMoveByPutawayLogic(WhWmsGradeOfQuality wmsGrade,
                                                      List<WhWmsGradeOfQualityBarcode> gradeBarcodes){
        List<WhWmsStartPutawayVO> pvoList = new ArrayList<>();
        if(EmptyUtil.isNotEmpty(wmsGrade) && CollectionUtils.isNotEmpty(gradeBarcodes)){
            for(WhWmsGradeOfQualityBarcode item : gradeBarcodes){
                WhWmsStartPutawayVO pt = new WhWmsStartPutawayVO();
                WhWmsHouseShelvesVO shelvesVO = whWmsHouseShelvesService.getHouseShelvesByCode(item.getShelvesCode());
                pt.setSourceShelvesVO(shelvesVO);
                pt.setPhysicalWarehouseCode(shelvesVO.getPhysicalWarehouseCode());
                pt.setPutawayAmount(0);
                pt.setSkuCode(item.getSkuCode());
                pt.setBarCode(item.getBarcode());
                pt.setSkuStatus(wmsGrade.getTargetGrade());
                pt.setReceiveAmount(item.getAdjustAmount());
                pt.setStorageType(0);
                pvoList.add(pt);
            }
        }
        if(EmptyUtil.isNotEmpty(pvoList)){
            List<WhWmsMoveSkuVO> moveskuList = whWmsWaitPutawayService.recommendWmsHouseShelves(pvoList);
            if (CollectionUtils.isNotEmpty(moveskuList)){
                WhWmsMoveStockVO moveStockVO = new WhWmsMoveStockVO();
                moveStockVO.setPhysicalWarehouseCode(wmsGrade.getPhysicalWarehouseCode());
                moveStockVO.setCreateUserId(wmsGrade.getSubmitUserId());
                moveStockVO.setMoveType(WhWmsMoveStockVO.MOVE_TYPE_COMMON);
                moveStockVO.setReferenceCode(wmsGrade.getCode());
                moveStockVO.setMoveSkuList(moveskuList);
                return whWmsMoveStockService.newMoveStock(moveStockVO);
            }
        }
        return false;
    }

    @Override
    public boolean rejectWmsGradeOfQuality(WhWmsGradeOfQualityVO vo, Long wmsGradeSkuId) {
        if (wmsReleaseOccupy(vo,wmsGradeSkuId)){
            // WhWmsGradeOfQualitySku gradeSku = whWmsGradeOfQualitySkuMapper.selectByPrimaryKey(wmsGradeSkuId);
            WhWmsGradeOfQualitySku gradeOfQualitySku = new WhWmsGradeOfQualitySku();
            gradeOfQualitySku.setId(wmsGradeSkuId);
            gradeOfQualitySku.setStatus(WhWmsGradeOfQualitySkuVO.APPROVE_STATUS_REJECT);
            whWmsGradeOfQualitySkuMapper.updateByPrimaryKeySelective(gradeOfQualitySku);

            WhWmsGradeOfQualityVO wmsGradeOfQualityVO = findWhWmsGradeOfQualityVOById(vo.getId(),true,false);
            Set<Integer> statusSet = new HashSet<>();
            for (WhWmsGradeOfQualitySkuVO skuVO : wmsGradeOfQualityVO.getGradeSkuVOs()){
                statusSet.add(skuVO.getStatus());
            }
            int approveStatus = 0;
            if (statusSet.contains(WhWmsGradeOfQualitySkuVO.APPROVE_STATUS_INIT) &&
                    (statusSet.contains(WhWmsGradeOfQualitySkuVO.APPROVE_STATUS_PASS)
                        || statusSet.contains(WhWmsGradeOfQualitySkuVO.APPROVE_STATUS_REJECT))){
                approveStatus = WhWmsGradeOfQualityVO.APPROVE_STATUS_APPROVING;
            }else if(!statusSet.contains(WhWmsGradeOfQualitySkuVO.APPROVE_STATUS_INIT)
                    && !statusSet.contains(WhWmsGradeOfQualitySkuVO.APPROVE_STATUS_PASS)){
                approveStatus = WhWmsGradeOfQualityVO.APPROVE_STATUS_REJECT;
            }else if(!statusSet.contains(WhWmsGradeOfQualitySkuVO.APPROVE_STATUS_INIT)
                    && statusSet.contains(WhWmsGradeOfQualitySkuVO.APPROVE_STATUS_PASS)){
                approveStatus = WhWmsGradeOfQualityVO.APPROVE_STATUS_PARTLY_COMPLETED;
            }
            WhWmsGradeOfQuality gradeOfQuality = new WhWmsGradeOfQuality();
            gradeOfQuality.setId(vo.getId());
            gradeOfQuality.setApproveStatus(approveStatus);
            int result = whWmsGradeOfQualityMapper.updateByPrimaryKeySelective(gradeOfQuality);
            if (result > 0){
                // 满足条件 则生成一张"待出库"的移库单
                buildAndCreateMoveStock(gradeOfQuality);
            }
            return result>0;
        }
        return false;
    }

    private Map<String,WhWmsHouseShelves> getHouseTypeMapByBarcodes(List<WhWmsGradeOfQualityBarcode> gradeBarcodes){
        Set<String> shelvesCodes = new HashSet<>();
        for (WhWmsGradeOfQualityBarcode barcode : gradeBarcodes){
            shelvesCodes.add(barcode.getShelvesCode());
        }
        List<WhWmsHouseShelves> whWmsHouseShelves = whWmsHouseShelvesService.getHouseShelvesByCode(new ArrayList<>(shelvesCodes));
        Map<String,WhWmsHouseShelves> shelvesMap = new HashMap<>();
        for (WhWmsHouseShelves houseShelves : whWmsHouseShelves){
            shelvesMap.put(houseShelves.getCode(),houseShelves);
        }
        return shelvesMap;
    }

    private List<WhWmsGradeOfQualityBarcode> aggregateAdjustAmountByBarcodes(List<WhWmsGradeOfQualityBarcode> gradeBarcodes){
        Map<String,WhWmsGradeOfQualityBarcode> gradeBarcodeMap = new HashMap<>();
        for (WhWmsGradeOfQualityBarcode barcode : gradeBarcodes){
            String key = barcode.getBarcode()+barcode.getShelvesCode();
            WhWmsGradeOfQualityBarcode gradeBc = gradeBarcodeMap.get(key);
            if (EmptyUtil.isEmpty(gradeBc)){
                gradeBarcodeMap.put(key,barcode);
            }else{
                gradeBc.setAdjustAmount(gradeBc.getAdjustAmount()+barcode.getAdjustAmount());
            }
        }

        List<WhWmsGradeOfQualityBarcode> newGradeBarcodes = new ArrayList<>();
        for (String key :  gradeBarcodeMap.keySet()){
            newGradeBarcodes.add(gradeBarcodeMap.get(key));
        }
        return newGradeBarcodes;
    }

    public void wmsOccupy(WhWmsGradeOfQualityVO vo){
        //获取分布式锁
        String lockKey = "lock:whWmsGrade:wmsOccupy:"+vo.getId();
        try {
            Boolean getLock = redisDistributLock.tryLock(lockKey, 3L, TimeUnit.SECONDS);
            if (!getLock) {
                log.error("获取锁失败   false  lockKey="+lockKey);
                throw new WarehouseException(
                        WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED,
                        "请稍后重试!");
            }
            List<WhWmsGradeOfQualityBarcode> gradeBarcodes = listWhWmsGradeOfQualityBarcodesByWmsGradeId(vo.getId());
            if (CollectionUtils.isEmpty(gradeBarcodes)){
                throw new WarehouseException(WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED,"品级调整单详情信息为空，无法占用!");
            }

            // 获取库位类型
            Map<String,WhWmsHouseShelves> shelvesMap = getHouseTypeMapByBarcodes(gradeBarcodes);
            // 聚合调整数量 不需要
            // gradeBarcodes = aggregateAdjustAmountByBarcodes(gradeBarcodes);
            List<WhWmsOccupyVO> occupyList = new ArrayList<>();
            for (WhWmsGradeOfQualityBarcode bc : gradeBarcodes) {
                if (EmptyUtil.isNotEmpty(shelvesMap.get(bc.getShelvesCode()))){
                    //创建原库位占用
                    WhWmsOccupyVO occupyVO = new WhWmsOccupyVO();
                    occupyVO.setSkuCode(bc.getSkuCode());
                    occupyVO.setBarCode(bc.getBarcode());
                    occupyVO.setSkuStatus(vo.getSourceGrade());
                    occupyVO.setAmount(-bc.getAdjustAmount());
                    occupyVO.setHouseType(shelvesMap.get(bc.getShelvesCode()).getHouseType());
                    occupyVO.setPhysicalWarehouseCode(vo.getPhysicalWarehouseCode());
                    occupyVO.setReceiptsNo(vo.getCode()+"_"+bc.getWmsGradeSkuId());
                    occupyVO.setOriShelvesCode(bc.getShelvesCode());
                    occupyVO.setType(WhWmsOccupyVO.TYPE_GRADE_ADJUST);
                    occupyVO.setStatus(WhWmsOccupyVO.OCCUPIED);
                    occupyList.add(occupyVO);
                }
            }
            whWmsOccupyService.wmsBatchOccupy(occupyList);
        }finally {
            redisDistributLock.unLock(lockKey);
        }
    }

    @Override
    public boolean updateSkuStockByCond(WhWmsGradeOfQualityVO whWmsGradeVO,String skuCode,Long wmsGradeSkuId) {
        //获取分布式锁
        String lockKey = "lock:whWmsGrade:updateSkuStockByCond:"+whWmsGradeVO.getCode()+"_"+wmsGradeSkuId;
        try {
            Boolean getLock = redisDistributLock.tryLock(lockKey, 2L, TimeUnit.MINUTES);
            if (!getLock) {
                log.error("获取锁失败   false  lockKey="+lockKey);
                throw new WarehouseException(
                        WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED,
                        "请稍后重试!");
            }
            WhWmsOccupyVO cond = new WhWmsOccupyVO();
            cond.setReceiptsNo(whWmsGradeVO.getCode()+"_"+wmsGradeSkuId);
            cond.setType(WhWmsOccupyVO.TYPE_GRADE_ADJUST);
            cond.setStatus(WhWmsOccupyVO.OCCUPIED);
            cond.setSkuCode(skuCode);
            List<WhWmsOccupyVO> whWmsOccupyVOs = whWmsOccupyService.getWmsOccupyByCond(cond);
            for (WhWmsOccupyVO occupyVO : whWmsOccupyVOs){
                // 减库存
                whWmsSkuStockService.updateStockByCond(occupyVO.getAmount(),occupyVO.getPhysicalWarehouseCode(),
                        occupyVO.getHouseType(),occupyVO.getBarCode(),
                        occupyVO.getOriShelvesCode(),skuCode,whWmsGradeVO.getSourceGrade(),WhInvRcd.TYPE_GRADE_ADJUST_OUT,
                        whWmsGradeVO.getCode()+"_"+skuCode
                        ,whWmsGradeVO.getSubmitUserId(),"品级调整出库:"+whWmsGradeVO.getMemo(),2);

                // 加库存
                whWmsSkuStockService.updateStockByCond(Math.abs(occupyVO.getAmount()),occupyVO.getPhysicalWarehouseCode(),
                        occupyVO.getHouseType(),occupyVO.getBarCode(),
                        occupyVO.getOriShelvesCode(),skuCode,whWmsGradeVO.getTargetGrade(),WhInvRcd.TYPE_GRADE_ADJUST_IN,
                        whWmsGradeVO.getCode()+"_"+skuCode
                        ,whWmsGradeVO.getSubmitUserId(),"品级调整入库:"+whWmsGradeVO.getMemo(),2);
            }
            return true;
        }finally {
            redisDistributLock.unLock(lockKey);
        }
    }

    public boolean modifyStatus(WhWmsGradeOfQualityVO vo){
        WhWmsGradeOfQuality gradeOfQuality = BeanUtil.buildFrom(vo,WhWmsGradeOfQuality.class);
        int result = whWmsGradeOfQualityMapper.updateByPrimaryKeySelective(gradeOfQuality);
        if (result > 0){
            WhWmsGradeOfQualitySkuCond skuCond = new WhWmsGradeOfQualitySkuCond();
            skuCond.setWmsGradeId(vo.getId());
            WhWmsGradeOfQualitySkuExample skuExample = buildWhWmsGradeOfQualitySkuExampleByCond(skuCond);
            WhWmsGradeOfQualitySku record = new WhWmsGradeOfQualitySku();
            record.setStatus(vo.getApproveStatus());
            whWmsGradeOfQualitySkuMapper.updateByExampleSelective(record,skuExample);
            if (deleteGradeSkuAndBarcodes(vo.getId())){
                insertGradeSkuAndBarcodes(vo);
            }
        }
        return result > 0;
    }

    @Override
    @Transactional
    public boolean update(WhWmsGradeOfQualityVO vo) throws Exception{
        WhWmsGradeOfQuality gradeOfQuality = BeanUtil.buildFrom(vo,WhWmsGradeOfQuality.class);
        int result = whWmsGradeOfQualityMapper.updateByPrimaryKeySelective(gradeOfQuality);
        if (result > 0){
            if (deleteGradeSkuAndBarcodes(vo.getId())){
                insertGradeSkuAndBarcodes(vo);
            }
            // 如果是提交审批 则占用对应wms库存 并生成scm品级调整审批单
            if (WhWmsGradeOfQualityVO.APPROVE_STATUS_WAIT == vo.getApproveStatus()){
                submitWmsApprove(vo);
            }
        }
        return result > 0;
    }

    public boolean deleteGradeSkuAndBarcodes(Long gradeId){
        WhWmsGradeOfQualityBarcodeCond barcodeCond = new WhWmsGradeOfQualityBarcodeCond();
        barcodeCond.setWmsGradeId(gradeId);
        boolean result = deleteGradeBarcode(barcodeCond);
        if (result){
            WhWmsGradeOfQualitySkuCond skuCond = new WhWmsGradeOfQualitySkuCond();
            skuCond.setWmsGradeId(gradeId);
            result = deleteGradeSku(skuCond);
        }
        return result;
    }

    public boolean deleteGradeSku(WhWmsGradeOfQualitySkuCond skuCond){
        WhWmsGradeOfQualitySkuExample example = buildWhWmsGradeOfQualitySkuExampleByCond(skuCond);
        return whWmsGradeOfQualitySkuMapper.deleteByExample(example) > 0;
    }

    public boolean deleteGradeBarcode(WhWmsGradeOfQualityBarcodeCond barcodeCond){
        WhWmsGradeOfQualityBarcodeExample example = buildWhWmsGradeOfQualityBarcodeExampleByCond(barcodeCond);
        return whWmsGradeOfQualityBarcodeMapper.deleteByExample(example) > 0;
    }

    public List<WhWmsGradeOfQualityBarcode> selectGradeBarcodeByCond(WhWmsGradeOfQualityBarcodeCond barcodeCond){
        WhWmsGradeOfQualityBarcodeExample example = buildWhWmsGradeOfQualityBarcodeExampleByCond(barcodeCond);
        return whWmsGradeOfQualityBarcodeMapper.selectByExample(example);
    }

    private void insertGradeSkuAndBarcodes(WhWmsGradeOfQualityVO vo){
        if (CollectionUtils.isNotEmpty(vo.getGradeSkuVOs())){
            for (WhWmsGradeOfQualitySkuVO skuVO : vo.getGradeSkuVOs()){
                WhWmsGradeOfQualitySku gradeSku = BeanUtil.buildFrom(skuVO,WhWmsGradeOfQualitySku.class);
                gradeSku.setWmsGradeId(vo.getId());
                gradeSku.setStatus(WhWmsGradeOfQualitySkuVO.APPROVE_STATUS_INIT);
                if (EmptyUtil.isNotEmpty(gradeSku.getSkuImage()) && gradeSku.getSkuImage().endsWith(",")){
                    gradeSku.setSkuImage(gradeSku.getSkuImage().substring(0,gradeSku.getSkuImage().length()-1));
                }
                whWmsGradeOfQualitySkuMapper.insert(gradeSku);
                skuVO.setId(gradeSku.getId()); // 记录id scm那边用

                if (CollectionUtils.isNotEmpty(skuVO.getGradeBarcodes())){
                    for (WhWmsGradeOfQualityBarcode barcode : skuVO.getGradeBarcodes()){
                        barcode.setWmsGradeId(vo.getId());
                        barcode.setWmsGradeSkuId(gradeSku.getId());
                        whWmsGradeOfQualityBarcodeMapper.insert(barcode);
                    }
                }
            }
        }
    }

    @Override
    @Transactional
    public boolean updateWhWmsGradeOfQualitySku(WhWmsGradeOfQualityVO vo) {
        return false;
    }

    @Override
    @Transactional
    public boolean updateWhWmsGradeOfQualityBarcode(WhWmsGradeOfQualityVO vo) {
        return false;
    }

    @Override
    @Transactional
    public boolean batchCreateWhWmsGradeOfQualitySku(List<WhWmsGradeOfQualitySku> gradeOfQualitySkus) {
        return false;
    }

    @Override
    @Transactional
    public boolean batchCreateWhWmsGradeOfQualityBarcode(List<WhWmsGradeOfQualityBarcode> gradeOfQualityBarcodes) {
        return false;
    }

    @Override
    public boolean batchDeleteWhWmsGradeOfQualitySkuByCond(WhWmsGradeOfQualitySkuCond cond) {
        return false;
    }

    @Override
    public boolean batchDeleteWhWmsGradeOfQualityBarcodeByCond(WhWmsGradeOfQualityBarcodeCond cond) {
        return false;
    }

    @Override
    public WhWarehouse findGradeWarehouseByCond(String phyCode, Integer relateType, Integer skuStatus) throws Exception {
        WhWarehouse whWarehouse = whInfoService.findDefaultInOrOutWarehouseByPhyWhCode(phyCode,relateType,skuStatus);
        if (EmptyUtil.isEmpty(whWarehouse)){
            throw new BusinessException("创建失败，物理仓["+phyCode+"],商品状态["+WhWarehouseVO.getSkuStatusName(skuStatus)+"]找不到出库逻辑仓");
        }else if(!WhWarehouseVO.WAREHOUSE_STATUS_CAN_USE.equals(whWarehouse.getWarehouseStatus())){
            throw new BusinessException("创建失败，逻辑仓["+whWarehouse.getCode()+"],为禁用状态");
        }
        return whWarehouse;
    }
}
