/**
 * Copyright (C), 上海布鲁爱电子商务有限公司
 */
package com.thebeastshop.pegasus.service.warehouse.service.impl;

import com.thebeastshop.pegasus.service.warehouse.WMSConstants;
import com.thebeastshop.pegasus.service.warehouse.dao.WhWmsCommandPreOccupyIdxMapper;
import com.thebeastshop.pegasus.service.warehouse.dao.WhWmsConnectDistributionMapper;
import com.thebeastshop.pegasus.service.warehouse.exception.WarehouseException;
import com.thebeastshop.pegasus.service.warehouse.exception.WarehouseExceptionErrorCode;
import com.thebeastshop.pegasus.service.warehouse.exception.WmsExceptionErrorCode;
import com.thebeastshop.pegasus.service.warehouse.model.WhCommand;
import com.thebeastshop.pegasus.service.warehouse.model.WhWmsConnectDistribution;
import com.thebeastshop.pegasus.service.warehouse.model.WhWmsConnectDistributionExample;
import com.thebeastshop.pegasus.service.warehouse.service.*;
import com.thebeastshop.pegasus.service.warehouse.vo.*;
import com.thebeastshop.pegasus.util.PegasusConstants;
import com.thebeastshop.pegasus.util.comm.*;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.*;

/**
 * @author Eric.Lou
 * @version $Id: WhWmsConnectDistributionServiceImpl.java, v 0.1 2016-04-11 下午2:50
 */
@Service("whWmsConnectDistributionService")
public class WhWmsConnectDistributionServiceImpl implements WhWmsConnectDistributionService {

    @Autowired
    private WhWmsConnectDistributionMapper mapper;

    @Autowired
    private WhWmsConnectInfoService whWmsConnectInfoService;

    @Autowired
    private WhWmsConnectPickSkuService whWmsConnectPickSkuService;

    @Autowired
    private WhWmsCommandConnectService whWmsCommandConnectService;

    @Autowired
    private WhCommandService whCommandService;

    @Autowired
    private WhWmsMoveStockService whWmsMoveStockService;

    @Autowired
    private WhWmsCommandPreOccupyIdxMapper whWmsCommandPreOccupyIdxMapper;

    @Autowired
    private WhWmsOperationRcdService whWmsOperationRcdService;

    @Override
    public List<WhWmsConnectDistributionVO> create(List<WhWmsConnectDistributionVO> vos) {
        if(CollectionUtils.isEmpty(vos)){
            throw new WarehouseException(WarehouseExceptionErrorCode.ILLEGAL_PARAM);
        }
        Set<Long> connectIds = new HashSet<>();
        for (WhWmsConnectDistributionVO vo : vos) {
            connectIds.add(vo.getConnectId());
        }
        //同一个connectId的分拨数据
        if(CollectionUtils.isEmpty(connectIds)||connectIds.size()>1){
            throw new WarehouseException(WarehouseExceptionErrorCode.ILLEGAL_PARAM);
        }
        WhWmsConnectInfoVO connectVO = whWmsConnectInfoService.findById(vos.get(0).getConnectId());
        if(!connectVO.getConnectStatus().equals(WMSConstants.ConnectStatus.DISTRIBUTION)){
            throw new WarehouseException(WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED,"波次必须是待分拨状态");
        }
        //更新波次状态, 已完成
        boolean reslut = whWmsConnectInfoService.connectStatusTransform(connectVO.getId(),WMSConstants.ConnectStatus.DISTRIBUTION,WMSConstants.ConnectStatus.FINISHED);
        if(!reslut){
            throw new WarehouseException(WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED,"波次必须是待分拨状态");
        }
        //删除波次拣货周转箱
        whWmsConnectPickSkuService.deleteConnectBox(vos.get(0).getConnectId());
        //更新波次关联的所有command的状态,待分拨到待包装
        for (WhWmsCommandConnectVO commandConnectVO : whWmsCommandConnectService.findNotCanceledByConnectId(vos.get(0).getConnectId())) {
            whCommandService.commandInDistributionToPacking(commandConnectVO.getCommandCode());
        }

        //批量创建波次分拨数据
        for (WhWmsConnectDistributionVO vo : vos) {
            vo.setSkuStatus(connectVO.getSkuStatus());
            create(vo);
        }
        return vos;
    }

    private void buildOperationRcdThenInsert(WhWmsConnectDistributionVO vo ,Long operatorId){
        WhWmsOperationRcdVO rcd = new WhWmsOperationRcdVO();
        rcd.setReferenceCode(vo.getId().toString());
        rcd.setOperateType(WhWmsOperationRcdVO.OPERATOR_TYPE_DISTRIBUTION);
        rcd.setCommandCode(vo.getCommandCode());
        rcd.setOperatorId(operatorId);
        rcd.setOperateTime(DateUtil.getNow());
        //记录操作日志
        whWmsOperationRcdService.insert(rcd);
    }

    @Override
    @Transactional
    public List<WhWmsConnectDistributionVO> create(List<WhWmsConnectDistributionVO> vos, Long operatorId) {
        List<WhWmsConnectDistributionVO> result = create(vos);
        if(EmptyUtil.isNotEmpty(result)){
            Map<String,String> commandMap = new HashMap<>();
            for(WhWmsConnectDistributionVO vo : result){
                if(NullUtil.isNull(commandMap.get(vo.getCommandCode()))){
                    commandMap.put(vo.getCommandCode(),vo.getCommandCode());
                    buildOperationRcdThenInsert(vo,operatorId);
                }
            }
        }
        return result;
    }


    @Override
    @Transactional
    public boolean connectDistribute(WhWmsConnectDistributeInfoVO connectDistributeInfo) {
        //更新波次状态, 已完成
        boolean reslut = whWmsConnectInfoService.connectStatusTransform(connectDistributeInfo.getConnectId(),WMSConstants.ConnectStatus.DISTRIBUTION,WMSConstants.ConnectStatus.FINISHED);
        if(!reslut){
            throw new WarehouseException(WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED,"波次必须是待分拨状态");
        }
        //删除波次拣货周转箱
        whWmsConnectPickSkuService.deleteConnectBox(connectDistributeInfo.getConnectId());
        List<WhWmsCommandDistributeInfoVO> commandDistributeInfoList = connectDistributeInfo.getCommandDistributeList();
        List<String> commandDistributionToPackings = new ArrayList<>();
        if(EmptyUtil.isNotEmpty(commandDistributeInfoList)){
            List<WhWmsConnectDistributionVO> distributeList = new ArrayList<>();
            for(WhWmsCommandDistributeInfoVO commandDistributeInfo : commandDistributeInfoList){
                commandDistributionToPackings.add(commandDistributeInfo.getCommandCode());
                distributeList.addAll(commandDistributeInfo.getDistributeList());
            }
            //更新command的状态,待分拨到待包装
            whCommandService.batchUpdateCommandStatus(commandDistributionToPackings,WhCommand.STATUS_PACKING,WhCommand.STATUS_DISTRIBUTION);
            //记录验货波次分拨数据
            for (WhWmsConnectDistributionVO vo : distributeList) {
                create(vo);
                buildOperationRcdThenInsert(vo,connectDistributeInfo.getOperatorId());
            }
        }
        //缺货指令更新为初始，打缺货标记，波次指令关系打取消标记
        if(EmptyUtil.isNotEmpty(connectDistributeInfo.getShortageCommandCodes())){
            whCommandService.failureStartConnect(connectDistributeInfo.getShortageCommandCodes());
            whCommandService.batchUpdateCommandStatus(connectDistributeInfo.getShortageCommandCodes(),WhCommand.STATUS_IN_PROCESSING,WhCommand.STATUS_DISTRIBUTION);
            whWmsCommandConnectService.batchCancel(connectDistributeInfo.getShortageCommandCodes(),connectDistributeInfo.getConnectId());
            //删除预占记录key,否则无法重新占用
            whWmsCommandPreOccupyIdxMapper.batchDelete(connectDistributeInfo.getShortageCommandCodes());
        }
        //处理相关移库
        if(EmptyUtil.isNotEmpty(connectDistributeInfo.getMoveStockList())){
            for(WhWmsMoveStockVO moveStock : connectDistributeInfo.getMoveStockList()){
                if(NullUtil.isNull(moveStock)){
                    continue;
                }
                if(moveStock.isAutoFinish()){
                    whWmsMoveStockService.executeMoveStock(moveStock);
                }else{
                    whWmsMoveStockService.createMoveStock(moveStock);
                }
                if(WhWmsMoveStockVO.MOVE_TYPE_CANCEL_MAKE.equals(moveStock.getMoveType())){
                    connectDistributeInfo.setToCancelMoveCode(moveStock.getCode());
                }
            }
        }
        return true;
    }

    @Transactional
    private WhWmsConnectDistributionVO create(WhWmsConnectDistributionVO vo) {
        WhWmsConnectDistribution whWmsConnectDistribution = BeanUtil.buildFrom(vo, WhWmsConnectDistribution.class);
        mapper.insert(whWmsConnectDistribution);
        vo.setId(whWmsConnectDistribution.getId());
        return vo;
    }

    @Override
    public Boolean update(WhWmsConnectDistributionVO vo) {
        return mapper.updateByPrimaryKeySelective(BeanUtil.buildFrom(vo,WhWmsConnectDistribution.class))!=0;
    }

    @Override
    @Transactional
    public boolean deleteConnectDistribution(Long connectId, String commandCode) {
        //条件
        WhWmsConnectDistributionExample distributionExample = new WhWmsConnectDistributionExample();
        distributionExample.createCriteria()
                .andConnectIdEqualTo(connectId)
                .andCommandCodeEqualTo(commandCode);
        //软删除
        WhWmsConnectDistribution whWmsConnectDistribution = new WhWmsConnectDistribution();
        whWmsConnectDistribution.setCancelFlag(PegasusConstants.YES);
        int count = mapper.updateByExampleSelective(whWmsConnectDistribution,distributionExample);
//        if(count == 0){
//            //尝试删除拣货周转箱-大宗订单验货
//            whWmsConnectPickSkuService.deleteConnectBox(connectId);
//        }
        return true;
    }

    @Override
    @Transactional
    public boolean batchDeleteConnectDistribution(Long connectId, List<String> commandCodes) {
        //条件
        WhWmsConnectDistributionExample distributionExample = new WhWmsConnectDistributionExample();
        distributionExample.createCriteria()
                .andConnectIdEqualTo(connectId)
                .andCommandCodeIn(commandCodes);
        //软删除
        WhWmsConnectDistribution whWmsConnectDistribution = new WhWmsConnectDistribution();
        whWmsConnectDistribution.setCancelFlag(PegasusConstants.YES);
        int count = mapper.updateByExampleSelective(whWmsConnectDistribution,distributionExample);
        if(count == 0){
            //尝试删除拣货周转箱-大宗订单验货
            whWmsConnectPickSkuService.deleteConnectBox(connectId);
        }
        return true;
    }

    @Override
    public WhWmsConnectDistributionVO findById(Long id) {
        WhWmsConnectDistribution whWmsConnectDistribution = mapper.selectByPrimaryKey(id);
        if(whWmsConnectDistribution!=null){
            return BeanUtil.buildFrom(whWmsConnectDistribution,WhWmsConnectDistributionVO.class);
        }
        return null;
    }

    @Override
    public Map<String, List<String>> findPackageDistributionBoxNo(List<Long> commandIdList, List<String> commandCodeList) {
        Map<String, List<String>> map  = new HashMap<>();
        List<WhWmsConnectDistribution> list =  mapper.findPackageDistributionBoxNo(commandIdList,commandCodeList);
        if(list != null && list.size() > 0){
            for(WhWmsConnectDistribution whWmsConnectDistribution:list){
                List<String> packageCodeList = map.get(whWmsConnectDistribution.getCommandCode());
                if(packageCodeList == null){
                    packageCodeList = new ArrayList<>();
                }
                packageCodeList.add(whWmsConnectDistribution.getDistributionCode());
                map.put(whWmsConnectDistribution.getCommandCode(),packageCodeList);
            }
        }
        return map;
    }

    @Override
    public List<WhWmsConnectDistributionVO> findByCond(WhWmsConnectDistributionVO cond) {
        WhWmsConnectDistributionExample example = new WhWmsConnectDistributionExample();
        WhWmsConnectDistributionExample.Criteria criteria = example.createCriteria();

        if(cond.getConnectId()!=null){
            criteria.andConnectIdEqualTo(cond.getConnectId());
        }
        if(cond.getDistributionCode()!=null){
            criteria.andDistributionCodeEqualTo(cond.getDistributionCode());
        }
        if(cond.getCancelFlag()!=null){
            criteria.andCancelFlagEqualTo(cond.getCancelFlag());
        }

        if(cond.getCommandCode()!=null){
            criteria.andCommandCodeEqualTo(cond.getCommandCode());
        }else if(CollectionUtils.isNotEmpty(cond.getCommandCodes())){
            criteria.andCommandCodeIn(cond.getCommandCodes());
        }

        List<WhWmsConnectDistribution> whWmsConnectDistributions = mapper.selectByExample(example);
        if (CollectionUtils.isEmpty(whWmsConnectDistributions)) {
            return Collections.emptyList();
        }
        List<WhWmsConnectDistributionVO> vos = new ArrayList<WhWmsConnectDistributionVO>(whWmsConnectDistributions.size());
        for (WhWmsConnectDistribution whWmsConnectDistribution : whWmsConnectDistributions) {
            WhWmsConnectDistributionVO whWmsConnectDistributionVO = BeanUtil.buildFrom(whWmsConnectDistribution, WhWmsConnectDistributionVO.class);
            vos.add(whWmsConnectDistributionVO);
        }
        return vos;
    }

    @Override
    public boolean checkConnectDistribution(Long connectId, String commandCode, List<WhWmsCommandCheckVO> paras) {
        WhWmsConnectDistributionExample example = new WhWmsConnectDistributionExample();
        WhWmsConnectDistributionExample.Criteria criteria = example.createCriteria();
        criteria.andConnectIdEqualTo(connectId);
        criteria.andCommandCodeEqualTo(commandCode);
        criteria.andCancelFlagEqualTo(PegasusConstants.NO);
        List<WhWmsConnectDistribution> whWmsConnectDistributions = mapper.selectByExample(example);
        if(CollectionUtils.isEmpty(whWmsConnectDistributions)){
            throw new WarehouseException(WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED,"找不到cmd分拨数据");
        }
//        if(paras.size()!=whWmsConnectDistributions.size()){
//            throw new WarehouseException(WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED,"分拨数据数量匹配不上");
//        }
        for (WhWmsConnectDistribution cd : whWmsConnectDistributions) {
            for (WhWmsCommandCheckVO para : paras) {
                if(cd.getSkuCode().equals(para.getSkuCode())&&cd.getBarCode().equals(para.getBarCode())){
                    cd.setCheckAmount(para.getCheckAmount());
                }
            }
        }
        StringBuilder sb = new StringBuilder("");
        for (WhWmsConnectDistribution cd : whWmsConnectDistributions) {
            if(!cd.getDistributAmount().equals(cd.getCheckAmount())){

                sb.append("skucode:");
                sb.append(cd.getSkuCode());
                sb.append(",barcode:");
                sb.append(cd.getBarCode());
                sb.append(",分拨数:");
                sb.append(cd.getDistributAmount());
                sb.append(",实际分拨数:");
                sb.append(cd.getCheckAmount());
                sb.append("\n");
            }
        }
        String msg = sb.toString();
        if(EmptyUtil.isNotEmpty(msg)){
            throw new WarehouseException(WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED,"分拨数据barcode匹配不上\r\n"+msg);
        }
        return true;
    }

    @Override
    public List<WhWmsConnectDistribution> cancelCheckConnectDistribution(Long connectId, String commandCode, List<WhWmsCommandCheckVO> paras) {
        WhWmsConnectDistributionExample example = new WhWmsConnectDistributionExample();
        WhWmsConnectDistributionExample.Criteria criteria = example.createCriteria();
        criteria.andConnectIdEqualTo(connectId);
        criteria.andCommandCodeEqualTo(commandCode);
        criteria.andCancelFlagEqualTo(PegasusConstants.NO);
        List<WhWmsConnectDistribution> whWmsConnectDistributions = mapper.selectByExample(example);
        if(CollectionUtils.isEmpty(whWmsConnectDistributions)){
            throw new WarehouseException(WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED,"找不到cmd分拨数据");
        }
        for (WhWmsConnectDistribution cd : whWmsConnectDistributions) {
            for (WhWmsCommandCheckVO para : paras) {
                if(cd.getSkuCode().equals(para.getSkuCode())&&cd.getBarCode().equals(para.getBarCode())){
                    cd.setCheckAmount(para.getCheckAmount());
                }
            }
        }
        return whWmsConnectDistributions;
    }

    @Override
    public boolean checkBoxNo(String code) {
        WhWmsConnectDistributionExample example = new WhWmsConnectDistributionExample();
        WhWmsConnectDistributionExample.Criteria criteria = example.createCriteria();
        criteria.andDistributionCodeEqualTo(code);//分拨箱号
        criteria.andCancelFlagEqualTo(PegasusConstants.NO);//未取消
        List<WhWmsConnectDistribution> whWmsConnectDistributions = mapper.selectByExample(example);
        if(CollectionUtils.isNotEmpty(whWmsConnectDistributions)){
            return false;
        }
        return true;
    }


}
