package com.thebeastshop.pegasus.service.warehouse.service.impl;

import com.thebeastshop.pegasus.service.warehouse.cond.PhyWhStockCond;
import com.thebeastshop.pegasus.service.warehouse.cond.WhInventoryCond;
import com.thebeastshop.pegasus.service.warehouse.cond.WhInventoryDetailCond;
import com.thebeastshop.pegasus.service.warehouse.dao.*;
import com.thebeastshop.pegasus.service.warehouse.exception.WarehouseException;
import com.thebeastshop.pegasus.service.warehouse.exception.WarehouseExceptionErrorCode;
import com.thebeastshop.pegasus.service.warehouse.model.*;
import com.thebeastshop.pegasus.service.warehouse.service.*;
import com.thebeastshop.pegasus.service.warehouse.vo.*;
import com.thebeastshop.pegasus.util.PegasusUtilFacade;
import com.thebeastshop.pegasus.util.comm.*;
import com.thebeastshop.pegasus.util.model.CommSku;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import page.Pagination;

import java.util.*;

/**
 * Created by Administrator on 2018/8/2.
 */
@Service("whInventoryService")
public class WhInventoryServiceImpl implements WhInventoryService {

    @Autowired
    private WhInventoryMapper whInventoryMapper;

    @Autowired
    private WhInventoryDetailMapper whInventoryDetailMapper;

    @Autowired
    private WhInventorySkuCategoryMapper whInventorySkuCategoryMapper;

    @Autowired
    private WhInventoryReasionDicMapper whInventoryReasionDicMapper;

    @Autowired
    private WhInventoryConfigMapper whInventoryConfigMapper;

    @Autowired
    private WhInventoryEmailMapper whInventoryEmailMapper;

    @Autowired
    private WhInfoService whInfoService;

    @Autowired
    private WhWmsSkuStockService wmsSkuStockService;

    @Autowired
    private WhInvService whInvService;

    @Autowired
    private WhTakeStockService whTakeStockService;


    @Override
    @Transactional
    public boolean createWhInventory(WhInventoryCreateVO inventoryInfo) {
        preCheckCreateInfo(inventoryInfo);//检查
        Date time = DateUtil.getNow();
        List<Long> idList = new ArrayList<>();
        for(String physicalWarehouseCode : inventoryInfo.getPhysicalWarehouseCodeList()){
            WhInventoryVO inventory = BeanUtil.buildFrom(inventoryInfo,WhInventoryVO.class);
            inventory.setPhysicalWarehouseCode(physicalWarehouseCode);
            inventory.setCreateTime(time);
            inventory.setStatus(WhInventoryVO.STATUS_INIT);
            inventory.setSkuCategoryIdList(inventoryInfo.getSkuCategoryList());
            createWhInventory(inventory);
            idList.add(inventory.getId());
        }
        inventoryInfo.setInventoryIdList(idList);
        return true;
    }

    private void preCheckCreateInfo(WhInventoryCreateVO inventoryInfo){
        if(EmptyUtil.isEmpty(inventoryInfo.getPhysicalWarehouseCodeList())){
            throw new WarehouseException(WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED
                    ,"必须至少指定一个需要盘点的物理仓");
        }
        if(EmptyUtil.isEmpty(inventoryInfo.getSkuCategoryList())){
            throw new WarehouseException(WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED
                    ,"必须至少指定一个需要盘点的品类");
        }
        List<Integer> statusList = new ArrayList<>();
        statusList.add(WhInventoryVO.STATUS_INIT);
        statusList.add(WhInventoryVO.STATUS_PROCESSING);
        statusList.add(WhInventoryVO.STATUS_WAIT_AUDIT_FINANCE);
        statusList.add(WhInventoryVO.STATUS_WAIT_AUDIT_MANAGER);
        WhInventoryCond cond = new WhInventoryCond();
        cond.setPhyWhCodeList(inventoryInfo.getPhysicalWarehouseCodeList());
        cond.setStatusList(statusList);
        List<WhInventoryVO> inventoryList = findWhInventoryByCond(cond);
        if(EmptyUtil.isNotEmpty(inventoryList)){
            Map<String,WhPhysicalWarehouseVO> phyWhMap = whInfoService.findAllPhysicalWarehouseMap();
            StringBuilder buff = new StringBuilder();
            for(WhInventoryVO inventory : inventoryList){
                WhPhysicalWarehouseVO phyWh = phyWhMap.get(inventory.getPhysicalWarehouseCode());
                buff.append(String.format("[%s]%s存在进行中的任务,结束后才能创建新的任务<br/>"
                        ,phyWh.getCode(),phyWh.getName()));
            }
            throw new WarehouseException(WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED
                    ,buff.toString());
        }
    }

    private boolean createWhInventory(WhInventoryVO inventory){
        whInventoryMapper.insert(inventory);
        List<WhInventorySkuCategory> skuCategoryList = new ArrayList<>();
        for(Long skuCategoryId : inventory.getSkuCategoryIdList()){
            WhInventorySkuCategory skuCategory = new WhInventorySkuCategory();
            skuCategory.setInventoryId(inventory.getId());
            skuCategory.setSkuCategoryId(skuCategoryId);
            skuCategoryList.add(skuCategory);
        }
        whInventorySkuCategoryMapper.batchInsert(skuCategoryList);
        //存储盘点配置
        saveInventoryConfig(inventory);
        return true;
    }

    private void saveInventoryConfig(WhInventoryVO inventory){
        List<WhInventoryConfig> configList = new ArrayList<>();
        if(EmptyUtil.isNotEmpty(inventory.getSkuStatusList())){
            for(Integer skuStatus : inventory.getSkuStatusList()){
                WhInventoryConfig config = new WhInventoryConfig();
                config.setInventoryId(inventory.getId());
                config.setAttrValue(skuStatus.toString());
                config.setAttrType(WhInventoryConfigVO.TYPE_SKU_STATUS);
                configList.add(config);
            }
        }
        if(EmptyUtil.isNotEmpty(inventory.getSkuTypeList())){
            for(Integer skuType : inventory.getSkuTypeList()){
                WhInventoryConfig config = new WhInventoryConfig();
                config.setInventoryId(inventory.getId());
                config.setAttrValue(skuType.toString());
                config.setAttrType(WhInventoryConfigVO.TYPE_SKU_TYPE);
                configList.add(config);
            }
        }
        if(EmptyUtil.isNotEmpty(configList)){
            whInventoryConfigMapper.batchInsert(configList);
        }
    }

    @Override
    public WhInventoryVO findWhInventory(Long id) {
        return whInventoryMapper.findInventoryById(id);
    }

    @Override
    public WhInventoryVO findWhInventory(Long invenoryId, boolean cascade) {
        WhInventoryVO inventory = findWhInventory(invenoryId);
        if(NullUtil.isNotNull(inventory) && cascade){
            inventory.setSkuCategoryIdList(
                    findInventorySkuCategoryIdList(invenoryId));
            fullInventoryConfig(inventory);
        }
        return inventory;
    }

    private void fullInventoryConfig(WhInventoryVO inventory){
        List<WhInventoryConfig> configList = findInventoryConfig(inventory.getId());
        inventory.setSkuStatusList(getInventorySkuStatus(configList));
        inventory.setSkuTypeList(getInventorySkuType(configList));
    }

    private List<Integer> getInventorySkuType(List<WhInventoryConfig> configList){
        List<Integer> skuTypeList = new ArrayList<>();
        if(EmptyUtil.isNotEmpty(configList)){
            for(WhInventoryConfig config : configList){
                if(WhInventoryConfigVO.TYPE_SKU_TYPE.equals(config.getAttrType())){
                    skuTypeList.add(Integer.parseInt(config.getAttrValue()));
                }
            }
        }
        return skuTypeList;
    }

    private List<Integer> getInventorySkuStatus(List<WhInventoryConfig> configList){
        List<Integer> skuStatusList = new ArrayList<>();
        if(EmptyUtil.isNotEmpty(configList)){
            for(WhInventoryConfig config : configList){
                if(WhInventoryConfigVO.TYPE_SKU_STATUS.equals(config.getAttrType())){
                    skuStatusList.add(Integer.parseInt(config.getAttrValue()));
                }
            }
        }
        return skuStatusList;
    }

    private List<WhInventoryConfig> findInventoryConfig(Long inventoryId){
        WhInventoryConfigExample example = new WhInventoryConfigExample();
        example.createCriteria().andInventoryIdEqualTo(inventoryId);
        return whInventoryConfigMapper.selectByExample(example);
    }

    private List<Long> findInventorySkuCategoryIdList(Long inventoryId){
        List<Long> skuCategoryIdList = new ArrayList<>();
        WhInventorySkuCategoryExample example = new WhInventorySkuCategoryExample();
        example.createCriteria().andInventoryIdEqualTo(inventoryId);
        List<WhInventorySkuCategory> list = whInventorySkuCategoryMapper.selectByExample(example);
        if(EmptyUtil.isNotEmpty(list)){
            for(WhInventorySkuCategory skuCategory : list){
                skuCategoryIdList.add(skuCategory.getSkuCategoryId());
            }
        }
        return skuCategoryIdList;
    }

    @Override
    public List<WhInventoryVO> findWhInventoryByCond(WhInventoryCond cond) {
        return whInventoryMapper.findInventoryByCond(cond);
    }

    @Override
    public Pagination<WhInventoryVO> findWhInventoryByCondPage(WhInventoryCond cond) {
        Pagination<WhInventoryVO> page = new Pagination<>(cond.getCurrpage(),cond.getPagenum());
        Integer total = whInventoryMapper.countInventory(cond);
        page.setRecord(total);
        if(!NumberUtil.isNullOrZero(total)){
            List<WhInventoryVO> inventoryList = findWhInventoryByCond(cond);
            page.setResultList(inventoryList);
        }
        return page;
    }


    @Override
    @Transactional
    public boolean initWhInventory(Long invenotryId) {
        WhInventoryVO inventory = findWhInventory(invenotryId);
        if(NullUtil.isNull(inventory)){
            throw new WarehouseException(WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED,"初始化失败");
        }
        if(!WhInventoryVO.STATUS_INIT.equals(inventory.getStatus())){
            throw new WarehouseException(WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED
                    ,String.format("盘点任务当前状态[%s]",inventory.getStatusName()));
        }
        //初始化盘点记录行
        List<WhInventoryDetail> detailList = buildInitWhInventoryDetail(inventory);
        if(EmptyUtil.isNotEmpty(detailList)){
            whInventoryDetailMapper.batchInsert(detailList);
        }
        //更新盘点任务状态
        boolean success = updateWhInventoryStatus(inventory.getId()
                ,WhInventoryVO.STATUS_INIT,WhInventoryVO.STATUS_PROCESSING);
        if(!success){
            throw new WarehouseException(WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED
                    ,String.format("当前状态[%s],初始化失败!",inventory.getStatusName()));
        }
        return true;
    }

    private List<WhInventoryDetail> buildInitWhInventoryDetail(WhInventoryVO inventory){
        List<WhInventoryDetail> detailList = new ArrayList<>();
        List<PhyWhStockVO> phyWhStockList = findPhyWhSkuStock(inventory.getPhysicalWarehouseCode(),null,null);
        if(EmptyUtil.isNotEmpty(phyWhStockList)){
            for(PhyWhStockVO phyWhStock : phyWhStockList){
                WhInventoryDetailVO inventoryDetail = BeanUtil.buildFrom(phyWhStock,WhInventoryDetailVO.class);
                inventoryDetail.setStatus(WhInventoryDetailVO.STATUS_NONE);
                inventoryDetail.setCreateTime(DateUtil.getNow());
                inventoryDetail.setCreateUserId(1L);
                inventoryDetail.setInventoryId(inventory.getId());
                inventoryDetail.setOriAmount(phyWhStock.getQuantityTotal());
                inventoryDetail.setRealAmount(0);
                inventoryDetail.setDiffAmount(-phyWhStock.getQuantityTotal());
                inventoryDetail.setAuditStatus(WhInventoryDetailVO.AUDIT_STATUS_INIT);
                detailList.add(inventoryDetail);
            }
        }
        return detailList;
    }

    @Override
    @Transactional
    public boolean cancelWhInventory(Long inventoryId) {
        WhInventoryVO inventory = findWhInventory(inventoryId);
        if(WhInventoryVO.STATUS_INIT.equals(inventory.getStatus())
                || WhInventoryVO.STATUS_PROCESSING.equals(inventory.getStatus())){
            boolean success = updateWhInventoryStatus(inventory.getId()
                    ,inventory.getStatus(),WhInventoryVO.STATUS_CANCEL);
            if(!success){
                throw new WarehouseException(WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED
                        ,String.format("当前状态[%s],无法取消",inventory.getStatusName()));
            }
        }
        return true;
    }

    @Override
    public boolean whInventoryExistDiff(Long inventoryId) {
        List<WhInventoryDetailVO> inventoryDetailList = findWhInventoryDetailByInventoryId(inventoryId);
        return isExistDiff(inventoryDetailList);
    }

    @Override
    @Transactional
    public boolean submitInventoryForManagerAudit(Long inventoryId) {
        WhInventoryVO inventory = findWhInventory(inventoryId,true);
        if(!WhInventoryVO.STATUS_PROCESSING.equals(inventory.getStatus())){
            throw new WarehouseException(WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED
                    ,String.format("当前任务状态[%s]",inventory.getStatusName()));
        }
        //检查物理仓库存是否足够
        checkPhyWhStockForLossout(inventory);

        boolean success = updateWhInventoryStatus(inventoryId
                ,WhInventoryVO.STATUS_PROCESSING,WhInventoryVO.STATUS_WAIT_AUDIT_MANAGER);
        if(!success){
            throw new WarehouseException(WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED
                    ,"操作失败请重试!");
        }
        List<Integer> currentAuditStatusList = new ArrayList<>();
        currentAuditStatusList.add(WhInventoryDetailVO.AUDIT_STATUS_INIT);
        currentAuditStatusList.add(WhInventoryDetailVO.AUDIT_STATUS_REJECT);
        updateWhInventoryDetailAuditStatus(inventoryId
                ,currentAuditStatusList
                ,WhInventoryDetailVO.AUDIT_STATUS_WAIT_AUDIT_MANAGER);
        return true;
    }

    private void checkPhyWhStockForLossout(WhInventoryVO inventory){
        List<PhyWhStockVO> phyWhStockList = findPhyWhSkuStock(inventory.getPhysicalWarehouseCode(),null,null);
        Map<String,Integer> stockMap = new HashMap<>();
        if(NullUtil.isNotNull(phyWhStockList)){
            for(PhyWhStockVO whSkuStock : phyWhStockList){
                stockMap.put(whSkuStock.getSkuCode()+":"+whSkuStock.getSkuStatus(),whSkuStock.getQuantityTotal());
            }
        }
        List<WhInventoryDetailVO> inventoryDetailList = findWhInventoryDetailByInventoryId(inventory.getId());
        StringBuilder buff = new StringBuilder();
        for(WhInventoryDetailVO inventoryDetail : inventoryDetailList){
            int diffQuantity = inventoryDetail.getRealAmount() - inventoryDetail.getOriAmount();
            if(diffQuantity >=0
                    || !WhInventoryDetailVO.STATUS_DONE.equals(inventoryDetail.getStatus())){
                continue;
            }
            Integer phyWhSkuStock = stockMap.get(inventoryDetail.getSkuCode()+":"+inventoryDetail.getSkuStatus());
            if(NullUtil.isNull(phyWhSkuStock)){
                phyWhSkuStock = 0;
            }
            if(-diffQuantity > phyWhSkuStock){
                buff.append(String.format("[%s][%s]库存发生变化，当前库存小于盘亏数量,请重新盘点该条记录"
                        ,inventoryDetail.getSkuCode()
                        ,inventoryDetail.getSkuStatusName()));
            }
        }
        String warningMsg = buff.toString();
        if(EmptyUtil.isNotEmpty(warningMsg)){
            throw new WarehouseException(WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED,warningMsg);
        }
    }

    private boolean updateWhInventoryStatus(Long inventoryId, Integer currentStatus, Integer nextStatus){
        WhInventory updateRecord = new WhInventory();
        updateRecord.setStatus(nextStatus);
        WhInventoryExample example = new WhInventoryExample();
        example.createCriteria().andIdEqualTo(inventoryId).andStatusEqualTo(currentStatus);
        return whInventoryMapper.updateByExampleSelective(updateRecord,example) == 1;
    }

    //只更新已盘点记录
    private boolean updateWhInventoryDetailAuditStatus(
            Long inventoryId
            ,List<Integer> currentAuditStatusList
            ,Integer nextAuditStatus){
        WhInventoryDetail update = new WhInventoryDetail();
        update.setAuditStatus(nextAuditStatus);
        WhInventoryDetailExample example = new WhInventoryDetailExample();
        example.createCriteria()
                .andInventoryIdEqualTo(inventoryId)
                .andAuditStatusIn(currentAuditStatusList);
        whInventoryDetailMapper.updateByExampleSelective(update,example);
        return true;
    }

    private List<PhyWhStockVO> findPhyWhSkuStock(String physicalWarehouseCode, String skuCode, Integer skuStatus){
        PhyWhStockCond cond = new PhyWhStockCond();
        cond.setPhysicalWarehouseCode(physicalWarehouseCode);
        cond.setGtZero(true);
        cond.setSkuCode(skuCode);
        cond.setSkuStatus(skuStatus);
        List<Integer> skuStatusList = new ArrayList<>();
        skuStatusList.add(WhWarehouseVO.COMMODITY_STATUS_FOR_NONDEFECTIVE);
        skuStatusList.add(WhWarehouseVO.COMMODITY_STATUS_FOR_MILD_DAMAGED);
        skuStatusList.add(WhWarehouseVO.COMMODITY_STATUS_FOR_SAMPLE);
        skuStatusList.add(WhWarehouseVO.COMMODITY_STATUS_FOR_DISPLAY);
        cond.setSkuStatusList(skuStatusList);
        return wmsSkuStockService.findPhyWhSkuTotalQuantityByCond(cond);
    }

    @Override
    public WhInventoryDetailVO findWhInventoryDetail(Long detailId) {
        WhInventoryDetailCond detailCond = new WhInventoryDetailCond();
        detailCond.setId(detailId);
        List<WhInventoryDetailVO> list = findWhInventoryDetailByCond(detailCond);
        if(EmptyUtil.isNotEmpty(list)){
            return list.get(0);
        }
        return null;
    }

    @Override
    public List<WhInventoryDetailVO> findWhInventoryDetailByCond(WhInventoryDetailCond detailCond) {
        List<WhInventoryDetailVO> detailList = whInventoryDetailMapper.findWhInventoryDetailByCond(detailCond);
        if(EmptyUtil.isNotEmpty(detailList)){
            //数据库 差异数据字段 不准确 ，弃用
            detailList.forEach(detail->detail.setDiffAmount(detail.getRealAmount() - detail.getOriAmount()));
        }
        return detailList;
    }

    List<WhInventoryDetailVO> findWhInventoryDetailByInventoryId(Long inventoryId){
        WhInventoryDetailCond detailCond = new WhInventoryDetailCond();
        detailCond.setInventoryId(inventoryId);
        return findWhInventoryDetailByCond(detailCond);
    }

    @Override
    public Pagination<WhInventoryDetailVO> findWhInventoryDetailByCondPage(WhInventoryDetailCond detailCond) {
        Pagination<WhInventoryDetailVO> page = new Pagination<>(detailCond.getCurrpage(),detailCond.getPagenum());
        Integer total = whInventoryDetailMapper.countWhInventoryDetail(detailCond);
        page.setRecord(total);
        if(!NumberUtil.isNullOrZero(total)){
            List<WhInventoryDetailVO> list = findWhInventoryDetailByCond(detailCond);
            page.setResultList(list);
        }
        return page;
    }

    @Override
    public List<WhInventoryReasionDicVO> findAllWhInventoryReasionDic() {
        WhInventoryReasionDicExample example = new WhInventoryReasionDicExample();
        example.createCriteria();
        List<WhInventoryReasionDic> list = whInventoryReasionDicMapper.selectByExample(example);
        return BeanUtil.buildListFrom(list,WhInventoryReasionDicVO.class);
    }

    @Override
    @Transactional
    public boolean submitWhInventoryDetail(WhInventoryDetailVO inventoryDetail) {
        processingWhInventoryDetailCheck(inventoryDetail.getId());
        WhInventoryDetailVO currentDetail = findWhInventoryDetail(inventoryDetail.getId());
        if(NullUtil.isNull(currentDetail)){
            throw new WarehouseException(WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED
                    ,"盘点记录不存在!");
        }
        WhInventoryVO inventory = findWhInventory(currentDetail.getInventoryId());
        if(!WhInventoryVO.STATUS_PROCESSING.equals(inventory.getStatus())){
            throw new WarehouseException(WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED
                    ,String.format("当前盘点任务状态[%s]",inventory.getStatusName()));
        }
        if(currentDetail.getOriAmount().equals(currentDetail.getRealAmount())){
            currentDetail.setDiffReasionType(null);
            currentDetail.setMark(null);
        }
        inventoryDetail.setStatus(WhInventoryDetailVO.STATUS_DONE);
        whInventoryDetailMapper.submitWhInventory(inventoryDetail);
        return true;
    }

    @Override
    @Transactional
    public boolean reInitInventoryDetail(Long inventoryDetailId) {
        WhInventoryDetailVO currentDetail = findWhInventoryDetail(inventoryDetailId);
        if(NullUtil.isNull(currentDetail)){
            throw new WarehouseException(WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED
                    ,"盘点记录不存在!");
        }
        WhInventoryVO inventory = findWhInventory(currentDetail.getInventoryId());
        if(!WhInventoryVO.STATUS_PROCESSING.equals(inventory.getStatus())){
            throw new WarehouseException(WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED
                    ,String.format("当前盘点任务状态[%s]",inventory.getStatusName()));
        }
//        List<Integer> auditStatusList = new ArrayList<>();
//        auditStatusList.add(WhInventoryDetailVO.AUDIT_STATUS_INIT);
//        auditStatusList.add(WhInventoryDetailVO.AUDIT_STATUS_REJECT);
//        auditStatusList.add(WhInventoryDetailVO.AUDIT_STATUS_WAIT_AUDIT_FINANCE);
//        if(!auditStatusList.contains(currentDetail.getAuditStatus())){
//            throw new WarehouseException(WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED
//                    ,String.format("操作失败,当前记录审核状态[%s]",currentDetail.getAuditStatusName()));
//        }
        //查找sku当前库存
        int oriAmount = findPhyWhSkuTotalQuantity(
                inventory.getPhysicalWarehouseCode()
                ,currentDetail.getSkuCode()
                ,currentDetail.getSkuStatus());
        currentDetail.setOriAmount(oriAmount);
        currentDetail.setRealAmount(0);
        currentDetail.setDiffAmount(0-currentDetail.getOriAmount());
        currentDetail.setDiffReasionType(null);
        currentDetail.setMark(null);
        currentDetail.setStatus(WhInventoryDetailVO.STATUS_NONE);
        currentDetail.setAuditStatus(WhInventoryDetailVO.AUDIT_STATUS_INIT);
        WhInventoryDetailExample example = new WhInventoryDetailExample();
        example.createCriteria()
//                .andAuditStatusIn(auditStatusList)
                .andIdEqualTo(inventoryDetailId);
        boolean success = whInventoryDetailMapper.updateByExample(currentDetail,example) == 1;
        if(!success){
            throw new WarehouseException(WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED
                    ,"操作失败,请重试！");
        }
        return true;
    }

    @Override
    @Transactional
    public boolean addWhInventoryDetail(WhInventoryDetailVO inventoryDetail) {
        if(EmptyUtil.isEmpty(inventoryDetail.getSkuCode())
                || NullUtil.isNull(inventoryDetail.getSkuStatus())
                || NullUtil.isNull(inventoryDetail.getSkuStatus())){
            throw new WarehouseException(WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED
                    ,"操作失败,参数异常!");
        }
        WhInventoryVO inventory = findWhInventory(inventoryDetail.getInventoryId());
        if(WhInventoryVO.STATUS_CANCEL.equals(inventory.getStatus())){
            throw new WarehouseException(WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED
                    ,"操作失败,盘点任务已取消!");
        }
        WhInventoryDetailCond detailCond = new WhInventoryDetailCond();
        detailCond.setSkuStatus(inventoryDetail.getSkuStatus());
        detailCond.setSkuCode(inventoryDetail.getSkuCode());
        detailCond.setInventoryId(inventoryDetail.getInventoryId());
        List<WhInventoryDetailVO> detailList = findWhInventoryDetailByCond(detailCond);
        if(EmptyUtil.isNotEmpty(detailList)){
            throw new WarehouseException(WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED
                    ,String.format("已经存在[%s][%s]的记录,无法再次添加"
                    ,inventoryDetail.getSkuCode(),inventoryDetail.getSkuStatusName()));
        }
        //查找sku当前库存
        int oriAmount = findPhyWhSkuTotalQuantity(
                inventory.getPhysicalWarehouseCode()
                ,inventoryDetail.getSkuCode()
                ,inventoryDetail.getSkuStatus());
        inventoryDetail.setOriAmount(oriAmount);
        inventoryDetail.setRealAmount(0);
        inventoryDetail.setDiffAmount(0-inventoryDetail.getOriAmount());
        inventoryDetail.setStatus(WhInventoryDetailVO.STATUS_NONE);
        inventoryDetail.setAuditStatus(WhInventoryDetailVO.AUDIT_STATUS_INIT);
        inventoryDetail.setCreateTime(DateUtil.getNow());
        whInventoryDetailMapper.insert(inventoryDetail);
        return true;
    }

    private int findPhyWhSkuTotalQuantity(String physicalWarehouseCode, String skuCode, Integer skuStatus){
        List<PhyWhStockVO> phyWhStockList = findPhyWhSkuStock(
                physicalWarehouseCode
                ,skuCode
                ,skuStatus);
        if(EmptyUtil.isNotEmpty(phyWhStockList)){
            return phyWhStockList.get(0).getQuantityTotal();
        }
        return 0;
    }


    @Override
    @Transactional
    public boolean inventoryDetailManagerAudit(WhInventoryDetailAuditVO auditInfo) {
        WhInventoryDetailVO detail = findWhInventoryDetail(auditInfo.getInventoryDetailId());
        if(!WhInventoryVO.STATUS_WAIT_AUDIT_MANAGER.equals(detail.getInventoryStatus())){
            throw new WarehouseException(WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED
                    ,String.format("操作失败，当前盘点任务状态[%s]"
                    ,WhInventoryVO.getStatusName(detail.getInventoryStatus())));
        }
        if(!WhInventoryDetailVO.AUDIT_STATUS_WAIT_AUDIT_MANAGER.equals(detail.getAuditStatus())
                && !WhInventoryDetailVO.AUDIT_STATUS_WAIT_AUDIT_FINANCE.equals(detail.getAuditStatus())){
            throw new WarehouseException(WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED
                    ,String.format("操作失败，当前盘点记录状态[%s]"
                    ,detail.getAuditStatusName()));
        }
        Integer nextAuditStatus = auditInfo.isAuditPass()
                ?WhInventoryDetailVO.AUDIT_STATUS_WAIT_AUDIT_FINANCE
                :WhInventoryDetailVO.AUDIT_STATUS_REJECT;
        boolean success = updateInventoryDetailForAudit(auditInfo.getInventoryDetailId()
                ,detail.getAuditStatus()
                ,nextAuditStatus,auditInfo.getHandlingSuggestion());
        if(!success){
            throw new WarehouseException(WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED,"操作出错,请重试!");
        }
        return true;
    }

    @Override
    @Transactional
    public boolean inventoryDetailManagerBatchAudit(WhInventoryDetailAuditVO auditInfo) {

        if(EmptyUtil.isEmpty(auditInfo.getInventoryDetailIdList())){
            throw new WarehouseException(WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED,"操作失败，无驳回行数据");
        }

        List<Integer> statusList = new ArrayList<>();
        statusList.add(WhInventoryDetailVO.AUDIT_STATUS_WAIT_AUDIT_FINANCE);
        statusList.add(WhInventoryDetailVO.AUDIT_STATUS_WAIT_AUDIT_MANAGER);

        Integer nextAuditStatus = auditInfo.isAuditPass()
                ?WhInventoryDetailVO.AUDIT_STATUS_WAIT_AUDIT_FINANCE
                :WhInventoryDetailVO.AUDIT_STATUS_REJECT;

        WhInventoryDetail update = new WhInventoryDetail();
        update.setAuditStatus(nextAuditStatus);
        if(EmptyUtil.isNotEmpty(auditInfo.getHandlingSuggestion())){
            update.setHandlingSuggestion(auditInfo.getHandlingSuggestion());
        }

        WhInventoryDetailExample example = new WhInventoryDetailExample();
        example.createCriteria()
                .andIdIn(auditInfo.getInventoryDetailIdList())
                .andAuditStatusIn(statusList);

        whInventoryDetailMapper.updateByExampleSelective(update,example);

        return true;
    }

    @Override
    @Transactional
    public boolean inventoryFinishForManager(WhInventoryAuditVO auditInfo) {
        WhInventoryVO inventory = findWhInventory(auditInfo.getInventoryId());
        if(!WhInventoryVO.STATUS_WAIT_AUDIT_MANAGER.equals(inventory.getStatus())){
            throw new WarehouseException(WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED
                    ,String.format("操作失败，当前盘点任务状态[%s]"
                    ,inventory.getStatusName()));
        }
        //所有状态为【待经理审核】的记录更新为【待财务审核】
        updateWhInventoryDetailAuditStatus(inventory.getId()
                ,Collections.singletonList(WhInventoryDetailVO.AUDIT_STATUS_WAIT_AUDIT_MANAGER)
                ,WhInventoryDetailVO.AUDIT_STATUS_WAIT_AUDIT_FINANCE);
        List<WhInventoryDetailVO> inventoryDetailList = findWhInventoryDetailByInventoryId(inventory.getId());
        if(isExistReject(inventoryDetailList)){
            //如果存在【驳回】记录，则提示”由于存在驳回的记录，该盘点需要重新确认”,任务状态更新为【进行中】
            auditInfo.setAlertMsg("由于存在驳回的记录，该盘点需要重新确认!");
            auditInfo.setPass(false);
            boolean success = updateWhInventoryStatus(inventory.getId()
                    ,WhInventoryVO.STATUS_WAIT_AUDIT_MANAGER
                    ,WhInventoryVO.STATUS_PROCESSING);
            if(!success){
                throw new WarehouseException(WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED,"操作出错,请重试!");
            }
            return true;
        }
        boolean existDiff = isExistDiff(inventoryDetailList);
        boolean existNotDone = isExistNotDone(inventoryDetailList);
        //存在有差异的记录或者未盘点的记录，则提示”审批成功，已进入财务审批阶段”，任务状态更新为【待财务审批】
        auditInfo.setAlertMsg("审批成功，已进入财务审批阶段!");
        updateWhInventoryAuditManager(auditInfo);
        boolean success = updateWhInventoryStatus(inventory.getId()
                ,WhInventoryVO.STATUS_WAIT_AUDIT_MANAGER
                ,WhInventoryVO.STATUS_WAIT_AUDIT_FINANCE);
        if(!success){
            throw new WarehouseException(WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED,"操作出错,请重试!");
        }
        if(!existDiff && !existNotDone){
            //提示”审批成功，该盘点任务已完成”，任务状态更新为【已完成】
            return inventoryFinishForFinance(auditInfo);
        }
        return true;
    }

    private boolean updateWhInventoryAuditManager(WhInventoryAuditVO auditInfo){
        WhInventory updateRecord = new WhInventory();
        updateRecord.setManageAuditId(auditInfo.getAuditOperaterId());
        updateRecord.setManageAuditTime(DateUtil.getNow());
        WhInventoryExample example = new WhInventoryExample();
        example.createCriteria().andIdEqualTo(auditInfo.getInventoryId());
        whInventoryMapper.updateByExampleSelective(updateRecord,example);
        return true;
    }

    @Override
    @Transactional
    public boolean inventoryDetailFinanceAudit(WhInventoryDetailAuditVO auditInfo) {
        WhInventoryDetailVO detail = findWhInventoryDetail(auditInfo.getInventoryDetailId());
        if(!WhInventoryVO.STATUS_WAIT_AUDIT_FINANCE.equals(detail.getInventoryStatus())){
            throw new WarehouseException(WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED
                    ,String.format("操作失败，当前盘点任务状态[%s]"
                    ,WhInventoryVO.getStatusName(detail.getInventoryStatus())));
        }
        if(!WhInventoryDetailVO.AUDIT_STATUS_WAIT_AUDIT_FINANCE.equals(detail.getAuditStatus())
                && !WhInventoryDetailVO.AUDIT_STATUS_FINISH.equals(detail.getAuditStatus())){
            throw new WarehouseException(WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED
                    ,String.format("操作失败，当前盘点记录审核状态[%s]"
                    ,detail.getAuditStatusName()));
        }
        Integer nextAuditStatus = auditInfo.isAuditPass()
                ?WhInventoryDetailVO.AUDIT_STATUS_FINISH
                :WhInventoryDetailVO.AUDIT_STATUS_REJECT;
        boolean success = updateInventoryDetailForFinanceAudit(auditInfo.getInventoryDetailId()
                ,detail.getAuditStatus()
                ,nextAuditStatus,auditInfo.getHandlingSuggestion());
        if(!success){
            throw new WarehouseException(WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED,"操作出错,请重试!");
        }
        return true;
    }


    @Override
    @Transactional
    public boolean inventoryDetailFinanceBatchAudit(WhInventoryDetailAuditVO auditInfo) {

        if(EmptyUtil.isEmpty(auditInfo.getInventoryDetailIdList())){
            throw new WarehouseException(WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED,"操作失败，无驳回行数据");
        }

        List<Integer> statusList = new ArrayList<>();
        statusList.add(WhInventoryDetailVO.AUDIT_STATUS_WAIT_AUDIT_FINANCE);
        statusList.add(WhInventoryDetailVO.AUDIT_STATUS_FINISH);

        Integer nextAuditStatus = auditInfo.isAuditPass()
                ?WhInventoryDetailVO.AUDIT_STATUS_FINISH
                :WhInventoryDetailVO.AUDIT_STATUS_REJECT;

        WhInventoryDetail update = new WhInventoryDetail();
        update.setAuditStatus(nextAuditStatus);
        if(EmptyUtil.isNotEmpty(auditInfo.getHandlingSuggestion())){
            update.setFinanceHandlingSuggestion(auditInfo.getHandlingSuggestion());
        }

        WhInventoryDetailExample example = new WhInventoryDetailExample();
        example.createCriteria()
                .andIdIn(auditInfo.getInventoryDetailIdList())
                .andAuditStatusIn(statusList);

        whInventoryDetailMapper.updateByExampleSelective(update,example);

        return true;
    }

    @Override
    @Transactional
    public boolean inventoryFinishForFinance(WhInventoryAuditVO auditInfo) {
        WhInventoryVO inventory = findWhInventory(auditInfo.getInventoryId());
        if(!WhInventoryVO.STATUS_WAIT_AUDIT_FINANCE.equals(inventory.getStatus())){
            throw new WarehouseException(WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED
                    ,String.format("操作失败，当前盘点任务状态[%s]"
                    ,inventory.getStatusName()));
        }
        //所有状态为【待财务审核】的记录更新为【已完成】
        updateWhInventoryDetailAuditStatus(inventory.getId()
                ,Collections.singletonList(WhInventoryDetailVO.AUDIT_STATUS_WAIT_AUDIT_FINANCE)
                ,WhInventoryDetailVO.AUDIT_STATUS_FINISH);
        List<WhInventoryDetailVO> inventoryDetailList = findWhInventoryDetailByInventoryId(inventory.getId());
        if(isExistReject(inventoryDetailList)){
            //如果存在【驳回】记录，则提示”由于存在驳回的记录，该盘点需要重新确认”,任务状态更新为【进行中】
            auditInfo.setAlertMsg("由于存在驳回的记录，该盘点需要重新确认!");
            auditInfo.setPass(false);
            boolean success = updateWhInventoryStatus(inventory.getId()
                    ,WhInventoryVO.STATUS_WAIT_AUDIT_FINANCE
                    ,WhInventoryVO.STATUS_PROCESSING);
            if(!success){
                throw new WarehouseException(WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED,"操作出错,请重试!");
            }
            return true;
        }
        //如果不存在【驳回】记录，则提示”审批成功，该盘点任务已完成”。任务状态更新为【已完成】，差异库存生成盘点记录，更新库存
        auditInfo.setAlertMsg("审批成功，该盘点任务已完成!");
        updateWhInventoryAuditFinance(auditInfo);
        boolean success = updateWhInventoryStatus(inventory.getId()
                ,WhInventoryVO.STATUS_WAIT_AUDIT_FINANCE
                ,WhInventoryVO.STATUS_FINISH);
        if(!success){
            throw new WarehouseException(WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED,"操作出错,请重试!");
        }
        inventory.setInventoryDetailList(inventoryDetailList);
        inventory.setAuditOperaterId(auditInfo.getAuditOperaterId());
        //更新库存
        processInventorySkuStock(inventory);
        return true;
    }

    private boolean updateWhInventoryAuditFinance(WhInventoryAuditVO auditInfo){
        WhInventory updateRecord = new WhInventory();
        updateRecord.setFinanceAuditId(auditInfo.getAuditOperaterId());
        updateRecord.setFinanceAuditTime(DateUtil.getNow());
        WhInventoryExample example = new WhInventoryExample();
        example.createCriteria().andIdEqualTo(auditInfo.getInventoryId());
        whInventoryMapper.updateByExampleSelective(updateRecord,example);
        return true;
    }

    private void processInventorySkuStock(WhInventoryVO inventory){
        List<WhInventoryDetailVO> inventoryDetailList = inventory.getInventoryDetailList();
        if(EmptyUtil.isEmpty(inventoryDetailList)){
            return;
        }
        //更新物理仓库存
        processWhWmsSkuStock(inventory);
        //更新逻辑仓库存
        processWhSkuStock(inventory);
    }

    private void processWhWmsSkuStock(WhInventoryVO inventory){
        List<WhInventoryDetailVO> inventoryDetailList = inventory.getInventoryDetailList();
        if(EmptyUtil.isEmpty(inventoryDetailList)){
            return;
        }
        for(WhInventoryDetailVO inventoryDetail : inventoryDetailList){
            if(!WhInventoryDetailVO.STATUS_DONE.equals(inventoryDetail.getStatus())
                    || inventoryDetail.getOriAmount().equals(inventoryDetail.getRealAmount())){
                continue;
            }
            int diffQuantity = inventoryDetail.getRealAmount() - inventoryDetail.getOriAmount();
            wmsSkuStockService.updatePhyWhSkuStock(inventory.getPhysicalWarehouseCode()
                    ,inventoryDetail.getSkuCode(),inventoryDetail.getSkuStatus()
                    ,inventoryDetail.getRealAmount() - inventoryDetail.getOriAmount()
                    ,diffQuantity>0 ?WhCommand.TYPE_INVENTROY_PROFIT_IN
                            :WhCommand.TYPE_INVENTROY_LOSS_OUT
                    ,inventory.getId().toString(),inventory.getAuditOperaterId()
                    ,inventoryDetail.getMark());
        }
    }

    private void processWhSkuStock(WhInventoryVO inventory){
        List<WhInventoryDetailVO> inventoryDetailList = inventory.getInventoryDetailList();
        if(EmptyUtil.isEmpty(inventoryDetailList)){
            return;
        }
        Map<Integer,List<WhWarehouse>> whMap = whInfoService
                .findGroupWarehouseSortedMap(inventory.getPhysicalWarehouseCode(),null);
        List<WhInvRcd> invRcdList = new ArrayList<>();
        List<WhInvRcd> profitInInvRcdList = buildProfitInSkuStock(inventory,whMap);
        List<WhInvRcd> lossOutInvRcdList = buildLossOutSkuStock(inventory,whMap);
        invRcdList.addAll(profitInInvRcdList);//盘盈数
        invRcdList.addAll(lossOutInvRcdList);//盘盈数
        //whInvService.batchRecords(invRcdList);
        List<WhTakeStock> takeStockList = buildWhTakeStock(invRcdList,inventory);
        try{
            whTakeStockService.batchCreateTakeStock(takeStockList);
        }catch (Exception e){
            throw new WarehouseException(WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED,"同步SCM库存异常:"+e.getMessage());
        }
    }

    private List<WhTakeStock> buildWhTakeStock(List<WhInvRcd> invRcdList,WhInventoryVO inventory){
        //按逻辑仓聚合
        Map<String,List<WhInvRcd>> whGroup = new HashMap<>();
        for(WhInvRcd invRcd : invRcdList){
            List<WhInvRcd> list = whGroup.get(invRcd.getWarehouseCode());
            if(NullUtil.isNull(list)){
                list = new ArrayList<>();
                whGroup.put(invRcd.getWarehouseCode(),list);
            }
            list.add(invRcd);
        }
        List<WhTakeStock> takeStockList = new ArrayList<>();
        for(Map.Entry<String,List<WhInvRcd>> entry : whGroup.entrySet()){
            WhTakeStock whTakeStock = new WhTakeStock();
            whTakeStock.setUpdatePhyWhStock(false);
            whTakeStock.setWarehouseCode(entry.getKey());
            whTakeStock.setType(WhTakeStock.TYPE_STORE);
            whTakeStock.setReferenceId(inventory.getId());
            whTakeStock.setReferenceMemo(inventory.getName());
            whTakeStock.setSyncThirdpartStock(false);
            whTakeStock.setCreateUserId(inventory.getFinanceAuditId());
            List<WhTakeStockRcd> takeStockRcdList = new ArrayList<>();
            for(WhInvRcd invRcd : entry.getValue()){
                WhTakeStockRcd takeStockRcd = BeanUtil.buildFrom(invRcd,WhTakeStockRcd.class);
                takeStockRcd.setResultType(invRcd.getQuantity() > 0
                        ? WhTakeStockRcd.TYPE_PROFIT:WhTakeStockRcd.TYPE_LOSS);
                takeStockRcd.setReferenceDetailId(invRcd.getTakeStockRcdId().longValue());
                takeStockRcdList.add(takeStockRcd);
            }
            whTakeStock.setWhTakeStockRcdList(takeStockRcdList);
            takeStockList.add(whTakeStock);
        }
        return takeStockList;
    }

    private List<WhInvRcd> buildLossOutSkuStock(WhInventoryVO inventory,Map<Integer,List<WhWarehouse>> whMap){
        List<WhInvRcd> rcdList = new ArrayList<>();
        List<WhInventoryDetailVO> inventoryDetailList = inventory.getInventoryDetailList();
        if(EmptyUtil.isEmpty(inventoryDetailList)){
            return rcdList;
        }
        //盘亏sku可以库存
        Map<String,Map<String,WhInvVO>> whSkuStockMap = getLossOutCanUseWhSkuStockMap(inventory,whMap);
        StringBuilder buff = new StringBuilder();
        for(WhInventoryDetailVO inventoryDetail : inventoryDetailList){
            if(!WhInventoryDetailVO.STATUS_DONE.equals(inventoryDetail.getStatus())
                    || inventoryDetail.getOriAmount().equals(inventoryDetail.getRealAmount())){
                continue;
            }
            List<WhWarehouse> whListSorted = whMap.get(inventoryDetail.getSkuStatus());
            if(EmptyUtil.isEmpty(whListSorted)){
                throw new WarehouseException(WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED
                        ,String.format("[%s]不存在[%s]逻辑仓!"
                        ,inventory.getPhysicalWarehouseCode()
                        ,WhWarehouseVO.getSkuStatusName(inventoryDetail.getSkuStatus())));
            }
            int diffQuantity = inventoryDetail.getRealAmount() - inventoryDetail.getOriAmount();
            if(diffQuantity < 0){
                //盘亏//优先可用库存
                int needQuantity = -diffQuantity;
                for(WhWarehouse wh : whListSorted){
                    WhInvVO whSkuInv = findWhSkuInv(inventoryDetail.getSkuCode(),wh.getCode(),whSkuStockMap);
                    if(NullUtil.isNotNull(whSkuInv) && whSkuInv.getCanUseInv() > 0){
                        int canUse = Math.min(needQuantity,whSkuInv.getCanUseInv());
                        needQuantity -= canUse;
                        whSkuInv.setQuantityInRcd(whSkuInv.getQuantityInRcd() - canUse);//更新实际数量
                        WhInvRcdVO invRcd = buildWhInvRcd(inventoryDetail
                                ,wh
                                ,-canUse
                                ,inventory.getAuditOperaterId());
                        rcdList.add(invRcd);
                    }
                    if(needQuantity == 0){
                        break;
                    }
                }
                if(needQuantity != 0){////可用库存不足，使用总数
                    for(WhWarehouse wh : whListSorted){
                        WhInvVO whSkuInv = findWhSkuInv(inventoryDetail.getSkuCode(),wh.getCode(),whSkuStockMap);
                        if(NullUtil.isNotNull(whSkuInv) && whSkuInv.getQuantityInRcd() > 0){
                            int canUse = Math.min(needQuantity,whSkuInv.getQuantityInRcd());
                            needQuantity -= canUse;
                            whSkuInv.setQuantityInRcd(whSkuInv.getQuantityInRcd() - canUse);//更新实际数量
                            WhInvRcdVO invRcd = buildWhInvRcd(inventoryDetail
                                    ,wh
                                    ,-canUse
                                    ,inventory.getAuditOperaterId());
                            rcdList.add(invRcd);
                        }
                        if(needQuantity == 0){
                            break;
                        }
                    }
                }
                if(needQuantity != 0 ){
                    buff.append(String.format("[%s]库存不足<br/>",inventoryDetail.getSkuCode()));
                }
            }
        }
        String msg = buff.toString();
        if(EmptyUtil.isNotEmpty(msg)){
            throw new WarehouseException(WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED,msg);
        }
        return rcdList;
    }

    private List<WhInvRcd> buildProfitInSkuStock(WhInventoryVO inventory,Map<Integer,List<WhWarehouse>> whMap){
        List<WhInvRcd> rcdList = new ArrayList<>();
        List<WhInventoryDetailVO> inventoryDetailList = inventory.getInventoryDetailList();
        if(EmptyUtil.isEmpty(inventoryDetailList)){
            return rcdList;
        }
        for(WhInventoryDetailVO inventoryDetail : inventoryDetailList){
            if(!WhInventoryDetailVO.STATUS_DONE.equals(inventoryDetail.getStatus())
                    || inventoryDetail.getOriAmount().equals(inventoryDetail.getRealAmount())){
                continue;
            }
            List<WhWarehouse> whListSorted = whMap.get(inventoryDetail.getSkuStatus());
            if(EmptyUtil.isEmpty(whListSorted)){
                throw new WarehouseException(WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED
                        ,String.format("[%s]不存在[%s]逻辑仓!"
                        ,inventory.getPhysicalWarehouseCode()
                        ,WhWarehouseVO.getSkuStatusName(inventoryDetail.getSkuStatus())));
            }
            int diffQuantity = inventoryDetail.getRealAmount() - inventoryDetail.getOriAmount();
            if(diffQuantity > 0){
                //盘盈
                WhInvRcdVO invRcd = buildWhInvRcd(inventoryDetail
                        ,whListSorted.get(0)
                        ,diffQuantity
                        ,inventory.getAuditOperaterId());
                rcdList.add(invRcd);
            }
        }
        return rcdList;
    }

//    private void processSyncTmallSkuStock(List<WhInvRcd> invRcdList,Long inventoryId){
//        if(EmptyUtil.isEmpty(invRcdList)){
//            return;
//        }
//        Map<String,List<CommSku>> skuListMap = new HashMap<>();
//        for(WhInvRcd rcd : invRcdList){
//            String key = Constants.warehouseMap.get(rcd.getWarehouseCode());
//            if(EmptyUtil.isEmpty(key)){
//                return;
//            }
//            if(EmptyUtil.isEmpty(rcd.getMemo()) || rcd.getMemo().contains("预售")){
//                List<CommSku> list = skuListMap.get(key);
//                if(NullUtil.isNull(list)){
//                    list = new ArrayList<>();
//                    skuListMap.put(key,list);
//                }
//                CommSku pcsSku = new CommSku();
//                pcsSku.setCode(rcd.getSkuCode());
//                pcsSku.setQuantity(rcd.getQuantity());
//                list.add(pcsSku);
//            }
//        }
//        if(!skuListMap.isEmpty()){
//            try{
//                PegasusUtilFacade pegasusUtilFacade = PegasusUtilFacade.getInstance();
//                for(Map.Entry<String,List<CommSku>> entry : skuListMap.entrySet()){
//                    String sessionKey = entry.getKey();
//                    for(CommSku sku : entry.getValue()){
//                        whCommandService.synCommodityStocks(pegasusUtilFacade.getSkuNumIid(sku.getCode(), sessionKey), sku.getCode(),
//                                sku.getQuantity(), sessionKey, 2, inventoryId + "");
//                    }
//                }
//            }catch (Exception e){
//                throw new WarehouseException(WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED,e.getMessage());
//            }
//        }
//    }




    private Map<String,Map<String,WhInvVO>> getLossOutCanUseWhSkuStockMap(
            WhInventoryVO inventory
            ,Map<Integer,List<WhWarehouse>> whMap){
        List<String> lossOutSkuCodes = buildLossOutSkuCodes(inventory.getInventoryDetailList());//盘亏skucodes
        List<String> lossOutWhCodes = buildLossOutWhCodes(inventory.getInventoryDetailList(),whMap);//可能盘亏逻辑仓
        return whInvService.findSkuStocks(lossOutSkuCodes,lossOutWhCodes);
    }

    private int getWhSkuQuantity(String skuCode,String warehouseCode,Map<String,Map<String,WhInvVO>> whSkuStockMap){
        Map<String,WhInvVO> skuStockMap = whSkuStockMap.get(warehouseCode);
        if(NullUtil.isNotNull(skuStockMap)){
            WhInvVO skuInv = skuStockMap.get(skuCode);
            if(NullUtil.isNotNull(skuInv)){
                return skuInv.getCanUseInv();
            }
        }
        return 0;
    }

    private WhInvVO findWhSkuInv(String skuCode,String warehouseCode,Map<String,Map<String,WhInvVO>> whSkuStockMap){
        Map<String,WhInvVO> skuStockMap = whSkuStockMap.get(warehouseCode);
        if(NullUtil.isNotNull(skuStockMap)){
            WhInvVO skuInv = skuStockMap.get(skuCode);
            return skuInv;
        }
        return null;
    }

    private WhInvRcdVO buildWhInvRcd(WhInventoryDetailVO inventoryDetail
            ,WhWarehouse warehouse
            ,Integer quantity
            ,Long operaterId){
        WhInvRcdVO invRcd = new WhInvRcdVO();
        invRcd.setWarehouseCode(warehouse.getCode());
        invRcd.setSkuCode(inventoryDetail.getSkuCode());
        invRcd.setQuantity(quantity);
        invRcd.setCommandCode(inventoryDetail.getInventoryId().toString());
        invRcd.setMemo(inventoryDetail.getMark());
        invRcd.setSubmitUserId(operaterId);
        invRcd.setTakeStockRcdId(inventoryDetail.getId().intValue());
        invRcd.setInOutType(quantity>0
                ?WhCommand.TYPE_INVENTROY_PROFIT_IN
                :WhCommand.TYPE_INVENTROY_LOSS_OUT);
        return invRcd;
    }

    private List<String> buildLossOutSkuCodes(List<WhInventoryDetailVO> inventoryDetailList){
        Set<String> skuCodes = new HashSet<>();
        for(WhInventoryDetailVO inventoryDetail : inventoryDetailList){
            if(!WhInventoryDetailVO.STATUS_DONE.equals(inventoryDetail.getStatus())
                    || inventoryDetail.getOriAmount().equals(inventoryDetail.getRealAmount())){
                continue;
            }
            int diffQuantity = inventoryDetail.getRealAmount() - inventoryDetail.getOriAmount();
            if(diffQuantity < 0){
                //盘亏
                skuCodes.add(inventoryDetail.getSkuCode());
            }
        }
        return Arrays.asList(skuCodes.toArray(new String[skuCodes.size()]));
    }

    private List<String> buildLossOutWhCodes(List<WhInventoryDetailVO> inventoryDetailList,Map<Integer,List<WhWarehouse>> whMap){
        Set<String> codes = new HashSet<>();
        for(WhInventoryDetailVO inventoryDetail : inventoryDetailList){
            if(!WhInventoryDetailVO.STATUS_DONE.equals(inventoryDetail.getStatus())
                    || inventoryDetail.getOriAmount().equals(inventoryDetail.getRealAmount())){
                continue;
            }
            int diffQuantity = inventoryDetail.getRealAmount() - inventoryDetail.getOriAmount();
            if(diffQuantity < 0){
                //盘亏
                List<WhWarehouse> whList = whMap.get(inventoryDetail.getSkuStatus());
                if(EmptyUtil.isNotEmpty(whList)){
                    for(WhWarehouse wh : whList){
                        codes.add(wh.getCode());
                    }
                }
            }
        }
        return Arrays.asList(codes.toArray(new String[codes.size()]));
    }

    private boolean updateInventoryDetailForAudit(
            Long inventoryDetailId
            ,Integer currentAuditStatus
            ,Integer nextAuditStatus
            ,String handlingSuggestion){
        WhInventoryDetail update = new WhInventoryDetail();
        update.setAuditStatus(nextAuditStatus);
        if(EmptyUtil.isNotEmpty(handlingSuggestion)){
            update.setHandlingSuggestion(handlingSuggestion);
        }
        WhInventoryDetailExample example = new WhInventoryDetailExample();
        example.createCriteria()
                .andIdEqualTo(inventoryDetailId)
                .andAuditStatusEqualTo(currentAuditStatus);
        return whInventoryDetailMapper.updateByExampleSelective(update,example) == 1;
    }

    private boolean updateInventoryDetailForFinanceAudit(
            Long inventoryDetailId
            ,Integer currentAuditStatus
            ,Integer nextAuditStatus
            ,String handlingSuggestion){
        WhInventoryDetail update = new WhInventoryDetail();
        update.setAuditStatus(nextAuditStatus);
        if(EmptyUtil.isNotEmpty(handlingSuggestion)){
            update.setFinanceHandlingSuggestion(handlingSuggestion);
        }
        WhInventoryDetailExample example = new WhInventoryDetailExample();
        example.createCriteria()
                .andIdEqualTo(inventoryDetailId)
                .andAuditStatusEqualTo(currentAuditStatus);
        return whInventoryDetailMapper.updateByExampleSelective(update,example) == 1;
    }

    private boolean isExistReject(List<WhInventoryDetailVO> inventoryDetailList){
        if(EmptyUtil.isNotEmpty(inventoryDetailList)){
            for(WhInventoryDetailVO detail : inventoryDetailList){
                if(WhInventoryDetailVO.AUDIT_STATUS_REJECT.equals(detail.getAuditStatus())){
                    return true;
                }
            }
        }
        return false;
    }

    private boolean isExistDiff(List<WhInventoryDetailVO> inventoryDetailList){
        if(EmptyUtil.isNotEmpty(inventoryDetailList)){
            for(WhInventoryDetailVO detail : inventoryDetailList){
                if(!detail.getOriAmount().equals(detail.getRealAmount())){
                    return true;
                }
            }
        }
        return false;
    }

    private boolean isExistNotDone(List<WhInventoryDetailVO> inventoryDetailList){
        if(EmptyUtil.isNotEmpty(inventoryDetailList)){
            for(WhInventoryDetailVO detail : inventoryDetailList){
                if(WhInventoryDetailVO.STATUS_NONE.equals(detail.getStatus())){
                    return true;
                }
            }
        }
        return false;
    }


    @Override
    public WhInventoryCountInfoVO countProcessingWhInventory(String physicalWarehouseCode) {
        WhInventoryCountInfoVO countInfo = new WhInventoryCountInfoVO();
        countInfo.setPhysicalWarehouseCode(physicalWarehouseCode);
        List<WhCountVO> countList = whInventoryDetailMapper.countProcessingInventoryDetail(physicalWarehouseCode);
        Map<Integer,Integer> countMap = new HashMap<>();
        if(EmptyUtil.isNotEmpty(countList)){
            for(WhCountVO count : countList){
                countMap.put(Integer.valueOf(count.getCode()),count.getAmount());
            }
        }
        List<Integer> skuStatusList = new ArrayList<>();
        skuStatusList.add(WhWarehouseVO.COMMODITY_STATUS_FOR_NONDEFECTIVE);
        skuStatusList.add(WhWarehouseVO.COMMODITY_STATUS_FOR_MILD_DAMAGED);
        skuStatusList.add(WhWarehouseVO.COMMODITY_STATUS_FOR_SAMPLE);
        skuStatusList.add(6);
        List<WhCountVO> resultCountList = new ArrayList<>();
        for(Integer skuStatus : skuStatusList){
            WhCountVO count = new WhCountVO();
            count.setCode(skuStatus.toString());
            count.setName(WhWarehouseVO.getSkuStatusName(skuStatus));
            count.setAmount(0);
            Integer amount = countMap.get(skuStatus);
            if(NullUtil.isNotNull(amount)){
                count.setAmount(amount);
            }
            resultCountList.add(count);
        }
        countInfo.setCountList(resultCountList);
        return countInfo;
    }

    @Override
    public boolean processingWhInventoryDetailInSingle(WhInventoryDetailVO inventoryDetail) {
        processingWhInventoryDetailCheck(inventoryDetail.getId());
        if(NullUtil.isNull(inventoryDetail.getRealAmount())){
            throw new WarehouseException(WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED
                    ,"应有数量不可为空！");
        }
        if(inventoryDetail.getRealAmount()< 0){
            throw new WarehouseException(WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED
                    ,"应有数量必须大于等于0！");
        }
        whInventoryDetailMapper.processingWhInventoryDetailInSingle(inventoryDetail);
        return true;
    }

    @Override
    public boolean processingWhInventoryDetailInContinue(WhInventoryDetailVO inventoryDetail) {
        processingWhInventoryDetailCheck(inventoryDetail.getId());
        whInventoryDetailMapper.processingWhInventoryDetailInContinue(inventoryDetail);
        return true;
    }

    private void processingWhInventoryDetailCheck(Long inventoryDetailId){
        WhInventoryDetailVO detail = findWhInventoryDetail(inventoryDetailId);
        if(NullUtil.isNull(detail)){
            throw new WarehouseException(WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED
                    ,"该盘点记录不存在！");
        }
        if(!WhInventoryVO.STATUS_PROCESSING.equals(detail.getInventoryStatus())){
            throw new WarehouseException(WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED
                    ,String.format("当前盘点任务状态[%s]",WhInventoryVO.getStatusName(detail.getInventoryStatus())));
        }
        if(WhInventoryDetailVO.AUDIT_STATUS_WAIT_AUDIT_FINANCE.equals(detail.getAuditStatus())
                || WhInventoryDetailVO.AUDIT_STATUS_FINISH.equals(detail.getAuditStatus())){
            throw new WarehouseException(WarehouseExceptionErrorCode.SPECIAL_ALERT_CODE
                    ,"该任务已提交,如需重新盘点,请点击重盘!");
        }
        if(!WhInventoryDetailVO.AUDIT_STATUS_INIT.equals(detail.getAuditStatus())
                && !WhInventoryDetailVO.AUDIT_STATUS_REJECT.equals(detail.getAuditStatus())){
            throw new WarehouseException(WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED
                    ,String.format("该任务当前状态[%s]！",detail.getAuditStatusName()));
        }
    }

    @Override
    public List<WhInventoryEmailVO> findAllWhInventoryEmail() {
        WhInventoryEmailExample example = new WhInventoryEmailExample();
        example.createCriteria();
        List<WhInventoryEmail> list = whInventoryEmailMapper.selectByExample(example);
        return BeanUtil.buildListFrom(list,WhInventoryEmailVO.class);
    }

    @Override
    public List<WhInventoryEmailVO> findWhInventoryEmail(String physicalWarehouseCode) {
        if(EmptyUtil.isEmpty(physicalWarehouseCode)){
            return Collections.emptyList();
        }
        WhInventoryEmailExample example = new WhInventoryEmailExample();
        example.createCriteria().andPhysicalWarehouseCodeEqualTo(physicalWarehouseCode);
        List<WhInventoryEmail> list = whInventoryEmailMapper.selectByExample(example);
        return BeanUtil.buildListFrom(list,WhInventoryEmailVO.class);
    }

    @Override
    @Transactional
    public boolean saveWhInventoryEmail(List<WhInventoryEmailVO> emailList) {
        //删除全部
        WhInventoryEmailExample example = new WhInventoryEmailExample();
        example.createCriteria();
        whInventoryEmailMapper.deleteByExample(example);
        //写入
        if(EmptyUtil.isNotEmpty(emailList)){
            whInventoryEmailMapper.batchInsert(emailList);
        }
        return true;
    }
}
