/**
 * Copyright (C), 上海布鲁爱电子商务有限公司
 */
package com.thebeastshop.pegasus.service.warehouse.service.impl;

import com.google.common.collect.Lists;
import com.thebeastshop.commdata.service.ChannelQueryService;
import com.thebeastshop.commdata.vo.ChannelWarehouseVO;
import com.thebeastshop.common.PageQueryResp;
import com.thebeastshop.common.ServiceResp;
import com.thebeastshop.common.PageCond;
import com.thebeastshop.pegasus.merchandise.service.McOpChannelService;
import com.thebeastshop.pegasus.merchandise.service.McPcsSkuService;
import com.thebeastshop.pegasus.merchandise.vo.OpChannelVO;
import com.thebeastshop.pegasus.merchandise.vo.PcsSkuDTO;
import com.thebeastshop.pegasus.merchandise.vo.PcsSkuVO;
import com.thebeastshop.pegasus.service.warehouse.cond.WhInvRcdCond;
import com.thebeastshop.pegasus.service.warehouse.cond.WhWarehouseCond;
import com.thebeastshop.pegasus.service.warehouse.dao.WhCommandMapper;
import com.thebeastshop.pegasus.service.warehouse.dao.WhInvRcdExtendMapper;
import com.thebeastshop.pegasus.service.warehouse.dao.WhInvRcdMapper;
import com.thebeastshop.pegasus.service.warehouse.dao.WhWarehouseMapper;
import com.thebeastshop.pegasus.service.warehouse.exception.WarehouseException;
import com.thebeastshop.pegasus.service.warehouse.model.*;
import com.thebeastshop.pegasus.service.warehouse.service.WhInfoService;
import com.thebeastshop.pegasus.service.warehouse.service.WhInvService;
import com.thebeastshop.pegasus.service.warehouse.service.WhJitPackageSkuReferenceService;
import com.thebeastshop.pegasus.service.warehouse.vo.*;
import com.thebeastshop.pegasus.util.comm.BeanUtil;
import com.thebeastshop.pegasus.util.comm.DateUtil;
import com.thebeastshop.pegasus.util.comm.EmptyUtil;
import com.thebeastshop.pegasus.util.comm.NullUtil;
import com.thebeastshop.stock.dto.SStockOccupyDTO;
import com.thebeastshop.stock.dto.SStockRecordConditionDTO;
import com.thebeastshop.stock.dto.SStockReleaseDTO;
import com.thebeastshop.stock.enums.SStockInOutTypeEnum;
import com.thebeastshop.stock.enums.SStockOccupyTypeEnum;
import com.thebeastshop.stock.enums.SStockOperationTypeEnum;
import com.thebeastshop.stock.service.SStockService;
import com.thebeastshop.stock.vo.SOccupyResultVO;
import com.thebeastshop.stock.vo.SReleaseResultVO;
import com.thebeastshop.stock.vo.SSkuStockQuantityVO;
import com.thebeastshop.stock.vo.SSkuStockVO;
import com.thebeastshop.stock.vo.SStockOccupyRecordVO;
import com.thebeastshop.stock.vo.SStockRecordVO;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;


/**
 * @author Royan
 * @version $Id: WhInvServiceImpl.java, v 0.1 2015-07-08 下午10:38
 */
@Service("whInvService")
public class WhInvServiceImpl implements WhInvService {

    private final Logger      log = LoggerFactory.getLogger(WhInvServiceImpl.class);

    @Autowired
    private WhCommandMapper   whCommandMapper;

    @Autowired
    private WhWarehouseMapper whWarehouseMapper;
    @Autowired
    private WhInvRcdExtendMapper whInvRcdExtendMapper;

    @Autowired
    private WhJitPackageSkuReferenceService whJitPackageSkuReferenceService;

    /*@Autowired
    private RedisLockHandler redisLockHandler;*/

    @Autowired
    private SStockService sStockService;

    @Autowired
    private McOpChannelService mcOpChannelService;

    @Autowired
    private McPcsSkuService mcPcsSkuService;

    @Autowired
    private WhInvRcdMapper whInvRcdMapper;
    
    @Autowired
    private ChannelQueryService channelQueryService;

    @Autowired
    private WhInfoService whInfoService;


    /**
     * 记录SKU库存记录
     *
     * @param whInvRcd SKU库存记录
     * @return 库存记录ID
     */
    @Override
    @Transactional
    public Boolean record(final WhInvRcd whInvRcd) {
        SStockRecordVO sStockRecordVO = new SStockRecordVO();
        BeanUtils.copyProperties(whInvRcd, sStockRecordVO);
        if(NullUtil.isNotNull(whInvRcd.getTakeStockRcdId())){
            sStockRecordVO.setTakeStockRcdId(whInvRcd.getTakeStockRcdId());
        }
        if(NullUtil.isNotNull(whInvRcd.getSubmitUserId())){
            sStockRecordVO.setSubmitUserId(whInvRcd.getSubmitUserId().intValue());
        }
        List<SStockRecordVO> rollbackRecordList = new ArrayList<SStockRecordVO>();
        ServiceResp<SStockRecordVO> serviceResp = sStockService.record(sStockRecordVO);
        if (!serviceResp.isSuccess()) {
            throw  new  WarehouseException(serviceResp.getRespCode(), serviceResp.getRespMsg());
        }else{
            rollbackRecordList.add(serviceResp.getBean());
            whInvRcd.setRollbackRecordList(rollbackRecordList);
        }
        return serviceResp.isSuccess();
    }

    @Override
    public Boolean batchRecord(List<WhInvRcd> whInvRcdList) {
        List<SStockRecordVO> sStockRecordVOArrayList = Lists.newArrayList();
        for (WhInvRcd whInvRcd : whInvRcdList) {
            SStockRecordVO sStockRecordVO = new SStockRecordVO();
            BeanUtils.copyProperties(whInvRcd, sStockRecordVO);
            if(NullUtil.isNotNull(whInvRcd.getTakeStockRcdId())){
                sStockRecordVO.setTakeStockRcdId(whInvRcd.getTakeStockRcdId());
            }
            if(NullUtil.isNotNull(whInvRcd.getSubmitUserId())){
                sStockRecordVO.setSubmitUserId(whInvRcd.getSubmitUserId().intValue());
            }
            sStockRecordVOArrayList.add(sStockRecordVO);
        }

        List<SStockRecordVO> rollbackRecordList = new ArrayList<SStockRecordVO>();
        ServiceResp<List<SStockRecordVO>> serviceResp = sStockService.batchRecord(sStockRecordVOArrayList);
        if (serviceResp.isSuccess()) {
            rollbackRecordList.addAll(serviceResp.getBean());
        } else {
            throw new  WarehouseException(serviceResp.getRespCode(), serviceResp.getRespMsg());
        }
        return serviceResp.isSuccess();
    }

    /**
     * 记录SKU库存占用
     *
     * @param whInvOccupy SKU库存占用
     * @return 库存占用ID
     */
    @Override
    @Transactional
    public Long occupy(final WhInvOccupy whInvOccupy) {
        ServiceResp<SOccupyResultVO> resp = sStockService.occupy(convertWhInvOccupy2DTO(whInvOccupy));
        if (!resp.isSuccess()) {
            throw  new  WarehouseException(resp.getRespCode(), resp.getRespMsg());
        }
        SOccupyResultVO resultVO = resp.getBean();
        return    resultVO.getId();
    }

    @Override
    @Transactional
    public boolean batchRecords(List<WhInvRcd> whInvRcdList) {
        if(EmptyUtil.isEmpty(whInvRcdList)){
            return true;
        }
        List<SStockRecordVO> rollbackRecordList = new ArrayList<>();
        List<SStockRecordVO> stockRcdList = new ArrayList<>();
        for(WhInvRcd invRcd : whInvRcdList){
            stockRcdList.add(convertWhInv2DTO(invRcd));
        }
        ServiceResp<List<SStockRecordVO>> serviceResp = sStockService.batchRecord(stockRcdList);
        if (serviceResp.isSuccess()) {
            rollbackRecordList.addAll(serviceResp.getBean());
        } else {
            throw new  WarehouseException(serviceResp.getRespCode(), serviceResp.getRespMsg());
        }
        return true;
    }


    private SStockRecordVO convertWhInv2DTO(WhInvRcd whInvRcd){
        SStockRecordVO sStockRecordVO = new SStockRecordVO();
        BeanUtils.copyProperties(whInvRcd, sStockRecordVO);
        if(NullUtil.isNotNull(whInvRcd.getTakeStockRcdId())){
            sStockRecordVO.setTakeStockRcdId(whInvRcd.getTakeStockRcdId());
        }
        if(NullUtil.isNotNull(whInvRcd.getSubmitUserId())){
            sStockRecordVO.setSubmitUserId(whInvRcd.getSubmitUserId().intValue());
        }
        return sStockRecordVO;
    }

    /**
     * 记录SKU库存占用
     *
     * @param whInvOccupyList SKU库存占用列表
     * @return 库存占用ID
     */
    @Override
    @Transactional
    public List<Long> occupy(final List<WhInvOccupy> whInvOccupyList) {
        final List<Long> result = new ArrayList<>();
        ServiceResp<List<SOccupyResultVO>> serviceResp = sStockService.occupy(convertWhInvOccupy2DTO(whInvOccupyList));
        if (!serviceResp.isSuccess()) {
            throw  new  WarehouseException(serviceResp.getRespCode(), serviceResp.getRespMsg());
        }
        List<SOccupyResultVO> occupyResultVOs = serviceResp.getBean();
        if (EmptyUtil.isNotEmpty(occupyResultVOs)) {
            for (SOccupyResultVO occupyResultVO : occupyResultVOs) {
                result.add(occupyResultVO.getId());
            }
        }
        return result;
    }

    @Override
    public List<Long> occupyAndNoNeedCheckStock(List<WhInvOccupy> whInvOccupyList, boolean noNeedCheckStock) {
        final List<Long> result = new ArrayList<>();
        List<SStockOccupyDTO> stockOccupyDTOs = convertWhInvOccupy2DTO(whInvOccupyList);
        for (SStockOccupyDTO occupyDTO : stockOccupyDTOs){
            // needToCheckStock为 false 则不校验库存
            occupyDTO.setNeedToCheckStock(!noNeedCheckStock);
        }
        ServiceResp<List<SOccupyResultVO>> serviceResp = sStockService.occupy(stockOccupyDTOs);
        if (!serviceResp.isSuccess()) {
            throw  new  WarehouseException(serviceResp.getRespCode(), serviceResp.getRespMsg());
        }
        List<SOccupyResultVO> occupyResultVOs = serviceResp.getBean();
        if (EmptyUtil.isNotEmpty(occupyResultVOs)) {
            for (SOccupyResultVO occupyResultVO : occupyResultVOs) {
                result.add(occupyResultVO.getId());
            }
        }
        return result;
    }

    @Override
    @Transactional
    public List<Long> occupy(List<WhInvOccupy> whInvOccupyList, List<WhJitPackageSkuReferenceVO> whJitPackageSkuReferenceVOList) {
        createWhJitPackageSkuReference(whJitPackageSkuReferenceVOList);
        return occupy(whInvOccupyList);
    }

    @Override
    @Transactional
    public List<Long> occupy(List<WhInvOccupy> whInvOccupyList, List<WhJitPackageSkuReferenceVO> whJitPackageSkuReferenceVOList, String allotCode) {
        whJitPackageSkuReferenceService.finishByReferenceCode(allotCode);
        return occupy(whInvOccupyList,whJitPackageSkuReferenceVOList);
    }

    private void createWhJitPackageSkuReference(List<WhJitPackageSkuReferenceVO> whJitPackageSkuReferenceVOList){
        if(EmptyUtil.isNotEmpty(whJitPackageSkuReferenceVOList)){
            whJitPackageSkuReferenceService.createWhJitPackageSkuReference(whJitPackageSkuReferenceVOList);
        }
    }

    /**
     * 分布式锁的机制
     * @return
     */
    private static List<String> lockSkuCodes(){
        List<String> skuCodes = new ArrayList<>();
        skuCodes.add("781161392");
        skuCodes.add("605161182");
        skuCodes.add("692160022");
        return skuCodes;
    }


    @Override
    @Transactional
    public Map<String, List<Long>> interestOccupy(Map<String, List<WhInvOccupy>> params) {
        Map<String, List<Long>> result = new HashMap<>();
        Set<String> interestIds = params.keySet();
        if(CollectionUtils.isNotEmpty(interestIds)) {
            List<Long> occupyIds;
            for (String interestId : interestIds) {
                List<WhInvOccupy> whInvOccupies = params.get(interestId);
                if(CollectionUtils.isNotEmpty(whInvOccupies)) {
                    for (WhInvOccupy whInvOccupy : whInvOccupies) {
                        String skuCode = whInvOccupy.getSkuCode();
                        Integer quantity = whInvOccupy.getQuantity();
                        Boolean releaseResult = releaseInterestOccupy(interestId, skuCode, quantity);
                        if(!releaseResult) {
                            throw new WarehouseException("库存不足");
                        }
                    }
                    occupyIds = occupy(whInvOccupies);
                    result.put(interestId, occupyIds);
                }
            }
        }
        return result;
    }

    @Override
    @Transactional
    public Map<String, List<Long>> campaignOccupy(Map<String, List<WhInvOccupy>> params) {
        Map<String, List<Long>> result = new HashMap<>();
        Set<String> campaignCodes = params.keySet();
        if(CollectionUtils.isNotEmpty(campaignCodes)) {
            List<Long> occupyIds;
            for (String campaignCode : campaignCodes) {
                List<WhInvOccupy> whInvOccupies = params.get(campaignCode);
                if(CollectionUtils.isNotEmpty(whInvOccupies)) {
                    for (WhInvOccupy whInvOccupy : whInvOccupies) {
                        String skuCode = whInvOccupy.getSkuCode();
                        Integer quantity = whInvOccupy.getQuantity();
                        Boolean releaseResult = releaseCampaignOccupy(campaignCode, skuCode, quantity);
                        if(!releaseResult) {
                            throw new WarehouseException("库存不足");
                        }
                    }
                    occupyIds = occupy(whInvOccupies);
                    result.put(campaignCode, occupyIds);
                }
            }
        }
        return result;
    }

    /**
     * 释放SKU库存后占用
     *
     * @param whInvOccupyList
     * @param whReleaseOccupationVOList
     * @return 库存占用ID
     */
    @Override
    @Transactional
    public List<Long> occupyAfterRelease(final List<WhInvOccupy> whInvOccupyList,
                                         final List<WhReleaseOccupationVO> whReleaseOccupationVOList) {
        ServiceResp<List<SReleaseResultVO>> serviceResp = sStockService.release(convertWhRelease2DTO(whReleaseOccupationVOList));
        if (!serviceResp.isSuccess()) {
            throw new WarehouseException(serviceResp.getRespCode(), serviceResp.getRespMsg());
        }
        List<Long> result;
        try {
            result = occupy(whInvOccupyList);
        }catch (Exception e) {
            log.info("调用释放库存回滚接口");
            sStockService.rollbackRelease(serviceResp);
            throw  e;
        }
        return result;
    }


    @Override
    @Transactional
    public List<Long> occupyAfterRelease(List<WhInvOccupy> whInvOccupyList, List<WhReleaseOccupationVO> whReleaseOccupationVOList,
                                         String referenceCode,
                                         List<WhJitPackageSkuReferenceVO> whJitPackageSkuReferenceVOList) {
        whJitPackageSkuReferenceService.deleteByReferenceCode(referenceCode);
        createWhJitPackageSkuReference(whJitPackageSkuReferenceVOList);
        return occupyAfterRelease(whInvOccupyList,whReleaseOccupationVOList);
    }

    @Override
    @Transactional
    public void releaseJitPackageOccupy(String packageCode) {
        SStockReleaseDTO stockReleaseDTO = new SStockReleaseDTO();
        stockReleaseDTO.setPackageCode(packageCode);
        stockReleaseDTO.setOperationType(SStockOperationTypeEnum.JIT);
        ServiceResp<SReleaseResultVO> serviceResp = sStockService.release(stockReleaseDTO);
        if (serviceResp.isFailure()) {
            throw  new  WarehouseException(serviceResp.getRespCode(), serviceResp.getRespMsg());
        }
    }

    /**
     * SKU库存占用释放
     *
     * @param id SKU库存占用Id
     */
    @Override
    @Transactional
    public void releaseOccupationById(final Long id) {
        ServiceResp<SReleaseResultVO> serviceResp = sStockService.releaseById(id);
        if (!serviceResp.isSuccess()) {
            throw new WarehouseException(serviceResp.getRespCode(), serviceResp.getRespMsg());
        }
    }

    /**
     * 根据占用类型和相关单据号释放SKU库存占用
     *
     * @param whReleaseOccupationVO          释放占用
     */
    @Override
    @Transactional
    public void releaseOccupation(final WhReleaseOccupationVO whReleaseOccupationVO) {
        ServiceResp<SReleaseResultVO> serviceResp = sStockService.release(convertWhRelease2DTO(whReleaseOccupationVO));
        if (!serviceResp.isSuccess()) {
            throw new WarehouseException(serviceResp.getRespCode(), serviceResp.getRespMsg());
        }
    }

    /**
     * 根据占用类型和相关单据号释放SKU库存占用
     *
     * @param whReleaseOccupationVOList 释放占用
     */
    @Override
    @Transactional
    public void releaseOccupation(final List<WhReleaseOccupationVO> whReleaseOccupationVOList) {
        ServiceResp<List<SReleaseResultVO>> serviceResp = sStockService.release(convertWhRelease2DTO(whReleaseOccupationVOList));
        if (!serviceResp.isSuccess()) {
            throw new WarehouseException(serviceResp.getRespCode(), serviceResp.getRespMsg());
        }
    }

    @Override
    @Transactional
    public void releaseOccupation(List<WhReleaseOccupationVO> whReleaseOccupationVOList, List<WhJitPackageSkuReferenceVO> whJitPackageSkuReferenceVOList) {
        if(EmptyUtil.isNotEmpty(whJitPackageSkuReferenceVOList)){
            Set<String> referenceCodes = new HashSet<>();
            for(WhJitPackageSkuReferenceVO referenceVO : whJitPackageSkuReferenceVOList){
                referenceCodes.add(referenceVO.getReferenceCode());
            }
            for(String referenceCode : referenceCodes){
                whJitPackageSkuReferenceService.finishByReferenceCode(referenceCode);
            }
        }
        releaseOccupation(whReleaseOccupationVOList);
    }

    @Override
    public List<WhInvOccupy> getWhInvOccupy(List<String> referenceCodes) {
        return BeanUtil.buildListFrom(sStockService.getOccupyRecordsByReferenceCodes(referenceCodes), WhInvOccupy.class);
    }

    /**
     * 根据SKU编码查找库存数量
     *
     * @param skuCode SKU编码
     * @return Map<仓库编码,库存数量>
     */
    @Override
    public List<WhWarehouseInvQttVO> findInvQttBySkuCode(final String skuCode) {
        Map<String, SSkuStockVO> whStockMap = sStockService.getSkuStocksInAllWarehouses(skuCode);
        List<WhWarehouseInvQttVO> whWarehouseInvQttVOs = new ArrayList<WhWarehouseInvQttVO>();
        for (Map.Entry<String, SSkuStockVO> entry : whStockMap.entrySet()) {
            WhWarehouseInvQttVO whWarehouseInvQttVO = new  WhWarehouseInvQttVO();
            BeanUtils.copyProperties(entry.getValue(), whWarehouseInvQttVO);
            whWarehouseInvQttVOs.add(whWarehouseInvQttVO);
        }
        return  whWarehouseInvQttVOs;
    }

    /**
     * 根据SKU编码查找待入库数量
     *
     * @param skuCode SKU编码
     * @return Map<仓库编码,库存数量>
     */
    @Override
    public List<WhWarehouseInvQttVO> findWillInQttBySkuCode(final String skuCode) {
        return whCommandMapper.findWillInQttBySkuCode(skuCode);
    }
    
    /**
     * 根据SKU编码查找待入库数量
     *
     * @param skuCode SKU编码
     * @return Map<仓库编码,库存数量>
     */
    @Override
    public List<WhWarehouseInvQttVO> findNewWillInQttBySkuCode(final String skuCode) {
        return whCommandMapper.findNewWillInQttBySkuCode(skuCode);
    }

    /**
     * 根据SKU编码和仓库编码查找库存数量
     *
     * @param skuCode       SKU编码
     * @param warehouseCode 仓库编码
     * @return 库存数量
     */
    @Override
    public int findInvQttBySkuCodeAndWarehouseCode(final String skuCode, final String warehouseCode) {
        List<String> skuCodeList = new ArrayList<>();
        skuCodeList.add(skuCode);
        Map<String, SSkuStockQuantityVO> map = sStockService.getSkuRealStocks(skuCodeList, warehouseCode);
        if (EmptyUtil.isEmpty(map)) {
            return  0;
        }
        return map.get(skuCode).getQuantity();
    }

    /**
     * 根据SKU编码查找所有仓库的可用库存数量
     * @param skuCode       SKU编码
     * @return 可用库存数量
     */
    @Override
    public int findQttBySkuCode(final String skuCode) {
        final Map<String, String> params = new HashMap<>();
        params.put("skuCode", skuCode);
        return whInvRcdMapper.findQttBySkuCode(params);
    }

    /**
     * 根据SKU编码列表和仓库编码查找库存数量
     *
     * @param skuCodes      SKU编码列表
     * @param warehouseCode 仓库编码
     * @return Map<SKU编码,库存数量>
     */
    @Override
    public List<WhSkuInvQttVO> findInvQttBySkuCodesAndWarehouseCode(final List<String> skuCodes,
                                                                    final String warehouseCode) {
        Map<String, SSkuStockQuantityVO> map =  sStockService.getSkuRealStocks(skuCodes, warehouseCode);
        List<WhSkuInvQttVO> resultList = new ArrayList<>();
        for (Map.Entry<String, SSkuStockQuantityVO> entry : map.entrySet()) {
            resultList.add(BeanUtil.buildFrom(entry.getValue(), WhSkuInvQttVO.class));
        }
        return resultList;
    }


    /**
     * 根据仓库编码查找SKU库存数量
     *
     * @param warehouseCode 仓库编码
     * @return Map<SKU编码,库存数量>
     */
    @Override
    public List<WhSkuInvQttVO> findSkuInvQttByWarehouseCode(final String warehouseCode) {
        List<SSkuStockVO>  skuStockVOList = getSkuStockInAllWarehouses(warehouseCode);
        Map<String, PcsSkuDTO> skuNameMap = getSkuNameMap(skuStockVOList);
        List<WhSkuInvQttVO> whSkuInvQttVOList = new ArrayList<>();
        if (skuStockVOList != null) {
            WhWarehouse whWarehouse = whWarehouseMapper.findWarehouseByCode(warehouseCode);
            for (SSkuStockVO sSkuStockVO : skuStockVOList) {
                WhSkuInvQttVO whSkuInvQttVO = new WhSkuInvQttVO();
                whSkuInvQttVO.setSkuCode(sSkuStockVO.getSkuCode());
                whSkuInvQttVO.setWarehouseName(whWarehouse != null ? whWarehouse.getName() : "");
                whSkuInvQttVO.setQuantity(sSkuStockVO.getRealQuantity());
                PcsSkuDTO pcsSkuDTO = skuNameMap.get(sSkuStockVO.getSkuCode());
                if (pcsSkuDTO == null) {
                    System.out.print(11);

                }
                whSkuInvQttVO.setNameCn(pcsSkuDTO == null ? "" : pcsSkuDTO.getNameCn());
                whSkuInvQttVOList.add(whSkuInvQttVO);
            }
        }
        return whSkuInvQttVOList;
    }

    /**
     * 根据物理仓库编码查找SKU库存数量
     *
     * @param physicalWarehouseCode 物理仓库编码
     * @return Map<SKU编码,库存数量>
     */
    @Override
    public List<WhSkuInvQttVO> findSkuInvQttByPhysicalWarehouseCode(final String physicalWarehouseCode) {
        List<WhWarehouse> whWarehouseList = whWarehouseMapper.findWarehouseByPhysicalCodeAndStatus(physicalWarehouseCode, null,null);
        List<WhSkuInvQttVO> whSkuInvQttVOList = new ArrayList<>();
        if (EmptyUtil.isNotEmpty(whWarehouseList)) {
            for (WhWarehouse whWarehouse : whWarehouseList) {
                whSkuInvQttVOList.addAll(this.findSkuInvQttByWarehouseCode(whWarehouse.getCode()));
            }
        }
        return whSkuInvQttVOList;
    }

    /**
     * 根据仓库编码列表查找SKU库存数量
     *
     * @param warehouseCodeList 仓库编码列表
     * @return Map<SKU编码,库存数量>
     */
    @Override
    public List<WhSkuInvQttVO> findSkuInvQttByWarehouseCodes(final List<String> warehouseCodeList) {
        List<WhSkuInvQttVO> whSkuInvQttVOList = new ArrayList<>();
        if (EmptyUtil.isNotEmpty(warehouseCodeList)) {
            for (String warehouseCode : warehouseCodeList) {
                whSkuInvQttVOList.addAll(this.findSkuInvQttByWarehouseCode(warehouseCode));
            }
        }
        return whSkuInvQttVOList;
    }

    /**
     * 根据SKU编码查找占用数量
     *
     * @param warehouseCode 仓库编码
     * @return Map<SKU编码,库存数量>
     */
    @Override
    public List<WhSkuInvQttVO> findOccupyQttByWarehouseCode(final String warehouseCode) {
        List<SSkuStockVO>  skuStockVOList = getSkuStockInAllWarehouses(warehouseCode);
        List<WhSkuInvQttVO> whSkuInvQttVOList = new ArrayList<>();
        if (skuStockVOList != null) {
          WhWarehouse whWarehouse = whWarehouseMapper.findWarehouseByCode(warehouseCode);
          for (SSkuStockVO sSkuStockVO : skuStockVOList) {
              WhSkuInvQttVO whSkuInvQttVO = new WhSkuInvQttVO();
              whSkuInvQttVO.setSkuCode(sSkuStockVO.getSkuCode());
              whSkuInvQttVO.setWarehouseName(whWarehouse != null ? whWarehouse.getName() : "");
              whSkuInvQttVO.setQuantity(sSkuStockVO.getOccupyQuantity());
              whSkuInvQttVOList.add(whSkuInvQttVO);
          }
        }
        return whSkuInvQttVOList;
    }

    /**
     * 根据SKU编码查找待入库数量
     *
     * @param warehouseCode 仓库编码
     * @return Map<SKU编码,库存数量>
     */
    @Override
    public List<WhSkuInvQttVO> findWillInQttByWarehouseCode(final String warehouseCode) {
        return whCommandMapper.findWillInQttByWarehouseCode(warehouseCode);
    }

    /**
     * 根据SKU编码查找占用数量
     *
     * @param skuCode SKU编码
     * @return Map<仓库编码,库存数量>
     */
    @Override
    public List<WhWarehouseInvQttVO> findOccupyQttBySkuCode(final String skuCode) {
        Map<String, SSkuStockQuantityVO> map = sStockService.getSkuOccupyStocksInAllWarehouses(skuCode);
        List<WhWarehouseInvQttVO> resultList = new ArrayList<WhWarehouseInvQttVO>();
        if (EmptyUtil.isEmpty(map)) {
           return  resultList;
        }
        for (Map.Entry<String, SSkuStockQuantityVO> entry : map.entrySet()) {
            resultList.add(BeanUtil.buildFrom(entry.getValue(), WhWarehouseInvQttVO.class));
        }
        return resultList;
    }

    /**
     * 根据SKU编码和仓库编码查找占用数量
     *
     * @param skuCode       SKU编码
     * @param warehouseCode 仓库编码
     * @return 库存数量
     */
    @Override
    public int findOccupyQttBySkuCodeAndWarehouseCode(final String skuCode, final String warehouseCode) {
        List<String> skuCodeList = new ArrayList<String>();
        skuCodeList.add(skuCode);
        Map<String, SSkuStockQuantityVO> map = sStockService.getSkuOccupyStocks(skuCodeList, warehouseCode);
        if (EmptyUtil.isEmpty(map)) {
            return  0;
        }
        SSkuStockQuantityVO skuStockQuantityVO = map.get(skuCode);
        return skuStockQuantityVO == null ? 0 : skuStockQuantityVO.getQuantity();
    }

    /**
     * 根据SKU编码列表和仓库编码查找占用数量
     *
     * @param skuCodes      SKU编码列表
     * @param warehouseCode 仓库编码
     * @return Map<SKU编码,库存数量>
     */
    @Override
    public List<WhSkuInvQttVO> findOccupyQttBySkuCodesAndWarehouseCode(final List<String> skuCodes,
                                                                       final String warehouseCode) {
        if (EmptyUtil.isEmpty(skuCodes)) {
            return new ArrayList<>();
        }
        List<WhSkuInvQttVO> resultList = new ArrayList<WhSkuInvQttVO>();
        Map<String, SSkuStockQuantityVO> map = sStockService.getSkuOccupyStocks(skuCodes, warehouseCode);
        if (EmptyUtil.isEmpty(map)) {
            return  resultList;
        }
        for (Map.Entry<String, SSkuStockQuantityVO> entry : map.entrySet()) {
            resultList.add(BeanUtil.buildFrom(entry.getValue(), WhSkuInvQttVO.class));
        }
        return resultList;
    }

    /**
     * 根据SKU编码查找可用库存数量
     *
     * @param skuCode SKU编码
     * @return Map<仓库编码,库存数量>
     */
    @Override
    public Map<String, WhInvVO> findCanUseQttBySkuCode(final String skuCode) {
        Map<String, WhInvVO> resultMap = new HashMap<String, WhInvVO>();
        Map<String, SSkuStockVO> map = sStockService.getSkuStocksInAllWarehouses(skuCode);
        final List<WhSkuInvQttVO> quantityAllotWillIn = whCommandMapper.findAllotWillInQttBySkuCode(skuCode);
        if (EmptyUtil.isNotEmpty(map)) {
            for (Map.Entry<String, SSkuStockVO> entry : map.entrySet()) {
                resultMap.put(entry.getKey(), convertSkuStockVO2WhInvVO(entry.getValue()));
            }
        }
        final List<WhWarehouseInvQttVO> quantityWillIn = findNewWillInQttBySkuCode(skuCode);
        // willIn
        for (final WhWarehouseInvQttVO e : quantityWillIn) {
            final String warehouseCode = e.getWarehouseCode();
            final Integer quantity = e.getQuantity();

            WhInvVO whInvVO = resultMap.get(warehouseCode);
            if (whInvVO == null) {
                whInvVO = new WhInvVO();
                whInvVO.setSkuCode(skuCode);
                whInvVO.setWarehouseCode(warehouseCode);
                whInvVO.setQuantityInRcd(0);
                whInvVO.setQuantityInOccupy(0);
                whInvVO.setQuantityAllotWillIn(0);
                resultMap.put(warehouseCode, whInvVO);
            }
            whInvVO.setQuantityWillIn(quantity == null ? 0 : quantity);
        }

        for (final WhSkuInvQttVO e : quantityAllotWillIn) {
            WhInvVO whInvVO = resultMap.get(e.getWarehouseCode());
            if (whInvVO == null) {
                whInvVO = new WhInvVO();
                whInvVO.setSkuCode(skuCode);
                whInvVO.setWarehouseCode(e.getWarehouseCode());
                whInvVO.setQuantityInRcd(0);
                whInvVO.setQuantityInOccupy(0);
                whInvVO.setQuantityAllotWillIn(0);
                resultMap.put(e.getWarehouseCode(), whInvVO);
            }
            whInvVO.setQuantityAllotWillIn(e.getQuantity());
        }
        return resultMap;
    }

    @Override
    public List<WhWarehouseSkuInvVO> findWarehousesSkuInvByCond(List<String> warehouseCodeList, String skuCode) {
        Map<String, WhWarehouse> warehouseMap = getWarehouseMap(warehouseCodeList);
        Map<String, WhInvVO> skuInvMap = findCanUseQttBySkuCode(skuCode);
        List<WhWarehouseSkuInvVO> whWarehouseSkuInvVOList = new ArrayList<>();
        for(Map.Entry<String, WhInvVO> whSkuInvEntry : skuInvMap.entrySet()){
            String warehouseCode = whSkuInvEntry.getKey();
            if(warehouseCodeList.contains(warehouseCode)){
                WhWarehouseSkuInvVO whWarehouseSkuInvVO = new WhWarehouseSkuInvVO();
                BeanUtils.copyProperties(whSkuInvEntry.getValue(), whWarehouseSkuInvVO);
                whWarehouseSkuInvVO.setWarehouseCode(warehouseCode);
                whWarehouseSkuInvVOList.add(whWarehouseSkuInvVO);
                /*if(NullUtil.isNotNull(pcsSkuVO.getNameCn())){
                    whWarehouseSkuInvVO.setSkuName(pcsSkuVO.getNameCn());
                }*/
                WhWarehouse warehouse = warehouseMap.get(warehouseCode);
                if(NullUtil.isNotNull(warehouse)){
                    whWarehouseSkuInvVO.setWarehouseName(warehouse.getName());
                }
            }
        }
        return whWarehouseSkuInvVOList;
    }

    @Override
    public Map<String, Map<String, WhInvVO>> findSkuStocks(List<String> skuCodes, List<String> warhouseCodes) {
        Map<String, Map<String, WhInvVO>> map = new HashMap<>();
        if(EmptyUtil.isEmpty(warhouseCodes)){
            return map;
        }
        Map<String, Map<String, SSkuStockVO>> whStockMap = sStockService.getSkuStocks(skuCodes,warhouseCodes);
        if(NullUtil.isNull(whStockMap) || whStockMap.isEmpty()){
            return map;
        }
        for(Map.Entry<String, Map<String, SSkuStockVO>> whEntry : whStockMap.entrySet()){
            Map<String,WhInvVO> whSkuStockMap = new HashMap<>();
            map.put(whEntry.getKey(),whSkuStockMap);
            for(Map.Entry<String, SSkuStockVO> stockEntry : whEntry.getValue().entrySet()){
                whSkuStockMap.put(stockEntry.getKey(),convertSkuStockVO2WhInvVO(stockEntry.getValue()));
            }
        }
        return map;
    }

    // 查找仓库
    @Override
    public Map<String, WhWarehouse> getWarehouseMap(List<String> warehouseCodeList){
        Map<String, WhWarehouse> warehouseMap = new HashMap<>();
        for (WhWarehouse warehouse : whInfoService.findWarehouseByCodes(warehouseCodeList)) {
            warehouseMap.put(warehouse.getCode(), warehouse);
        }
        return warehouseMap;
    }

    /**
     * 根据仓库编码查找可用库存数量
     *
     * @param warehouseCode 仓库编码
     * @return Map<SKU编码,库存数量>
     */
    @Override
    public Map<String, WhInvVO> findCanUseQttByWarehouseCode(final String warehouseCode) {
        final List<WhSkuInvQttVO> quantityWillIn = findWillInQttByWarehouseCode(warehouseCode);
        final List<WhSkuInvQttVO> quantityAllotWillIn = findAllotWillInQttByWarehouseCode(warehouseCode);
        List<SSkuStockVO>  skuStockVOList = getSkuStockInAllWarehouses(warehouseCode);
        List<WhInvVO> whInvVOList = new ArrayList<>();
        if (skuStockVOList != null) {
            whInvVOList = convertSkuStockVO2WhInvVO(skuStockVOList);
        }

        final Map<String, WhInvVO> result = new HashMap<>();
        WhWarehouse whWarehouse = whWarehouseMapper.findWarehouseByCode(warehouseCode);
        String warehouseName = whWarehouse != null ? whWarehouse.getName() : "";


        Set<String>  skuCodes = new HashSet<>();
        if (EmptyUtil.isNotEmpty(skuStockVOList)) {
            for (SSkuStockVO sSkuStockVO : skuStockVOList) {
                skuCodes.add(sSkuStockVO.getSkuCode());
            }
        }

        for (final WhSkuInvQttVO e : quantityWillIn) {
            skuCodes.add(e.getSkuCode());
        }

        Map<String, PcsSkuDTO> skuNameMap = getSkuNameMapBySkuCodes(skuCodes);
        PcsSkuDTO pcsSkuDTO;
        // record
        for (final WhInvVO e : whInvVOList) {
            e.setWarehouseCode(warehouseCode);
            e.setWarehouseName(warehouseName);
            if (EmptyUtil.isNotEmpty(skuNameMap) && EmptyUtil.isNotEmpty(e)) {
                pcsSkuDTO = skuNameMap.get(e.getSkuCode());
                if (EmptyUtil.isNotEmpty(pcsSkuDTO)) {
                    e.setSkuNameCn(pcsSkuDTO.getNameCn());
                    e.setSkuTypeStr(pcsSkuDTO.getSkuTypeStr());
                    e.setNameCn(pcsSkuDTO.getNameCn());
                }
            }
            result.put(e.getSkuCode(), e);
        }

        // willIn
        for (final WhSkuInvQttVO e : quantityWillIn) {
            final String skuCode = e.getSkuCode();
            final Integer quantity = e.getQuantity();
            pcsSkuDTO = skuNameMap.get(skuCode);
            WhInvVO whInvVO = result.get(skuCode);
            if (whInvVO == null) {
                whInvVO = new WhInvVO();
                whInvVO.setSkuCode(skuCode);
                whInvVO.setWarehouseCode(warehouseCode);
                whInvVO.setWarehouseName(warehouseName);
                whInvVO.setQuantityInRcd(0);
                whInvVO.setQuantityInOccupy(0);
                whInvVO.setSkuName(pcsSkuDTO.getNameCn());
                whInvVO.setSkuTypeStr(pcsSkuDTO.getSkuTypeStr());
                whInvVO.setSkuNameCn(e.getNameCn());
                whInvVO.setNameCn(e.getNameCn());
                result.put(skuCode, whInvVO);
            }
            whInvVO.setQuantityWillIn(quantity);
        }

        for (final WhSkuInvQttVO e : quantityAllotWillIn) {
            WhInvVO whInvVO = result.get(e.getSkuCode());
            whInvVO.setQuantityAllotWillIn(e.getQuantity());
        }

        return result;
    }

    /**
     * 根据SKU编码和仓库编码查找可用库存数量
     *
     * @param skuCode       SKU编码
     * @param warehouseCode 仓库编码
     * @return 库存数量
     */
    @Override
    public WhInvVO findCanUseQttBySkuCodeAndWarehouseCode(final String skuCode, final String warehouseCode) {
        SSkuStockVO sSkuStockVO = sStockService.getSkuStock(skuCode, warehouseCode);
        sSkuStockVO.setWarehouseCode(warehouseCode);
        return convertSkuStockVO2WhInvVO(sSkuStockVO);
    }

    /**
     * 根据SKU编码列表和仓库编码查找可用库存数量
     *
     * @param skuCodes      SKU编码列表
     * @param warehouseCode 仓库编码
     * @return Map<SKU编码,库存数量>
     */
    @Override
    public Map<String, WhInvVO> findCanUseQttBySkuCodesAndWarehouseCode(final List<String> skuCodes,
                                                                        final String warehouseCode) {
        final Map<String, WhInvVO> result = new HashMap<>();
        Map<String, SSkuStockVO> map = sStockService.getSkuStocks(skuCodes, warehouseCode);
        if (EmptyUtil.isNotEmpty(map)) {
            for (Map.Entry<String, SSkuStockVO> entry : map.entrySet()) {
                result.put(entry.getKey(), convertSkuStockVO2WhInvVO(entry.getValue()));
            }
        }
        return result;
    }

    /**
     * 计算可用库存报表
     */
    @Override
    public List<WhInvVO> calCanUseQtt() {
        List<WhWarehouse> whWarehouseList = whWarehouseMapper.findWarehouseByCond(new WhWarehouseCond());
        List<WhInvVO> resultList = new ArrayList<>();
        if (resultList != null) {
            for (WhWarehouse whWarehouse : whWarehouseList) {
                Map<String, WhInvVO> map = findCanUseQttByWarehouseCode(whWarehouse.getCode());
                if (map != null) {
                    for (Map.Entry<String, WhInvVO> entry : map.entrySet()) {
                        resultList.add(entry.getValue());
                    }
                }
            }
        }
        return  resultList;
    }

    @Override
    public List<WhInvRcdVO> getInvRcdByCond(final WhInvRcdCond cond) {
        cond.setCurrpage(cond.getCurrpage() == null ? 1 : cond.getCurrpage());
        SStockRecordConditionDTO sStockRecordConditionDTO = BeanUtil.buildFrom(cond, SStockRecordConditionDTO.class);
        sStockRecordConditionDTO.setInOutType(SStockInOutTypeEnum.getEnumByCode(cond.getInOutType()));
        sStockRecordConditionDTO.setSubmitTimeStart(EmptyUtil.isNotEmpty(cond.getSubmitTimeStart()) ? DateUtil.dayStart(DateUtil.parse(cond.getSubmitTimeStart(), DateUtil.DEFAULT_DATE_FORMAT)) : null);
        sStockRecordConditionDTO.setSubmitTimeEnd(EmptyUtil.isNotEmpty(cond.getSubmitTimeEnd()) ? DateUtil.dayEnd(DateUtil.parse(cond.getSubmitTimeEnd(), DateUtil.DEFAULT_DATE_FORMAT)) : null);
        List<SStockRecordVO> recordVOList = sStockService.findStockRecordByCond(sStockRecordConditionDTO);
        List<WhInvRcdVO> rcdList = new ArrayList<>();
        if(EmptyUtil.isNotEmpty(recordVOList)){
            for(SStockRecordVO sStockRecord : recordVOList){
                WhInvRcdVO rcd = BeanUtil.buildFrom(sStockRecord,WhInvRcdVO.class);
                if(NullUtil.isNotNull(rcd)
                        && NullUtil.isNotNull(sStockRecord.getSubmitUserId())){
                    rcd.setSubmitUserId(sStockRecord.getSubmitUserId().longValue());
                }
                rcdList.add(rcd);
            }
        }
        return rcdList;
    }

    @Override
    public List<WhInvRcdExtend> findWhInvRcdExtendListByIdList(List<Long> idList) {
        WhInvRcdExtendExample extendExample = new WhInvRcdExtendExample();
        extendExample.createCriteria().andRcdIdIn(idList);
        return whInvRcdExtendMapper.selectByExample(extendExample);
    }

    @Override
    public List<WhInvOccupy> getInvOccupyByCond(final WhInvRcdCond cond) {
        cond.setCurrpage(cond.getCurrpage() == null ? 1 : cond.getCurrpage());
        SStockOccupyDTO sStockOccupyDTO = BeanUtil.buildFrom(cond, SStockOccupyDTO.class);
        sStockOccupyDTO.setSkuCode(cond.getSkuCode());
        if (EmptyUtil.isNotEmpty(cond.getWarehouseCode())){
            sStockOccupyDTO.setWarehouseCode(cond.getWarehouseCode());
        }else if(CollectionUtils.isNotEmpty(cond.getWarehouseCodes())){
            sStockOccupyDTO.setWarehouseCodes(cond.getWarehouseCodes());
        }
        if(EmptyUtil.isNotEmpty(cond.getInOutType())){
        	sStockOccupyDTO.setOccupyType(SStockOccupyTypeEnum.getEnumByCode(cond.getInOutType()));
        }else if(CollectionUtils.isNotEmpty(cond.getInOutTypes())){
            sStockOccupyDTO.setOccupyTypes(cond.getInOutTypes());
        }
        sStockOccupyDTO.setOccupyTimeStart(EmptyUtil.isNotEmpty(cond.getSubmitTimeStart()) ? DateUtil.dayStart(DateUtil.parse(cond.getSubmitTimeStart(), DateUtil.DEFAULT_DATE_FORMAT)) : null);
        sStockOccupyDTO.setOccupyTimeEnd(EmptyUtil.isNotEmpty(cond.getSubmitTimeEnd()) ? DateUtil.dayEnd(DateUtil.parse(cond.getSubmitTimeEnd(), DateUtil.DEFAULT_DATE_FORMAT)) : null);
        List<SStockOccupyRecordVO> recordVOList = sStockService.findStockRecordInvOccupyByCond(sStockOccupyDTO);
        List<WhInvOccupy> occupyList = new ArrayList<>();
        if(EmptyUtil.isNotEmpty(recordVOList)){
            for(SStockOccupyRecordVO sStockOccupyRecordVO : recordVOList){
            	WhInvOccupy rcd = BeanUtil.buildFrom(sStockOccupyRecordVO,WhInvOccupy.class);
                occupyList.add(rcd);
            }
        }
        return occupyList;
    }
    
    @Override
    public WhInvOccupy getCountInvRcdWillInByCond(final WhInvRcdCond cond) {
        cond.setCurrpage(cond.getCurrpage() == null ? 1 : cond.getCurrpage());
        SStockOccupyDTO sStockOccupyDTO = BeanUtil.buildFrom(cond, SStockOccupyDTO.class);
        sStockOccupyDTO.setSkuCode(cond.getSkuCode());
        sStockOccupyDTO.setWarehouseCode(cond.getWarehouseCode());
        if(EmptyUtil.isNotEmpty(cond.getInOutType())){
        	sStockOccupyDTO.setInvOccupyType(cond.getInOutType());
        }
//        if(EmptyUtil.isNotEmpty(cond.getIsSplit())){
//        	sStockOccupyDTO.setIsSplit(cond.getIsSplit());
//        }
        sStockOccupyDTO.setWillInTimeStart(EmptyUtil.isNotEmpty(cond.getSubmitTimeStart()) ? cond.getSubmitTimeStart().replace("-", "") : null);
        sStockOccupyDTO.setWillInTimeEnd(EmptyUtil.isNotEmpty(cond.getSubmitTimeEnd()) ? cond.getSubmitTimeEnd().replace("-", "") : null);
        SStockOccupyRecordVO recordVO = sStockService.findCountStockRecordInvRcdWillInByCond(sStockOccupyDTO);
        WhInvOccupy rcd = BeanUtil.buildFrom(recordVO,WhInvOccupy.class);
        return rcd;
    }
    
    @Override
    public List<WhInvOccupy> getInvRcdWillInByCond(final WhInvRcdCond cond) {
        cond.setCurrpage(cond.getCurrpage() == null ? 1 : cond.getCurrpage());
        SStockOccupyDTO sStockOccupyDTO = BeanUtil.buildFrom(cond, SStockOccupyDTO.class);
        sStockOccupyDTO.setSkuCode(cond.getSkuCode());
        sStockOccupyDTO.setWarehouseCode(cond.getWarehouseCode());
        if(EmptyUtil.isNotEmpty(cond.getInOutType())){
        	sStockOccupyDTO.setInvOccupyType(cond.getInOutType());
        }
//        if(EmptyUtil.isNotEmpty(cond.getIsSplit())){
//        	sStockOccupyDTO.setIsSplit(cond.getIsSplit());
//        }
        sStockOccupyDTO.setWillInTimeStart(EmptyUtil.isNotEmpty(cond.getSubmitTimeStart()) ? cond.getSubmitTimeStart().replace("-", "") : null);
        sStockOccupyDTO.setWillInTimeEnd(EmptyUtil.isNotEmpty(cond.getSubmitTimeEnd()) ? cond.getSubmitTimeEnd().replace("-", "") : null);
        List<SStockOccupyRecordVO> recordVOList = sStockService.findStockRecordInvRcdWillInByCond(sStockOccupyDTO);
        List<WhInvOccupy> occupyList = new ArrayList<>();
        if(EmptyUtil.isNotEmpty(recordVOList)){
            for(SStockOccupyRecordVO sStockOccupyRecordVO : recordVOList){
            	WhInvOccupy rcd = BeanUtil.buildFrom(sStockOccupyRecordVO,WhInvOccupy.class);
                occupyList.add(rcd);
            }
        }
        return occupyList;
    }


    @Override
    public WhInvOccupy findOccupyQttByReferenceCode(String referenceCode) {
        return BeanUtil.buildFrom(sStockService.getOccupyRecordsByReferenceCode(referenceCode), WhInvOccupy.class);
    }

    @Override
    public List<WhInvOccupy> findOccupyQttByReferenceCodeLike(String referenceCode) {
        return BeanUtil.buildListFrom(sStockService.getOccupyRecordsByReferenceCodeLike(referenceCode), WhInvOccupy.class);
    }

    @Override
    public List<WhInvQttVO> findInvQttBySkuCode_IPOS(String channelCode, String skuCode) {
    	 List<WhInvQttVO> list = Lists.newArrayList();
        if (StringUtils.isBlank(channelCode) || StringUtils.isEmpty(skuCode)) {
            return list;
        }
        // 获取IPOS 渠道
        List<OpChannelVO> channelVOs = mcOpChannelService.getIPOSChannels();
        if (CollectionUtils.isEmpty(channelVOs)) {
        	return list;
        }
        // 获取SKU库存
        Map<String, SSkuStockVO> stockMap = sStockService.getSkuStocksInAllWarehouses(skuCode);
        if (MapUtils.isEmpty(stockMap)) {
        	return list;
        }
        
        // 取当前渠道对应的配送仓库存
        List<ChannelWarehouseVO> channelWarehouseList = channelQueryService.getWarehouseByCode(channelCode);
        for (ChannelWarehouseVO warehouse : channelWarehouseList) {
        	SSkuStockVO sSkuStockVO = stockMap.get(warehouse.getCode());
        	if (sSkuStockVO == null) {
    			continue;
    		}
        	WhWarehouse whWarehouse = whWarehouseMapper.findWarehouseByCode(warehouse.getCode());
        	WhInvQttVO whInvQttVO = new  WhInvQttVO();
            whInvQttVO.setChannelCode(channelCode);
            whInvQttVO.setQuantity(sSkuStockVO.getCanUseQuantity());
            whInvQttVO.setSkuCode(skuCode);
            whInvQttVO.setName(whWarehouse.getName());
            list.add(whInvQttVO);
        }
        
        for (OpChannelVO opChannelVO : channelVOs) {
    		SSkuStockVO sSkuStockVO = stockMap.get(opChannelVO.getWarehouseForSales());
    		if (sSkuStockVO == null) {
    			continue;
    		}
	        WhInvQttVO whInvQttVO = new  WhInvQttVO();
            whInvQttVO.setChannelCode(opChannelVO.getCode());
            whInvQttVO.setQuantity(sSkuStockVO.getCanUseQuantity());
            whInvQttVO.setSkuCode(skuCode);
            whInvQttVO.setName(opChannelVO.getName());
            if (opChannelVO.getCode().equals(channelCode)) {
            	list.add(0, whInvQttVO);
            } else {
            	list.add(whInvQttVO);
            }
        }
        return list;
    }
    
    @Override
    public int updateOccupyType(WhInvOccupy record) {
        log.warn("-------------record.getReferenceCode():" + record.getReferenceCode() + ";record.getOccupyType():" + record.getOccupyType());
        ServiceResp<Boolean> serviceResp = sStockService.updateOccupyType(record.getReferenceCode(), record.getOccupyType());
        boolean resultStatus = serviceResp.getBean();
        return resultStatus ? 1 : 0;
    }



    @Override
    public Boolean releaseInterestOccupy(String interestId, String skuCode, Integer quantity) {
        SStockReleaseDTO stockReleaseDTO = new SStockReleaseDTO();
        stockReleaseDTO.setOccupyType(SStockOccupyTypeEnum.getEnumByCode(SStockOccupyTypeEnum.INTEREST.getCode()));
        stockReleaseDTO.setPreparedSkuCode(skuCode);
        stockReleaseDTO.setReferenceCode(interestId);
        stockReleaseDTO.setPreparedReleaseQuantity(quantity);
        ServiceResp<SReleaseResultVO> serviceResp = sStockService.release(stockReleaseDTO);
        if (!serviceResp.isSuccess()) {
            throw  new  WarehouseException(serviceResp.getRespCode(), serviceResp.getRespMsg());
        }
        return serviceResp.isSuccess() ? true : false;
    }

    @Override
    public Boolean releaseCampaignOccupy(String campaignCode, String skuCode, Integer quantity) {
        SStockReleaseDTO stockReleaseDTO = new SStockReleaseDTO();
        stockReleaseDTO.setOccupyType(SStockOccupyTypeEnum.getEnumByCode(SStockOccupyTypeEnum.CAMPAIGN_OUT.getCode()));
        stockReleaseDTO.setPreparedSkuCode(skuCode);
        stockReleaseDTO.setBusinessCode(campaignCode);
        stockReleaseDTO.setPreparedReleaseQuantity(quantity);
        ServiceResp<SReleaseResultVO> serviceResp = sStockService.release(stockReleaseDTO);
        if (!serviceResp.isSuccess()) {
            throw  new  WarehouseException(serviceResp.getRespCode(), serviceResp.getRespMsg());
        }
        return serviceResp.isSuccess() ? true : false;
    }

    @Override
    public Integer findOccupyByCampaignCodeAndSkuCode(String warehouseCode, String campaignCode, String skuCode) {
        /*SCampaignStockQueryDTO queryDTO = new SCampaignStockQueryDTO();
        queryDTO.setWarehouseCode(warehouseCode);
        queryDTO.setCampaignCode(campaignCode);
        queryDTO.setSkuCode(skuCode);
        SPreparedStockVO sPreparedStockVO = sStockService.getPreparedQuantity(queryDTO);
        return  EmptyUtil.isEmpty(sPreparedStockVO) ? 0 : sPreparedStockVO.getPreparedQuantity();*/
        return  0;
    }

    @Override
    public void deleteCampaignOccupation(List<WhReleaseOccupationVO> releaseList) {
        // sStockService.deleteOccupation(BeanUtil.buildListFrom(releaseList, SStockDeleteOccupationDTO.class));
    }

    /**
     * 释放库存占用
     *
     * @param whInvOccupyList SKU库存占用列表
     * @return 库存占用ID
     */
    @Override
    @Transactional
    public int releaseOccupy(List<WhInvOccupy> whInvOccupyList) {
        ServiceResp<List<SReleaseResultVO>>  serviceResp = sStockService.release(BeanUtil.buildListFrom(whInvOccupyList, SStockReleaseDTO.class));
        if (!serviceResp.isSuccess()) {
            throw  new  WarehouseException(serviceResp.getRespCode(), serviceResp.getRespMsg());
        }
       boolean resultStatus = serviceResp.isSuccess();
        return resultStatus ? 1 : 0;
    }

    @Override
    public int updateReferenceCode(String oldReferenceCode,
                                   String newReferenceCode) {
        ServiceResp<Boolean> serviceResp = sStockService.updateReferenceCode(oldReferenceCode, newReferenceCode);
        if (!serviceResp.isSuccess()) {
            throw  new  WarehouseException(serviceResp.getRespCode(), serviceResp.getRespMsg());
        }
        boolean resultStatus = serviceResp.getBean();
        return resultStatus ? 1 : 0;
    }


    private WhInvVO convertSkuStockVO2WhInvVO(SSkuStockVO sSkuStockVO) {
        WhInvVO whInvVO = new WhInvVO();
        whInvVO.setSkuCode(sSkuStockVO.getSkuCode());
        whInvVO.setQuantityInOccupy(sSkuStockVO.getOccupyQuantity());
        whInvVO.setQuantityInRcd(sSkuStockVO.getRealQuantity());
        whInvVO.setWarehouseCode(sSkuStockVO.getWarehouseCode());
        return  whInvVO;
    }

    private List<WhInvVO> convertSkuStockVO2WhInvVO(List<SSkuStockVO> sSkuStockVOList) {
        List<WhInvVO> whInvVOList = new ArrayList<>();
        if (whInvVOList != null) {
            for (SSkuStockVO sSkuStockVO : sSkuStockVOList) {
                whInvVOList.add(convertSkuStockVO2WhInvVO(sSkuStockVO));
            }
        }
        return  whInvVOList;
    }
    private SStockOccupyDTO convertWhInvOccupy2DTO(WhInvOccupy whInvOccupy) {
        SStockOccupyDTO sStockOccupyDTO = new SStockOccupyDTO();
        BeanUtils.copyProperties(whInvOccupy, sStockOccupyDTO);
        sStockOccupyDTO.setOccupyTime(new Date());
        sStockOccupyDTO.setOccupyType(SStockOccupyTypeEnum.getEnumByCode(whInvOccupy.getOccupyType()));
        return  sStockOccupyDTO;
    }

    private List<SStockOccupyDTO> convertWhInvOccupy2DTO(List<WhInvOccupy> whInvOccupyList) {
        List<SStockOccupyDTO> dtoList = new ArrayList<SStockOccupyDTO>();
        if (EmptyUtil.isNotEmpty(whInvOccupyList)) {
            for(WhInvOccupy whInvOccupy : whInvOccupyList) {
                dtoList.add(convertWhInvOccupy2DTO(whInvOccupy));
            }
        }
        return  dtoList;
    }

    @Override
    public List<SStockReleaseDTO> convertWhRelease2DTO(List<WhReleaseOccupationVO> whReleaseOccupationVOList) {
        List<SStockReleaseDTO>  dtoList = new ArrayList<SStockReleaseDTO>();
        if(EmptyUtil.isNotEmpty(whReleaseOccupationVOList)) {
            for(WhReleaseOccupationVO whReleaseOccupationVO : whReleaseOccupationVOList) {
                SStockReleaseDTO dto = new SStockReleaseDTO();
                BeanUtils.copyProperties(whReleaseOccupationVO, dto);
                dto.setOperationType(SStockOperationTypeEnum.DEFAULT);
                dto.setOccupyType(SStockOccupyTypeEnum.getEnumByCode(whReleaseOccupationVO.getOccupyType()));
                dto.setBackToPrepared(false);
                dtoList.add(dto);
            }
        }
        return  dtoList;
    }

    @Override
    public List<SStockRecordVO> convertWhInv2DTO(List<WhInvRcd> whInvRcdList) {
        List<SStockRecordVO> dtoList = new ArrayList<>();
        if(EmptyUtil.isNotEmpty(whInvRcdList)){
            for (WhInvRcd invRcd : whInvRcdList){
                dtoList.add(convertWhInv2DTO(invRcd));
            }
        }
        return dtoList;
    }

    @Override
    public List<WhSkuInvQttVO> findAllotWillInQttByWarehouseCode(String warehouseCode) {
        return whCommandMapper.findAllotWillInQttByWarehouseCode(warehouseCode);
    }

    private SStockReleaseDTO convertWhRelease2DTO(WhReleaseOccupationVO whReleaseOccupationVO) {
        SStockReleaseDTO dto = new SStockReleaseDTO();
        BeanUtils.copyProperties(whReleaseOccupationVO, dto);
        dto.setOperationType(SStockOperationTypeEnum.DEFAULT);
        dto.setOccupyType(SStockOccupyTypeEnum.getEnumByCode(whReleaseOccupationVO.getOccupyType()));
        return  dto;
    }

    private List<WhSkuInvQttVO> convertDTO2WhSkuInvQttVO(List<SSkuStockVO> sSkuStockVOList) {
        List<WhSkuInvQttVO> whSkuInvQttVOList = new ArrayList<>();
        if (sSkuStockVOList != null) {
            for (SSkuStockVO sSkuStockVO : sSkuStockVOList) {

            }
        }
        return  null;
    }

    private final static  int PAGE_NUM = 500;
    public  List<SSkuStockVO> getSkuStockInAllWarehouses(String warehouseCode) {
        PageCond page = new PageCond();
        int currentPage = 1;
        page.setPagenum(PAGE_NUM);
        //当前页记录数
        int currentPageNum = PAGE_NUM;
        List<SSkuStockVO> resultList = new ArrayList<SSkuStockVO>();
        while(currentPageNum == PAGE_NUM ) {
          page.setCurrpage(currentPage);
          PageQueryResp<SSkuStockVO> pageQueryResp = sStockService.getAllSkuStocksInWarehouse(warehouseCode, page);
          if (EmptyUtil.isNotEmpty(warehouseCode)) {
              pageQueryResp = sStockService.getAllSkuStocksInWarehouse(warehouseCode, page);
          } else {

          }
          List<SSkuStockVO> skuStockVOList = pageQueryResp.getBeanList();
          /*for (SSkuStockVO skuStockVO : skuStockVOList) {
              if (skuStockVO.getSkuCode().equals("10010001")) {
                    System.out.print(currentPage);
              }
          }*/
          resultList.addAll(skuStockVOList);
          currentPageNum = skuStockVOList.size();
          currentPage ++;
        }

        return  resultList;
    }

    private  Map<String, PcsSkuDTO> getSkuNameMap(List<SSkuStockVO>  skuStockVOList) {
        Set<String>  skuCodes = new HashSet<>();
        if (EmptyUtil.isNotEmpty(skuStockVOList)) {
            for (SSkuStockVO sSkuStockVO : skuStockVOList) {
                skuCodes.add(sSkuStockVO.getSkuCode());
            }
        }
        return   getSkuNameMapBySkuCodes(skuCodes);
    }


    private  Map<String, PcsSkuDTO> getSkuNameMapBySkuCodes(Set<String>  skuCodes) {
        List<String> skuCodeList = new ArrayList<>();
        skuCodeList.addAll(skuCodes);
        int skuCodeNum = skuCodeList.size();
        int maxSearchNum = 500;
        int index = 0;
        Map<String, PcsSkuDTO> resultMap = new HashMap<>();
        if (maxSearchNum < skuCodeNum) {
            while (index < skuCodeNum) {
                List<String> subList = skuCodeList.subList(index, (index + maxSearchNum) < skuCodeNum ? (index + maxSearchNum) : skuCodeNum);
                index += maxSearchNum;
                Set<String> subSkuCodes = new HashSet<>();
                subSkuCodes.addAll(subList);
                resultMap.putAll(mcPcsSkuService.getSkuNameByCodes(subSkuCodes));
            }
        } else {
            return  mcPcsSkuService.getSkuNameByCodes(skuCodes);
        }
        return   resultMap;
    }
    public static void  main(String[] args) {
        int skuCodeNum = 501;
        int maxSearchNum = 500;
        int totalPage = skuCodeNum/maxSearchNum;

        List<String> skuCodeList = new ArrayList<>();
        skuCodeList.add("1");
        skuCodeList.add("2");
        skuCodeList.add("3");
        skuCodeList.add("4");
        skuCodeList.add("5");
        List test =  skuCodeList.subList(1,4);
        System.out.print(totalPage);
    }
}


