package com.thebeastshop.pegasus.service.warehouse.service.impl;

import com.thebeastshop.common.ServiceResp;
import com.thebeastshop.pegasus.integration.email.EmailUtil;
import com.thebeastshop.pegasus.integration.email.vo.EmailVO;
import com.thebeastshop.pegasus.service.warehouse.cond.WhGradeAffectInfoCond;
import com.thebeastshop.pegasus.service.warehouse.cond.WhGradeOfQualityCond;
import com.thebeastshop.pegasus.service.warehouse.cond.WhGradeOfQualityDetailCond;
import com.thebeastshop.pegasus.service.warehouse.cond.WhWarehouseGroupCond;
import com.thebeastshop.pegasus.service.warehouse.dao.WhGradeAffectInfoMapper;
import com.thebeastshop.pegasus.service.warehouse.dao.WhGradeOfQualityDetailMapper;
import com.thebeastshop.pegasus.service.warehouse.dao.WhGradeOfQualityMapper;
import com.thebeastshop.pegasus.service.warehouse.dao.custom.WhGradeOfQualityCustomMapper;
import com.thebeastshop.pegasus.service.warehouse.exception.WarehouseException;
import com.thebeastshop.pegasus.service.warehouse.exception.WarehouseExceptionErrorCode;
import com.thebeastshop.pegasus.service.warehouse.model.*;
import com.thebeastshop.pegasus.service.warehouse.service.*;
import com.thebeastshop.pegasus.service.warehouse.vo.*;
import com.thebeastshop.pegasus.util.PegasusUtilFacade;
import com.thebeastshop.pegasus.util.comm.*;
import com.thebeastshop.pegasus.util.model.CommGlobalConfig;
import com.thebeastshop.stock.dto.SStockOccupyDTO;
import com.thebeastshop.stock.enums.SStockOccupyTypeEnum;
import com.thebeastshop.stock.enums.SStockOperationTypeEnum;
import com.thebeastshop.stock.service.SStockService;
import com.thebeastshop.stock.vo.SOccupyResultVO;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.task.TaskExecutor;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import page.Pagination;

import java.util.*;

/**
 * @author xw
 * @create 2018-04-23 15:26
 */
@Service("whGradeOfQualityService")
public class WhGradeOfQualityServiceImpl implements WhGradeOfQualityService {

    private final static Logger log = LoggerFactory.getLogger(WhGradeOfQualityServiceImpl.class);

    @Autowired
    private WhGradeOfQualityMapper whGradeOfQualityMapper;
    @Autowired
    private WhGradeOfQualityDetailMapper whGradeOfQualityDetailMapper;
    @Autowired
    private WhGradeAffectInfoMapper whGradeAffectInfoMapper;
    @Autowired
    private WhGradeOfQualityCustomMapper whGradeOfQualityCustomMapper;

    @Autowired
    private SStockService sStockService;

    @Autowired
    private WhInvService whInvService;

    @Autowired
    private WhCommandService whCommandService;

    @Autowired
    private WhWmsGradeOfQualityService whWmsGradeOfQualityService;

    @Autowired
    private TaskExecutor gradeAdjustScheduler;

    @Autowired
    private WhWarehouseGroupService whWarehouseGroupService;

    // private PegasusUtilFacade pegasusUtilFacade = PegasusUtilFacade.getInstance();

    @Override
    public Pagination<WhGradeOfQualityVO> pageWhGradeOfQualityVOByCond(WhGradeOfQualityCond cond) {
        Pagination<WhGradeOfQualityVO> page = new Pagination<>(cond.getCurrpage(), cond.getPagenum());
        int count = countWhGradeOfQuality(cond);
        page.setRecord(count);
        if (NumberUtil.isNullOrZero(count))
            return page;
        List<WhGradeOfQualityVO> gradeOfQualityVOs = whGradeOfQualityCustomMapper.listWhGradeOfQualityVOByCond(cond);
        page.setResultList(gradeOfQualityVOs);
        return page;
    }

    private int countWhGradeOfQuality(WhGradeOfQualityCond cond){
        return whGradeOfQualityCustomMapper.countWhGradeOfQualityByCond(cond);
    }

    @Override
    public WhGradeOfQualityVO findWhGradeOfQualityVOById(Long id, WhGradeOfQualityCond cond) {
        WhGradeOfQualityVO vo = whGradeOfQualityCustomMapper.findVOById(id);
        if (EmptyUtil.isNotEmpty(vo)){
            fetchGradeDetailVO(vo,cond.isFetchGradeDetail());
            fetchGradeAppectInfo(vo,cond.isFetchGradeAppectInfo());
        }
        return vo;
    }

    public WhGradeOfQualityVO findWhGradeOfQualityVOById(Long id, boolean fetchDetail,boolean fetchAffect) {
        WhGradeOfQualityVO vo = whGradeOfQualityCustomMapper.findVOById(id);
        if (EmptyUtil.isNotEmpty(vo)){
            if (fetchDetail){
                WhGradeOfQualityDetailCond cond = new WhGradeOfQualityDetailCond();
                cond.setRefId(id);
                vo.setWhGradeDetails(listWhGradeOfQualityDetailByCond(cond));
            }
            if (fetchAffect){
                vo.setWhGradeAffectInfos(listWhGradeAffectInfoByWhGradeId(id));
            }
        }
        return vo;
    }

    @Override
    public List<WhGradeAffectInfo> listWhGradeAffectInfoByWhGradeId(Long whGradeId){
        WhGradeAffectInfoCond cond = new WhGradeAffectInfoCond();
        cond.setWhGradeId(whGradeId);
        return listWhGradeAffectInfoByCond(cond);
    }

    @Override
    public List<WhGradeOfQualityDetail> listWhGradeOfQualityDetailByCond(WhGradeOfQualityDetailCond cond){
        WhGradeOfQualityDetailExample example = buildWhGradeOfQualityDetailExampleByCond(cond);
        return whGradeOfQualityDetailMapper.selectByExample(example);
    }

    @Override
    public List<WhGradeAffectInfo> listWhGradeAffectInfoByCond(WhGradeAffectInfoCond cond){
        WhGradeAffectInfoExample example = buildWhGradeAffectInfoExampleByCond(cond);
        return whGradeAffectInfoMapper.selectByExample(example);
    }

    @Override
    public List<WhGradeAffectInfoVO> listWhGradeAffectInfoVOByCond(WhGradeAffectInfoCond cond) {
        return whGradeOfQualityCustomMapper.listWhGradeAffectInfoVOByCond(cond);
    }

    private void fetchGradeDetailVO(WhGradeOfQualityVO vo, boolean fetchGradeDetail){
        if (fetchGradeDetail){
            WhGradeOfQualityDetailCond detailCond = new WhGradeOfQualityDetailCond();
            detailCond.setRefId(vo.getId());
            List<WhGradeOfQualityDetailVO> gradeDetailVOs = whGradeOfQualityCustomMapper.listWhGradeOfQualityDetailVOByCond(detailCond);
            vo.setWhGradeDetailVOs(gradeDetailVOs);
        }
    }

    @Override
    public boolean updateWhGradeAffectInfo(WhGradeAffectInfo affectInfo) {
        return whGradeAffectInfoMapper.updateByPrimaryKeySelective(affectInfo)>0;
    }

    @Override
    public boolean updateWhGradeAffectInfoOptFlag(List<Long> ids,Integer newOptFlag,Integer oldOptFlag) {
        return whGradeOfQualityCustomMapper.updateWhGradeAffectInfoOptFlag(ids,newOptFlag,oldOptFlag);
    }

    @Override
    public boolean updateWhGradeAffectInfoSendMailFlag(List<Long> affectIds, Integer newSendMailFlag, Integer oldSendMailFlag) {
        WhGradeAffectInfo affectInfo = new WhGradeAffectInfo();
        affectInfo.setSendMailFlag(newSendMailFlag);
        WhGradeAffectInfoCond affectCond = new WhGradeAffectInfoCond();
        affectCond.setIds(affectIds);
        affectCond.setSendMailFlag(oldSendMailFlag);
        return updateWhGradeAffectInfoByExampleSelective(affectInfo,affectCond);
    }

    @Override
    public boolean updateWhGradeAffectInfoByExampleSelective(WhGradeAffectInfo affectInfo,WhGradeAffectInfoCond affectCond) {
        WhGradeAffectInfoExample example = buildWhGradeAffectInfoExampleByCond(affectCond);
        return whGradeAffectInfoMapper.updateByExampleSelective(affectInfo,example)>0;
    }

    public boolean updateWhGradeAffectInfoByExample(WhGradeAffectInfo affectInfo,WhGradeAffectInfoExample example) {
        return whGradeAffectInfoMapper.updateByExample(affectInfo,example)>0;
    }

    private void fetchGradeAppectInfo(WhGradeOfQualityVO vo, boolean fetchGradeAppectInfo){
        if (fetchGradeAppectInfo){
            WhGradeAffectInfoCond cond = new WhGradeAffectInfoCond();
            cond.setWhGradeId(vo.getId());
            WhGradeAffectInfoExample example = buildWhGradeAffectInfoExampleByCond(cond);
            List<WhGradeAffectInfo> whGradeAffectInfos = whGradeAffectInfoMapper.selectByExample(example);
            vo.setWhGradeAffectInfos(whGradeAffectInfos);
        }
    }

    @Override
    public WhGradeOfQualityVO findWhGradeOfQualityVOByCond(WhGradeOfQualityCond cond) {
        WhGradeOfQualityExample example = buildWhGradeOfQualityExampleByCond(cond);
        List<WhGradeOfQuality> gradeOfQualities = whGradeOfQualityMapper.selectByExample(example);
        if (CollectionUtils.isNotEmpty(gradeOfQualities)){
            WhGradeOfQualityVO vo = BeanUtil.buildFrom(gradeOfQualities.get(0),WhGradeOfQualityVO.class);
            fetchGradeDetailVO(vo,cond.isFetchGradeDetail());
            fetchGradeAppectInfo(vo,cond.isFetchGradeAppectInfo());
            return vo;
        }
        return null;
    }

    @Override
    public List<WhGradeOfQualityVO> listWhGradeOfQualityVOByCond(WhGradeOfQualityCond cond) {
        WhGradeOfQualityExample example = buildWhGradeOfQualityExampleByCond(cond);
        List<WhGradeOfQuality> gradeOfQualities = whGradeOfQualityMapper.selectByExample(example);
        if (CollectionUtils.isNotEmpty(gradeOfQualities)){
            List<WhGradeOfQualityVO> whGradeOfQualityVOs = new ArrayList<>();
            for (WhGradeOfQuality whGrade : gradeOfQualities){
                WhGradeOfQualityVO vo = BeanUtil.buildFrom(whGrade,WhGradeOfQualityVO.class);
                fetchGradeDetailVO(vo,cond.isFetchGradeDetail());
                fetchGradeAppectInfo(vo,cond.isFetchGradeAppectInfo());
                whGradeOfQualityVOs.add(vo);
            }
            return whGradeOfQualityVOs;
        }
        return Collections.emptyList();
    }

    @Override
    public List<WhGradeOfQualityVO> listWhGradeOfQualityVOsByCond(WhGradeOfQualityCond cond) {
        return whGradeOfQualityCustomMapper.listWhGradeOfQualityVOByCond(cond);
    }

    private WhGradeOfQualityExample buildWhGradeOfQualityExampleByCond(WhGradeOfQualityCond cond){
        WhGradeOfQualityExample example = new WhGradeOfQualityExample();
        WhGradeOfQualityExample.Criteria criteria = example.createCriteria();

        if (EmptyUtil.isNotEmpty(cond.getId())){
            criteria.andIdEqualTo(cond.getId());
        }else if(CollectionUtils.isNotEmpty(cond.getIds())){
            criteria.andIdIn(cond.getIds());
        }

        if (EmptyUtil.isNotEmpty(cond.getRefId())){
            criteria.andRefIdEqualTo(cond.getRefId());
        }else if(CollectionUtils.isNotEmpty(cond.getRefIds())){
            criteria.andRefIdIn(cond.getRefIds());
        }

        if (EmptyUtil.isNotEmpty(cond.getRefCode())){
            criteria.andRefCodeEqualTo(cond.getRefCode());
        }else if(CollectionUtils.isNotEmpty(cond.getRefCodes())){
            criteria.andRefCodeIn(cond.getRefCodes());
        }

        if (EmptyUtil.isNotEmpty(cond.getSkuCode())){
            criteria.andSkuCodeEqualTo(cond.getSkuCode());
        }

        if (EmptyUtil.isNotEmpty(cond.getSourceGrade())){
            criteria.andSourceGradeEqualTo(cond.getSourceGrade());
        }

        if (EmptyUtil.isNotEmpty(cond.getTargetGrade())){
            criteria.andTargetGradeEqualTo(cond.getTargetGrade());
        }

        if (EmptyUtil.isNotEmpty(cond.getPhysicalWarehouseCode())){
            criteria.andPhysicalWarehouseCodeEqualTo(cond.getPhysicalWarehouseCode());
        }

        if (EmptyUtil.isNotEmpty(cond.getApproveStatus())){
            criteria.andApproveStatusEqualTo(cond.getApproveStatus());
        }

        if (EmptyUtil.isNotEmpty(cond.getCreateTimeStart())){
            criteria.andCreateTimeGreaterThanOrEqualTo(DateUtil.parse(cond.getCreateTimeStart(),DateUtil.DEFAULT_DATE_FORMAT));
        }

        if (EmptyUtil.isNotEmpty(cond.getCreateTimeEnd())){
            criteria.andCreateTimeLessThanOrEqualTo(DateUtil.parse(cond.getCreateTimeEnd(),DateUtil.DEFAULT_DATE_FORMAT));
        }

        if (EmptyUtil.isNotEmpty(cond.getApproveTimeStart())){
            criteria.andApproveTimeGreaterThanOrEqualTo(DateUtil.parse(cond.getApproveTimeStart(),DateUtil.DEFAULT_DATE_FORMAT));
        }

        if (EmptyUtil.isNotEmpty(cond.getApproveTimeEnd())){
            criteria.andApproveTimeLessThanOrEqualTo(DateUtil.parse(cond.getApproveTimeEnd(),DateUtil.DEFAULT_DATE_FORMAT));
        }

        if (EmptyUtil.isNotEmpty(cond.getCreateUserId())){
            criteria.andCreateUserIdEqualTo(cond.getCreateUserId());
        }

        if (EmptyUtil.isNotEmpty(cond.getApproveUserId())){
            criteria.andApproveUserIdEqualTo(cond.getApproveUserId());
        }

        if (EmptyUtil.isNotEmpty(cond.getOriginType())){
            criteria.andOriginTypeEqualTo(cond.getOriginType());
        }

        return example;
    }

    private WhGradeOfQualityDetailExample buildWhGradeOfQualityDetailExampleByCond(WhGradeOfQualityDetailCond cond){
        WhGradeOfQualityDetailExample example = new WhGradeOfQualityDetailExample();
        WhGradeOfQualityDetailExample.Criteria criteria = example.createCriteria();

        if (EmptyUtil.isNotEmpty(cond.getId())){
            criteria.andIdEqualTo(cond.getId());
        }else if(CollectionUtils.isNotEmpty(cond.getIds())){
            criteria.andIdIn(cond.getIds());
        }

        if (EmptyUtil.isNotEmpty(cond.getRefId())){
            criteria.andRefIdEqualTo(cond.getRefId());
        }else if(CollectionUtils.isNotEmpty(cond.getRefIds())){
            criteria.andRefIdIn(cond.getRefIds());
        }

        if (EmptyUtil.isNotEmpty(cond.getWmsGradeId())){
            criteria.andWmsGradeIdEqualTo(cond.getWmsGradeId());
        }else if (CollectionUtils.isNotEmpty(cond.getWmsGradeIds())){
            criteria.andWmsGradeIdIn(cond.getWmsGradeIds());
        }

        if (EmptyUtil.isNotEmpty(cond.getWmsGradeSkuId())){
            criteria.andWmsGradeSkuIdEqualTo(cond.getWmsGradeSkuId());
        }else if (CollectionUtils.isNotEmpty(cond.getWmsGradeSkuIds())){
            criteria.andWmsGradeSkuIdIn(cond.getWmsGradeSkuIds());
        }

        if (EmptyUtil.isNotEmpty(cond.getSkuCode())){
            criteria.andSkuCodeEqualTo(cond.getSkuCode());
        }

        if (EmptyUtil.isNotEmpty(cond.getApproveStatus())){
            criteria.andApproveStatusEqualTo(cond.getApproveStatus());
        }

        if (EmptyUtil.isNotEmpty(cond.getSourceWarehouseCode())){
            criteria.andSourceWarehouseCodeEqualTo(cond.getSourceWarehouseCode());
        }

        if (EmptyUtil.isNotEmpty(cond.getTargetWarehouseCode())){
            criteria.andTargetWarehouseCodeEqualTo(cond.getTargetWarehouseCode());
        }
        return example;
    }

    private WhGradeAffectInfoExample buildWhGradeAffectInfoExampleByCond(WhGradeAffectInfoCond cond){
        WhGradeAffectInfoExample example = new WhGradeAffectInfoExample();
        WhGradeAffectInfoExample.Criteria criteria = example.createCriteria();

        if (EmptyUtil.isNotEmpty(cond.getId())){
            criteria.andIdEqualTo(cond.getId());
        }else if(CollectionUtils.isNotEmpty(cond.getIds())){
            criteria.andIdIn(cond.getIds());
        }

        if (EmptyUtil.isNotEmpty(cond.getWhGradeId())){
            criteria.andWhGradeIdEqualTo(cond.getWhGradeId());
        }else if (CollectionUtils.isNotEmpty(cond.getWhGradeIds())){
            criteria.andWhGradeIdIn(cond.getWhGradeIds());
        }

        if (EmptyUtil.isNotEmpty(cond.getType())){
            criteria.andTypeEqualTo(cond.getType());
        }else if (CollectionUtils.isNotEmpty(cond.getTypes())){
            criteria.andTypeIn(cond.getTypes());
        }

        if (EmptyUtil.isNotEmpty(cond.getAffectCode())){
            criteria.andAffectCodeEqualTo(cond.getAffectCode());
        }

        if (EmptyUtil.isNotEmpty(cond.getWarehouseCode())){
            criteria.andWarehouseCodeEqualTo(cond.getWarehouseCode());
        }

        if (EmptyUtil.isNotEmpty(cond.getOptFlag())){
            criteria.andOptFlagEqualTo(cond.getOptFlag());
        }

        if (EmptyUtil.isNotEmpty(cond.getWhGradeStatus())){
            criteria.andWhGradeStatusEqualTo(cond.getWhGradeStatus());
        }

        if (EmptyUtil.isNotEmpty(cond.getSendMailFlag())){
            criteria.andSendMailFlagEqualTo(cond.getSendMailFlag());
        }
        return example;
    }

    @Override
    public List<WhGradeOfQuality> listWhGradeOfQualityByCond(WhGradeOfQualityCond cond) {
        WhGradeOfQualityExample example = buildWhGradeOfQualityExampleByCond(cond);
        return whGradeOfQualityMapper.selectByExample(example);
    }

    @Override
    @Transactional
    public boolean create(WhGradeOfQualityVO vo) throws Exception{
        WhGradeOfQuality gradeOfQuality = BeanUtil.buildFrom(vo,WhGradeOfQuality.class);
        gradeOfQuality.setCreateTime(DateUtil.getNow());
        int result = whGradeOfQualityMapper.insert(gradeOfQuality);
        vo.setId(gradeOfQuality.getId());
        if (result > 0){
            insertGradeDetails(vo);

            // 生成CODE
            Map<String, Object> params = new HashMap<>();
            params.put("createTime", gradeOfQuality.getCreateTime());
            params.put("sourceGrade", gradeOfQuality.getSourceGrade());
            params.put("id", gradeOfQuality.getId());
            String code = CodeGenerator.getInstance().generate("GA_CODE", params);
            gradeOfQuality.setCode(code);
            whGradeOfQualityMapper.updateByPrimaryKeySelective(gradeOfQuality);

            vo.setCode(code);
        }
        return result > 0;
    }

    @Override
    public boolean batchCreateWhGradeAndFetch(List<WhGradeOfQualityVO> vos) throws Exception {
        boolean result = false;
        for (WhGradeOfQualityVO vo : vos){
            result = createOrUpdate(vo);
        }
        return result;
    }

    @Override
    @Transactional
    public boolean createOrUpdate(WhGradeOfQualityVO vo) throws Exception{
        boolean result = false;
        if (EmptyUtil.isNotEmpty(vo.getId()) && EmptyUtil.isNotEmpty(whGradeOfQualityMapper.selectByPrimaryKey(vo.getId()))){
            result = update(vo);
        }else{
            result = create(vo);
        }
        if (result && WhGradeOfQualityVO.APPROVE_STATUS_WAIT == vo.getApproveStatus()){
            // 生成scm占用
            scmOccupy(vo);
            // 记录受影响的单据包裹
            insertGradeAffectInfos(vo);
        }
        return result;
    }

    @Override
    @Transactional
    public boolean auditWhGradeOfQuality(WhGradeOfQualityVO vo) throws Exception{
        WhGradeOfQuality whGradeOfQuality = whGradeOfQualityMapper.selectByPrimaryKey(vo.getId());
        if (EmptyUtil.isEmpty(whGradeOfQuality)){
            throw new WarehouseException(WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED,
                    "品级调整单不存在,请刷新后重新处理!");
        }
        if (WhWmsGradeOfQualityVO.APPROVE_STATUS_WAIT != whGradeOfQuality.getApproveStatus()){
            throw new WarehouseException(WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED,
                    "品级调整状态不是[待审批],请确认状态后重新处理!");
        }
        WhWmsGradeOfQualityVO whWmsGradeOfQualityVO = whWmsGradeOfQualityService.findWhWmsGradeOfQualityVOById(whGradeOfQuality.getRefId(),false,false);

        boolean optPass = vo.getOptPass();
        WhGradeOfQualityVO whGradeVo = findWhGradeOfQualityVOById(vo.getId(),true,false);
        if (EmptyUtil.isEmpty(whGradeVo.getApproveUserId())){
            whGradeVo.setApproveUserId(vo.getApproveUserId());
        }
        whGradeVo.setMemo(vo.getMemo());
        whGradeVo.setRejectReason(vo.getRejectReason());
        whGradeVo.setApproveTime(DateUtil.getNow());
        if (optPass){
            // 扣减scm库存
            updateScmStock(whGradeVo);
            // 源逻辑仓库存释放
            scmReleaseOccupy(whGradeVo);
            // 品级调整单，行状态变更
            whGradeVo.setApproveStatus(WhGradeOfQualityVO.APPROVE_STATUS_PASS);
            for (WhGradeOfQualityDetail whGradeDetail : whGradeVo.getWhGradeDetails()){
                whGradeDetail.setApproveStatus(WhGradeOfQualityDetailVO.APPROVE_STATUS_PASS);
            }
            // 由于 扣减scm库存时 可能更新affectInfo,因此 此处重新查询数据
            whGradeVo.setWhGradeAffectInfos(listWhGradeAffectInfoByWhGradeId(vo.getId()));
            // 更新库存已经变动的标志，即 品级单状态为：(通过2)
            for (WhGradeAffectInfo whGradeAffectInfo : whGradeVo.getWhGradeAffectInfos()){
                whGradeAffectInfo.setWhGradeStatus(WhGradeAffectInfoVO.WH_GRADE_STATUS_PASS);
            }
            // 更新品级单及行状态
            modifyAndFetch(whGradeVo);

            whWmsGradeOfQualityVO.setSubmitUserId(whGradeVo.getApproveUserId());
            // 通过后 则同步scm
            whWmsGradeOfQualityService.passWmsGradeOfQuality(whWmsGradeOfQualityVO,whGradeVo.getWmsGradeSkuId());
        }else{
            // 驳回
            // 源逻辑仓库存释放
            scmReleaseOccupy(whGradeVo);

            // 品级调整单，行状态变更
            whGradeVo.setApproveStatus(WhGradeOfQualityVO.APPROVE_STATUS_REJECT);
            for (WhGradeOfQualityDetail whGradeDetail : whGradeVo.getWhGradeDetails()){
                whGradeDetail.setApproveStatus(WhGradeOfQualityDetailVO.APPROVE_STATUS_REJECT);
            }
            whGradeVo.setWhGradeAffectInfos(listWhGradeAffectInfoByWhGradeId(vo.getId()));
            for (WhGradeAffectInfo affectInfo : whGradeVo.getWhGradeAffectInfos()){
                affectInfo.setOptFlag(WhGradeAffectInfoVO.OPT_FLAG_CANCELED);
            }
            modifyAndFetch(whGradeVo);
            // 同步WMS
            whWmsGradeOfQualityService.rejectWmsGradeOfQuality(whWmsGradeOfQualityVO,whGradeVo.getWmsGradeSkuId());
        }
        return true;
    }

    @Override
    @Transactional
    public List<Long> batchAuditWhGradeOfQuality(List<WhGradeOfQualityVO> gradeOfQualityVOs,Long approveUserId) throws Exception {
        boolean result = false;
        List<Long> whGradeIds = new ArrayList<>();
        for (WhGradeOfQualityVO whGradeVO : gradeOfQualityVOs){
            whGradeVO.setRejectReason("审批单批量审核");
            whGradeVO.setMemo(whGradeVO.getRejectReason());
            whGradeVO.setOptPass(true);
            whGradeVO.setApproveUserId(approveUserId);
            result = auditWhGradeOfQuality(whGradeVO);
            if (!result){
                throw new WarehouseException(WarehouseExceptionErrorCode.RESULT_NOT_EXPECTED,
                        "批量品级调整失败,请刷新后重新处理!");
            }
            whGradeIds.add(whGradeVO.getId());
        }
        return whGradeIds;
    }

    public boolean updateScmStock(WhGradeOfQualityVO vo){
        boolean result = false;
        List<WhGradeTmallInfo> tmallInfos = new ArrayList<>();
        if (CollectionUtils.isNotEmpty(vo.getWhGradeDetails())){
            // 逻辑仓出库
            for (WhGradeOfQualityDetail gradeDetail : vo.getWhGradeDetails()){
                WhInvRcd whInvRcd = new WhInvRcd();
                whInvRcd.setCommandCode(vo.getCode());
                whInvRcd.setWarehouseCode(gradeDetail.getSourceWarehouseCode());
                whInvRcd.setInOutType(WhInvRcd.TYPE_GRADE_ADJUST_OUT);
                whInvRcd.setSkuCode(gradeDetail.getSkuCode());
                whInvRcd.setQuantity(-gradeDetail.getAdjustAmount());
                whInvRcd.setSubmitUserId(vo.getApproveUserId());
                whInvRcd.setMemo(vo.getAdjustReason()+vo.getMemo());
                whInvRcd.setSubmitTime(Calendar.getInstance().getTime());
                whInvService.record(whInvRcd);
                if (WhGradeOfQualityVO.WAREHOUSE_CHANNEL_TMALL_MAP.containsKey(gradeDetail.getSourceWarehouseCode())){
                    WhGradeTmallInfo tmallInfo = new WhGradeTmallInfo();
                    tmallInfo.setSkuCode(gradeDetail.getSkuCode());
                    tmallInfo.setChannelCode(WhGradeOfQualityVO.WAREHOUSE_CHANNEL_TMALL_MAP.get(gradeDetail.getSourceWarehouseCode()));
                    tmallInfo.setRefCode(vo.getCode());
                    tmallInfo.setQuality(-gradeDetail.getAdjustAmount());
                    tmallInfo.setWarehoseCode(gradeDetail.getSourceWarehouseCode());
                    tmallInfos.add(tmallInfo);
                }
            }
            // 逻辑仓入库
            WhInvRcd whInvRcd = new WhInvRcd();
            whInvRcd.setCommandCode(vo.getCode());
            whInvRcd.setWarehouseCode(vo.getWhGradeDetails().get(0).getTargetWarehouseCode());
            whInvRcd.setInOutType(WhInvRcd.TYPE_GRADE_ADJUST_IN);
            whInvRcd.setSkuCode(vo.getSkuCode());
            whInvRcd.setQuantity(vo.getAdjustAmount());
            whInvRcd.setSubmitUserId(vo.getApproveUserId());
            whInvRcd.setMemo(vo.getAdjustReason()+vo.getMemo());
            whInvRcd.setSubmitTime(Calendar.getInstance().getTime());
            result = whInvService.record(whInvRcd);

            if ((WhGradeOfQualityVO.WAREHOUSE_CHANNEL_TMALL_MAP.containsKey(whInvRcd.getWarehouseCode()))){
                WhGradeTmallInfo tmallInfo = new WhGradeTmallInfo();
                tmallInfo.setSkuCode(whInvRcd.getSkuCode());
                tmallInfo.setChannelCode(WhGradeOfQualityVO.WAREHOUSE_CHANNEL_TMALL_MAP.get(whInvRcd.getWarehouseCode()));
                tmallInfo.setRefCode(vo.getCode());
                tmallInfo.setQuality(whInvRcd.getQuantity());
                tmallInfo.setWarehoseCode(whInvRcd.getWarehouseCode());
                syncTmallStocks(Collections.singletonList(tmallInfo),null,vo.getApproveUserId());
            }

            // 同步天猫库存
            if (CollectionUtils.isNotEmpty(tmallInfos)){
                syncTmallStocks(tmallInfos,vo.getId(),vo.getApproveUserId());
            }

            //同步jd库存
            trySyncJdStock(vo);
        }
        return result;
    }

    private void trySyncJdStock(WhGradeOfQualityVO vo){
        List<WhGradeTmallInfo> tmallInfos = new ArrayList<>();
        PegasusUtilFacade utilFacade = PegasusUtilFacade.getInstance();
        if(EmptyUtil.isNotEmpty(vo.getWhGradeDetails())){
            //逻辑仓出库
            for (WhGradeOfQualityDetail gradeDetail : vo.getWhGradeDetails()){
                if(utilFacade.needSyncJdStock(gradeDetail.getSourceWarehouseCode())){
                    WhGradeTmallInfo tmallInfo = new WhGradeTmallInfo();
                    tmallInfo.setSkuCode(gradeDetail.getSkuCode());
                    tmallInfo.setRefCode(vo.getCode());
                    tmallInfo.setQuality(-gradeDetail.getAdjustAmount());
                    tmallInfo.setWarehoseCode(gradeDetail.getSourceWarehouseCode());
                    tmallInfos.add(tmallInfo);
                }
            }
            // 逻辑仓入库
            if(utilFacade.needSyncJdStock(vo.getWhGradeDetails().get(0).getTargetWarehouseCode())){
                WhGradeTmallInfo tmallInfo = new WhGradeTmallInfo();
                tmallInfo.setSkuCode(vo.getSkuCode());
                tmallInfo.setRefCode(vo.getCode());
                tmallInfo.setQuality(vo.getAdjustAmount());
                tmallInfo.setWarehoseCode(vo.getWhGradeDetails().get(0).getTargetWarehouseCode());
                tmallInfos.add(tmallInfo);
            }
        }
        if(EmptyUtil.isEmpty(tmallInfos)){
            return;
        }
        //按仓库sku聚合
        Map<String,Map<String,Integer>> whSkuQuantityMap = new HashMap<>();
        for(WhGradeTmallInfo tmallInfo : tmallInfos){
            Map<String,Integer> skuQuantityMap = whSkuQuantityMap.get(tmallInfo.getWarehoseCode());
            if(NullUtil.isNull(skuQuantityMap)){
                skuQuantityMap = new HashMap<>();
                whSkuQuantityMap.put(tmallInfo.getWarehoseCode(),skuQuantityMap);
            }
            Integer total = skuQuantityMap.get(tmallInfo.getSkuCode());
            if(NullUtil.isNull(total)){
                total = 0;
            }
            total += tmallInfo.getQuality();
            skuQuantityMap.put(tmallInfo.getSkuCode(),total);
        }
        for(Map.Entry<String,Map<String,Integer>> entry : whSkuQuantityMap.entrySet()){
            List<String> referenceCodes = new LinkedList<String>();
            referenceCodes.add(vo.getCode());
            whCommandService.syncJdSkuStock(entry.getKey(),entry.getValue()
                    ,Constants.STOCK_SYNC_TYPE_GRADE
                    ,referenceCodes,vo.getApproveUserId());
        }

    }

    private void scmReleaseOccupy(WhGradeOfQualityVO vo){
        if (CollectionUtils.isEmpty(vo.getWhGradeDetails())){
            vo = findWhGradeOfQualityVOById(vo.getId(),true,false);
        }
        for (WhGradeOfQualityDetail gradeDetail : vo.getWhGradeDetails()){
            // 释放占用
            WhReleaseOccupationVO whReleaseOccupationVO = new WhReleaseOccupationVO();
            whReleaseOccupationVO.setOccupyType(WhInvOccupy.TYPE_GRADE_ADJUST_OCCUPY);
            whReleaseOccupationVO.setReferenceCode(vo.getCode()+"_"+gradeDetail.getId());
            whInvService.releaseOccupation(whReleaseOccupationVO);
        }
    }

    private void scmOccupy(WhGradeOfQualityVO vo){
        List<SOccupyResultVO> rollbackOccupyList = new ArrayList<>();
        for (WhGradeOfQualityDetail gradeDetail : vo.getWhGradeDetails()) {
            SStockOccupyDTO occupyDTO = new SStockOccupyDTO();
            occupyDTO.setWarehouseCode(gradeDetail.getSourceWarehouseCode());
            occupyDTO.setOccupyType(SStockOccupyTypeEnum.getEnumByCode(SStockOccupyTypeEnum.GRADE_ADJUST_OUT.getCode()));
            occupyDTO.setOperationType(SStockOperationTypeEnum.DEFAULT);
            occupyDTO.setReferenceCode(vo.getCode()+"_"+gradeDetail.getId());
            occupyDTO.setSkuCode(gradeDetail.getSkuCode());
            occupyDTO.setQuantity(gradeDetail.getAdjustAmount());
            occupyDTO.setNeedToCheckStock(false);
            ServiceResp<SOccupyResultVO> serviceResp = sStockService.occupy(occupyDTO);
            //成功记录，接下来可能需要手动回滚
            if (serviceResp.isSuccess()) {
                rollbackOccupyList.add(serviceResp.getBean());
            } else {
                throw  new WarehouseException(serviceResp.getRespCode(), serviceResp.getRespMsg());
            }
        }
        vo.setRollbackOccupyList(rollbackOccupyList);
    }

    @Override
    @Transactional
    public boolean update(WhGradeOfQualityVO vo) {
        WhGradeOfQuality gradeOfQuality = BeanUtil.buildFrom(vo,WhGradeOfQuality.class);
        int result = whGradeOfQualityMapper.updateByPrimaryKeySelective(gradeOfQuality);
        if (result > 0){
            if (deleteGradeDetails(vo.getId())){
                insertGradeDetails(vo);
            }
        }
        return result > 0;
    }

    @Override
    @Transactional
    public boolean modifyAndFetch(WhGradeOfQualityVO vo) {
        WhGradeOfQuality gradeOfQuality = BeanUtil.buildFrom(vo,WhGradeOfQuality.class);
        int result = whGradeOfQualityMapper.updateByPrimaryKeySelective(gradeOfQuality);
        if (result > 0){
            if (CollectionUtils.isNotEmpty(vo.getWhGradeDetails())){
                for (WhGradeOfQualityDetail whGradeDetail : vo.getWhGradeDetails()){
                    whGradeOfQualityDetailMapper.updateByPrimaryKeySelective(whGradeDetail);
                }
            }
            if (CollectionUtils.isNotEmpty(vo.getWhGradeAffectInfos())){
                for (WhGradeAffectInfo affectInfo : vo.getWhGradeAffectInfos()){
                    whGradeAffectInfoMapper.updateByPrimaryKeySelective(affectInfo);
                }
            }
        }
        return result > 0;
    }

    public boolean deleteGradeDetails(Long gradeId){
        WhGradeOfQualityDetailCond detailCond = new WhGradeOfQualityDetailCond();
        detailCond.setRefId(gradeId);
        return deleteGradeDetail(detailCond);
    }

    public boolean deleteGradeDetail(WhGradeOfQualityDetailCond detailCond){
        WhGradeOfQualityDetailExample example = buildWhGradeOfQualityDetailExampleByCond(detailCond);
        return whGradeOfQualityDetailMapper.deleteByExample(example) > 0;
    }

    private void insertGradeDetails(WhGradeOfQualityVO vo){
        if (CollectionUtils.isNotEmpty(vo.getWhGradeDetails())){
            for (WhGradeOfQualityDetail gradeDetail : vo.getWhGradeDetails()){
                gradeDetail.setRefId(vo.getId());
                whGradeOfQualityDetailMapper.insert(gradeDetail);
            }
        }
    }

    private void insertGradeAffectInfos(WhGradeOfQualityVO vo){
        if (CollectionUtils.isNotEmpty(vo.getWhGradeAffectInfos())){
            for (WhGradeAffectInfo affectInfo : vo.getWhGradeAffectInfos()){
                affectInfo.setWhGradeId(vo.getId());
                whGradeAffectInfoMapper.insert(affectInfo);
            }
        }
    }

    @Override
    @Transactional
    public boolean updateWhGradeOfQualityDetail(WhGradeOfQualityDetail whGradeDetail) {
        return whGradeOfQualityDetailMapper.updateByPrimaryKeySelective(whGradeDetail)>0;
    }

    @Override
    @Transactional
    public boolean batchCreateWhGradeOfQuality(List<WhGradeOfQuality> gradeOfQualitys) {
        return whGradeOfQualityCustomMapper.batchInsertWhGradeOfQuality(gradeOfQualitys) > 0;
    }

    @Override
    @Transactional
    public boolean batchCreateWhGradeOfQualityDetail(List<WhGradeOfQualityDetail> gradeOfQualityDetails) {
        return whGradeOfQualityCustomMapper.batchInsertWhGradeOfQualityDetail(gradeOfQualityDetails) > 0;
    }

    @Override
    public boolean batchDeleteWhGradeOfQualityDetailByCond(WhGradeOfQualityDetailCond cond) {
        return deleteGradeDetail(cond);
    }

    public void syncTmallStocks(final List<WhGradeTmallInfo> tmallInfos,final Long whGradeId,Long operatorId){
        String result = "";
        for (WhGradeTmallInfo tmallInfo : tmallInfos){
            try{
                Long skuNumId = PegasusUtilFacade.getInstance().getSkuNumIid(tmallInfo.getSkuCode(),tmallInfo.getChannelCode());
                result = whCommandService.synCommodityStocks(skuNumId,tmallInfo.getSkuCode(),tmallInfo.getQuality(),tmallInfo.getChannelCode(),8,tmallInfo.getRefCode(),operatorId);

                // 同步天猫后，更新是否成功标志
                if(NullUtil.isNotNull(whGradeId)){
                    updateWhGradeAffectInfoAfterSyncTmall(EmptyUtil.isNotEmpty(result),whGradeId,tmallInfo.getWarehoseCode());
                }

            }catch (Exception e){
                if(NullUtil.isNotNull(whGradeId)){
                    updateWhGradeAffectInfoAfterSyncTmall(false,whGradeId,tmallInfo.getWarehoseCode());
                }
                e.printStackTrace();
                log.error(e.getMessage());
            }
        }
    }

    /**
     * 同步天猫库存后，记录同步成功标志
     * @param success
     * @param whGradeId
     * @param warehouseCode
     */
    private void updateWhGradeAffectInfoAfterSyncTmall(boolean success,Long whGradeId,String warehouseCode){
        WhGradeAffectInfo affectInfo = new WhGradeAffectInfo();
        if (success){
            affectInfo.setSyncResult(WhGradeAffectInfoVO.SYNC_RESULT_SUCCESS);
        }else{
            affectInfo.setSyncResult(WhGradeAffectInfoVO.SYNC_RESULT_FAILURE);
        }
        WhGradeAffectInfoCond affectCond = new WhGradeAffectInfoCond();
        affectCond.setWhGradeId(whGradeId);
        affectCond.setWarehouseCode(warehouseCode);
        affectCond.setSyncResult(WhGradeAffectInfoVO.SYNC_RESULT_DEFAULT);
        boolean result = updateWhGradeAffectInfoByExampleSelective(affectInfo,affectCond);
        log.info("更新结果:"+result);
    }

    /**
     * 商品品级调整申请 邮件 [quality@thebeastshop.com]
     */
    @Override
    public void gradeAdjustApplySendMail(final List<WhGradeOfQualityVO> whGradeVOs){
        gradeAdjustScheduler.execute(new Runnable() {
            @Override
            public void run() {
                sendMailGradeAdjustApply(whGradeVOs);
            }
        });
    }

    /**
     * 商品品级调整申请 邮件 [quality@thebeastshop.com]
     * @param whGradeVOs
     */
    private void sendMailGradeAdjustApply(List<WhGradeOfQualityVO> whGradeVOs) {
        if (EmptyUtil.isEmpty(whGradeVOs)){
            return;
        }
        final CommGlobalConfig config = PegasusUtilFacade.getInstance().findConfigByKey(WhGradeOfQualityVO.GRADE_ADJUST_APPLY_EMAIL);
        if(EmptyUtil.isEmpty(config) || EmptyUtil.isEmpty(config.getConfigValue())) {
            return;
        }
        String configValue = config.getConfigValue();
        StringBuffer emailContent = new StringBuffer();
        emailContent.append("<html><style>#table-5 thead th {background-color: rgb(156, 186, 95);color: #fff;border-bottom-width: 0;}");
        emailContent.append("#table-5 td {color: #000;}");
        emailContent.append("#table-5 tr, #table-5 th {border-width: 1px;border-style: solid;border-color: rgb(156, 186, 95);}");
        emailContent.append("#table-5 td, #table-5 th {padding: 5px 10px;font-size: 12px;font-family: Verdana;font-weight: bold;}</style>");
        emailContent.append("<table id='table-5'><thead><th>品级调整CODE</th><th>源品级</th><th>目标品级</th><th>SKU</th><th>调整数量</th><th>调整原因</th></thead>");
        emailContent.append("<tbody>");
        for (WhGradeOfQualityVO whGradeVO : whGradeVOs) {
            emailContent.append("<tr>");
            emailContent.append("<td>" + whGradeVO.getCode() + "</td>");
            emailContent.append("<td>" + whGradeVO.getSourceGradeStr() + "</td>");
            emailContent.append("<td>" + whGradeVO.getTargetGradeStr() + "</td>");
            emailContent.append("<td>[" + whGradeVO.getSkuCode() + "]"+whGradeVO.getSkuName()+"</td>");
            emailContent.append("<td>" + whGradeVO.getAdjustAmount() + "</td>");
            emailContent.append("<td>" + whGradeVO.getAdjustReason() + "</td>");
            emailContent.append("</tr>");
        }
        emailContent.append("</tbody></table></html>");
        // 邮件VO
        EmailVO emailVO = new EmailVO();
        emailVO.setToAddressList(Arrays.asList(configValue.split(",")));
        emailVO.setSubject("商品品级调整申请");
        emailVO.setContent(emailContent.toString());
        // 发邮件
        if (CollectionUtils.isNotEmpty(emailVO.getToAddressList())) {
            // 正式环境发邮件
            EmailUtil.getInstance().send(emailVO);
        }
    }

    /**
     * 商品品级调整导致订单缺货 邮件 [发给配置邮箱 及 渠道邮箱]
     * @param whGradeVO
     * @param affectInfoVOs
     */
    @Override
    public void stockOutByGradeAdjustSendMail(final WhGradeOfQualityVO whGradeVO,final List<WhGradeAffectInfoVO> affectInfoVOs){
        log.info("stockOutByGradeAdjustSendMail Thread="+Thread.currentThread().getName()+"  whGradeId="+whGradeVO.getId());
        gradeAdjustScheduler.execute(new Runnable() {
            @Override
            public void run() {
                sendMailStockOutByGradeAdjust(whGradeVO,affectInfoVOs,null);
            }
        });
    }

    /**
     * 商品品级调整导致订单缺货 [发给配置邮箱 及 渠道邮箱]
     * @param whGradeVO
     * @param affectInfoVOs
     */
    public void sendMailStockOutByGradeAdjust(final WhGradeOfQualityVO whGradeVO,final List<WhGradeAffectInfoVO> affectInfoVOs,
            Map<String,WhGradeAffectInfoVO> waitSendEmailMap) {
        if (EmptyUtil.isEmpty(whGradeVO) || CollectionUtils.isEmpty(affectInfoVOs)){
            return;
        }
        final CommGlobalConfig config = PegasusUtilFacade.getInstance().findConfigByKey(WhGradeOfQualityVO.GRADE_ADJUST_STOCK_OUT_EMAIL);
        StringBuilder emailContent = new StringBuilder();
        emailContent.append("<html><style>#table-5 thead th {background-color: rgb(156, 186, 95);color: #fff;border-bottom-width: 0;}");
        emailContent.append("#table-5 td {color: #000;}");
        emailContent.append("#table-5 tr, #table-5 th {border-width: 1px;border-style: solid;border-color: rgb(156, 186, 95);}");
        emailContent.append("#table-5 td, #table-5 th {padding: 5px 10px;font-size: 12px;font-family: Verdana;font-weight: bold;}</style>");
        emailContent.append("<table id='table-5'><thead><th>订单号</th><th>包裹号</th><th>SKU</th><th>采购员</th><th>缺货数量</th><th>变动原因</th><th>省市区</th><th>逻辑仓所属分组名</th></thead>");
        emailContent.append("<tbody>");
        List<Long> gradeAffectIds = new ArrayList<>();
        StringBuffer allEmailContent = new StringBuffer();
        Map<String,List<WhGradeAffectInfoVO>> gradeAffectMap = new HashMap<>();

        // 根据sku获取 可用库存的逻辑仓分组 (分组非仓库、逻辑仓商品状态：良品)
        Map<String,String> skuWhGroupName = buildAvailableStockWarehouseGroupBySku(whGradeVO.getSkuCode());
        // 构建全部行数据，发给配置收件人
        buildStockOutGradeAdjust(allEmailContent,affectInfoVOs,gradeAffectIds,gradeAffectMap,whGradeVO,skuWhGroupName);

        // 针对对应渠道邮箱的信息
        Map<String,StringBuffer> channelEmailContentMap = new HashMap<>();
        if (gradeAffectMap.size() > 0){
            for (Map.Entry<String, List<WhGradeAffectInfoVO>> entry : gradeAffectMap.entrySet()) {
                StringBuffer channelEmailContent = channelEmailContentMap.get(entry.getKey());
                if (EmptyUtil.isEmpty(channelEmailContent)){
                    channelEmailContent = new StringBuffer();
                    channelEmailContentMap.put(entry.getKey(),channelEmailContent);
                }
                buildStockOutGradeAdjust(channelEmailContent,entry.getValue(),null,null,whGradeVO,skuWhGroupName);
            }
            /*for (Map.Entry<String,StringBuffer> channelContentEntry : channelEmailContentMap.entrySet()){
                channelContentEntry.getValue().append("</tbody></table></html>");
            }*/
        }

        String tableBottom = "</tbody></table></html>";
        // 发送给 配置邮箱邮件
        if(EmptyUtil.isNotEmpty(config) && EmptyUtil.isNotEmpty(config.getConfigValue())) {
            List<String> toAddressList = Arrays.asList(config.getConfigValue().split(","));
            if (whGradeVO.isSystemGradeEmail() && waitSendEmailMap != null){
                buildWaitSendGradeEmailObj(waitSendEmailMap,WhGradeAffectInfoVO.SEND_EMAIL_ALL,
                        toAddressList,config.getConfigValue(),emailContent.toString(),allEmailContent.toString(),gradeAffectIds);
            }else{
                allEmailContent.append(tableBottom);
                //sendStockOutEmailForGradeAdjust(toAddressList,emailContent.toString()+allEmailContent.toString(),gradeAffectIds);
                // 此处不再更新，统一定时30分发送库存变化时 更新
                log.info("All Thread="+Thread.currentThread().getName()+"  gradeAffectIds="+gradeAffectIds.toString());
                sendStockOutEmailForGradeAdjust(toAddressList,emailContent.toString()+allEmailContent.toString(),null);
            }
        }

        //  针对 渠道发送对应的行信息
        if (channelEmailContentMap.size() > 0){
            if (whGradeVO.isSystemGradeEmail() && waitSendEmailMap != null){
                for (Map.Entry<String,StringBuffer> entry : channelEmailContentMap.entrySet()){
                    buildWaitSendGradeEmailObj(waitSendEmailMap,WhGradeAffectInfoVO.SEND_EMAIL_CHANNEL+"-"+entry.getKey(),
                            Arrays.asList(entry.getKey().split(",")),entry.getKey(),emailContent.toString(),entry.getValue().toString(),null);
                }
            }else{
                for (Map.Entry<String,StringBuffer> entry : channelEmailContentMap.entrySet()){
                    entry.getValue().append(tableBottom);
                    log.info("Channel Thread="+Thread.currentThread().getName()+"  emailAddressList="+entry.getKey().toString());
                    sendStockOutEmailForGradeAdjust(Arrays.asList(entry.getKey().split(",")),
                            emailContent.toString()+entry.getValue().toString(),null);
                }
            }
        }
    }

    private void buildWaitSendGradeEmailObj(Map<String,WhGradeAffectInfoVO> waitSendEmailMap,String keyFlag,
                                                    List<String> toAddressList,String emailAddressKey, String emailHead,String emailContent,List<Long> affectIds){
        WhGradeAffectInfoVO emailObj = waitSendEmailMap.get(keyFlag);
        if (EmptyUtil.isNotEmpty(emailObj)){
            emailObj.setWaitSendContent(emailObj.getWaitSendContent().append(emailContent));
        }else{
            emailObj = new WhGradeAffectInfoVO();
            emailObj.setToAddressList(toAddressList);
            emailObj.setEmailAddressKey(emailAddressKey);
            emailObj.setWaitSendHead(emailHead);
            emailObj.setWaitSendContent(new StringBuilder(emailContent));
            emailObj.setWaitSendBottom("</tbody></table></html>");
            emailObj.setWaitUpdateAffectIds(affectIds);
            waitSendEmailMap.put(keyFlag,emailObj);
        }
    }

    public void sendStockOutEmailForGradeAdjust(List<String> toAddressList,String emailContent,List<Long> affectIds){
        EmailVO emailVO = new EmailVO();
        emailVO.setToAddressList(toAddressList);
        emailVO.setSubject("商品品级调整导致订单缺货");
        emailVO.setContent(emailContent);
        // 发邮件
        if (CollectionUtils.isNotEmpty(emailVO.getToAddressList())) {
            // 正式环境发邮件
            try {
                log.info("开始发送 商品品级调整导致订单缺货 邮件：" + emailVO.getToAddressList());
                EmailUtil.getInstance().send(emailVO);
                // 无需更新
                /*boolean result = EmailUtil.getInstance().send(emailVO);
                if (result && CollectionUtils.isNotEmpty(affectIds)){
                    updateWhGradeAffectInfoSendMailFlag(affectIds,WhGradeAffectInfoVO.SEND_MAIL_FLAG_YES,
                            WhGradeAffectInfoVO.SEND_MAIL_FLAG_NO);
                }*/
                log.info("成功发送 商品品级调整导致订单缺货 邮件：" + emailVO.getToAddressList());
            }catch (Exception e){
                e.printStackTrace();
            }
        }
    }

    private void buildStockOutGradeAdjust(StringBuffer emailContent, List<WhGradeAffectInfoVO> affectInfoVOs,List<Long> affectIds,
                                      Map<String,List<WhGradeAffectInfoVO>> gradeAffectMap,WhGradeOfQualityVO whGradeVO,
                                          Map<String,String> skuWhGroupName){
        for (WhGradeAffectInfoVO affectInfoVO : affectInfoVOs) {
            emailContent.append("<tr>");
            emailContent.append("<td>" + affectInfoVO.getSalesOrderCode() + "</td>");
            emailContent.append("<td>" + affectInfoVO.getAffectCode() + "</td>");
            emailContent.append("<td>[" + whGradeVO.getSkuCode() + "]"+whGradeVO.getSkuName()+"</td>");
            emailContent.append("<td>" + whGradeVO.getBuyerName() + "</td>");
            emailContent.append("<td>" + affectInfoVO.getAmount() + "</td>");
            emailContent.append("<td>" + whGradeVO.getAdjustReason() + "</td>");
            emailContent.append("<td>" + removeEmpty(affectInfoVO.getFullAddress()) + "</td>");
            emailContent.append("<td>" + removeEmpty(skuWhGroupName.get(whGradeVO.getSkuCode())) + "</td>");
            emailContent.append("</tr>");
            if (affectIds != null){
                affectIds.add(affectInfoVO.getId());
            }
            // 以渠道邮箱分组
            if (gradeAffectMap != null){
                if (EmptyUtil.isNotEmpty(affectInfoVO.getChannelNoticeEmails())){
                    List<WhGradeAffectInfoVO> gradeAffectVOList = gradeAffectMap.get(affectInfoVO.getChannelNoticeEmails());
                    if (CollectionUtils.isEmpty(gradeAffectVOList)){
                        gradeAffectVOList = new ArrayList<>();
                        gradeAffectMap.put(affectInfoVO.getChannelNoticeEmails(),gradeAffectVOList);
                    }
                    gradeAffectVOList.add(affectInfoVO);
                }
            }
        }
    }

    private String removeEmpty(String value){
        if (EmptyUtil.isNotEmpty(value)){
            return value;
        }
        return "";
    }

    private Map<String,String> buildAvailableStockWarehouseGroupBySku(String skuCode){
        // SKU可用逻辑仓分组=[仓库分组类型：非仓库],[逻辑仓商品状态：良品],[库存可用]
        Map<String,String> skuWarehouseGroupNameMap = new HashMap<>();
        Map<String, WhInvVO> skuInvMap = whInvService.findCanUseQttBySkuCode(skuCode);
        if (EmptyUtil.isNotEmpty(skuInvMap)){
            Set<String> warehouseSet = new HashSet<>();
            // 单个sku对应的 逻辑仓库存列表
            for(Map.Entry<String, WhInvVO> whSkuInvEntry : skuInvMap.entrySet()){
                // 获取有可用库存的逻辑仓
                if (whSkuInvEntry.getValue().getCanUseInv() > 0){
                    warehouseSet.add(whSkuInvEntry.getKey());
                }
            }
            WhWarehouseGroupCond groupCond = new WhWarehouseGroupCond();
            groupCond.setRelateWhCodes(new ArrayList<>(warehouseSet));
            groupCond.setNotType(WhWarehouseGroupVO.WAREHOUSE_YES);
            groupCond.setCommodityStatus(WhWarehouseVO.COMMODITY_STATUS_FOR_NONDEFECTIVE);
            List<WhWarehouseGroupVO> groupVOs = whWarehouseGroupService.findWarehouseGroupByWhCodes(groupCond);
            if (CollectionUtils.isNotEmpty(groupVOs)){
                String groupName = "";
                for (WhWarehouseGroupVO groupVO : groupVOs){
                    groupName += groupVO.getName()+",";
                }
                groupName = groupName.substring(0,groupName.lastIndexOf(","));
                skuWarehouseGroupNameMap.put(skuCode,groupName);
            }
        }
        return skuWarehouseGroupNameMap;
    }

    //获取仓库sku库存
    private List<Map<String, WhInvVO>> getSkuWhInvBySkuCodes(List<String> skuCodes){
        List<Map<String, WhInvVO>> invMapList = new ArrayList<>();
        for(String skuCode : skuCodes){
            Map<String, WhInvVO> skuInvMap = whInvService.findCanUseQttBySkuCode(skuCode);
            if(EmptyUtil.isNotEmpty(skuInvMap)){
                invMapList.add(skuInvMap);
            }
        }
        return invMapList;
    }
    /**
     * 商品品级调整导致单据异常 邮件
     * @param whGradeVO
     * @param affectInfoVOs
     */
    @Override
    public void orderExceptionByGradeAdjustSendMail(final WhGradeOfQualityVO whGradeVO,final List<WhGradeAffectInfoVO> affectInfoVOs){
        gradeAdjustScheduler.execute(new Runnable() {
            @Override
            public void run() {
                sendMailOrderExceptionByGradeAdjust(whGradeVO,affectInfoVOs,null);
            }
        });
    }

    /**
     * 商品品级调整导致单据异常 邮件 发给[planning@thebeastshop.com]
     * @param whGradeVO
     * @param affectInfoVOs
     */
    public void sendMailOrderExceptionByGradeAdjust(final WhGradeOfQualityVO whGradeVO,final List<WhGradeAffectInfoVO> affectInfoVOs,
                                                                               Map<String,WhGradeAffectInfoVO> waitSendEmailMap) {
        if (EmptyUtil.isEmpty(whGradeVO) || CollectionUtils.isEmpty(affectInfoVOs)){
            return;
        }
        final CommGlobalConfig config = PegasusUtilFacade.getInstance().findConfigByKey(WhGradeOfQualityVO.GRADE_ADJUST_ORDER_EXCEPTION_EMAIL);
        if(EmptyUtil.isEmpty(config) || EmptyUtil.isEmpty(config.getConfigValue())) {
            return;
        }
        String configValue = config.getConfigValue();
        StringBuffer emailContent = new StringBuffer();
        emailContent.append("<html><style>#table-5 thead th {background-color: rgb(156, 186, 95);color: #fff;border-bottom-width: 0;}");
        emailContent.append("#table-5 td {color: #000;}");
        emailContent.append("#table-5 tr, #table-5 th {border-width: 1px;border-style: solid;border-color: rgb(156, 186, 95);}");
        emailContent.append("#table-5 td, #table-5 th {padding: 5px 10px;font-size: 12px;font-family: Verdana;font-weight: bold;}</style>");
        emailContent.append("<table id='table-5'><thead><th>单据号</th><th>SKU</th><th>采购员</th><th>缺货数量</th></thead>");
        emailContent.append("<tbody>");
        StringBuffer allEmailContent = new StringBuffer();
        List<Long> ids = new ArrayList<>();
        for (WhGradeAffectInfoVO affectInfoVO : affectInfoVOs) {
            allEmailContent.append("<tr>");
            allEmailContent.append("<td>" + affectInfoVO.getAffectCode() + "</td>");
            allEmailContent.append("<td>[" + whGradeVO.getSkuCode() + "]"+whGradeVO.getSkuName()+"</td>");
            allEmailContent.append("<td>" + whGradeVO.getBuyerName() + "</td>");
            allEmailContent.append("<td>" + affectInfoVO.getAmount() + "</td>");
            allEmailContent.append("</tr>");
            ids.add(affectInfoVO.getId());
        }
        if (whGradeVO.isSystemGradeEmail() && waitSendEmailMap != null){
            buildWaitSendGradeEmailObj(waitSendEmailMap,WhGradeAffectInfoVO.SEND_EMAIL_ALL,
                    Arrays.asList(configValue.split(",")),configValue,emailContent.toString(),allEmailContent.toString(),ids);
        }else{
            // 邮件VO
            allEmailContent.append("</tbody></table></html>");
            // sendOrderExceptionEmailForGradeAdjust(Arrays.asList(configValue.split(",")),emailContent.toString()+allEmailContent.toString(),ids);
            sendOrderExceptionEmailForGradeAdjust(Arrays.asList(configValue.split(",")),emailContent.toString()+allEmailContent.toString(),null);
        }
    }

    @Override
    public void sendOrderExceptionEmailForGradeAdjust(List<String> toAddressList,String emailContent,List<Long> affectIds) {
        // 邮件VO
        EmailVO emailVO = new EmailVO();
        emailVO.setToAddressList(toAddressList);
        emailVO.setSubject("商品品级调整导致单据异常");
        emailVO.setContent(emailContent);
        // 发邮件
        if (CollectionUtils.isNotEmpty(emailVO.getToAddressList())) {
            // 正式环境发邮件
            boolean result = EmailUtil.getInstance().send(emailVO);
            if (result && CollectionUtils.isNotEmpty(affectIds)){
                // affectIds 传值为null,不用在此处更新
                /*updateWhGradeAffectInfoSendMailFlag(affectIds,WhGradeAffectInfoVO.SEND_MAIL_FLAG_YES,
                        WhGradeAffectInfoVO.SEND_MAIL_FLAG_NO);*/
            }
        }
    }

        /**
         * 商品品级调整导致库存变化 邮件 发给[逻辑仓邮箱 和 planning@thebeastshop.com]
         * @param whGradeVO
         * @param affectInfoVOs
         */
    @Override
    public void stockChangeByGradeAdjustSendMail(final WhGradeOfQualityVO whGradeVO,final List<WhGradeAffectInfoVO> affectInfoVOs){
        gradeAdjustScheduler.execute(new Runnable() {
            @Override
            public void run() {
                sendMailStockChangeByGradeAdjust(whGradeVO,affectInfoVOs,null);
            }
        });
    }

    /**
     * 商品品级调整导致库存变化 邮件 发给[逻辑仓邮箱 和 planning@thebeastshop.com]
     * @param whGradeVO
     * @param affectInfoVOs
     */
    @Override
    public void sendMailStockChangeByGradeAdjust(final WhGradeOfQualityVO whGradeVO,final List<WhGradeAffectInfoVO> affectInfoVOs,
                                                 Map<String,WhGradeAffectInfoVO> waitSendEmailMap) {
        if (EmptyUtil.isEmpty(whGradeVO) || CollectionUtils.isEmpty(affectInfoVOs)){
            return;
        }
        final CommGlobalConfig config = PegasusUtilFacade.getInstance().findConfigByKey(WhGradeOfQualityVO.GRADE_ADJUST_STOCK_CHANGE_EMAIL);

        StringBuilder emailContent = new StringBuilder();
        emailContent.append("<html><style>#table-5 thead th {background-color: rgb(156, 186, 95);color: #fff;border-bottom-width: 0;}");
        emailContent.append("#table-5 td {color: #000;}");
        emailContent.append("#table-5 tr, #table-5 th {border-width: 1px;border-style: solid;border-color: rgb(156, 186, 95);}");
        emailContent.append("#table-5 td, #table-5 th {padding: 5px 10px;font-size: 12px;font-family: Verdana;font-weight: bold;}</style>");
        emailContent.append("<table id='table-5'><thead><th>逻辑仓</th><th>SKU</th><th>变动数量</th><th>变动原因</th><th>天猫库存同步结果</th></thead>");
        emailContent.append("<tbody>");

        StringBuffer allEmailContent = new StringBuffer();
        Map<String,List<WhGradeAffectInfoVO>> gradeAffectMap = new HashMap<>();
        List<Long> affectIds = whGradeVO.getWhAffectGradeIds();
        // 构建全部行数据，发给配置收件人
        buildStockChangeGradeAdjust(allEmailContent,affectInfoVOs,gradeAffectMap,whGradeVO);

        // 针对 对应逻辑仓邮箱的信息
        Map<String,StringBuffer> warehouseEmailContentMap = new HashMap<>();
        if (gradeAffectMap.size() > 0){
            for (Map.Entry<String, List<WhGradeAffectInfoVO>> entry : gradeAffectMap.entrySet()) {
                StringBuffer warehouseEmailContent = warehouseEmailContentMap.get(entry.getKey());
                if (EmptyUtil.isEmpty(warehouseEmailContent)){
                    warehouseEmailContent = new StringBuffer();
                    warehouseEmailContentMap.put(entry.getKey(),warehouseEmailContent);
                }
                buildStockChangeGradeAdjust(warehouseEmailContent,entry.getValue(),null,whGradeVO);
            }
            /*for (Map.Entry<String,StringBuffer> warehouseContentEntry : warehouseEmailContentMap.entrySet()){
                warehouseContentEntry.getValue().append("</tbody></table></html>");
            }*/
        }

        String tableBottom = "</tbody></table></html>";
        // 发送给 配置邮箱邮件
        if(EmptyUtil.isNotEmpty(config) && EmptyUtil.isNotEmpty(config.getConfigValue())) {
            List<String> toAddressList = Arrays.asList(config.getConfigValue().split(","));
            if (whGradeVO.isSystemGradeEmail() && waitSendEmailMap != null){
                buildWaitSendGradeEmailObj(waitSendEmailMap,WhGradeAffectInfoVO.SEND_EMAIL_ALL,
                        toAddressList,config.getConfigValue(),emailContent.toString(),allEmailContent.toString(),affectIds);
            }else{
                allEmailContent.append(tableBottom);
                // sendStockChangeEmailForGradeAdjust(toAddressList,emailContent.toString()+allEmailContent.toString(),affectIds);
                sendStockChangeEmailForGradeAdjust(toAddressList,emailContent.toString()+allEmailContent.toString(),null);
            }
        }

        //  针对 逻辑仓邮箱 发送对应的行信息
        if (warehouseEmailContentMap.size() > 0){
            if (whGradeVO.isSystemGradeEmail() && waitSendEmailMap != null){
                for (Map.Entry<String,StringBuffer> entry : warehouseEmailContentMap.entrySet()){
                    buildWaitSendGradeEmailObj(waitSendEmailMap,WhGradeAffectInfoVO.SEND_EMAIL_WAREHOUSE+"-"+entry.getKey(),
                            Arrays.asList(entry.getKey().split(",")),entry.getKey(),emailContent.toString(),entry.getValue().toString(),null);
                }
            }else{
                for (Map.Entry<String,StringBuffer> entry : warehouseEmailContentMap.entrySet()){
                    entry.getValue().append(tableBottom);
                    sendStockChangeEmailForGradeAdjust(Arrays.asList(entry.getKey().split(",")),
                            emailContent.toString()+entry.getValue().toString(),null);
                }
            }
        }
    }

    @Override
    public void sendStockChangeEmailForGradeAdjust(List<String> toAddressList,String emailContent,List<Long> affectIds){
        EmailVO emailVO = new EmailVO();
        emailVO.setToAddressList(toAddressList);
        emailVO.setSubject("商品品级调整导致库存变化");
        emailVO.setContent(emailContent);
        // 发邮件
        if (CollectionUtils.isNotEmpty(emailVO.getToAddressList())) {
            // 正式环境发邮件
            try {
                log.info("开始发送 商品品级调整导致库存变化 邮件：" + emailVO.getToAddressList());
                boolean result = EmailUtil.getInstance().send(emailVO);
                if (result && CollectionUtils.isNotEmpty(affectIds)){
                    updateWhGradeAffectInfoSendMailFlag(affectIds,WhGradeAffectInfoVO.SEND_MAIL_FLAG_YES,
                            WhGradeAffectInfoVO.SEND_MAIL_FLAG_NO);
                }
                log.info("成功发送 商品品级调整导致库存变化 邮件：" + emailVO.getToAddressList());
            }catch (Exception e){
                e.printStackTrace();
            }
        }
    }

    private void buildStockChangeGradeAdjust(StringBuffer emailContent, List<WhGradeAffectInfoVO> affectInfoVOs,
                                          Map<String,List<WhGradeAffectInfoVO>> gradeAffectMap,WhGradeOfQualityVO whGradeVO){

        // 按仓库正序排列
        if (EmptyUtil.isNotEmpty(gradeAffectMap)){
            Collections.sort(affectInfoVOs, new Comparator<WhGradeAffectInfoVO>() {
                @Override
                public int compare(WhGradeAffectInfoVO vo1, WhGradeAffectInfoVO vo2) {
                    return vo1.getWarehouseCode().compareTo(vo2.getWarehouseCode());
                }
            });
        }
        for (WhGradeAffectInfoVO affectInfoVO : affectInfoVOs) {
            emailContent.append("<tr>");
            emailContent.append("<td>[" + affectInfoVO.getWarehouseCode() + "]"+affectInfoVO.getWarehouseName()+"</td>");
            emailContent.append("<td>[" + whGradeVO.getSkuCode() + "]"+whGradeVO.getSkuName()+"</td>");
            emailContent.append("<td>" + affectInfoVO.getAmount() + "</td>");
            emailContent.append("<td>" + whGradeVO.getAdjustReason() + "</td>");
            emailContent.append("<td>" + affectInfoVO.getSyncResultStr() + "</td>");
            emailContent.append("</tr>");

            // 以逻辑仓邮箱分组
            if (gradeAffectMap != null){
                if (EmptyUtil.isNotEmpty(affectInfoVO.getWarehouseContactEmails())){
                    List<WhGradeAffectInfoVO> gradeAffectVOList = gradeAffectMap.get(affectInfoVO.getWarehouseContactEmails());
                    if (CollectionUtils.isEmpty(gradeAffectVOList)){
                        gradeAffectVOList = new ArrayList<>();
                        gradeAffectMap.put(affectInfoVO.getWarehouseContactEmails(),gradeAffectVOList);
                    }
                    gradeAffectVOList.add(affectInfoVO);
                }
            }
        }
    }
}
