/**
 * Copyright (C), 上海布鲁爱电子商务有限公司
 */
package com.thebeastshop.pegasus.service.warehouse.service.impl;

import com.thebeastshop.pegasus.service.warehouse.cond.WhDamagedRcdCond;
import com.thebeastshop.pegasus.service.warehouse.dao.WhDamagedRcdMapper;
import com.thebeastshop.pegasus.service.warehouse.dao.WhWastedRcdMapper;
import com.thebeastshop.pegasus.service.warehouse.exception.WarehouseException;
import com.thebeastshop.pegasus.service.warehouse.exception.WarehouseExceptionErrorCode;
import com.thebeastshop.pegasus.service.warehouse.model.*;
import com.thebeastshop.pegasus.service.warehouse.service.WhAllotService;
import com.thebeastshop.pegasus.service.warehouse.service.WhDamageWasteService;
import com.thebeastshop.pegasus.service.warehouse.service.WhInfoService;
import com.thebeastshop.pegasus.service.warehouse.vo.WhWarehouseVO;
import com.thebeastshop.pegasus.util.comm.EmptyUtil;
import com.thebeastshop.pegasus.util.exception.CommExceptionErrorCode;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.*;

/**
 * @author Royan
 * @version $Id: WhDamageWasteServiceImpl.java, v 0.1 2015-07-09 下午4:19
 */
@Service("whDamageWasteService")
public class WhDamageWasteServiceImpl implements WhDamageWasteService {

    private final Logger       log = LoggerFactory.getLogger(WhDamageWasteServiceImpl.class);

    @Autowired
    private WhDamagedRcdMapper whDamagedRcdMapper;

    @Autowired
    private WhWastedRcdMapper  whWastedRcdMapper;

    @Autowired
    private WhAllotService     whAllotService;

    @Autowired
    private WhInfoService      whInfoService;

    /**
     * 记录耗损
     *
     * @param whDamagedRcd 耗损
     * @return 记录ID
     */
    @Override
    @Transactional
    public Long recordDamage(WhDamagedRcd whDamagedRcd)throws Exception {
        String skuCode = whDamagedRcd.getSkuCode();
        Integer quantity = whDamagedRcd.getQuantity();

        // 如果target仓为空，则自动填入
        if (EmptyUtil.isEmpty(whDamagedRcd.getTargetWarehouseCode())) {
            WhWarehouse targetWarehouse = whInfoService.findUniqDamagedWarehouse(whDamagedRcd
                .getSourceWarehouseCode());
            if (targetWarehouse == null) {
                throw new WarehouseException(WarehouseExceptionErrorCode.WAREHOUSE_NOT_FIND,
                    "找不到目标仓库");
            }
            whDamagedRcd.setTargetWarehouseCode(targetWarehouse.getCode());
        }

        whDamagedRcd.setCreateTime(new Date());
        whDamagedRcdMapper.insert(whDamagedRcd);

        // 创建调拨
        WhAllotRcd whAllotRcd = new WhAllotRcd();
        whAllotRcd.setAllotType(WhAllotRcd.TYPE_DAMAGED);
        whAllotRcd.setReferenceCode(whDamagedRcd.getId() + "");
        whAllotRcd.setSourceWarehouseCode(whDamagedRcd.getSourceWarehouseCode());
        whAllotRcd.setTargetWarehouseCode(whDamagedRcd.getTargetWarehouseCode());

        // 调拨行
        List<WhAllotRcdSku> whAllotRcdSkuList = new ArrayList<>();
        WhAllotRcdSku whAllotRcdSku = new WhAllotRcdSku();
        whAllotRcdSku.setSkuCode(skuCode);
        whAllotRcdSku.setQuantity(quantity);
        whAllotRcdSkuList.add(whAllotRcdSku);
        whAllotRcd.setWhAllotRcdSkuList(whAllotRcdSkuList);
        whAllotService.createAllotRcd(whAllotRcd);

        return whDamagedRcd.getId();
    }

    /**
     * 记录报废
     *
     * @param whWastedRcd 报废
     * @return 记录ID
     */
    @Override
    @Transactional
    public Long recordWaste(WhWastedRcd whWastedRcd)throws Exception {
        whWastedRcd.setCreateTime(new Date());
        whWastedRcdMapper.insert(whWastedRcd);

        // 创建调拨
        WhAllotRcd whAllotRcd = new WhAllotRcd();
        whAllotRcd.setAllotType(WhAllotRcd.TYPE_WASTED);
        whAllotRcd.setReferenceCode(whWastedRcd.getId() + "");
        whAllotRcd.setSourceWarehouseCode(whWastedRcd.getSourceWarehouseCode());
        whAllotRcd.setTargetWarehouseCode(whWastedRcd.getTargetWarehouseCode());

        // 调拨行
        List<WhAllotRcdSku> whAllotRcdSkuList = new ArrayList<>();
        WhAllotRcdSku whAllotRcdSku = new WhAllotRcdSku();
        whAllotRcdSku.setSkuCode(whWastedRcd.getSkuCode());
        whAllotRcdSku.setQuantity(whWastedRcd.getQuantity());
        whAllotRcd.setWhAllotRcdSkuList(whAllotRcdSkuList);
        whAllotService.createAllotRcd(whAllotRcd);

        return whWastedRcd.getId();
    }

    /**
     * 根据SKU编码查找耗损记录
     *
     * @param skuCode SKU编码
     * @return Map<来源仓库编码,耗损记录列表>
     */
    @Override
    public Map<String, List<WhDamagedRcd>> findDamagedRcdBySkuCode(String skuCode) {
        WhDamagedRcdCond cond = new WhDamagedRcdCond();
        cond.setIncludeSkuCode(skuCode);
        List<WhDamagedRcd> whDamagedRcdList = findDamagedRcdByCond(cond);

        Map<String, List<WhDamagedRcd>> result = new HashMap<>();
        for (WhDamagedRcd whDamagedRcd : whDamagedRcdList) {
            String sourceWarehouseCode = whDamagedRcd.getSourceWarehouseCode();
            List<WhDamagedRcd> whDamagedRcdListInMap = result.get(sourceWarehouseCode);
            if (whDamagedRcdListInMap == null) {
                whDamagedRcdListInMap = new ArrayList<>();
                result.put(sourceWarehouseCode, whDamagedRcdListInMap);
            }
            whDamagedRcdListInMap.add(whDamagedRcd);
        }

        return result;
    }

    /**
     * 根据SKU编码和来源仓库编码查找耗损记录
     *
     * @param skuCode             SKU编码
     * @param sourceWarehouseCode 来源仓库编码
     * @return 耗损记录列表
     */
    @Override
    public List<WhDamagedRcd> findDamagedRcdBySkuCodeAndWarehouseCode(String skuCode,
                                                                      String sourceWarehouseCode) {
        WhDamagedRcdCond cond = new WhDamagedRcdCond();
        cond.setIncludeSkuCode(skuCode);
        cond.setSourceWarehouseCode(sourceWarehouseCode);
        return findDamagedRcdByCond(cond);
    }

    /**
     * 根据条件查找残次记录
     *
     * @param cond 条件
     * @return 残次记录列表
     */
    @Override
    public List<WhDamagedRcd> findDamagedRcdByCond(WhDamagedRcdCond cond) {
        return whDamagedRcdMapper.findDamagedRcdByCond(cond);
    }

    /**
     * 根据SKU编码查找报废记录
     *
     * @param skuCode SKU编码
     * @return Map<来源仓库编码,报废记录列表>
     */
    @Override
    public Map<String, List<WhWastedRcd>> findWastedRcdBySkuCode(String skuCode) {
        WhWastedRcdExample example = new WhWastedRcdExample();
        example.createCriteria().andSkuCodeEqualTo(skuCode);
        example.setOrderByClause("id");
        List<WhWastedRcd> whWastedRcdList = whWastedRcdMapper.selectByExample(example);

        Map<String, List<WhWastedRcd>> result = new HashMap<>();
        for (WhWastedRcd whWastedRcd : whWastedRcdList) {
            String sourceWarehouseCode = whWastedRcd.getSourceWarehouseCode();
            List<WhWastedRcd> whWastedRcdListInMap = result.get(sourceWarehouseCode);
            if (whWastedRcdListInMap == null) {
                whWastedRcdListInMap = new ArrayList<>();
                result.put(sourceWarehouseCode, whWastedRcdListInMap);
            }
            whWastedRcdListInMap.add(whWastedRcd);
        }

        return result;
    }

    /**
     * 根据SKU编码和来源仓库编码查找报废记录
     *
     * @param skuCode             SKU编码
     * @param sourceWarehouseCode 来源仓库编码
     * @return 报废记录列表
     */
    @Override
    public List<WhWastedRcd> findWastedRcdBySkuCodeAndWarehouseCode(String skuCode,
                                                                    String sourceWarehouseCode) {
        WhWastedRcdExample example = new WhWastedRcdExample();
        example.createCriteria().andSkuCodeEqualTo(skuCode)
            .andSourceWarehouseCodeEqualTo(sourceWarehouseCode);
        example.setOrderByClause("id");
        return whWastedRcdMapper.selectByExample(example);
    }

    /**
     * 处理残次
     *
     * @param warehouseCode 残次仓编码
     * @param skuCode   sku编码
     * @param quantity  处理数量
     * @param type  类型（waste,fix）
     * @return
     */
    @Override
    public boolean dealDamaged(String warehouseCode, String skuCode, Integer quantity, String type)throws Exception {
        if (EmptyUtil.isEmpty(warehouseCode)
            || EmptyUtil.isEmpty(skuCode)
            || EmptyUtil.isEmpty(quantity)
            || (!WhDamagedRcd.DEAL_TYPE_WASTE.equals(type) && !WhDamagedRcd.DEAL_TYPE_FIX
                .equals(type))) {
            throw new WarehouseException(CommExceptionErrorCode.ILLEGAL_PARAM, "参数异常");
        }

        WhWarehouse whWarehouse = whInfoService.findWarehouseByCode(warehouseCode);
        if (!WhWarehouseVO.TYPE_DAMAGED.equals(whWarehouse.getWarehouseType())) {
            throw new WarehouseException(WarehouseExceptionErrorCode.WRONG_WAREHOUSE_TYPE,
                "仓库类型错误，必须是残次仓");
        }

        boolean isWaste = WhDamagedRcd.DEAL_TYPE_WASTE.equals(type);
        WhWarehouse targetWarehouse = isWaste ? whInfoService
            .findUniqWastedWarehouse(warehouseCode) : whInfoService
            .findUniqWaitForDispatchingWarehouse(warehouseCode);

        if (targetWarehouse == null) {
            throw new WarehouseException(WarehouseExceptionErrorCode.WAREHOUSE_NOT_FIND, "目标仓不存在");
        }

        // 创建调拨
        WhAllotRcd whAllotRcd = new WhAllotRcd();
        whAllotRcd.setAllotType(isWaste ? WhAllotRcd.TYPE_WASTED : WhAllotRcd.TYPE_FIXED);
        whAllotRcd.setSourceWarehouseCode(whWarehouse.getCode());
        whAllotRcd.setTargetWarehouseCode(targetWarehouse.getCode());

        // 调拨行
        List<WhAllotRcdSku> whAllotRcdSkuList = new ArrayList<>();
        WhAllotRcdSku whAllotRcdSku = new WhAllotRcdSku();
        whAllotRcdSku.setSkuCode(skuCode);
        whAllotRcdSku.setQuantity(quantity);
        whAllotRcdSkuList.add(whAllotRcdSku);
        whAllotRcd.setWhAllotRcdSkuList(whAllotRcdSkuList);
        String allotCode = whAllotService.createAllotRcd(whAllotRcd);

        return EmptyUtil.isNotEmpty(allotCode);
    }
}