package com.thebeastshop.pegasus.service.warehouse.service.impl;

import com.alibaba.fastjson.JSON;
import com.github.pagehelper.PageInfo;
import com.github.pagehelper.PageRowBounds;
import com.google.common.collect.Lists;
import com.thebeastshop.pegasus.merchandise.service.McPcsSkuService;
import com.thebeastshop.pegasus.merchandise.vo.PcsSkuDTO;
import com.thebeastshop.pegasus.service.warehouse.cond.WhAutoAllotCond;
import com.thebeastshop.pegasus.service.warehouse.dao.*;
import com.thebeastshop.pegasus.service.warehouse.enums.WhAllotTypeEnum;
import com.thebeastshop.pegasus.service.warehouse.enums.WhAutoAllotDifferenceEnum;
import com.thebeastshop.pegasus.service.warehouse.enums.WhAutoAllotStatusEnum;
import com.thebeastshop.pegasus.service.warehouse.exception.WarehouseException;
import com.thebeastshop.pegasus.service.warehouse.exception.WarehouseExceptionErrorCode;
import com.thebeastshop.pegasus.service.warehouse.model.*;
import com.thebeastshop.pegasus.service.warehouse.service.WhAllotService;
import com.thebeastshop.pegasus.service.warehouse.service.WhAutoAllotService;
import com.thebeastshop.pegasus.service.warehouse.service.WhInfoService;
import com.thebeastshop.pegasus.service.warehouse.vo.*;
import com.thebeastshop.pegasus.util.PegasusUtilFacade;
import com.thebeastshop.pegasus.util.comm.*;
import com.thebeastshop.pegasus.util.model.CommDictionary;
import com.thebeastshop.stock.dto.SSkuStockQueryDTO;
import com.thebeastshop.stock.service.SPresaleService;
import com.thebeastshop.stock.service.SStockService;
import com.thebeastshop.stock.vo.SPresaleVO;
import com.thebeastshop.stock.vo.SSkuStockVO;
import org.apache.commons.collections4.ListUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.*;
import java.util.stream.Collectors;

@Service("whAutoAllotService")
public class WhAutoAllotServiceImpl implements WhAutoAllotService {

    private final static Logger logger = LoggerFactory.getLogger(WhAutoAllotServiceImpl.class);

    @Autowired
    private WhAutoAllotRuleMapper whAutoAllotRuleMapper;
    @Autowired
    private WhAutoAllotSkuMapper whAutoAllotSkuMapper;
    @Autowired
    private WhAutoAllotExcludeCategoryMapper whAutoAllotExcludeCategoryMapper;
    @Autowired
    private WhAutoAllotGroupSortMapper whAutoAllotGroupSortMapper;
    @Autowired
    private WhAutoAllotCallbackWhMapper whAutoAllotCallbackWhMapper;
    @Autowired
    private WhAutoAllotRecordMapper whAutoAllotRecordMapper;
    @Autowired
    private WhAutoAllotRecordDetailMapper whAutoAllotRecordDetailMapper;
    @Autowired
    private WhCommandMapper whCommandMapper;
    @Autowired
    private WhLbLimitAllotSkuMapper whLbLimitAllotSkuMapper;

    @Autowired
    private WhAllotService whAllotService;
    @Autowired
    private WhInfoService whInfoService;

    @Autowired
    private SPresaleService sPresaleService;
    @Autowired
    private SStockService sStockService;
    @Autowired
    private McPcsSkuService mcPcsSkuService;
    @Autowired
    private WhAutoAllotCallbackExcludeCategoryMapper whAutoAllotCallbackExcludeCategoryMapper;


    private final static String[] onlineWhArray = Constants.onlineWhArray;
    private final static String WAREHOUSECODE = "WH020600010096";

    @Override
    @Transactional
    public int saveAutoAllotRule(WhAutoAllotRuleVO whAutoAllotRule) throws Exception {

        short allotChannel = whAutoAllotRule.getAllotChannel();
        WhAutoAllotRuleVO currentRule = allotChannel == 0 ? findCurrentRule(whAutoAllotRule.getBu(), false) : findCurrentRuleOnline(false);

        whAutoAllotRuleMapper.invalidAvailableRule(whAutoAllotRule.getBu(), allotChannel);
        WhAutoAllotRule po;
        po = BeanUtil.buildFrom(whAutoAllotRule, WhAutoAllotRule.class);
        whAutoAllotRuleMapper.insertSelective(po);
        whAutoAllotRule.setId(po.getId());

        Integer currentRuleId = EmptyUtil.isNotEmpty(currentRule) ? currentRule.getId() : null;

        saveExcludeCategory(whAutoAllotRule, currentRuleId);
        //线下分货
        if (allotChannel == 0) {
            saveCallbackExcludeCategory(whAutoAllotRule, currentRuleId);
            saveGroupSort(whAutoAllotRule, currentRuleId);
            saveCallbackWh(whAutoAllotRule, currentRuleId);
        }
        saveAllotSku(whAutoAllotRule, currentRuleId);

        return 0;
    }

    @Override
    public List<WhAutoAllotRuleVO> findHisRules(String bu, short allotChannel) {
        return whAutoAllotRuleMapper.selectHisRules(bu, allotChannel);
    }

    @Override
    public List<WhAutoAllotExcludeCategory> findExcludeCategorysByRuleId(Integer ruleId) {
        return whAutoAllotExcludeCategoryMapper.selectByRuleId(ruleId);
    }

    @Override
    public List<WhAutoAllotCallbackExcludeCategory> findCallbackExcludeCategorysByRuleId(Integer ruleId) {
        return whAutoAllotCallbackExcludeCategoryMapper.selectByRuleId(ruleId);
    }

    @Override
    @Transactional
    public void executeAutoAllot(Long operatorId, String bu, boolean isAuto) throws Exception {
        WhAutoAllotRuleVO currentRule = findCurrentRule(bu, true);
        if (EmptyUtil.isEmpty(currentRule)) {
            throw new WarehouseException(WarehouseExceptionErrorCode.ILLEGAL_PARAM, "不存在可用分货规则");
        }
        //定时任务自动分货，校验开关状态
        if (!currentRule.getRuleSwitch() && isAuto) {
            throw new WarehouseException(WarehouseExceptionErrorCode.ILLEGAL_PARAM, "分货规则关闭");
        }
        List<WhAllotRcd> rcdList = createAutoAllotRcd(currentRule);
        if (EmptyUtil.isNotEmpty(rcdList)) {
            List<WhAutoAllotRecordDetail> notNeedAutoAllotList = excludeNotNeedAutoAllotData(rcdList, currentRule);
            List<String> rcdCodeList = EmptyUtil.isNotEmpty(rcdList) ? whAllotService.createAllotRcds(rcdList) : null;
            generateAutoAllotRecords(rcdList, rcdCodeList, currentRule, operatorId, notNeedAutoAllotList);
        }
    }

    @Override
    public void cronExecuteAutoAllot(Long operatorId) throws Exception {
        List<CommDictionary> bus = PegasusUtilFacade.getInstance().findDictionaryByType("BU");
        for (CommDictionary bu : bus) {
            try {
                executeAutoAllot(operatorId, bu.getDictValue(), true);
            } catch (Exception e) {
                logger.error("BU：{}自动分货失败", bu.getDictValue());
                e.printStackTrace();
            }

        }
    }

    @Override
    public void executeAutoAllotOnline(Long operatorId, boolean isAuto) throws Exception {
        WhAutoAllotRuleVO currentRule = findCurrentRuleOnline(true);
        currentRule.setAllotChannel((short) 1);
        if (EmptyUtil.isEmpty(currentRule)) {
            throw new WarehouseException(WarehouseExceptionErrorCode.ILLEGAL_PARAM, "不存在可用分货规则");
        }
        //定时任务自动分货，校验开关状态
        if (!currentRule.getRuleSwitch() && isAuto) {
            throw new WarehouseException(WarehouseExceptionErrorCode.ILLEGAL_PARAM, "分货规则关闭");
        }
        List<WhAllotRcd> rcdList = createAutoAllotRcdOnline(currentRule, operatorId);
        if (EmptyUtil.isNotEmpty(rcdList)) {
            List<WhAutoAllotRecordDetail> notNeedAutoAllotList = excludeNotNeedAutoAllotData(rcdList, currentRule);
            List<WhAllotRcd> preRcds = ListUtil.deepCopyList(rcdList);
            buildRealAllotQuantity(preRcds, rcdList);
            //返回分货记录ID
            Integer recordId = generateAutoAllotRecords(preRcds, null, currentRule, operatorId, notNeedAutoAllotList);
            List<WhAllotRcd> newAllotRcds = whAllotService.autoAllotCreateRcds(preRcds);
            if (preRcds != null && preRcds.size() > 0) {
                updateAllotFailRecordData(recordId, newAllotRcds);
            }
        }
    }

    //更新分货失败数据
    private void updateAllotFailRecordData(Integer recordId, List<WhAllotRcd> rcds) {
        List<WhAutoAllotRecordDetail> details = whAutoAllotRecordDetailMapper.listDetailByRecordId(recordId);
        if (details != null) {
            int listSize = details.size();
            for (int i = 0; i < listSize; i++) {
                WhAutoAllotRecordDetail detail = details.get(i);
                String recordLineCode = detail.getRecordLineCode();
                //是否同步成功
                boolean isAllotSuccess = false;
                if (rcds != null && rcds.size() > 0) {
                    for (WhAllotRcd rcd : rcds) {
                        detail.setAllotRcdCode(rcd.getCode());
                        if (isAllotSuccess) {
                            break;
                        }
                        List<WhAllotRcdSku> whAllotRcdSkuList = rcd.getWhAllotRcdSkuList();
                        if (whAllotRcdSkuList != null) {
                            for (WhAllotRcdSku rcdSku : whAllotRcdSkuList) {
                                if (recordLineCode.equals(rcdSku.getPreOccupyRefCode())) {
                                    isAllotSuccess = true;
                                    break;
                                }
                            }
                        }
                    }
                }
                if (!isAllotSuccess) {
                    detail.setAllotQuantity(0);
                    detail.setAllotResult((short) WhAutoAllotDifferenceEnum.THIRDPARTY__GOODS_ERROR.getVal());
                }
            }
            batchUpdateRecordLine(details);
        }
    }

    //排除不需要自动分货的数据
    private List<WhAutoAllotRecordDetail> excludeNotNeedAutoAllotData(List<WhAllotRcd> rcdList, WhAutoAllotRuleVO currentRule) {
        List<WhAutoAllotSku> skuList = currentRule.getSkuList();
        Set<String> skuCodes = new HashSet<>();
        if (EmptyUtil.isNotEmpty(skuList)) {
            for (WhAutoAllotSku allotSku : skuList) {
                skuCodes.add(allotSku.getSkuCode());
            }
        }
        Map<String, Boolean> skuMap = checkPresaleSku(skuCodes);
        Map<String, PcsSkuDTO> skuDTOMap = pagingQuerySku(skuCodes);
        int rcdTotal = rcdList.size();
        List<WhAutoAllotRecordDetail> recordDetails = new ArrayList<>();
        List<WhAutoAllotExcludeCategory> excludeCategories = currentRule.getExcludeCategoryList();

        short allotChannel = currentRule.getAllotChannel();
        Map<String, Boolean> skuPresalePlanedDeliveryDateMap = allotChannel == 1 ? checkPresaleSkuPlanedDeliveryDate(skuCodes) : new HashMap<String, Boolean>();
        List<WhAllotRcd> newRcdList = new ArrayList<>();
        for (int i = 0; i < rcdTotal; i++) {
            WhAllotRcd whAllotRcd = rcdList.get(i);
            String targetWarehouseCode = whAllotRcd.getTargetWarehouseCode();
            List<WhAllotRcdSku> newRcdSkuList = new ArrayList<>();
            List<WhAllotRcdSku> rcdSkuList = whAllotRcd.getWhAllotRcdSkuList();
            if (EmptyUtil.isNotEmpty(rcdSkuList)) {
                for (WhAllotRcdSku whAllotRcdSku : rcdSkuList) {
                    Integer planedQuantity = whAllotRcdSku.getPlanedQuantity();
                    Integer quantity = whAllotRcdSku.getQuantity();
                    WhAutoAllotRecordDetail recordDetail = new WhAutoAllotRecordDetail();
                    String skuCode = whAllotRcdSku.getSkuCode();
                    PcsSkuDTO pcsSkuDTO = skuDTOMap.get(skuCode);
                    recordDetail.setSkuCode(skuCode);
                    recordDetail.setPlanAllotQuantity(planedQuantity);
                    recordDetail.setAllotQuantity(quantity);
                    recordDetail.setTargetWarehouseCode(targetWarehouseCode);

                    if (pcsSkuDTO.getIsJit() == 1) {
                        recordDetail.setAllotResult((short) WhAutoAllotDifferenceEnum.IS_JIT.getVal());
                        recordDetail.setAllotQuantity(0);
                        recordDetails.add(recordDetail);
                    } else if (skuMap.containsKey(skuCode)) {
                        recordDetail.setAllotResult((short) WhAutoAllotDifferenceEnum.PRESALE_NOT_COMPLETE_ALLOT.getVal());
                        recordDetail.setAllotQuantity(0);
                        recordDetails.add(recordDetail);
                    } else if (isExcludeCategory(excludeCategories, pcsSkuDTO.getCategoryId().intValue())) {
                        recordDetail.setAllotResult((short) WhAutoAllotDifferenceEnum.EXCLUDE_CATEGORY.getVal());
                        recordDetail.setAllotQuantity(0);
                        recordDetails.add(recordDetail);
                    } else if (planedQuantity <= 0) { //无需分货
                        recordDetail.setAllotResult((short) WhAutoAllotDifferenceEnum.NO_DIFFERENCE.getVal());
                        recordDetail.setAllotQuantity(0);
                        recordDetails.add(recordDetail);
                    } else if (quantity <= 0) { //无货可分
                        recordDetail.setAllotResult((short) WhAutoAllotDifferenceEnum.LOW_STOCK.getVal());
                        recordDetail.setAllotQuantity(0);
                        recordDetails.add(recordDetail);
                    } else if (pcsSkuDTO.getCrossBorderFlag() == 1 && allotChannel == 1) { //线上分货要排除跨境购商品
                        recordDetail.setAllotResult((short) WhAutoAllotDifferenceEnum.CROSS_BORDER.getVal());
                        recordDetail.setAllotQuantity(0);
                        recordDetails.add(recordDetail);
                    } else if (allotChannel == 1 && skuPresalePlanedDeliveryDateMap.get(skuCode) != null && skuPresalePlanedDeliveryDateMap.get(skuCode)) { //
                        recordDetail.setAllotResult((short) WhAutoAllotDifferenceEnum.PRESALE_NOT_SHIPPED.getVal());
                        recordDetail.setAllotQuantity(0);
                        recordDetails.add(recordDetail);
                    } else {
                        newRcdSkuList.add(whAllotRcdSku);
                    }
                }
            }
            if (EmptyUtil.isNotEmpty(newRcdSkuList)) {
                whAllotRcd.setWhAllotRcdSkuList(newRcdSkuList);
                newRcdList.add(whAllotRcd);
            }
        }
        rcdList.clear();
        rcdList.addAll(newRcdList);
        return recordDetails;
    }

    private Map<String, PcsSkuDTO> pagingQuerySku(Set<String> skuSet) {
        List<String> list = new ArrayList<String>(skuSet);
        List<List<String>> temps = Lists.partition(list, 1000);
        Map<String, PcsSkuDTO> skuDTOMap = new HashMap<String, PcsSkuDTO>();
        for (List<String> subList : temps) {
            Map<String, PcsSkuDTO> skuMap = mcPcsSkuService.getSkuNameByCodes(new HashSet<String>(subList));
            if (EmptyUtil.isNotEmpty(skuMap)) {
                skuDTOMap.putAll(skuMap);
            }
        }
        return skuDTOMap;
    }

    private boolean isExcludeCategory(List<WhAutoAllotExcludeCategory> excludeCategories, Integer categoryId) {
        boolean isExclude = false;
        if (EmptyUtil.isNotEmpty(excludeCategories)) {
            for (WhAutoAllotExcludeCategory excludeCategory : excludeCategories) {
                //属于不自动分货类目
                if (excludeCategory.getCategoryId().equals(categoryId)) {
                    isExclude = true;
                }
            }
        }
        return isExclude;
    }

    private Integer generateAutoAllotRecords(List<WhAllotRcd> rcdList, List<String> rcdCodeList,
                                             WhAutoAllotRuleVO currentRule, Long operatorId,
                                             List<WhAutoAllotRecordDetail> notNeedAutoAllotList) {

        boolean difference = false;
        WhAutoAllotRecord autoAllotRecord = new WhAutoAllotRecord();
        boolean notNeedAllot = EmptyUtil.isNotEmpty(notNeedAutoAllotList) ? true : false;
        List<WhAutoAllotRecordDetail> recordDetails = notNeedAutoAllotList;
        short allotChannel = currentRule.getAllotChannel();
        autoAllotRecord.setAllotChannel(currentRule.getAllotChannel());
        if (EmptyUtil.isNotEmpty(rcdList)) {
            int rcdTotal = rcdList.size();
            int refIndex = 0;
            for (int i = 0; i < rcdTotal; i++) {
                WhAllotRcd whAllotRcd = rcdList.get(i);
                String allotRcdCode = allotChannel == 1 ? null : rcdCodeList.get(i);
                //TODO 临时方案，以后需处理
                if (allotRcdCode != null && allotRcdCode.contains("ALT")) {
                    allotRcdCode = allotRcdCode.substring(0, 21);
                } else {
                    allotRcdCode = null;
                }
                String targetWarehouseCode = whAllotRcd.getTargetWarehouseCode();
                List<WhAllotRcdSku> rcdSkuList = whAllotRcd.getWhAllotRcdSkuList();
                if (EmptyUtil.isNotEmpty(rcdSkuList)) {
                    for (WhAllotRcdSku whAllotRcdSku : rcdSkuList) {
                        Integer planedQuantity = whAllotRcdSku.getPlanedQuantity();
                        Integer quantity = whAllotRcdSku.getQuantity();
                        WhAutoAllotRecordDetail recordDetail = new WhAutoAllotRecordDetail();
                        //用来匹配
                        recordDetail.setRefIndex(refIndex);
                        whAllotRcdSku.setRefIndex(refIndex);

                        //存在差异
                        if (planedQuantity > quantity) {
                            if (!difference) {
                                difference = true;
                            }
                            recordDetail.setAllotResult((short) WhAutoAllotDifferenceEnum.LOW_STOCK.getVal());
                        } else {
                            recordDetail.setAllotResult((short) WhAutoAllotDifferenceEnum.NO_DIFFERENCE.getVal());
                        }
                        recordDetail.setSkuCode(whAllotRcdSku.getSkuCode());
                        recordDetail.setPlanAllotQuantity(planedQuantity);
                        recordDetail.setAllotQuantity(quantity);
                        recordDetail.setAllotRcdCode(allotRcdCode);
                        recordDetail.setTargetWarehouseCode(targetWarehouseCode);
                        recordDetails.add(recordDetail);
                        refIndex++;
                    }
                }
            }
        }
        autoAllotRecord.setAllotTime(new Date());
        //线上分货自动完成
        autoAllotRecord.setAllotStatus(allotChannel == 0 ? (short) WhAutoAllotStatusEnum.TO_BE_CONFIRMED.getVal() : (short) WhAutoAllotStatusEnum.COMPLETED.getVal());
        autoAllotRecord.setRuleId(currentRule.getId());
        autoAllotRecord.setAllotResult(difference ? (short) 0 : 1);

        if (notNeedAllot) {
            autoAllotRecord.setAllotResult((short) 0);
        }
        autoAllotRecord.setCreator(operatorId.intValue());
        whAutoAllotRecordMapper.insertSelective(autoAllotRecord);

        Integer recordId = autoAllotRecord.getId();
        if (EmptyUtil.isNotEmpty(recordDetails)) {
            for (WhAutoAllotRecordDetail recordDetail : recordDetails) {
                recordDetail.setRecordId(recordId);
            }
            whAutoAllotRecordDetailMapper.batchInsert(recordDetails);
            batchUpdateRecordLineCode(recordDetails);
        }
        buildRcdSkuPreRefCode(recordDetails, rcdList);
        return autoAllotRecord.getId();
    }

    private void batchUpdateRecordLineCode(List<WhAutoAllotRecordDetail> recordDetails) {
        if (recordDetails == null || recordDetails.size() == 0) {
            return;
        }
        int lineSize = recordDetails.size();
        for (int i = 0; i < lineSize; i++) {
            WhAutoAllotRecordDetail allotRecordDetail = recordDetails.get(i);
            allotRecordDetail.setRecordLineCode(generatorRecordLineCode(allotRecordDetail.getId()));
        }
        batchUpdateRecordLine(recordDetails);
    }

    /**
     * list 分片
     *
     * @param recordDetails
     */
    private void batchUpdateRecordLine(List<WhAutoAllotRecordDetail> recordDetails) {
        List<List<WhAutoAllotRecordDetail>> allList = ListUtils.partition(recordDetails, 500);
        for (List<WhAutoAllotRecordDetail> list : allList) {
            logger.info("自动分货分片更新记录数{}, 明细{}", list == null ? 0 : list.size(), JSON.toJSONString(list));
            whAutoAllotRecordDetailMapper.batchUpdateRecordLine(list);
        }
    }

    /**
     * 构造预占用 code
     *
     * @param recordDetails
     * @param rcdList
     */
    private void buildRcdSkuPreRefCode(List<WhAutoAllotRecordDetail> recordDetails, List<WhAllotRcd> rcdList) {
        if (recordDetails == null || recordDetails.size() == 0 || rcdList == null || rcdList.size() == 0) {
            return;
        }
        for (WhAllotRcd rcd : rcdList) {
            List<WhAllotRcdSku> whAllotRcdSkuList = rcd.getWhAllotRcdSkuList();
            for (WhAllotRcdSku rcdSku : whAllotRcdSkuList) {
                for (WhAutoAllotRecordDetail recordDetail : recordDetails) {
                    if (rcdSku.getRefIndex().equals(recordDetail.getRefIndex())) {
                        rcdSku.setPreOccupyRefCode(recordDetail.getRecordLineCode());
                        break;
                    }
                }
            }
        }

    }

    /*存在预售 存在预售 存在预售 并且 预售数量 预售数量 >已分配数量的不自动货、
     结果差异原因 已分配数量的不自动分货
     */
    private Map<String, Boolean> checkPresaleSku(Set<String> skuCodes) {
        Map<String, Boolean> skuMap = new HashMap<>();
        List<SPresaleVO> sPresaleVOList = sPresaleService.findUnfinishedAllotPreSale(new ArrayList<String>(skuCodes));
        if (EmptyUtil.isNotEmpty(sPresaleVOList)) {
            for (SPresaleVO sPresaleVO : sPresaleVOList) {
                skuMap.put(sPresaleVO.getSkuCode(), true);
            }
        }
        return skuMap;
    }

    /**
     * 检查预售预计发货日期是否为今天
     *
     * @param skuCodes
     * @return
     */
    private Map<String, Boolean> checkPresaleSkuPlanedDeliveryDate(Set<String> skuCodes) {
        Map<String, Boolean> skuMap = new HashMap<>();
        List<SPresaleVO> sPresaleVOList = sPresaleService.findByCond(new ArrayList<String>(skuCodes),
                DateUtil.parse(DateUtil.format(new Date(), DateUtil.DEFAULT_DATE_FORMAT), DateUtil.DEFAULT_DATE_FORMAT));
        if (EmptyUtil.isNotEmpty(sPresaleVOList)) {
            for (SPresaleVO sPresaleVO : sPresaleVOList) {
                skuMap.put(sPresaleVO.getSkuCode(), true);
            }
        }
        return skuMap;
    }

    private List<WhAllotRcd> createAutoAllotRcd(WhAutoAllotRuleVO currentRule) throws Exception {
        List<WhAutoAllotSku> skuList = currentRule.getSkuList();
        List<WhAutoAllotGroupSort> groupSortList = currentRule.getGroupSortList();
        Map<String, WhAutoAllotGroupSort> groupSortMap = buildGroupSortMap(groupSortList);
        if (EmptyUtil.isNotEmpty(skuList)) {
            //sku 分货仓分组map
            Map<String, Set<String>> skuGroupMap = new HashMap<>();
            List<SSkuStockQueryDTO> skuStockQueryDTOList = new ArrayList<>();
            Map<String, List<Map<String, List<String>>>> skuMap = new HashMap<>();
            List<String> skuCodes = new ArrayList<>();
            List<String> warehouseCodes = new ArrayList<>();
            Map<String, Integer> planMap = new HashMap<>();
            //调拨在途
            Map<String, Integer> invQttMap = new HashMap<>();
            for (WhAutoAllotSku autoAllotSku : skuList) {
                SSkuStockQueryDTO skuStockQueryDTO = new SSkuStockQueryDTO();
                String skuCode = autoAllotSku.getSkuCode();
                String warehouseCode = autoAllotSku.getTargetWarehouseCode();

                String calculateWarehouseCode = autoAllotSku.getCalculateWarehouseCode();
                skuStockQueryDTO.setSkuCode(skuCode);
                skuStockQueryDTO.setWarehouseCode(calculateWarehouseCode);
                skuStockQueryDTOList.add(skuStockQueryDTO);

                List<Map<String, List<String>>> targetWhList = skuMap.get(skuCode);
                if (EmptyUtil.isNotEmpty(targetWhList)) {
                    for (Map<String, List<String>> targetWhMap : targetWhList) {
                        List<String> calculateWhList = targetWhMap.get(warehouseCode);
                        if (EmptyUtil.isEmpty(calculateWhList)) {
                            calculateWhList = new ArrayList<>();
                        }
                        calculateWhList.add(calculateWarehouseCode);
                        targetWhMap.put(warehouseCode, calculateWhList);
                    }
                } else {
                    targetWhList = new ArrayList<>();
                    Map<String, List<String>> targetWhMap = new HashMap<>();
                    List<String> calculateWhList = new ArrayList<>();
                    calculateWhList.add(calculateWarehouseCode);
                    targetWhMap.put(warehouseCode, calculateWhList);
                    targetWhList.add(targetWhMap);
                }
                skuMap.put(skuCode, targetWhList);

                String key = new StringBuffer().append(calculateWarehouseCode)
                        .append(":")
                        .append(skuCode)
                        .toString();
                planMap.put(key, autoAllotSku.getSafeStock());
                skuCodes.add(skuCode);
                warehouseCodes.add(warehouseCode);

                Set<String> groupSet = skuGroupMap.get(skuCode);
                if (EmptyUtil.isEmpty(groupSet)) {
                    groupSet = new HashSet<>();
                }
                groupSet.add(groupSortMap.get(warehouseCode).getOwnGroup());
                skuGroupMap.put(skuCode, groupSet);
            }

            //批量查询可用库存
            List<SSkuStockVO> sSkuStockVOList = pagingQueryStock(skuStockQueryDTOList);

            //减去当前可用库存
            if (EmptyUtil.isNotEmpty(sSkuStockVOList)) {
                for (SSkuStockVO skuStockVO : sSkuStockVOList) {
                    String key = new StringBuffer().append(skuStockVO.getWarehouseCode())
                            .append(":")
                            .append(skuStockVO.getSkuCode())
                            .toString();
                    if (planMap.containsKey(key)) {
                        planMap.put(key, planMap.get(key) - skuStockVO.getCanUseQuantity());
                    }
                }
            }

            //调拨在途=状态 in (待出库、待入库、待确认)、系统类型为【自动分货】的调拨单计划数
            List<WhWarehouseInvQttVO> invQttVOList = whCommandMapper.findAutoAllotWillInQttBySkuCode(skuCodes, warehouseCodes);
            //减去调拨在途数量
            if (EmptyUtil.isNotEmpty(sSkuStockVOList)) {
                for (WhWarehouseInvQttVO invQttVO : invQttVOList) {
                    String key = new StringBuffer().append(invQttVO.getWarehouseCode())
                            .append(":")
                            .append(invQttVO.getSkuCode())
                            .toString();
                    if (!invQttMap.containsKey(key)) {
                        invQttMap.put(key, invQttVO.getQuantity());
                    }
                }
            }
            List<WhAutoAllotSkuVO> autoAllotSkuVOList = buildSkuPlanQuantityVO(groupSortMap, skuMap, planMap, skuGroupMap, invQttMap);
            bulidCanUseQuantity(skuCodes, autoAllotSkuVOList, currentRule.getCallbackExcludeCategoryList(), currentRule.getCallbackWhList(), false);
            buildCanAllotQunatity(autoAllotSkuVOList);
            return buildAllotRcd(autoAllotSkuVOList);
        }
        return null;
    }

    private List<WhAllotRcd> createAutoAllotRcdOnline(WhAutoAllotRuleVO currentRule, Long operatorId) throws Exception {
        List<WhAutoAllotSku> skuList = currentRule.getSkuList();
        if (EmptyUtil.isNotEmpty(skuList)) {
            //sku 分货仓分组map
            List<SSkuStockQueryDTO> skuStockQueryDTOList = new ArrayList<>();
            Set<String> skuCodes = new HashSet<>();
            List<String> warehouseCodes = Arrays.asList(Constants.onlineWhArray);
            Map<String, Integer> planMap = new HashMap<>();

            Map<String, Integer> safeStockMap = new TreeMap<String, Integer>();
            Map<String, Integer> skuCanUseStockMap = new HashMap<String, Integer>();
            Map<String, Integer> stockMap = new HashMap<String, Integer>();
            //调拨在途
            Map<String, Integer> invQttMap = new HashMap<>();
            for (WhAutoAllotSku autoAllotSku : skuList) {
                SSkuStockQueryDTO skuStockQueryDTO = new SSkuStockQueryDTO();
                String skuCode = autoAllotSku.getSkuCode();
                String warehouseCode = autoAllotSku.getTargetWarehouseCode();

                skuStockQueryDTO.setSkuCode(skuCode);
                skuStockQueryDTO.setWarehouseCode(warehouseCode);
                skuStockQueryDTOList.add(skuStockQueryDTO);

                String key = new StringBuffer().append(warehouseCode).append(":").append(skuCode).toString();
                planMap.put(key, autoAllotSku.getSafeStock());
                safeStockMap.put(key, autoAllotSku.getSafeStock());
                skuCodes.add(skuCode);
            }


            //批量查询可用库存
            List<SSkuStockVO> sSkuStockVOList = pagingQueryStock(skuStockQueryDTOList);

            //减去当前可用库存
            if (EmptyUtil.isNotEmpty(sSkuStockVOList)) {
                for (SSkuStockVO skuStockVO : sSkuStockVOList) {
                    String key = new StringBuffer().append(skuStockVO.getWarehouseCode())
                            .append(":")
                            .append(skuStockVO.getSkuCode())
                            .toString();
                    String skuCode = skuStockVO.getSkuCode();
                    if (skuCanUseStockMap.containsKey(skuCode)) {
                        skuCanUseStockMap.put(skuCode, skuCanUseStockMap.get(skuCode) + skuStockVO.getCanUseQuantity());
                    } else {
                        skuCanUseStockMap.put(skuCode, skuStockVO.getCanUseQuantity());
                    }
                    stockMap.put(key, skuStockVO.getCanUseQuantity());
                    if (planMap.containsKey(key)) {
                        planMap.put(key, planMap.get(key) - skuStockVO.getCanUseQuantity());
                    }
                }
            }


            //调拨在途=状态 in (待出库、待入库、待确认)、系统类型为【自动分货】的调拨单计划数
            List<WhWarehouseInvQttVO> invQttVOList = whCommandMapper.findAutoAllotWillInQttBySkuCode(new ArrayList<String>(skuCodes), warehouseCodes);
            //减去调拨在途数量
            if (EmptyUtil.isNotEmpty(sSkuStockVOList)) {
                for (WhWarehouseInvQttVO invQttVO : invQttVOList) {
                    String key = new StringBuffer().append(invQttVO.getWarehouseCode())
                            .append(":")
                            .append(invQttVO.getSkuCode())
                            .toString();
                    if (!invQttMap.containsKey(key)) {
                        invQttMap.put(key, invQttVO.getQuantity());
                    }
                }
            }

            List<WhAutoAllotVO> autoAllotList = new ArrayList<WhAutoAllotVO>();
            for (String skuCode : skuCodes) {
                WhAutoAllotOnlineSkuVO onlineSku = new WhAutoAllotOnlineSkuVO();
                onlineSku.setSkuCode(skuCode);
                Integer totalSafeStock = 0;
                int totalNeedAllotQuantity = 0;
                boolean needAllot = false;
                Map<String, Integer> overStockMap = new TreeMap<String, Integer>();
                Map<String, Integer> needStockMap = new HashMap<String, Integer>();
                Map<String, Integer> skuSafeStockMap = new TreeMap<String, Integer>();
                for (int i = 0; i < Constants.onlineWhArray.length; i++) {

                    String warehouseCode = Constants.onlineWhArray[i];
                    String key = new StringBuilder().append(warehouseCode).append(":").append(skuCode).toString();
                    int safeStock = safeStockMap.get(key);
                    skuSafeStockMap.put(warehouseCode, safeStock);
                    totalSafeStock += safeStock;
                    int needAllotQuantity = planMap.get(key);
                    needAllot = needAllotQuantity > 0 ? true : needAllot;
                    totalNeedAllotQuantity += needAllotQuantity;

                    if (needAllotQuantity < 0) {
                        overStockMap.put(warehouseCode, -needAllotQuantity);
                    } else {
                        needStockMap.put(warehouseCode, needAllotQuantity);
                    }
                    switch (i) {
                        case 0:
                            onlineSku.setWH020600010102(safeStock);
                            onlineSku.setWH020600010102NeedAllotQunatity(needAllotQuantity);
                            break;
                        case 1:
                            onlineSku.setWH020600010121(safeStock);
                            onlineSku.setWH020600010121NeedAllotQunatity(needAllotQuantity);
                            break;
                        case 2:
                            onlineSku.setWH000275910275(safeStock);
                            onlineSku.setWH000275910275NeedAllotQunatity(needAllotQuantity);
                            break;
                        case 3: //[WH020600010195]颛兴路 - 京东渠道销售仓
                            onlineSku.setWH020600010195(safeStock);
                            onlineSku.setWH020600010195NeedAllotQunatity(needAllotQuantity);
                            break;
                        case 4://[WH000332910332]京东家居仓
                            onlineSku.setWH000332910332(safeStock);
                            onlineSku.setWH000332910332NeedAllotQunatity(needAllotQuantity);
                            break;
                        default:
                            break;
                    }
                }

                //如果部分仓的可用库存+调拨在途>安全库存,不需要分货
                if (needAllot) {
                    //三个仓总可用＞=三个仓总安全库存 ，需要自动分货
                    if (totalNeedAllotQuantity < 0) {
                        overStockMap = sortMapByValue(overStockMap, false);
                        for (Map.Entry<String, Integer> overStockEntry : overStockMap.entrySet()) {
                            Integer canAllotQuantity = overStockEntry.getValue();
                            for (Map.Entry<String, Integer> needStockEntry : needStockMap.entrySet()) {
                                Integer needAllotQuantity = needStockEntry.getValue();
                                if (canAllotQuantity > 0 && needAllotQuantity > 0) {
                                    WhAutoAllotVO whAutoAllotVO = new WhAutoAllotVO();
                                    whAutoAllotVO.setSourceWarehouseCode(overStockEntry.getKey());
                                    whAutoAllotVO.setTargetWarehouseCode(needStockEntry.getKey());
                                    whAutoAllotVO.setPlanedQuantity(needAllotQuantity);
                                    whAutoAllotVO.setSkuCode(skuCode);
                                    //够分
                                    if (canAllotQuantity - needAllotQuantity >= 0) {
                                        whAutoAllotVO.setAllotQuantity(needAllotQuantity);
                                        canAllotQuantity = canAllotQuantity - needAllotQuantity;
                                        needStockEntry.setValue(0);
                                    } else {
                                        whAutoAllotVO.setAllotQuantity(canAllotQuantity);
                                        canAllotQuantity = 0;
                                        needStockEntry.setValue(needAllotQuantity - canAllotQuantity);
                                    }
                                    autoAllotList.add(whAutoAllotVO);
                                }
                            }
                        }
                    } else {
                        int canUseQuantity = skuCanUseStockMap.get(skuCode);
                        Map<String, Integer> needAllotOutMap = new HashMap<String, Integer>();
                        Map<String, Integer> needAllotInMap = new HashMap<String, Integer>();
                        //按安全库存数量排序
                        skuSafeStockMap = sortMapByValue(skuSafeStockMap, false);

                        Integer alreadyAllotQuantity = 0;
                        for (Map.Entry<String, Integer> safeStockEntry : skuSafeStockMap.entrySet()) {
                            int canAllotQuantity = canUseQuantity * safeStockEntry.getValue() / totalSafeStock;
                            alreadyAllotQuantity += canAllotQuantity;
                            String warehouseCode = safeStockEntry.getKey();
                            int alreadyQuantity = stockMap.get(new StringBuilder().append(warehouseCode).append(":").append(skuCode).toString());
                            if (canAllotQuantity == alreadyQuantity) {
                                continue;
                            } else if (canAllotQuantity < alreadyQuantity) {//分配库存小于已有库存，需要调出
                                needAllotOutMap.put(warehouseCode, alreadyQuantity - canAllotQuantity);
                            } else { //分配库存小于于已有库存，需要调入
                                needAllotInMap.put(warehouseCode, canAllotQuantity - alreadyQuantity);
                            }
                        }

                        //按比例分配后多余的数量
                        Integer scaleAllotOverQuantity = canUseQuantity - alreadyAllotQuantity;

                        boolean firstElm = true;
                        //将剩余库存分配到安全库存比例最高的仓
                        for (Map.Entry<String, Integer> safeStockEntry : skuSafeStockMap.entrySet()) {
                            if (firstElm) {
                                Integer needAllotOutQuantity = needAllotOutMap.get(safeStockEntry.getKey());
                                Integer needAllotInQuantity = (EmptyUtil.isNotEmpty(needAllotInMap.get(safeStockEntry.getKey()))
                                        ? needAllotInMap.get(safeStockEntry.getKey()) : 0) + scaleAllotOverQuantity;
                                //调出数量
                                needAllotOutQuantity = needAllotOutQuantity == null ? 0 : needAllotOutQuantity;
                                needAllotInMap.put(safeStockEntry.getKey(), needAllotInQuantity - needAllotOutQuantity);
                                needAllotOutMap.put(safeStockEntry.getKey(), needAllotOutQuantity - needAllotInQuantity);
                                firstElm = false;
                                break;
                            }
                        }
                        for (Map.Entry<String, Integer> allotIn : needAllotInMap.entrySet()) {
                            int needAllotQuantity = allotIn.getValue();
                            String targetWarehouseCode = allotIn.getKey();
                            for (Map.Entry<String, Integer> allotOut : needAllotOutMap.entrySet()) {
                                int allotOutQuantity = allotOut.getValue();
                                String sourceWarehouseCode = allotOut.getKey();
                                //调入仓和调出仓不能一样
                                if (!targetWarehouseCode.equals(sourceWarehouseCode)) {
                                    if (allotOutQuantity > 0 && needAllotQuantity > 0) {
                                        WhAutoAllotVO whAutoAllotVO = new WhAutoAllotVO();
                                        whAutoAllotVO.setSourceWarehouseCode(sourceWarehouseCode);
                                        whAutoAllotVO.setTargetWarehouseCode(targetWarehouseCode);
                                        whAutoAllotVO.setSkuCode(skuCode);
                                        whAutoAllotVO.setPlanedQuantity(needStockMap.get(targetWarehouseCode));
                                        if (allotOutQuantity == needAllotQuantity) {
                                            whAutoAllotVO.setAllotQuantity(needAllotQuantity);
                                            autoAllotList.add(whAutoAllotVO);
                                            needAllotQuantity = 0;
                                            allotIn.setValue(0);
                                            allotOut.setValue(0);
                                            break;
                                        } else if (allotOutQuantity > needAllotQuantity) {
                                            whAutoAllotVO.setAllotQuantity(needAllotQuantity);
                                            autoAllotList.add(whAutoAllotVO);
                                            allotIn.setValue(0);
                                            allotOut.setValue(allotOutQuantity - needAllotQuantity);
                                            break;
                                        } else {
                                            whAutoAllotVO.setAllotQuantity(allotOutQuantity);
                                            needAllotQuantity = needAllotQuantity - allotOutQuantity;
                                            autoAllotList.add(whAutoAllotVO);
                                            allotIn.setValue(needAllotQuantity);
                                            allotOut.setValue(0);
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }
            return buildAllotRcdOnLine(autoAllotList, operatorId);
        }
        return null;
    }


    private List<SSkuStockVO> pagingQueryStock(List<SSkuStockQueryDTO> skuStockQueryDTOList) {
        List<List<SSkuStockQueryDTO>> temps = Lists.partition(skuStockQueryDTOList, 1000);
        List<SSkuStockVO> allStockList = new ArrayList<SSkuStockVO>();
        for (List<SSkuStockQueryDTO> subList : temps) {
            List<SSkuStockVO> stockList = sStockService.getSkuStocks(subList);
            if (EmptyUtil.isNotEmpty(stockList)) {
                allStockList.addAll(stockList);
            }
        }
        return allStockList;
    }

    //生成调拨单
    private List<WhAllotRcd> buildAllotRcd(List<WhAutoAllotSkuVO> autoAllotSkuVOList) throws Exception {
        if (EmptyUtil.isNotEmpty(autoAllotSkuVOList)) {
            Map<String, List<WhAllotRcdSku>> allotRcdMap = new HashMap<>();
            for (WhAutoAllotSkuVO autoAllotSkuVO : autoAllotSkuVOList) {
                String skuCode = autoAllotSkuVO.getSkuCode();
                List<WhAutoAllotGroupVO> groupList = autoAllotSkuVO.getGroupList();
                if (EmptyUtil.isNotEmpty(groupList)) {
                    for (WhAutoAllotGroupVO whAutoAllotGroupVO : groupList) {
                        List<WhAutoAllotGroupWhSortVO> groupWhList = whAutoAllotGroupVO.getGroupWhList();
                        if (EmptyUtil.isNotEmpty(groupWhList)) {
                            for (WhAutoAllotGroupWhSortVO autoAllotGroupWhSortVO : groupWhList) {
                                int targetCanUseQuantity = autoAllotGroupWhSortVO.getCanUseQuantity();
                                String targetWarehouseCode = autoAllotGroupWhSortVO.getWarehouseCode();
                                WhAllotRcdSku whAllotRcdSku = new WhAllotRcdSku();
                                whAllotRcdSku.setSkuCode(skuCode);
                                whAllotRcdSku.setPlanedQuantity(autoAllotGroupWhSortVO.getPlanAllotQuantity());
                                whAllotRcdSku.setQuantity(targetCanUseQuantity);
                                List<WhAllotRcdSku> allotRcdSkuList = allotRcdMap.get(targetWarehouseCode);
                                if (allotRcdSkuList == null) {
                                    allotRcdSkuList = new ArrayList<>();
                                }
                                allotRcdSkuList.add(whAllotRcdSku);
                                allotRcdMap.put(targetWarehouseCode, allotRcdSkuList);
                            }
                        }
                    }
                }
            }
            if (EmptyUtil.isNotEmpty(allotRcdMap)) {
                List<WhAllotRcd> rcdList = new ArrayList<>();
                List<String> warehouseCodeList = removeSameGroupWh(new ArrayList<String>(allotRcdMap.keySet()));
                Map<String, WhPhysicalWarehouse> whPhysicalWarehouseMap = warehouseCodeList.size() == 0
                        ? new HashMap<String, WhPhysicalWarehouse>()
                        : whInfoService.findPhyWhMapByWarehouseCodes(new LinkedList<String>(warehouseCodeList));
                for (Map.Entry<String, List<WhAllotRcdSku>> entry : allotRcdMap.entrySet()) {
                    String targetWarehouseCode = entry.getKey();
                    List<WhAllotRcdSku> allotRcdSkuList = entry.getValue();
                    WhAllotRcd whAllotRcd = new WhAllotRcd();
                    whAllotRcd.setSourcePhysicalWarehouseCode(Constants.PHYSICAL_WAREHOUSE);
                    whAllotRcd.setSourceWarehouseCode(WAREHOUSECODE);
                    //目标物理仓
                    WhPhysicalWarehouse whPhysicalWarehouse = whPhysicalWarehouseMap.get(targetWarehouseCode);
                    whAllotRcd.setTargetPhysicalWarehouseCode(EmptyUtil.isNotEmpty(whPhysicalWarehouse) ? whPhysicalWarehouse.getCode() : Constants.PHYSICAL_WAREHOUSE);
                    whAllotRcd.setTargetWarehouseCode(targetWarehouseCode);
                    whAllotRcd.setAllotType(WhAllotTypeEnum.AUTO_ALLOT.getVal());
                    whAllotRcd.setRemark("自动分货调拨");
                    whAllotRcd.setWhAllotRcdSkuList(allotRcdSkuList);
                    rcdList.add(whAllotRcd);
                }
                return rcdList;
            }
        }
        return null;
    }

    //排除颛兴路物理仓同分组的逻辑仓
    private List<String> removeSameGroupWh(List<String> warehouseCodeList) {
        List<String> whCodeList = new ArrayList<String>();
        List<WhWarehouse> whWarehouseList = whInfoService.findWarehouseByCodes(warehouseCodeList);
        if (EmptyUtil.isNotEmpty(whWarehouseList)) {
            int listSize = whWarehouseList.size();
            for (int i = 0; i < listSize; i++) {
                WhWarehouse whWarehouse = whWarehouseList.get(i);
                if (!whWarehouse.getWarehouseGroupId().equals(Constants.PHYSICAL_WAREHOUSE_DEFAULT_GROUP_ID)) {
                    whCodeList.add(whWarehouse.getCode());
                }
            }
        }
        return whCodeList;
    }

    private List<WhAllotRcd> buildAllotRcdOnLine(List<WhAutoAllotVO> autoAllotList, Long operatorId) {
        List<WhAllotRcd> rcdList = new ArrayList<WhAllotRcd>();
        Set<String> rcdKeyList = new HashSet<String>();
        for (WhAutoAllotVO whAutoAllotVO : autoAllotList) {
            rcdKeyList.add(new StringBuilder(whAutoAllotVO.getSourceWarehouseCode()).append(":").append(whAutoAllotVO.getTargetWarehouseCode()).toString());
        }

        for (String rcdKey : rcdKeyList) {
            WhAllotRcd whAllotRcd = new WhAllotRcd();
            String[] rcdWarehouseCodeArray = rcdKey.split(":");
            whAllotRcd.setSourcePhysicalWarehouseCode(Constants.PHYSICAL_WAREHOUSE);
            whAllotRcd.setSourceWarehouseCode(rcdWarehouseCodeArray[0]);
            whAllotRcd.setTargetWarehouseCode(rcdWarehouseCodeArray[1]);
            whAllotRcd.setTargetPhysicalWarehouseCode(Constants.PHYSICAL_WAREHOUSE);
            whAllotRcd.setAllotType(WhAllotTypeEnum.AUTO_ALLOT_ONLINE.getVal());
            whAllotRcd.setCreateUserId(operatorId == null ? null : operatorId.intValue());
            whAllotRcd.setRemark("线上自动分货调拨");
            List<WhAllotRcdSku> whAllotRcdSkuList = new ArrayList<WhAllotRcdSku>();
            for (WhAutoAllotVO whAutoAllotVO : autoAllotList) {
                if (rcdKey.equals(new StringBuilder(whAutoAllotVO.getSourceWarehouseCode()).append(":").append(whAutoAllotVO.getTargetWarehouseCode()).toString())) {
                    WhAllotRcdSku whAllotRcdSku = new WhAllotRcdSku();
                    whAllotRcdSku.setQuantity(whAutoAllotVO.getAllotQuantity());
                    whAllotRcdSku.setSkuCode(whAutoAllotVO.getSkuCode());
                    whAllotRcdSku.setPlanedQuantity(whAutoAllotVO.getAllotQuantity());
                    whAllotRcdSkuList.add(whAllotRcdSku);
                }
            }
            whAllotRcd.setWhAllotRcdSkuList(whAllotRcdSkuList);
            rcdList.add(whAllotRcd);
        }

        return rcdList;
    }

    //计算各分货仓可分配数量
    private void buildCanAllotQunatity(List<WhAutoAllotSkuVO> autoAllotSkuVOList) {
        if (EmptyUtil.isNotEmpty(autoAllotSkuVOList)) {
            for (WhAutoAllotSkuVO autoAllotSkuVO : autoAllotSkuVOList) {
                int planAllotQuantity = autoAllotSkuVO.getPlanAllotQuantity();
                int canUseQuantity = autoAllotSkuVO.getCanUseQuantity();
                //可用库存不足
                if (canUseQuantity < planAllotQuantity && planAllotQuantity > 0) {
                    List<WhAutoAllotGroupVO> groupList = autoAllotSkuVO.getGroupList();
                    for (WhAutoAllotGroupVO whAutoAllotGroupVO : groupList) {
                        Integer groupCanUseQuantity = (int) Math.floor((canUseQuantity * whAutoAllotGroupVO.getPlanAllotQuantity()) / planAllotQuantity);
                        int groupPlanAllotQunatity = whAutoAllotGroupVO.getPlanAllotQuantity();
                        if (groupPlanAllotQunatity > 0) {
                            whAutoAllotGroupVO.setCanUseQuantity(groupCanUseQuantity);
                            List<WhAutoAllotGroupWhSortVO> groupWhList = whAutoAllotGroupVO.getGroupWhList();
                            //组内剩余数量
                            int overQuantity = groupCanUseQuantity;
                            for (WhAutoAllotGroupWhSortVO autoAllotGroupWhSortVO : groupWhList) {
                                int targetWhPlanAllotQunatity = autoAllotGroupWhSortVO.getPlanAllotQuantity();
                                int targetCanUseQuantity = (int) Math.floor((groupCanUseQuantity * targetWhPlanAllotQunatity)
                                        / groupPlanAllotQunatity);
                                autoAllotGroupWhSortVO.setCanUseQuantity(targetCanUseQuantity);
                                overQuantity -= targetCanUseQuantity;
                            }
                            //按优先级排序
                            Collections.sort(groupWhList);
                            if (overQuantity > 0) {
                                for (WhAutoAllotGroupWhSortVO autoAllotGroupWhSortVO : groupWhList) {
                                    int targetWhPlanAllotQunatity = autoAllotGroupWhSortVO.getPlanAllotQuantity();
                                    int targetCanUseQuantity = autoAllotGroupWhSortVO.getCanUseQuantity();
                                    int lessQuantity = targetWhPlanAllotQunatity - targetCanUseQuantity;
                                    if (lessQuantity > 0) {
                                        if (overQuantity <= lessQuantity) {
                                            autoAllotGroupWhSortVO.setCanUseQuantity(targetCanUseQuantity + overQuantity);
                                            overQuantity = 0;
                                        } else {
                                            autoAllotGroupWhSortVO.setCanUseQuantity(targetCanUseQuantity + lessQuantity);
                                            overQuantity -= lessQuantity;
                                        }
                                    }
                                }
                            }
                        }
                    }
                } else {
                    List<WhAutoAllotGroupVO> groupList = autoAllotSkuVO.getGroupList();
                    for (WhAutoAllotGroupVO whAutoAllotGroupVO : groupList) {
                        whAutoAllotGroupVO.setCanUseQuantity(whAutoAllotGroupVO.getPlanAllotQuantity());
                        List<WhAutoAllotGroupWhSortVO> groupWhList = whAutoAllotGroupVO.getGroupWhList();
                        for (WhAutoAllotGroupWhSortVO autoAllotGroupWhSortVO : groupWhList) {
                            autoAllotGroupWhSortVO.setCanUseQuantity(autoAllotGroupWhSortVO.getPlanAllotQuantity());
                        }
                    }
                }
            }
        }
    }


    //获取良品仓可用库存，不够则从回调仓回调
    private void bulidCanUseQuantity(List<String> skuCodes, List<WhAutoAllotSkuVO> autoAllotSkuVOList,
                                     List<WhAutoAllotCallbackExcludeCategory> callbackExcludeCategoryList,
                                     List<WhAutoAllotCallbackWh> callbackWhList, boolean isCallback) throws Exception {
        //查询颛兴路良品仓库存
        Map<String, SSkuStockVO> stockMap = sStockService.getSkuStocks(skuCodes, WAREHOUSECODE);
        if (EmptyUtil.isNotEmpty(autoAllotSkuVOList)) {
            for (WhAutoAllotSkuVO whAutoAllotSkuVO : autoAllotSkuVOList) {
                String skuCode = whAutoAllotSkuVO.getSkuCode();
                SSkuStockVO skuStockVO = stockMap.get(skuCode);
                whAutoAllotSkuVO.setCanUseQuantity(skuStockVO.getCanUseQuantity());
            }
            //库存不足sku 需要回调库存,只回调一次
            if (!isCallback) {
                isCallback = callBackStock(skuCodes, callbackWhList, stockMap, callbackExcludeCategoryList);
                //回调后重新计算可用库存
                bulidCanUseQuantity(skuCodes, autoAllotSkuVOList, null, callbackWhList, isCallback);
            }
        }
    }

    //回调库存
    private boolean callBackStock(List<String> skuCodes, List<WhAutoAllotCallbackWh> callbackWhList, Map<String, SSkuStockVO> stockMap,
                                  List<WhAutoAllotCallbackExcludeCategory> callbackExcludeCategoryList) throws Exception {
        if (EmptyUtil.isNotEmpty(callbackWhList)) {
            List<String> warehouseCodeList = new ArrayList<>();
            for (WhAutoAllotCallbackWh autoAllotCallbackWh : callbackWhList) {
                warehouseCodeList.add(autoAllotCallbackWh.getWarehouseCode());
            }
            Map<String, PcsSkuDTO> skuDTOMap = pagingQuerySku(new HashSet<String>(skuCodes));
            if (EmptyUtil.isNotEmpty(skuCodes)) {
                List<Long> allbackExcludeCategoryIds = new ArrayList<Long>();
                if (EmptyUtil.isNotEmpty(callbackExcludeCategoryList)) {
                    for (WhAutoAllotCallbackExcludeCategory whAutoAllotCallbackExcludeCategory : callbackExcludeCategoryList) {
                        allbackExcludeCategoryIds.add(whAutoAllotCallbackExcludeCategory.getCategoryId().longValue());
                    }
                }
                List<String> needCallbackSkuList = new ArrayList<String>();
                for (String skuCode : skuCodes) {
                    PcsSkuDTO pcsSkuDTO = skuDTOMap.get(skuCode);
                    if (EmptyUtil.isEmpty(pcsSkuDTO) || !allbackExcludeCategoryIds.contains(pcsSkuDTO.getCategoryId())) {
                        needCallbackSkuList.add(skuCode);
                    }
                }
                skuCodes = needCallbackSkuList;
            }
            Map<String, Map<String, SSkuStockVO>> stockVOMap = sStockService.getSkuStocks(skuCodes, warehouseCodeList);
            if (EmptyUtil.isNotEmpty(stockVOMap)) {
                List<WhAllotRcd> rcdList = new ArrayList<>();
                for (Map.Entry<String, Map<String, SSkuStockVO>> entry : stockVOMap.entrySet()) {
                    String warehouseCode = entry.getKey();
                    List<WhAllotRcdSku> whAllotRcdSkuList = new ArrayList<>();
                    Map<String, SSkuStockVO> skuStockMap = entry.getValue();
                    for (Map.Entry<String, SSkuStockVO> skuStockEntry : skuStockMap.entrySet()) {
                        String skuCode = skuStockEntry.getKey();
                        SSkuStockVO skuStockVO = skuStockEntry.getValue();
                        int canUseQuantity = skuStockVO.getCanUseQuantity();
                        if (canUseQuantity > 0) {
                            WhAllotRcdSku whAllotRcdSku = new WhAllotRcdSku();
                            whAllotRcdSku.setPlanedQuantity(skuStockVO.getCanUseQuantity());
                            whAllotRcdSku.setQuantity(skuStockVO.getCanUseQuantity());
                            whAllotRcdSku.setSkuCode(skuCode);
                            whAllotRcdSkuList.add(whAllotRcdSku);

                            //增加可调拨库存
                            SSkuStockVO canAllotStock = stockMap.get(skuCode);
                            canAllotStock.setCanUseQuantity(canAllotStock.getCanUseQuantity() + canUseQuantity);
                        }
                    }
                    //有可用库存，则创建调拨单
                    if (EmptyUtil.isNotEmpty(whAllotRcdSkuList)) {
                        WhAllotRcd whAllotRcd = new WhAllotRcd();
                        whAllotRcd.setSourcePhysicalWarehouseCode(Constants.PHYSICAL_WAREHOUSE);
                        whAllotRcd.setSourceWarehouseCode(warehouseCode);
                        whAllotRcd.setTargetPhysicalWarehouseCode(Constants.PHYSICAL_WAREHOUSE);
                        whAllotRcd.setTargetWarehouseCode(WAREHOUSECODE);
                        whAllotRcd.setAllotType(WhAllotTypeEnum.AUTO_ALLOT.getVal());
                        whAllotRcd.setRemark("自动分货回调库存");
                        whAllotRcd.setWhAllotRcdSkuList(whAllotRcdSkuList);
                        rcdList.add(whAllotRcd);
                    }
                }
                whAllotService.createAllotRcds(rcdList);
            }
        }
        return true;

    }


    /**
     * 计算sku 计划分货总数， 各分组计划分货总数，各分货仓计划分货数
     *
     * @param groupSortMap
     * @param skuMap
     * @param planMap
     */
    private List<WhAutoAllotSkuVO> buildSkuPlanQuantityVO(Map<String, WhAutoAllotGroupSort> groupSortMap,
                                                          Map<String, List<Map<String, List<String>>>> skuMap,
                                                          Map<String, Integer> planMap,
                                                          Map<String, Set<String>> skuGroupMap
            , Map<String, Integer> invQttMap) {
        List<WhAutoAllotSkuVO> autoAllotSkuVOList = new ArrayList<>();
        if (EmptyUtil.isNotEmpty(skuMap)) {
            for (Map.Entry<String, List<Map<String, List<String>>>> skuEntry : skuMap.entrySet()) {
                String skuCode = skuEntry.getKey();
                WhAutoAllotSkuVO autoAllotSkuVO = new WhAutoAllotSkuVO();
                autoAllotSkuVO.setSkuCode(skuCode);
                Set<String> groupSet = skuGroupMap.get(skuCode);
                List<WhAutoAllotGroupVO> groupList = new ArrayList<>();
                List<Map<String, List<String>>> targetWhList = skuEntry.getValue();
                for (String group : groupSet) {
                    WhAutoAllotGroupVO autoAllotGroupVO = new WhAutoAllotGroupVO();
                    autoAllotGroupVO.setOwnGroup(group);
                    List<WhAutoAllotGroupWhSortVO> groupWhList = new ArrayList<>();
                    if (EmptyUtil.isNotEmpty(targetWhList)) {
                        for (Map<String, List<String>> targetWhMap : targetWhList) {
                            for (Map.Entry<String, List<String>> targetEntry : targetWhMap.entrySet()) {
                                String targetWhCode = targetEntry.getKey();
                                WhAutoAllotGroupSort whAutoAllotGroupSort = groupSortMap.get(targetWhCode);
                                String ownGroup = whAutoAllotGroupSort.getOwnGroup();
                                if (ownGroup.equals(group)) {
                                    WhAutoAllotGroupWhSortVO whAutoAllotGroupWhSortVO = new WhAutoAllotGroupWhSortVO();
                                    whAutoAllotGroupWhSortVO.setWarehouseCode(targetWhCode);
                                    whAutoAllotGroupWhSortVO.setPriorityInGroup(whAutoAllotGroupSort.getPriorityInGroup());
                                    String allotOnTheWayKey = new StringBuffer().append(targetWhCode).append(":").append(skuCode)
                                            .toString();
                                    whAutoAllotGroupWhSortVO.setAllotOnTheWayQunatity(invQttMap.containsKey(allotOnTheWayKey)
                                            ? invQttMap.get(allotOnTheWayKey) : 0);
                                    List<WhAutoAllotCalculateWhVO> calculateWhList = new ArrayList<>();
                                    List<String> calculateWhCodeList = targetEntry.getValue();
                                    for (String calculateCode : calculateWhCodeList) {
                                        WhAutoAllotCalculateWhVO allotCalculateWhVO = new WhAutoAllotCalculateWhVO();
                                        allotCalculateWhVO.setCalculateWarehouseCode(calculateCode);
                                        String key = new StringBuffer().append(calculateCode).append(":").append(skuCode)
                                                .toString();

                                        allotCalculateWhVO.setPlanAllotQuantity(planMap.get(key) == null ? 0 : planMap.get(key));
                                        calculateWhList.add(allotCalculateWhVO);
                                    }
                                    whAutoAllotGroupWhSortVO.setCalculateWhList(calculateWhList);
                                    groupWhList.add(whAutoAllotGroupWhSortVO);
                                }
                            }
                        }
                    }
                    autoAllotGroupVO.setGroupWhList(groupWhList);
                    groupList.add(autoAllotGroupVO);
                }
                autoAllotSkuVO.setGroupList(groupList);
                autoAllotSkuVOList.add(autoAllotSkuVO);
            }
        }
        return autoAllotSkuVOList;
    }

    /*
      封装成分货仓所属组以及在组中的优先级Map
     */
    private Map<String, WhAutoAllotGroupSort> buildGroupSortMap(List<WhAutoAllotGroupSort> groupSortList) {
        Map<String, WhAutoAllotGroupSort> groupSortMap = new HashMap<>();
        for (WhAutoAllotGroupSort whAutoAllotGroupSort : groupSortList) {
            groupSortMap.put(whAutoAllotGroupSort.getWarehouseCode(), whAutoAllotGroupSort);
        }
        return groupSortMap;
    }

    @Override
    public WhAutoAllotRuleVO findCurrentRule(String bu, boolean cascade) {
        List<WhAutoAllotRuleVO> ruleList = whAutoAllotRuleMapper.selectCurrentRules(bu, (short) 0);
        if (EmptyUtil.isNotEmpty(ruleList)) {
            WhAutoAllotRuleVO currentRule = ruleList.get(0);
            if (cascade) {
                Integer ruleId = currentRule.getId();
                currentRule.setSkuList(whAutoAllotSkuMapper.selectByRuleId(ruleId));
                currentRule.setCallbackWhList(whAutoAllotCallbackWhMapper.selectByRuleId(ruleId));
                currentRule.setGroupSortList(whAutoAllotGroupSortMapper.selectByRuleId(ruleId));
                currentRule.setCallbackExcludeCategoryList(whAutoAllotCallbackExcludeCategoryMapper.selectByRuleId(ruleId));
            }
            return currentRule;
        } else {
            return null;
        }
    }

    @Override
    public WhAutoAllotRuleVO findCurrentRuleOnline(boolean cascade) {
        List<WhAutoAllotRuleVO> ruleList = whAutoAllotRuleMapper.selectCurrentRules(null, (short) 1);
        if (EmptyUtil.isNotEmpty(ruleList)) {
            WhAutoAllotRuleVO currentRule = ruleList.get(0);
            if (cascade) {
                Integer ruleId = currentRule.getId();
                currentRule.setSkuList(whAutoAllotSkuMapper.selectByRuleId(ruleId));
                currentRule.setExcludeCategoryList(whAutoAllotExcludeCategoryMapper.selectByRuleId(ruleId));
            }
            return currentRule;
        } else {
            return null;
        }
    }

    @Override
    public PageInfo<WhAutoAllotRecordVO> findAutoAllotRecords(WhAutoAllotCond whAutoAllotCond) {
        int offset = whAutoAllotCond.getCurrpage() * whAutoAllotCond.getPagenum();
        PageRowBounds pageRowBounds = new PageRowBounds(offset, whAutoAllotCond.getPagenum());
        PageInfo<WhAutoAllotRecordVO> pageInfo = new PageInfo(whAutoAllotRecordMapper.selectAutoAllotRecords(whAutoAllotCond, pageRowBounds));
        pageInfo.setTotal(pageRowBounds.getTotal());
        return pageInfo;
    }

    @Override
    public List<WhAutoAllotRecordDetailVO> findDetailByRecordId(Integer recordId) {
        return whAutoAllotRecordDetailMapper.selectDetailByRecordId(recordId);
    }

    @Override
    @Transactional
    public Set<String> confirmExecuteAutoAllot(Integer recordId) throws Exception {
        WhAutoAllotRecord whAutoAllotRecord = whAutoAllotRecordMapper.selectByPrimaryKey(recordId);
        whAutoAllotRecord.setAllotStatus((short) WhAutoAllotStatusEnum.COMPLETED.getVal());
        whAutoAllotRecordMapper.updateByPrimaryKeySelective(whAutoAllotRecord);
        Set<String> allotRcdCodeList = findAllotRcdCodeByRecordId(recordId);
        if (allotRcdCodeList != null) {
            for (String allotRcdCode : allotRcdCodeList) {
                whAllotService.confirmExecuteAutoAllot(allotRcdCode);
            }
            return allotRcdCodeList;
        }
        return null;
    }


    private Set<String> findAllotRcdCodeByRecordId(Integer recordId) {
        List<WhAutoAllotRecordDetailVO> detailList = findDetailByRecordId(recordId);
        if (detailList != null) {
            Set<String> allotRcdCodeList = new HashSet<>();
            for (WhAutoAllotRecordDetailVO detailVO : detailList) {
                if (EmptyUtil.isNotEmpty(detailVO.getAllotRcdCode())) {
                    allotRcdCodeList.add(detailVO.getAllotRcdCode());
                }
            }
            return allotRcdCodeList;
        }
        return null;
    }

    @Override
    @Transactional
    public Set<String> cancleExecuteAutoAllot(Integer recordId) throws Exception {
        WhAutoAllotRecord whAutoAllotRecord = whAutoAllotRecordMapper.selectByPrimaryKey(recordId);
        whAutoAllotRecord.setAllotStatus((short) WhAutoAllotStatusEnum.CANCLE.getVal());
        whAutoAllotRecordMapper.updateByPrimaryKeySelective(whAutoAllotRecord);
        Set<String> allotRcdCodeList = findAllotRcdCodeByRecordId(recordId);
        if (allotRcdCodeList != null) {
            for (String allotRcdCode : allotRcdCodeList) {
                whAllotService.cancelAllotRcdByCode(allotRcdCode);
            }
            return allotRcdCodeList;
        }
        return null;
    }

    @Override
    public List<String> checkWhetherNeedAutoAllot(List<String> skuCodes, String warehouseCode) {

        WhAutoAllotRuleVO whAutoAllotRuleVO = findCurrentRule(null, false);
        if (whAutoAllotRuleVO == null) {
            return new ArrayList<String>();
        }
        WhAutoAllotSkuExample example = new WhAutoAllotSkuExample();
        example.createCriteria().andSkuCodeIn(skuCodes).andTargetWarehouseCodeEqualTo(warehouseCode).andRuleIdEqualTo(whAutoAllotRuleVO.getId());
        List<WhAutoAllotSku> list = whAutoAllotSkuMapper.selectByExample(example);
        if (EmptyUtil.isEmpty(list)) {
            return new ArrayList<String>();
        }

        List<String> results = new ArrayList<String>();
        int listSize = list.size();
        for (int i = 0; i < listSize; i++) {
            results.add(list.get(i).getSkuCode());
        }

        return results;
    }

    @Override
    public void autoAllotForLbLimitSale() {
        List<WhLbLimitAllotSku> allotSkus = getTodayLbLimitAllotSkus();

        List<String> warehouseCodeList = allotSkus.stream().map(allotSku -> allotSku.getWarehouseCode())
                .collect(Collectors.toList());
        warehouseCodeList = removeSameGroupWh(warehouseCodeList);
        Map<String, WhPhysicalWarehouse> whPhysicalWarehouseMap = warehouseCodeList.size() == 0
                ? new HashMap<String, WhPhysicalWarehouse>() : whInfoService.findPhyWhMapByWarehouseCodes(new LinkedList<String>(warehouseCodeList));

        allotSkus.stream().forEach(allotSku -> {
            buildAllotRcd(allotSku, whPhysicalWarehouseMap);
        });
    }

    private List<WhLbLimitAllotSku> getTodayLbLimitAllotSkus() {
        String today = new StringBuffer().append(DateUtil.getCurrMonth()).append("月")
                .append(DateUtil.getCurrDayOfMonth()).append("日").toString();
        WhLbLimitAllotSkuExample example = new WhLbLimitAllotSkuExample();
        example.createCriteria().andLimitDateEqualTo(today);
        return whLbLimitAllotSkuMapper.selectByExample(example);
    }

    private void buildAllotRcd(WhLbLimitAllotSku allotSku, Map<String, WhPhysicalWarehouse> whPhysicalWarehouseMap) {
        try {
            //限量大于0，需要自动分货
            if (allotSku != null && allotSku.getLimitQuantity() != null
                    && allotSku.getLimitQuantity() > 0) {
                WhAllotRcd rcd = new WhAllotRcd();
                rcd.setSourceWarehouseCode(WAREHOUSECODE);
                rcd.setSourcePhysicalWarehouseCode(Constants.PHYSICAL_WAREHOUSE);

                String targetWarehouseCode = allotSku.getWarehouseCode();
                rcd.setTargetPhysicalWarehouseCode(whPhysicalWarehouseMap.containsKey(targetWarehouseCode)
                        ? whPhysicalWarehouseMap.get(targetWarehouseCode).getCode() : Constants.PHYSICAL_WAREHOUSE);
                rcd.setTargetWarehouseCode(targetWarehouseCode);
                rcd.setAllotType(WhAllotTypeEnum.TYPE_NORMAL.getVal());
                rcd.setCreateUserId(1);
                rcd.setRemark("LB门店门店限量活动，系统自动调拨");
                rcd.setEstimatedAllocationDate(new Date());

                List<WhAllotRcdSku> whAllotRcdSkuList = new ArrayList<WhAllotRcdSku>();
                WhAllotRcdSku whAllotRcdSku = new WhAllotRcdSku();
                whAllotRcdSku.setQuantity(allotSku.getLimitQuantity());
                whAllotRcdSku.setSkuCode(allotSku.getSkuCode());
                whAllotRcdSkuList.add(whAllotRcdSku);

                rcd.setWhAllotRcdSkuList(whAllotRcdSkuList);
                whAllotService.createAllotRcd(rcd);
            }
        } catch (Exception e) {
            logger.error("LB商品限量销售分货失败：{}", JSON.toJSONString(allotSku), e);
        }
    }
    public static void main(String[] args) {
        System.out.println(DateUtil.getCurrMonth());
        System.out.println(DateUtil.getCurrDayOfMonth());
    }

    //保存排除分类
    private int saveExcludeCategory(WhAutoAllotRuleVO whAutoAllotRule, Integer currentRuleId) {
        Integer ruleId = whAutoAllotRule.getId();
        if (ruleId == null) {
            throw new WarehouseException("ruleId不能为空！");
        }
        List<WhAutoAllotExcludeCategory> excludeCategoryList = whAutoAllotRule.getExcludeCategoryList();
        //导入产能不保存排除分类，从db取一次
        if (whAutoAllotRule.getOptType() == 2 && currentRuleId != null) {
            excludeCategoryList = whAutoAllotExcludeCategoryMapper.selectByRuleId(currentRuleId);
        }
        if (EmptyUtil.isNotEmpty(excludeCategoryList)) {
            for (WhAutoAllotExcludeCategory excludeCategory : excludeCategoryList) {
                excludeCategory.setRuleId(ruleId);
            }
            return whAutoAllotExcludeCategoryMapper.batchInsert(excludeCategoryList);
        }
        return 1;
    }


    //保存回调排除分类
    private int saveCallbackExcludeCategory(WhAutoAllotRuleVO whAutoAllotRule, Integer currentRuleId) {
        Integer ruleId = whAutoAllotRule.getId();
        if (ruleId == null) {
            throw new WarehouseException("ruleId不能为空！");
        }
        List<WhAutoAllotCallbackExcludeCategory> excludeCategoryList = whAutoAllotRule.getCallbackExcludeCategoryList();
        //导入产能不保存排除分类，从db取一次
        if (whAutoAllotRule.getOptType() == 2 && currentRuleId != null) {
            excludeCategoryList = whAutoAllotCallbackExcludeCategoryMapper.selectByRuleId(currentRuleId);
        }
        if (EmptyUtil.isNotEmpty(excludeCategoryList)) {
            for (WhAutoAllotCallbackExcludeCategory excludeCategory : excludeCategoryList) {
                excludeCategory.setRuleId(ruleId);
            }
            return whAutoAllotCallbackExcludeCategoryMapper.batchInsert(excludeCategoryList);
        }
        return 1;
    }

    //保存分组排序
    private int saveGroupSort(WhAutoAllotRuleVO whAutoAllotRule, Integer currentRuleId) {
        Integer ruleId = whAutoAllotRule.getId();
        if (ruleId == null) {
            throw new WarehouseException("ruleId不能为空！");
        }
        List<WhAutoAllotGroupSort> groupSortList = whAutoAllotRule.getGroupSortList();

        if (whAutoAllotRule.getOptType() == 1 && currentRuleId != null) {
            groupSortList = whAutoAllotGroupSortMapper.selectByRuleId(currentRuleId);
        }

        if (EmptyUtil.isNotEmpty(groupSortList)) {
            for (WhAutoAllotGroupSort groupSort : groupSortList) {
                groupSort.setRuleId(ruleId);
            }
            return whAutoAllotGroupSortMapper.batchInsert(groupSortList);
        }
        return 1;
    }

    //保存回调仓库
    private int saveCallbackWh(WhAutoAllotRuleVO whAutoAllotRule, Integer currentRuleId) {
        Integer ruleId = whAutoAllotRule.getId();
        if (ruleId == null) {
            throw new WarehouseException("ruleId不能为空！");
        }
        List<WhAutoAllotCallbackWh> callbackWhList = whAutoAllotRule.getCallbackWhList();

        if (whAutoAllotRule.getOptType() == 1 && currentRuleId != null) {
            callbackWhList = whAutoAllotCallbackWhMapper.selectByRuleId(currentRuleId);
        }

        if (EmptyUtil.isNotEmpty(callbackWhList)) {
            for (WhAutoAllotCallbackWh callbackWh : callbackWhList) {
                callbackWh.setRuleId(ruleId);
            }
            return whAutoAllotCallbackWhMapper.batchInsert(callbackWhList);
        }
        return 1;
    }

    //保存回分货sku
    private int saveAllotSku(WhAutoAllotRuleVO whAutoAllotRule, Integer currentRuleId) {
        Integer ruleId = whAutoAllotRule.getId();
        if (ruleId == null) {
            throw new WarehouseException("ruleId不能为空！");
        }
        List<WhAutoAllotSku> skuList = whAutoAllotRule.getSkuList();

        if (whAutoAllotRule.getOptType() == 1 && currentRuleId != null) {
            skuList = whAutoAllotSkuMapper.selectByRuleId(currentRuleId);
        }
        if (EmptyUtil.isNotEmpty(skuList)) {
            for (WhAutoAllotSku allotSku : skuList) {
                allotSku.setRuleId(ruleId);
            }
            return whAutoAllotSkuMapper.batchInsert(skuList);
        }
        return 1;
    }

    /**
     * 线上分货，可能部分失败，失败的实际分货数量设置为0（线上同步天猫可能失败）
     *
     * @param preRcds
     * @param rcds
     */
    private void buildRealAllotQuantity(List<WhAllotRcd> preRcds, List<WhAllotRcd> rcds) {
        if (preRcds != null && preRcds.size() > 0) {
            int preSize = preRcds.size();
            for (int i = 0; i < preSize; i++) {
                WhAllotRcd preRcd = preRcds.get(i);
                WhAllotRcd rcd = rcds.get(i);
                List<WhAllotRcdSku> preRcdSkuList = preRcd.getWhAllotRcdSkuList();
                if (preRcdSkuList != null && preRcdSkuList.size() > 0) {
                    List<WhAllotRcdSku> rcdSkuList = rcd.getWhAllotRcdSkuList();
                    for (WhAllotRcdSku prRcdSku : preRcdSkuList) {
                        //实际分货是否成功
                        boolean allotSuccess = false;
                        //单个调拨单下全部同步失败
                        if (rcdSkuList == null || rcdSkuList.size() == 0) {
                            prRcdSku.setQuantity(0);
                            break;
                        }
                        for (WhAllotRcdSku rcdSku : rcdSkuList) {
                            //有实际分货
                            if (prRcdSku.getSkuCode().equals(rcdSku.getSkuCode())) {
                                allotSuccess = true;
                                break;
                            }
                        }
                        if (!allotSuccess) {
                            prRcdSku.setQuantity(0);
                        }
                    }
                }
            }
        }
    }

    private String generatorRecordLineCode(Integer recodeLineId) {
        Map<String, Object> params = new HashMap<>();
        params.put("createTime", DateUtil.getNow());
        params.put("id", recodeLineId);
        return CodeGenerator.getInstance().generate("AUTO_ALLOT_RECORD_LINE_CODE", params);
    }

    /**
     * 使用 Map按value进行排序
     *
     * @param map
     * @return
     */
    public Map<String, Integer> sortMapByValue(Map<String, Integer> oriMap, boolean asc) {
        if (oriMap == null || oriMap.isEmpty()) {
            return null;
        }
        Map<String, Integer> sortedMap = new LinkedHashMap<String, Integer>();
        List<Map.Entry<String, Integer>> entryList = new ArrayList<Map.Entry<String, Integer>>(
                oriMap.entrySet());
        Collections.sort(entryList, asc ? new MapValueComparator() : new MapValueComparatorDesc());

        Iterator<Map.Entry<String, Integer>> iter = entryList.iterator();
        Map.Entry<String, Integer> tmpEntry = null;
        while (iter.hasNext()) {
            tmpEntry = iter.next();
            sortedMap.put(tmpEntry.getKey(), tmpEntry.getValue());
        }
        return sortedMap;
    }

    class MapValueComparator implements Comparator<Map.Entry<String, Integer>> {

        @Override
        public int compare(Map.Entry<String, Integer> me1, Map.Entry<String, Integer> me2) {

            return me1.getValue().compareTo(me2.getValue());
        }
    }

    class MapValueComparatorDesc implements Comparator<Map.Entry<String, Integer>> {

        @Override
        public int compare(Map.Entry<String, Integer> me1, Map.Entry<String, Integer> me2) {
            return -me1.getValue().compareTo(me2.getValue());
        }
    }
}
