/*
 * Copyright (C), 上海布鲁爱电子商务有限公司
 */
package com.thebeastshop.pegasus.service.purchase.service.impl;

import com.thebeastshop.pegasus.merchandise.enums.PcsSkuSalesPriceChangeStatusEnum;
import com.thebeastshop.pegasus.merchandise.service.McPcsSkuSalesPriceChangeService;
import com.thebeastshop.pegasus.merchandise.service.McPcsSkuService;
import com.thebeastshop.pegasus.merchandise.vo.PcsSkuVO;
import com.thebeastshop.pegasus.service.purchase.cond.PrdcRecipeCond;
import com.thebeastshop.pegasus.service.purchase.cond.PrdcRecipeMaterialCond;
import com.thebeastshop.pegasus.service.purchase.dao.PrdcRecipeMapper;
import com.thebeastshop.pegasus.service.purchase.dao.PrdcRecipeMaterialMapper;
import com.thebeastshop.pegasus.service.purchase.dao.PrdcRecipeSkuMapper;
import com.thebeastshop.pegasus.service.purchase.exception.PurchaseException;
import com.thebeastshop.pegasus.service.purchase.exception.PurchaseExceptionErrorCode;
import com.thebeastshop.pegasus.service.purchase.model.*;
import com.thebeastshop.pegasus.service.purchase.service.PcsSkuAutoApproveRuleService;
import com.thebeastshop.pegasus.service.purchase.service.PrdcRecipeService;
import com.thebeastshop.pegasus.service.purchase.vo.PrdcRecipeMaterialSimpleVO;
import com.thebeastshop.pegasus.service.purchase.vo.PrdcRecipeMaterialVO;
import com.thebeastshop.pegasus.service.purchase.vo.PrdcRecipeVO;
import com.thebeastshop.pegasus.util.comm.BeanUtil;
import com.thebeastshop.pegasus.util.comm.DateUtil;
import com.thebeastshop.pegasus.util.comm.EmptyUtil;
import com.thebeastshop.pegasus.util.comm.NullUtil;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.*;

/**
 * @author Eric.Lou
 * @version $Id: PrdcRecipeServiceImpl.java, v 0.1 2015-12-30 17:11
 */
@Service("prdcRecipeService")
public class PrdcRecipeServiceImpl implements PrdcRecipeService {

    @Autowired
    private McPcsSkuService mcPcsSkuService;

    @Autowired
    private PrdcRecipeMapper prdcRecipeMapper;

    @Autowired
    private PrdcRecipeMaterialMapper prdcRecipeMaterialMapper;

    @Autowired
    private PrdcRecipeSkuMapper prdcRecipeSkuMapper;

    @Autowired
    private McPcsSkuSalesPriceChangeService mcPcsSkuSalesPriceChangeService;

    @Autowired
    private PcsSkuAutoApproveRuleService pcsSkuAutoApproveRuleService;

    @Override
    public PrdcRecipeVO findRecipeVOById(Long id) {
        PrdcRecipeCond cond = new PrdcRecipeCond();
        cond.setRecipeId(id);
        cond.setCurrpage(1);
        cond.setFetchSkuImg(true);
        List<PrdcRecipeVO> prdcRecipeVOList = findRecipeVOByCond(cond, true);
        if(CollectionUtils.isEmpty(prdcRecipeVOList)){
            return null;
        }else{
            return prdcRecipeVOList.get(0);
        }
    }
    
    private List<PrdcRecipeMaterialVO> findRecipeMaterialVOByRecipeIds(List<Long> recipeIds){
    	return prdcRecipeMaterialMapper.findRecipeMaterialVOByRecipeIds(recipeIds);
    }

    private List<PrdcRecipeSku> findRecipeSkuByRecipeIds(List<Long> recipeIds){
        PrdcRecipeSkuExample example = new PrdcRecipeSkuExample();
        example.createCriteria().andRecipeIdIn(recipeIds);
    	return prdcRecipeSkuMapper.selectByExample(example);
    }

    @Override
    public List<PrdcRecipeMaterialVO> findRecipeMaterialVOByRecipeId(Long recipeId) {
        PrdcRecipeMaterialExample example = new PrdcRecipeMaterialExample();
        example.createCriteria().andRecipeIdEqualTo(recipeId);
        List<PrdcRecipeMaterial> prdcRecipeMaterialList = prdcRecipeMaterialMapper.selectByExample(example);
        if(CollectionUtils.isEmpty(prdcRecipeMaterialList)){
            return Collections.emptyList();
        }else{
            List<PrdcRecipeMaterialVO> prdcRecipeMaterialVOList = BeanUtil.buildListFrom(prdcRecipeMaterialList, PrdcRecipeMaterialVO.class);
            for (PrdcRecipeMaterialVO prdcRecipeMaterialVO : prdcRecipeMaterialVOList) {
                PcsSkuVO skuVO = mcPcsSkuService.findByCode(prdcRecipeMaterialVO.getSkuCode());
                if(EmptyUtil.isNotEmpty(skuVO)) {
                	prdcRecipeMaterialVO.setSkuName(skuVO.getNameCn()==null?"":skuVO.getNameCn());
                }
            }
            return prdcRecipeMaterialVOList;
        }
    }

    @Override
    public List<PrdcRecipeVO> findRecipeVOByCond(PrdcRecipeCond cond, Boolean cascade) {
        List<PrdcRecipeVO> prdcRecipeVOList = prdcRecipeMapper.findRecipeVOByCond(cond);
        if(CollectionUtils.isEmpty(prdcRecipeVOList)){
            return Collections.emptyList();
        }else{
            //List<PrdcRecipeVO> prdcRecipeVOList = prdcRecipeList;
            List<Long> ids = new ArrayList<>();
            if(cascade){
                for(PrdcRecipeVO recipe : prdcRecipeVOList){
                	ids.add(recipe.getId());
                }
            	List<PrdcRecipeMaterialVO> recipeMaterialList = findRecipeMaterialVOByRecipeIds(ids);
                Map<Long,List<PrdcRecipeMaterialVO>> recipeMap = new HashMap<>();
                for(PrdcRecipeMaterialVO m : recipeMaterialList){
                	List<PrdcRecipeMaterialVO> tmp = recipeMap.get(m.getRecipeId());
                	if(NullUtil.isNull(tmp)){
                		tmp = new ArrayList<>();
                		recipeMap.put(m.getRecipeId(), tmp);
                	}
                	tmp.add(m);
                }
                for (PrdcRecipeVO prdcRecipeVO : prdcRecipeVOList) {
                	List<PrdcRecipeMaterialVO> mList = recipeMap.get(prdcRecipeVO.getId());
                	if(NullUtil.isNull(mList)){
                		mList = new ArrayList<>();
                	}
                	prdcRecipeVO.setRecipeMaterialVOList(mList);
                }
            }
            if (cond.isFetchSkuImg()){
                // 获取配方关联的SKU图片
                List<PrdcRecipeSku> prdcRecipeSkus = findRecipeSkuByRecipeIds(ids);
                Map<Long,List<PrdcRecipeSku>> recipeSkuMap = new HashMap<>();
                for(PrdcRecipeSku receipeSku : prdcRecipeSkus){
                    List<PrdcRecipeSku> tmp = recipeSkuMap.get(receipeSku.getRecipeId());
                    if(NullUtil.isNull(tmp)){
                        tmp = new ArrayList<>();
                        recipeSkuMap.put(receipeSku.getRecipeId(), tmp);
                    }
                    tmp.add(receipeSku);
                }
                for (PrdcRecipeVO prdcRecipeVO : prdcRecipeVOList) {
                    List<PrdcRecipeSku> mList = recipeSkuMap.get(prdcRecipeVO.getId());
                    if(NullUtil.isNull(mList)){
                        mList = new ArrayList<>();
                    }
                    prdcRecipeVO.setPrdcRecipeSkus(mList);
                }
            }
            return prdcRecipeVOList;
        }
    }

    @Override
    public List<PrdcRecipeMaterialVO> findRecipeMaterialVOByCond(PrdcRecipeMaterialCond cond, Boolean cascade) {
        PrdcRecipeMaterialExample example = new PrdcRecipeMaterialExample();
        PrdcRecipeMaterialExample.Criteria criteria = example.createCriteria();
        if (EmptyUtil.isNotEmpty(cond.getSkuCode())){
            criteria.andSkuCodeEqualTo(cond.getSkuCode());
        }
        if (CollectionUtils.isNotEmpty(cond.getSkuCodes())){
            criteria.andSkuCodeIn(cond.getSkuCodes());
        }
        List<PrdcRecipeMaterial> prdcRecipeMaterialList = prdcRecipeMaterialMapper.selectByExample(example);
        if(CollectionUtils.isEmpty(prdcRecipeMaterialList)){
            return Collections.emptyList();
        }
        return BeanUtil.buildListFrom(prdcRecipeMaterialList,PrdcRecipeMaterialVO.class);
    }

    @Override
    public List<PrdcRecipeVO> findRecipeVOBySkuCode(String skuCode, Boolean cascade) {
        PrdcRecipeExample example = new PrdcRecipeExample();
        example.createCriteria().andSkuCodeEqualTo(skuCode);
        List<PrdcRecipe> prdcRecipeList = prdcRecipeMapper.selectByExample(example);
        if(CollectionUtils.isEmpty(prdcRecipeList)){
            return Collections.emptyList();
        }
        return BeanUtil.buildListFrom(prdcRecipeList,PrdcRecipeVO.class);
    }

    @Override
    public List<PrdcRecipeVO> findRecipeVOBySkuCode(List<String> skuCodes) {
        PrdcRecipeExample example = new PrdcRecipeExample();
        example.createCriteria().andSkuCodeIn(skuCodes);
        List<PrdcRecipe> prdcRecipeList = prdcRecipeMapper.selectByExample(example);
        if(CollectionUtils.isEmpty(prdcRecipeList)){
            return Collections.emptyList();
        }
        return BeanUtil.buildListFrom(prdcRecipeList,PrdcRecipeVO.class);
    }

    @Override
    public Map<String, PrdcRecipeVO> mapRecipeVOBySkuCode(List<String> skuCodes) {
        List<PrdcRecipeVO> list = findRecipeVOBySkuCode(skuCodes);
        Map<String, PrdcRecipeVO> map = new HashMap<String, PrdcRecipeVO>();
        if (CollectionUtils.isEmpty(list)) {
          return map;
        }
        int listSize = list.size();
        for (int i = 0 ; i < listSize; i ++) {
            PrdcRecipeVO recipeVO = list.get(i);
            map.put(recipeVO.getSkuCode(), recipeVO);
        }
        return map;
    }

    @Override
    public List<PrdcRecipeMaterialVO> findRecipeMaterialVOBySkuCode(String skuCode, Boolean cascade) {
        PrdcRecipeMaterialExample example = new PrdcRecipeMaterialExample();
        example.createCriteria().andSkuCodeEqualTo(skuCode);
        List<PrdcRecipeMaterial> prdcRecipeMaterialList = prdcRecipeMaterialMapper.selectByExample(example);
        if(CollectionUtils.isEmpty(prdcRecipeMaterialList)){
            return Collections.emptyList();
        }
        return BeanUtil.buildListFrom(prdcRecipeMaterialList,PrdcRecipeMaterialVO.class);
    }

    @Override
    public List<PrdcRecipeSku> findPrdcRecipeSkuByCond(PrdcRecipeCond cond) {
        PrdcRecipeSkuExample example = new PrdcRecipeSkuExample();
        example.createCriteria().andRecipeIdEqualTo(cond.getRecipeId()).andSkuCodeEqualTo(cond.getRecipeSkuCode());
        List<PrdcRecipeSku> prdcRecipeSkus = prdcRecipeSkuMapper.selectByExample(example);
        if(CollectionUtils.isEmpty(prdcRecipeSkus)){
            return Collections.emptyList();
        }
        return prdcRecipeSkus;
    }

    @Override
    @Transactional
    public Long create(PrdcRecipeVO vo) {;
        PrdcRecipe prdcRecipe = BeanUtil.buildFrom(vo, PrdcRecipe.class);
        List<PrdcRecipeMaterial> prdcRecipeMaterialList = BeanUtil.buildListFrom(vo.getRecipeMaterialVOList(), PrdcRecipeMaterial.class);
        createRecipe(prdcRecipe);
        saveRecipeMaterial(prdcRecipe,prdcRecipeMaterialList);

        // 保存配方成品SKU图片信息
        List<PrdcRecipeSku> prdcRecipeSkus = vo.getPrdcRecipeSkus();
        createRecipeSkuImg(prdcRecipe,prdcRecipeSkus);

        //validateCost(vo);
        return prdcRecipe.getId();
    }

    @Override
    @Transactional
    public Boolean update(PrdcRecipeVO vo) {
        PrdcRecipe prdcRecipe = BeanUtil.buildFrom(vo, PrdcRecipe.class);
        List<PrdcRecipeMaterial> prdcRecipeMaterialList = BeanUtil.buildListFrom(vo.getRecipeMaterialVOList(), PrdcRecipeMaterial.class);
        updateRecipe(prdcRecipe);
        saveRecipeMaterial(prdcRecipe,prdcRecipeMaterialList);

        // 保存配方成品SKU图片信息
        List<PrdcRecipeSku> prdcRecipeSkus = vo.getPrdcRecipeSkus();
        createRecipeSkuImg(prdcRecipe,prdcRecipeSkus);

        //validateCost(vo);
        return true;
    }

    @Override
    public Boolean updateRecipeHour(PrdcRecipeVO vo) {
        PrdcRecipe prdcRecipe = BeanUtil.buildFrom(vo, PrdcRecipe.class);
        updateRecipe(prdcRecipe);
        return true;
    }

    @Override
    public Boolean deleteRecipeMaterial(Long recipeId, Long recipeMaterialId) {
        if(recipeId==null||recipeMaterialId==null){
            throw new PurchaseException(PurchaseExceptionErrorCode.ILLEGAL_PARAM,"参数异常");
        }
        PrdcRecipeVO prdcRecipeVO = findRecipeVOById(recipeId);
        if(prdcRecipeVO==null){
            throw new PurchaseException(PurchaseExceptionErrorCode.ILLEGAL_PARAM,"匹配不到配方");
        }
        Boolean found = false;
        for (PrdcRecipeMaterialVO prdcRecipeMaterialVO : prdcRecipeVO.getRecipeMaterialVOList()) {
            if(!prdcRecipeMaterialVO.getId().equals(recipeMaterialId)){
                continue;
            }
            found = true;
        }
        if(found){
           if(prdcRecipeMaterialMapper.deleteByPrimaryKey(recipeMaterialId)==0){
               throw new PurchaseException(PurchaseExceptionErrorCode.ERROR_DELETE_DB,"删除配方行失败");
           }
        }else{
            throw new PurchaseException(PurchaseExceptionErrorCode.ILLEGAL_PARAM,"配方不包含该行数据");
        }
        return true;
    }

    @Override
    public List<PrdcRecipeMaterialSimpleVO> findBottomMaterialByRecipeSkuCodes(List<String> skuCodes) {

        List<PrdcRecipeMaterialSimpleVO> materialSimpleList = prdcRecipeMaterialMapper.findMaterialsByRecipeSkus(skuCodes);

        List<PrdcRecipeMaterialSimpleVO> querySku =  new ArrayList<>();
        List<String> querySkuCodes =  new ArrayList<>();
        //耗材SKU　List
        List<PrdcRecipeMaterialSimpleVO> materialSkuList = new ArrayList<>();

        if (EmptyUtil.isNotEmpty(materialSimpleList)) {
            for (PrdcRecipeMaterialSimpleVO simpleVO : materialSimpleList) {
                querySku.add(simpleVO);
                querySkuCodes.add(simpleVO.getMaterialSkuCode());
            }
            while (querySkuCodes.size() > 0) {
                materialSimpleList = prdcRecipeMaterialMapper.findMaterialsByRecipeSkus(querySkuCodes);
                List<String> newQuerySkuCodes = new ArrayList<>();
                List<PrdcRecipeMaterialSimpleVO> newQuerySku =  new ArrayList<>();
                int querySize = querySkuCodes.size();
                for (int i = 0; i < querySize; i ++) {
                    String skuCode = querySkuCodes.get(i);
                    boolean isBottom = true;
                    for (PrdcRecipeMaterialSimpleVO simple : materialSimpleList) {
                        //可能有下一层耗材
                        if (skuCode.contains(simple.getRecipeSkuCode())) {
                            PrdcRecipeMaterialSimpleVO newSimple = new PrdcRecipeMaterialSimpleVO();
                            newSimple.setRecipeSkuCode(querySku.get(i).getRecipeSkuCode());
                            newSimple.setMaterialSkuCode(simple.getMaterialSkuCode());
                            newQuerySku.add(newSimple);
                            newQuerySkuCodes.add(simple.getMaterialSkuCode());
                            isBottom  = false;
                        }
                    }
                    if (isBottom) {
                        materialSkuList.add(querySku.get(i));
                    }
                }
                querySkuCodes = newQuerySkuCodes;
                querySku = newQuerySku;
            }
        }
        return materialSkuList;
    }

    /**
     * 校验配方
     * @param isCreat
     * @param rcd
     */
    private void validateRecipe(Boolean isCreat,PrdcRecipe rcd){
        PrdcRecipeCond con = new PrdcRecipeCond();
        con.setRecipeSkuCode(rcd.getSkuCode());
        List<PrdcRecipeVO> list = prdcRecipeMapper.findRecipeVOByCond(con);
        if(isCreat){
            if(rcd.getId()!=null){
                throw new PurchaseException(PurchaseExceptionErrorCode.ILLEGAL_PARAM,"配方已存在ID["+rcd.getId()+"]，无需新增!");
            }
            if(list != null && list.size() > 0){
                throw new PurchaseException(PurchaseExceptionErrorCode.ILLEGAL_PARAM,"成品sku["+rcd.getSkuCode()+"]已有配方!");
            }
        }else{
            if(rcd.getId()==null){
                throw new PurchaseException(PurchaseExceptionErrorCode.ILLEGAL_PARAM,"参数异常");
            }
            if(list.size() > 1){
                throw new PurchaseException(PurchaseExceptionErrorCode.ILLEGAL_PARAM,"成品sku["+rcd.getSkuCode()+"]不能重复");
            }else if(list.size() == 1){
                PrdcRecipeVO prdcRecipe = list.get(0);
                if(!prdcRecipe.getId().equals(rcd.getId())){
                    throw new PurchaseException(PurchaseExceptionErrorCode.ILLEGAL_PARAM,"配方ID不匹配["+prdcRecipe.getId()+"<>"+rcd.getId()+"],不能更新!");
                }
            }
        }
        PcsSkuVO skuVO = mcPcsSkuService.findByCode(rcd.getSkuCode());
        if(skuVO==null){
            throw new PurchaseException(PurchaseExceptionErrorCode.ILLEGAL_PARAM,"skuCode匹配不到sku");
        }
    }

    /**
     * 创建配方
     * @param rcd
     */
    @Transactional
    private void createRecipe(PrdcRecipe rcd){
        validateRecipe(true,rcd);
        rcd.setCreateTime(DateUtil.getNow());
        if(prdcRecipeMapper.insert(rcd)==0){
            throw new PurchaseException(PurchaseExceptionErrorCode.ERROR_INSERT_DB,"数据库插入失败");
        }
    }

    /**
     * 更新配方
     * @param rcd
     */
    @Transactional
    private void updateRecipe(PrdcRecipe rcd){
        validateRecipe(false,rcd);
        if(prdcRecipeMapper.updateByPrimaryKeySelective(rcd)==0){
            throw new PurchaseException(PurchaseExceptionErrorCode.ERROR_UPDATE_DB,"数据库更新失败");
        }
    }

    /**
     * 保存配方行
     * @param prdcRecipe
     * @param rcdList
     */
    private void saveRecipeMaterial(PrdcRecipe prdcRecipe,List<PrdcRecipeMaterial> rcdList){
        Boolean flag = false;
        for (PrdcRecipeMaterial rcd : rcdList) {
            if (rcd.getMaterialType().equals(1)) {
                flag =true;
            }
        }
        if (!flag) {
            throw new PurchaseException(PurchaseExceptionErrorCode.ILLEGAL_PARAM,"配方至少需要1个主耗材");
        }
        for (PrdcRecipeMaterial rcd : rcdList) {
            if(rcd.getId() == null){
                createRecipeMaterial(prdcRecipe,rcd);
            }else{
                updateRecipeMaterial(prdcRecipe,rcd);
            }
        }
    }

    /**
     * 校验配方行
     * @param rcd
     */
    private void validateRecipeMaterial(PrdcRecipe prdcRecipe,PrdcRecipeMaterial rcd){
        PcsSkuVO skuVO = mcPcsSkuService.findByCode(rcd.getSkuCode());
        if(skuVO==null){
            throw new PurchaseException(PurchaseExceptionErrorCode.ILLEGAL_PARAM,"耗材skuCode匹配不到sku");
        }
        if(EmptyUtil.isEmpty(rcd.getQuantity())){
            throw new PurchaseException(PurchaseExceptionErrorCode.ILLEGAL_PARAM,"耗材sku数量不能为空");
        }
        if(rcd.getQuantity().compareTo(new Float(0))<=0){
            throw new PurchaseException(PurchaseExceptionErrorCode.ILLEGAL_PARAM,"耗材sku数量不能小于等于0");
        }
        if(rcd.getSkuCode().equals(prdcRecipe.getSkuCode())){
            throw new PurchaseException(PurchaseExceptionErrorCode.ILLEGAL_PARAM,"耗材SKU包含成品SKU");
        }
        if(EmptyUtil.isEmpty(mcPcsSkuService.findAuditRecordsByCodeAndStatus(rcd.getSkuCode(), PcsSkuSalesPriceChangeStatusEnum.COMPLETED_REVIEW.getStatus()))) {
    		throw new PurchaseException(PurchaseExceptionErrorCode.ILLEGAL_PARAM,"[" + skuVO.getCode() + "]" + skuVO.getNameCn()
    		+ "未通过审批");
    	}
    }

    /**
     * 创建配方行
     * @param prdcRecipe
     * @param rcd
     */
    @Transactional
    private void createRecipeMaterial(PrdcRecipe prdcRecipe,PrdcRecipeMaterial rcd){
        validateRecipeMaterial(prdcRecipe,rcd);
        rcd.setRecipeId(prdcRecipe.getId());
        if(prdcRecipeMaterialMapper.insert(rcd)==0){
            throw new PurchaseException(PurchaseExceptionErrorCode.ERROR_INSERT_DB,"数据库插入失败");
        }
    }

    /**
     * 创建更新配方SKU图片
     * @param prdcRecipeSku
     */
    @Transactional
    private void createOrUpdateRecipeSku(PrdcRecipeSku prdcRecipeSku){
        int result = 0;
        if (EmptyUtil.isNotEmpty(prdcRecipeSku.getId())){
            prdcRecipeSku.setUpdateTime(DateUtil.getNow());
            result = prdcRecipeSkuMapper.updateByPrimaryKeySelective(prdcRecipeSku);
        }else{
            prdcRecipeSku.setCreateTime(DateUtil.getNow());
            result = prdcRecipeSkuMapper.insert(prdcRecipeSku);
        }
        if(result==0){
            throw new PurchaseException(PurchaseExceptionErrorCode.ERROR_INSERT_DB,"数据库插入失败");
        }
    }

    /**
     * 创建配方SKU图片
     * @param prdcRecipe
     * @param prdcRecipeSkus
     */
    @Transactional
    private void createRecipeSkuImg(PrdcRecipe prdcRecipe,List<PrdcRecipeSku> prdcRecipeSkus){
        // 清空以前的关联图片
        deleteRecipeSkuImg(prdcRecipe);
        if (EmptyUtil.isNotEmpty(prdcRecipe.getId()) && CollectionUtils.isNotEmpty(prdcRecipeSkus)){
            for (PrdcRecipeSku prdcRecipeSku : prdcRecipeSkus){
                prdcRecipeSku.setCreateTime(DateUtil.getNow());
                prdcRecipeSku.setRecipeId(prdcRecipe.getId());
                prdcRecipeSku.setSkuCode(prdcRecipe.getSkuCode());
                if(prdcRecipeSkuMapper.insert(prdcRecipeSku)==0){
                    throw new PurchaseException(PurchaseExceptionErrorCode.ERROR_INSERT_DB,"配方SKU图片插入失败");
                }
            }
        }
    }

    @Transactional
    private void deleteRecipeSkuImg(PrdcRecipe prdcRecipe){
        if (EmptyUtil.isNotEmpty(prdcRecipe.getId())){
            PrdcRecipeSkuExample example = new PrdcRecipeSkuExample();
            PrdcRecipeSkuExample.Criteria criteria = example.createCriteria();
            criteria.andRecipeIdEqualTo(prdcRecipe.getId());
            if(EmptyUtil.isNotEmpty(prdcRecipe.getSkuCode())){
                criteria.andSkuCodeEqualTo(prdcRecipe.getSkuCode());
            }
            prdcRecipeSkuMapper.deleteByExample(example);
        }
    }

    /**
     * 更新配方行
     * @param prdcRecipe
     * @param rcd
     */
    @Transactional
    private void updateRecipeMaterial(PrdcRecipe prdcRecipe,PrdcRecipeMaterial rcd){
        validateRecipeMaterial(prdcRecipe,rcd);
        if(prdcRecipeMaterialMapper.updateByPrimaryKeySelective(rcd)==0){
            throw new PurchaseException(PurchaseExceptionErrorCode.ERROR_UPDATE_DB,"数据库更新失败");
        }
    }


}
