package com.thebeastshop.pegasus.merchandise.service;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Map;

import javax.annotation.Resource;

import org.apache.commons.collections.CollectionUtils;
import org.elasticsearch.common.collect.Lists;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import com.thebeastshop.pegasus.merchandise.IService.IInterestGoodsService;
import com.thebeastshop.pegasus.merchandise.exception.InterestException;
import com.thebeastshop.pegasus.merchandise.exception.InterestExceptionEnum;
import com.thebeastshop.pegasus.merchandise.util.DateUtils;
import com.thebeastshop.pegasus.util.comm.DateUtil;
import com.thebeastshop.scm.cond.InterestGoodsCond;
import com.thebeastshop.scm.dao.InterestGoodsDao;
import com.thebeastshop.scm.dao.InterestGoodsProvideDao;
import com.thebeastshop.scm.dao.InterestGoodsSkuDao;
import com.thebeastshop.scm.po.InterestGoods;
import com.thebeastshop.scm.po.InterestGoodsProvide;
import com.thebeastshop.scm.po.InterestGoodsSku;
import com.thebeastshop.scm.vo.interest.InterestGoodsGetVO;
import com.thebeastshop.scm.vo.interest.InterestGoodsProvideVO;
import com.thebeastshop.scm.vo.interest.InterestGoodsSaveVO;
import com.thebeastshop.scm.vo.interest.InterestGoodsSkuVO;
import com.thebeastshop.scm.vo.interest.InterestGoodsVO;

/**
 * 商品权益Service
 * @Class Name InterestGoodsService
 * @Author Xiang.Gao
 * @Create In 2017-5-9
 */
@Service
public class InterestGoodsService implements IInterestGoodsService {
	
	@Resource 
	private InterestGoodsDao interestGoodsDao;
	
	@Resource 
	private InterestGoodsSkuDao interestGoodsSkuDao;
	
	@Resource 
	private InterestGoodsProvideDao interestGoodsProvideDao;
	
	@SuppressWarnings("rawtypes")
	@Override
	public List<Map> list(InterestGoodsCond cond) {
		return interestGoodsDao.list(cond);
	}
	
	@Override
	public InterestGoods selectInterestById(Integer interestGoodsId) {
		return interestGoodsDao.selectOne(interestGoodsId);
	}
	
	@Override
	public InterestGoodsVO selectInterestGoodsSku(Integer interestGoodsId) {
		InterestGoods interestGoods = interestGoodsDao.selectOne(interestGoodsId);
		List<InterestGoodsSku> interestGoodsSkuList = interestGoodsSkuDao.byInterestGoodsId(interestGoodsId);
		return new InterestGoodsVO(interestGoods, interestGoodsSkuList);
	}
	
	@Override
	public int sumQuantityByInterestId(Integer interestGoodsId) {
		return interestGoodsSkuDao.sumQuantityByInterestId(interestGoodsId);
	}
	
	@Transactional("scmDataSourceTx")
	@Override
	public Integer save(InterestGoodsSaveVO vo) {
		InterestGoods interestGoods = vo.toInterestGoods();
		int res = interestGoodsDao.insertOrUpdate(interestGoods);
		if (res != 1) {
			throw new InterestException(InterestExceptionEnum.ERROR_SAVE_INTEREST);
		}
		
		List<InterestGoodsSku> interestGoodsSkuList = vo.toInterestGoodsSku(interestGoods.getId());
		for (InterestGoodsSku interestGoodsSku : interestGoodsSkuList) {
			res = interestGoodsSkuDao.insertOrUpdate(interestGoodsSku);
			if (res != 1) {
				throw new InterestException(InterestExceptionEnum.ERROR_SAVE_INTEREST_SKU);
			}
		}
		return interestGoods.getId();
	}
	
	@Transactional("scmDataSourceTx")
	@Override
	public void delete(int interestGoodsId) {
		int res = interestGoodsDao.delete(interestGoodsId);
		if(res == 0) {
			throw new InterestException(InterestExceptionEnum.ERROR_DELETE_INTEREST);
		}
		res = interestGoodsSkuDao.deleteByInterestGoodsId(interestGoodsId);
		if(res == 0) {
			throw new InterestException(InterestExceptionEnum.ERROR_DELETE_INTEREST_SKU);
		}
	}
	
	@Override
	public List<InterestGoodsProvideVO> getProvideRecord(InterestGoodsCond cond) {
		// TODO Auto-generated method stub
		return interestGoodsProvideDao.queryProvideRecord(cond);
	}
	
	
	/**-----------------------------------   以下接口提供给外部调用       --------------------------*/
	
	@Transactional("scmDataSourceTx")
	@Override
	public InterestGoodsGetVO getInterestGoods(Integer memberId, Integer interestGoodsId){
		if(memberId == null || interestGoodsId == null) {
			throw new InterestException(InterestExceptionEnum.ERROR_PARAM_MISS);
		}
		InterestGoods interestGoods = interestGoodsDao.selectOne(interestGoodsId);
		if (interestGoods == null) {
			throw new InterestException(InterestExceptionEnum.ERROR_NOT_FOUND);
		}
		// 权益领取时间校验
		if (interestGoods.getReceiptStartTime().after(DateUtils.getCurrentDateYMD()) 
				|| interestGoods.getReceiptEndTime().before(DateUtils.getCurrentDateYMD())) {
			throw new InterestException(InterestExceptionEnum.ERROR_RECEIVE_SPOCE);
		}
		// 乐观锁发放权益
		InterestGoodsSku interestGoodsSku = interestGoodsSkuDao.getInStockByInterestGoodsId(interestGoodsId);
		if (interestGoodsSku == null) {
			throw new InterestException(InterestExceptionEnum.ERROR_SKU_NO_STOCK);
		}
		boolean isSuccess = interestGoodsSkuDao.addReceiveQuantity(interestGoodsSku.getId());
		if(!isSuccess) {
			throw new InterestException(InterestExceptionEnum.ERROR_UPDATE_RECEIVE);
		}
		int res = interestGoodsProvideDao.save(memberId, interestGoodsId, interestGoodsSku.getId());
		if(res != 1) {
			throw new InterestException(InterestExceptionEnum.ERROR_PROVIDE_INTEREST);
		}
		return new InterestGoodsGetVO(interestGoods, interestGoodsSku);
	}
	
	@Transactional("scmDataSourceTx")
	@Override
	public void cash(Integer memberId, List<InterestGoodsSkuVO> interestGoodsSkuVOList) {
		if(ObjectUtils.isEmpty(memberId) || CollectionUtils.isEmpty(interestGoodsSkuVOList)) {
			throw new InterestException(InterestExceptionEnum.ERROR_PARAM_NULL);
		}
		// 权益兑取验证
		InterestGoods interestGoods = null;
		for(InterestGoodsSkuVO interestGoodsSkuVO : interestGoodsSkuVOList) {
			if (ObjectUtils.isEmpty(interestGoodsSkuVO.getCashQuantity()) 
					|| ObjectUtils.isEmpty(interestGoodsSkuVO.getInterestGoodsId())
					|| ObjectUtils.isEmpty(interestGoodsSkuVO.getSkuCode())) {
				throw new InterestException(InterestExceptionEnum.ERROR_PARAM_MISS);
			}
			Integer interestGoodsId = interestGoodsSkuVO.getInterestGoodsId();
			String skuCode = interestGoodsSkuVO.getSkuCode();
			InterestGoodsSku interestGoodsSku = interestGoodsSkuDao.getInterestGoodsSku(interestGoodsId, skuCode);
			if(ObjectUtils.isEmpty(interestGoodsSku)) {
				throw new InterestException(InterestExceptionEnum.ERROR_NOT_FOUND);
			}
			interestGoods = interestGoodsDao.selectOne(interestGoodsId);
			if(interestGoods.getCashStartTime().after(DateUtils.getCurrentDateYMD()) 
					|| interestGoods.getCashEndTime().before(DateUtils.getCurrentDateYMD())) {
				throw new InterestException(InterestExceptionEnum.ERROR_CASH_SPOCE);
			}
			int unCashQuantity = interestGoodsProvideDao.countCashQuantity(
					memberId, interestGoodsId, interestGoodsSku.getId() , true);
			if(interestGoodsSkuVO.getCashQuantity() > unCashQuantity) {
				throw new InterestException(InterestExceptionEnum.ERROR_CASH_EXCESS);
			}
			interestGoodsSkuVO.setInterestGoodsSkuId(interestGoodsSku.getId());
		}
		//  验证通过更新权益兑取数量
		if(!interestGoodsSkuDao.updateCash(interestGoodsSkuVOList)) {
			throw new InterestException(InterestExceptionEnum.ERROR_UPDATE_EXCESS);
		}
		// 验证通过更新用户权益兑取状态
		if(!interestGoodsProvideDao.updateCash(memberId, interestGoodsSkuVOList)) {
			throw new InterestException(InterestExceptionEnum.ERROR_UPDATE_STATUS);
		}
	}
	
	@Transactional("scmDataSourceTx")
	@Override
	public void back(Integer memberId, List<InterestGoodsSkuVO> interestGoodsSkuVOList) {
		if(ObjectUtils.isEmpty(memberId) || CollectionUtils.isEmpty(interestGoodsSkuVOList)) {
			throw new InterestException(InterestExceptionEnum.ERROR_PARAM_NULL);
		}
		for(InterestGoodsSkuVO interestGoodsSkuVO : interestGoodsSkuVOList) {
			if (ObjectUtils.isEmpty(interestGoodsSkuVO.getCashQuantity()) 
					|| ObjectUtils.isEmpty(interestGoodsSkuVO.getInterestGoodsId())
					|| ObjectUtils.isEmpty(interestGoodsSkuVO.getSkuCode())) {
				throw new InterestException(InterestExceptionEnum.ERROR_PARAM_MISS);
			}
			Integer interestGoodsId = interestGoodsSkuVO.getInterestGoodsId();
			String skuCode = interestGoodsSkuVO.getSkuCode();
			InterestGoodsSku interestGoodsSku = interestGoodsSkuDao.getInterestGoodsSku(interestGoodsId, skuCode);
			if(ObjectUtils.isEmpty(interestGoodsSku)) {
				throw new InterestException(InterestExceptionEnum.ERROR_NOT_FOUND.getErrCode(),
						InterestExceptionEnum.ERROR_NOT_FOUND.getErrMsg()+":["+interestGoodsSkuVO.getSkuCode()+"]");
			}
			int usedQuantity = interestGoodsProvideDao.countCashQuantity(
					memberId, interestGoodsId, interestGoodsSku.getId() , false);
			if(interestGoodsSkuVO.getCashQuantity() > usedQuantity) {
				throw new InterestException(InterestExceptionEnum.ERROR_BACK_EXCESS);
			}
			interestGoodsSkuVO.setInterestGoodsSkuId(interestGoodsSku.getId());
		}
		boolean success = interestGoodsProvideDao.resetCash(memberId, interestGoodsSkuVOList);
		if(!success) {
			throw new InterestException(InterestExceptionEnum.ERROR_UPDATE_STATUS);
		}
		success = interestGoodsSkuDao.reduceCashQuantity(interestGoodsSkuVOList);
		if(!success) {
			throw new InterestException(InterestExceptionEnum.ERROR_UPDATE_EXCESS);
		}
	}
	
	@Transactional("scmDataSourceTx")
	@Override
	public boolean changeInterestGoodsSku(Integer memberId, Integer sourceInterestGoodsSkuId, Integer destInterestGoodsSkuId) {
		// 减少原商品权益SKU领取数量
		boolean isSuccess = interestGoodsSkuDao.reduceReceiveQuantity(sourceInterestGoodsSkuId);
		if (!isSuccess) {
			return false;
		}
		//增加目标商品权益SKU领取数量
		isSuccess = interestGoodsSkuDao.addReceiveQuantity(destInterestGoodsSkuId);
		if (!isSuccess) {
			throw new InterestException(InterestExceptionEnum.ERROR_UPDATE_RECEIVE);
		}
		int res = interestGoodsProvideDao.changeInterestGoodsSkuId(memberId, sourceInterestGoodsSkuId, destInterestGoodsSkuId);
		if(res != 1) {
			throw new InterestException(InterestExceptionEnum.ERROR_UPDATE_CHANGE);
		}
		return true;
	}

	@Override
	public Map<Integer, Integer> getCashQuantity(List<Integer> interestGoodsSkuIds) {
		return interestGoodsProvideDao.getCashQuantity(interestGoodsSkuIds);
	}
	
	@Override
	public List<InterestGoodsSku> getIdByValidity() {
		List<InterestGoods> interestGoodsList = interestGoodsDao.getInterestByTime(new Date());
		List<InterestGoodsSku> result = new ArrayList<InterestGoodsSku>();
		for (InterestGoods interestGoods : interestGoodsList) {
			result.addAll(interestGoodsSkuDao.byInterestGoodsId(interestGoods.getId()));
		}
		return result;
	}

	@Override
	public List<InterestGoodsGetVO> getPreExpiration(Integer day) {
		List<InterestGoodsGetVO> InterestGoodsGetVOList = Lists.newArrayList();
		List<InterestGoods> interestGoodsList = interestGoodsDao.getInterestByTime(
				DateUtil.addDay(DateUtil.dayStart(new Date()), day));
		if(CollectionUtils.isEmpty(interestGoodsList)) {
			return InterestGoodsGetVOList;
		}
		for(InterestGoods interestGoods : interestGoodsList) {
			List<InterestGoodsProvide> interestGoodsProvideList = 
					interestGoodsProvideDao.queryInterestGoodsProvide(interestGoods.getId());
			for(InterestGoodsProvide interestGoodsProvide : interestGoodsProvideList) {
				InterestGoodsSku sku = interestGoodsSkuDao.selectOne(interestGoodsProvide.getInterestGoodsSkuId());
				InterestGoodsGetVO interestGoodsGetVO = new InterestGoodsGetVO(interestGoods, sku);
				interestGoodsGetVO.setMemberId(interestGoodsProvide.getId());
				InterestGoodsGetVOList.add(interestGoodsGetVO);
			}
		}
		return InterestGoodsGetVOList;
	}
	
	@Override
	public void batchUpdateStatus(List<Integer> interestGoodsIds) {
		// TODO Auto-generated method stub
		int res = interestGoodsDao.batchUpdateStatus(interestGoodsIds);
		if (res != interestGoodsIds.size()) {
			throw new InterestException(InterestExceptionEnum.ERROR_UPDATE_STATUS);
		}
	}

}
