package com.thebeastshop.pegasus.merchandise.price;

import java.math.BigDecimal;
import java.util.List;

import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.thebeastshop.pegasus.merchandise.exception.PublicException;
import com.thebeastshop.pegasus.merchandise.service.impl.McPsProductServiceImpl;
import com.thebeastshop.pegasus.merchandise.vo.PriceCalResultVO;
import com.thebeastshop.pegasus.merchandise.vo.PsCampaignSectionVO;
import com.thebeastshop.pegasus.merchandise.vo.PsCampaignVO;
import com.thebeastshop.pegasus.merchandise.vo.PsSkuVO;
import com.thebeastshop.pegasus.merchandise.vo.PsSpvVO;

/**
 * 价格计算工具类
 * @author gongjun
 * @since 2016-12-29
 */
public class PriceCalculators {
    private static final Logger logger = LoggerFactory.getLogger(McPsProductServiceImpl.class);

    /**
     * 计算PRODUCT折扣价格
     * @param productCode
     * @param allowVipDiscount
     * @param originalPrice
     * @param campaignList
     * @param discountAmount
     * @return
     */
    public static BigDecimal calculateProductDiscountPrice(String productCode, Boolean allowVipDiscount, BigDecimal originalPrice,
                                              List<PsCampaignVO> campaignList, BigDecimal discountAmount) {
        BigDecimal discountPrice = calculateDiscountPrice(campaignList, originalPrice, allowVipDiscount, discountAmount);
        logger.debug("计算商品价格,折扣价：{}", discountPrice);
        return discountPrice;
    }


    /**
     * 计算SPV折扣价格
     * @param spvVO
     * @param allowVipDiscount
     * @param originalPrice
     * @param campaignList
     * @param discountAmount
     * @return
     */
    public static BigDecimal calculateSpvDiscountPrice(PsSpvVO spvVO, Boolean allowVipDiscount,BigDecimal originalPrice,List<PsCampaignVO> campaignList,BigDecimal discountAmount) {
		return calculateSpvDiscountPrice(spvVO, allowVipDiscount, originalPrice, campaignList, discountAmount, null);
    }

	/**
	 * 计算SPV折扣价格
	 * 
	 * @param spvVO
	 * @param allowVipDiscount
	 * @param originalPrice
	 * @param campaignList
	 * @param discountAmount
	 * @param results
	 * @return
	 */
	public static BigDecimal calculateSpvDiscountPrice(PsSpvVO spvVO, Boolean allowVipDiscount,
			BigDecimal originalPrice, List<PsCampaignVO> campaignList, BigDecimal discountAmount,
			List<PriceCalResultVO> results) {

		if (discountAmount == null) {
			throw new PublicException("会员折扣率参数没有传");
		}

		BigDecimal discountPrice = calculateDiscountPrice(campaignList, originalPrice, allowVipDiscount, discountAmount,
				results);
		logger.debug("计算SPV价格,折扣价：{}", discountPrice);
		return discountPrice;
	}

    /**
     * 计算SPV生日折扣价格
     *
     * @param spvVO
     * @param allowBdDiscount
     * @param originalPrice
     * @param campaignList
     * @param bdDiscountAmount
     * @param results
     * @return
     */
    public static BigDecimal calculateSpvBdDiscountPrice(PsSpvVO spvVO, Boolean allowBdDiscount,
                                                       BigDecimal originalPrice, List<PsCampaignVO> campaignList, BigDecimal bdDiscountAmount,
                                                       List<PriceCalResultVO> results) {

        if (bdDiscountAmount == null) {
            throw new PublicException("生日折扣率参数没有传");
        }

        BigDecimal discountPrice = calculateBdDiscountPrice(campaignList, originalPrice, allowBdDiscount, bdDiscountAmount,
                results);
        logger.debug("计算SPV生日价格,折扣价：{}", discountPrice);
        return discountPrice;
    }

    /**
     * 计算SKU折扣价格
     * @param skuVO
     * @param allowVipDiscount
     * @param originalPrice
     * @param campaignList
     * @param discountAmount
     * @return
     */
    public static BigDecimal calculateSkuDiscountPrice(PsSkuVO skuVO, Boolean allowVipDiscount, BigDecimal originalPrice, List<PsCampaignVO> campaignList, BigDecimal discountAmount) {
        BigDecimal discountPrice = calculateDiscountPrice(campaignList, originalPrice, allowVipDiscount, discountAmount);
        logger.debug("计算SKU价格,折扣价：{}", discountPrice);
        return discountPrice;
    }

    /**
     *	价格计算的公共方法
     * @param campaignList 活动集合
     * @param originalPrice 初始价格
     * @param allowVipDiscount 是否允许会员折扣
     * @param discountAmount 会员折扣率
     * @param results 价格计算结果集
     * @return
     */
	private static BigDecimal calculateDiscountPrice(List<PsCampaignVO> campaignList, BigDecimal originalPrice,
			Boolean allowVipDiscount, BigDecimal discountAmount, List<PriceCalResultVO> results) {
		BigDecimal targetPrice = originalPrice;
        PriceCalculator originalPriceCalculator = new SourcePrice(targetPrice);
        PriceCalculator priceCalculator = originalPriceCalculator;
        Boolean hasRush = false;
        if(CollectionUtils.isNotEmpty(campaignList)) {
            for (PsCampaignVO campaign : campaignList) {
                Integer discountType = campaign.getDiscountTypeId();
                if (discountType != 1 && discountType != 6) {
                    continue;
                }
                List<PsCampaignSectionVO> sectionList = campaign.getCampaignSectionList();
                if (CollectionUtils.isEmpty(sectionList)){
                    continue;
                }
                PsCampaignSectionVO section = sectionList.get(0);
                BigDecimal factor = section.getFactor();
                Integer factorType = section.getFactorType();
                if (factor == null)
                    continue;
                if(null == factorType) {
                    continue;
                }
                if (discountType == 6) {
                    hasRush = true;
                    priceCalculator = new RushPriceCalculator(originalPriceCalculator, factor, factorType, campaign.getId());
                    // 有了抢购活动就不再计算其他活动
                    break;
                }
                else if (!hasRush) {
                    priceCalculator = new DiscountPriceCalculator(priceCalculator, factor, factorType, campaign.getId());
                }
            }
        }

        // 有了抢购不参加VIP折扣
        if(!hasRush) {
            priceCalculator = new VIPDiscountPriceCalculator(priceCalculator, discountAmount, hasRush, allowVipDiscount);
        }
        targetPrice = priceCalculator.calculate(results);
        return targetPrice;
	}

    /**
     *	价格计算的公共方法
     * @param campaignList 活动集合
     * @param originalPrice 初始价格
     * @param allowBdDiscount 是否允许生日折扣
     * @param bdDiscountAmount 生日折扣率
     * @param results 价格计算结果集
     * @return
     */
    private static BigDecimal calculateBdDiscountPrice(List<PsCampaignVO> campaignList, BigDecimal originalPrice,
                                                     Boolean allowBdDiscount, BigDecimal bdDiscountAmount, List<PriceCalResultVO> results) {
        BigDecimal targetPrice = originalPrice;
        PriceCalculator originalPriceCalculator = new SourcePrice(targetPrice);
        PriceCalculator priceCalculator = originalPriceCalculator;
        Boolean hasRush = false;
        if(CollectionUtils.isNotEmpty(campaignList)) {
            for (PsCampaignVO campaign : campaignList) {
                Integer discountType = campaign.getDiscountTypeId();
                if (discountType != 1 && discountType != 6) {
                    continue;
                }
                List<PsCampaignSectionVO> sectionList = campaign.getCampaignSectionList();
                if (CollectionUtils.isEmpty(sectionList)){
                    continue;
                }
                PsCampaignSectionVO section = sectionList.get(0);
                BigDecimal factor = section.getFactor();
                Integer factorType = section.getFactorType();
                if (factor == null)
                    continue;
                if(null == factorType) {
                    continue;
                }
                if (discountType == 6) {
                    hasRush = true;
                    priceCalculator = new RushPriceCalculator(originalPriceCalculator, factor, factorType, campaign.getId());
                    // 有了抢购活动就不再计算其他活动
                    break;
                }
                else if (!hasRush) {
                    priceCalculator = new DiscountPriceCalculator(priceCalculator, factor, factorType, campaign.getId());
                }
            }
        }

        // 有了抢购不参加生日折扣
        if(!hasRush) {
            priceCalculator = new BDDiscountPriceCalculator(priceCalculator, bdDiscountAmount, hasRush, allowBdDiscount);
        }
        targetPrice = priceCalculator.calculate(results);
        return targetPrice;
    }

    /**
     * 价格计算的公共方法
     * @param campaignList 活动集合
     * @param originalPrice 初始价格
     * @param allowVipDiscount 是否允许会员折扣
     * @param discountAmount 会员折扣率
     * @return
     */
    private static BigDecimal calculateDiscountPrice(List<PsCampaignVO> campaignList, BigDecimal originalPrice, Boolean allowVipDiscount,  BigDecimal discountAmount) {
		return calculateDiscountPrice(campaignList, originalPrice, allowVipDiscount, discountAmount, null);
    }
}
