package com.thebeastshop.pegasus.merchandise.domain.impl;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.thebeastshop.pegasus.merchandise.cond.PcsSkuCategoryCond;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;

import com.thebeastshop.pegasus.merchandise.cond.PcsSkuCategoryRelationCond;
import com.thebeastshop.pegasus.merchandise.dao.PcsSkuCategoryMapper;
import com.thebeastshop.pegasus.merchandise.domain.PcsSkuCategoryDomain;
import com.thebeastshop.pegasus.merchandise.exception.PurchaseException;
import com.thebeastshop.pegasus.merchandise.exception.PurchaseExceptionErrorCode;
import com.thebeastshop.pegasus.merchandise.model.PcsSkuCategory;
import com.thebeastshop.pegasus.merchandise.model.PcsSkuCategoryExample;
import com.thebeastshop.pegasus.merchandise.model.PcsSkuCategoryRelation;
import com.thebeastshop.pegasus.merchandise.vo.PcsFrontBackCategoryVO;
import com.thebeastshop.pegasus.merchandise.vo.PcsSkuCategoryVO;
import com.thebeastshop.pegasus.util.comm.BeanUtil;
import com.thebeastshop.pegasus.util.comm.EmptyUtil;
import com.thebeastshop.pegasus.util.inter.CheckAble;

/**
 * @author gongjun
 * @since 2016-08-30
 */
@Component("pcsSkuCategoryDomain")
public class PcsSkuCategoryDomainImpl extends AbstractBaseDomain<PcsSkuCategoryVO, PcsSkuCategory> implements PcsSkuCategoryDomain, CheckAble {

    private static final String CODE_SEPARATOR = "_";

    private static final String NAME_SEPARATOR = "/";

    @Autowired
    private PcsSkuCategoryMapper pcsSkuCategoryMapper;

    @Override
    @Transactional
    public Long create(PcsSkuCategory record) {
        if (check(record)) {
            if (check(record, true)) {
                record.setFullName(getFullName(record));
                record.setIsLeaf(1);
                pcsSkuCategoryMapper.insertSelective(record);
                try {
                    if (check(record.getCode(),record.getId())) {
                        pcsSkuCategoryMapper.updateByPrimaryKeySelective(record);
                        return record.getId();
                    } else {
                        throw new PurchaseException(PurchaseExceptionErrorCode.DUPLICATE_FIELD, "code字段数据库已有相同的值不能重复");
                    }
                } catch (Exception e) {
                    log.error("", e);
                    throw new PurchaseException(PurchaseExceptionErrorCode.ERROR_GENERATE_CODE, e.getLocalizedMessage());
                }
            }
        }
        return null;
    }

    @Override
    public Long createBackCategory(PcsSkuCategory record) {
        if (check(record)) {
            if (check(record, true)) {
                record.setFullName(getBackFullName(record));
                record.setIsLeaf(1);
                pcsSkuCategoryMapper.insertBackCategorySelective(record);
                try {
                    if (check(record.getCode(),record.getId())) {
                        pcsSkuCategoryMapper.updateBackCategoryByPrimaryKeySelective(record);
                        return record.getId();
                    } else {
                        throw new PurchaseException(PurchaseExceptionErrorCode.DUPLICATE_FIELD, "code字段数据库已有相同的值不能重复");
                    }
                } catch (Exception e) {
                    log.error("", e);
                    throw new PurchaseException(PurchaseExceptionErrorCode.ERROR_GENERATE_CODE, e.getLocalizedMessage());
                }
            }
        }
        return null;

    }

    @Override
    @Transactional
    public boolean update(PcsSkuCategory record) {
        if (check(record)) {
            if (check(record, false)) {
                record.setFullName(getFullName(record));
                return pcsSkuCategoryMapper.updateByPrimaryKeySelective(record) != 0;
            }
        }
        return false;
    }

    @Override
    public boolean updateBackCategory(PcsSkuCategory record) {
        if (check(record)) {
            if (check(record, false)) {
                record.setFullName(getBackFullName(record));
                return pcsSkuCategoryMapper.updateBackCategoryByPrimaryKeySelective(record) != 0;
            }
        }
        return false;
    }

    @Override
    public PcsSkuCategory findById(long id) {
        return pcsSkuCategoryMapper.selectByPrimaryKey(id);
    }

    @Override
    public PcsSkuCategory findBackCategoryById(long id) {
        return pcsSkuCategoryMapper.selectBackCategoryByPrimaryKey(id);
    }

    @Override
    public List<PcsSkuCategory> findFrontCategoriesBySkuCode(String skuCode) {
        Map<String, Object> params = new HashMap<>();
        params.put("skuCode", skuCode);
        return pcsSkuCategoryMapper.findFrontCategory(params);
    }

    @Override
    public List<PcsSkuCategory> findBackCategoriesBySkuCode(String skuCode) {
        Map<String, Object> params = new HashMap<>();
        params.put("skuCode", skuCode);
        return pcsSkuCategoryMapper.findBackCategory(params);
    }

    @Override
    public List<PcsSkuCategory> findFrontCategoriesBySkuCodeList(List<String> skuCodeList) {
        Map<String, Object> params = new HashMap<>();
        params.put("skuCodeList", skuCodeList);
        return pcsSkuCategoryMapper.findFrontCategory(params);
    }

    @Override
    public List<PcsSkuCategory> findAllChildrens(long parentId) {
        PcsSkuCategoryExample example = new PcsSkuCategoryExample();
        example.createCriteria().andParentIdEqualTo(parentId);
        List<PcsSkuCategory> childrens = pcsSkuCategoryMapper.selectByExample(example);
        if (CollectionUtils.isEmpty(childrens)) {
            return Collections.emptyList();
        }
        return childrens;
    }

    @Override
    public List<PcsSkuCategory> findAllBackCategoryChildrens(long parentId) {
        Map<String, Object> params = new HashMap<>();
        params.put("parentId", parentId);
        List<PcsSkuCategory> childrens = pcsSkuCategoryMapper.findBackCategory(params);
        if (CollectionUtils.isEmpty(childrens)) {
            return Collections.emptyList();
        }
        return childrens;

    }

    @Override
    public List<PcsSkuCategory> findAll() {
        PcsSkuCategoryExample example = new PcsSkuCategoryExample();
        example.setOrderByClause("id");
        List<PcsSkuCategory> recordList = pcsSkuCategoryMapper.selectByExample(example);
        if (CollectionUtils.isEmpty(recordList)) {
            return Collections.emptyList();
        }
        return recordList;
    }

    @Override
    public List<PcsSkuCategory> findAllFrontCategories() {
        Map<String, Object> params = new HashMap<>();
        params.put("withoutRelation", 1);
        params.put("orderBy", "cg.ID");
        List<PcsSkuCategory> recordList = pcsSkuCategoryMapper.findFrontCategory(params);
        if (CollectionUtils.isEmpty(recordList)) {
            return Collections.emptyList();
        }
        return recordList;

    }


    @Override
    public PcsSkuCategory findByCode(String code) {
        PcsSkuCategoryExample example = new PcsSkuCategoryExample();
        example.createCriteria().andCodeEqualTo(code);
        List<PcsSkuCategory> recordList = pcsSkuCategoryMapper.selectByExample(example);
        if (CollectionUtils.isEmpty(recordList)) {
            return null;
        } else {
            if (recordList.size() > 1) {
                throw new PurchaseException(PurchaseExceptionErrorCode.FINDBY_CODE_DUPLICATE, " 根据code能查出2条以上(含两条)记录");
            } else {
                return recordList.get(0);
            }
        }
    }

    @Override
    public List<PcsSkuCategory> findByName(String name) {
        PcsSkuCategoryExample example = new PcsSkuCategoryExample();
        example.createCriteria().andNameLike("%" + name + "%");
        example.setOrderByClause("ID LIMIT 20 ");
        List<PcsSkuCategory> recordList = pcsSkuCategoryMapper.selectByExample(example);
        if (CollectionUtils.isEmpty(recordList)) {
            return Collections.emptyList();
        }
        return recordList;
    }

    @Override
    public List<PcsSkuCategory> findBackCategoryByName(String name) {
        Map<String, Object> params = new HashMap<>();
        params.put("nameLike", "%" + name + "%");
        params.put("orderByClause", "ID LIMIT 20 ");
        List<PcsSkuCategory> recordList = pcsSkuCategoryMapper.findBackCategory(params);
        if (CollectionUtils.isEmpty(recordList)) {
            return Collections.emptyList();
        }
        return recordList;
    }

    @Override
    public List<PcsSkuCategory> findByIds(List<Long> ids) {
        List<PcsSkuCategory> recordList = new ArrayList<PcsSkuCategory>();
        for (long id : ids) {
            PcsSkuCategory record = findById(id);
            if (EmptyUtil.isNotEmpty(record) && EmptyUtil.isNotEmpty(record.getId())) {
                recordList.add(record);
            }
        }
        if (CollectionUtils.isEmpty(recordList)) {
            return Collections.emptyList();
        } else {
            return recordList;
        }
    }

    @Override
    public List<PcsSkuCategory> findBackCategoriesByIds(List<Long> ids) {
        List<PcsSkuCategory> recordList = new ArrayList<PcsSkuCategory>();
        for (long id : ids) {
            PcsSkuCategory record = findBackCategoryById(id);
            if (EmptyUtil.isNotEmpty(record) && EmptyUtil.isNotEmpty(record.getId())) {
                recordList.add(record);
            }
        }
        if (CollectionUtils.isEmpty(recordList)) {
            return Collections.emptyList();
        } else {
            return recordList;
        }
    }

    @Override
    public List<PcsSkuCategory> findByCodeAndRightLikeCode(String code) {
        List<PcsSkuCategory> recordList = new ArrayList<PcsSkuCategory>();
//        PcsSkuCategoryExample selfExample = new PcsSkuCategoryExample();
//        selfExample.createCriteria().andCodeEqualTo(code);
        Map<String, Object> selfMap = new HashMap<>();
        selfMap.put("withoutRelation", 1);
        selfMap.put("code", code);
        List<PcsSkuCategory> selfList = pcsSkuCategoryMapper.findFrontCategory(selfMap);
        if (CollectionUtils.isNotEmpty(selfList)) {
            recordList.addAll(selfList);
        }

        recordList = findChildrenCategoryList(selfList, recordList);

/*
        PcsSkuCategoryExample childrenExample = new PcsSkuCategoryExample();
        childrenExample.createCriteria().andCodeLike(code + "__");
        List<PcsSkuCategory> childrenList = pcsSkuCategoryMapper.selectByExample(childrenExample);
*/
        if (CollectionUtils.isEmpty(recordList)) {
            return Collections.emptyList();
        } else {
            return recordList;
        }
    }

    private List<PcsSkuCategory> findChildrenCategoryList(List<PcsSkuCategory> parentCategoryList, List<PcsSkuCategory> recordList) {
        if (CollectionUtils.isEmpty(parentCategoryList)) {
            return recordList;
        }
//        PcsSkuCategoryExample childrenExample = new PcsSkuCategoryExample();
        Map<String, Object> childrenMap = new HashMap<>();
        List<Long> parentIdList = new ArrayList<>();
        for (PcsSkuCategory parentCategory : parentCategoryList) {
            Integer isLeaf = parentCategory.getIsLeaf();
            if (isLeaf != null && isLeaf == 0) {
                parentIdList.add(parentCategory.getId());
            }
        }
        if (CollectionUtils.isEmpty(parentIdList)) {
            return recordList;
        }

//        childrenExample.createCriteria().andParentIdIn(parentIdList);
        childrenMap.put("withoutRelation", 1);
        childrenMap.put("parentIdList", parentIdList);
        List<PcsSkuCategory> childrenList = pcsSkuCategoryMapper.findFrontCategory(childrenMap);
        if (CollectionUtils.isNotEmpty(childrenList)) {
            recordList.addAll(childrenList);
        }
        return findChildrenCategoryList(childrenList, recordList);
    }

    @Override
    public List<PcsSkuCategory> findSkuCategoryByParentIdIsNull() {
//        PcsSkuCategoryExample example = new PcsSkuCategoryExample();
//        example.createCriteria().andParentIdIsNull();
        Map<String, Object> params = new HashMap<>();
        params.put("parentIdIsNull", 1);
        params.put("withoutRelation", 1);
        List<PcsSkuCategory> recordList = pcsSkuCategoryMapper.findFrontCategory(params);
        if (CollectionUtils.isEmpty(recordList)) {
            return Collections.emptyList();
        } else {
            return recordList;
        }
    }

    @Override
    public PcsSkuCategory findByParentID(Long parentID) {
        PcsSkuCategoryExample example = new PcsSkuCategoryExample();
        example.createCriteria().andIdEqualTo(parentID);
        List<PcsSkuCategory> recordList = pcsSkuCategoryMapper.selectByExample(example);
        if (CollectionUtils.isEmpty(recordList)) {
            return null;
        } else {
            return recordList.get(0);
        }

    }

    @Override
    public List<Long> findProductIdsBySkuCodes(List<String> skuCodes) {
        Map<String, Object> params = new HashMap<>();
        params.put("skuCodes", skuCodes);
        return pcsSkuCategoryMapper.findProductIdsBySkuCodes(params);
    }

    @Override
    public List<Long> findProductIdListByCategoryFrontRelationIds(List<Long> ids) {
        Map<String, Object> params = new HashMap<>();
        params.put("idList", ids);
        return pcsSkuCategoryMapper.findProductIdListByCategoryFrontRelation(params);
    }

    @Override
    public List<PcsFrontBackCategoryVO> findFrontBackCategoryList(Map<String, Object> params) {
        List<PcsFrontBackCategoryVO> list = pcsSkuCategoryMapper.findFrontBackCategory(params);
        for (int i = 0; i < list.size(); i++) {
            PcsFrontBackCategoryVO fbCategoryVO = list.get(i);
            Map<String, Object> fParams = new HashMap<>();
            Long fid = fbCategoryVO.getFrontSkuCategoryId();
            if (fid != null) {
                fParams.put("id", fid);
                List<PcsSkuCategoryVO> frontCategories = findFrontCategoryList(fParams);
                if (EmptyUtil.isNotEmpty(frontCategories)) {
                    fbCategoryVO.setFrontSkuCategory(frontCategories.get(0));
                }
            }

            Long bid = fbCategoryVO.getBackSKuCategoryId();
            if (bid != null) {
                Map<String, Object> bParams = new HashMap<>();
                bParams.put("id", bid);
                List<PcsSkuCategoryVO> backCategories = findBackCategoryList(bParams);
                if (EmptyUtil.isNotEmpty(backCategories)) {
                    fbCategoryVO.setBackSkuCategory(backCategories.get(0));
                }
            }

        }
        return list;
    }

    public List<PcsSkuCategoryVO> findFrontCategoryList(Map<String, Object> params) {
        return BeanUtil.buildListFrom(pcsSkuCategoryMapper.findFrontCategory(params), PcsSkuCategoryVO.class);
    }


    @Override
    public List<PcsSkuCategoryVO> findBackCategoryList(Map<String, Object> params) {
        return BeanUtil.buildListFrom(pcsSkuCategoryMapper.findBackCategory(params), PcsSkuCategoryVO.class);
    }

    @Override
    public List<PcsSkuCategoryRelation> findBackCategoryRelationList(PcsSkuCategoryRelationCond cond) {
        return pcsSkuCategoryMapper.findBackCategoryRelation(cond);
    }

    @Override
    @Transactional
    public void saveOrUpdateBackCategoryRelationList(List<PcsSkuCategoryRelation> relationList) {
        for (PcsSkuCategoryRelation relation : relationList) {
            saveOrUpdateBackCategoryRelation(relation);
        }
    }

    @Override
    public void saveOrUpdateBackCategoryRelation(PcsSkuCategoryRelation relation) {
        PcsSkuCategoryRelationCond cond = new PcsSkuCategoryRelationCond();
        cond.setSkuCode(relation.getSkuCode());
        cond.setCategoryId(relation.getCategoryId());
        List<PcsSkuCategoryRelation> list = pcsSkuCategoryMapper.findBackCategoryRelation(cond);
        if (EmptyUtil.isEmpty(list)) {
            addBackCategoryRelation(relation);
        }
    /*    else {
            PcsSkuCategoryRelation backRelation = list.get(0);
            backRelation.setSort1(relation.getSort1());
            backRelation.setSort2(relation.getSort2());
            backRelation.setActive(relation.getActive());
            updateBackCategoryRelation(backRelation);
        }*/
    }

    @Override
    @Transactional
    public Integer addFrontCategoryRelation(PcsSkuCategoryRelation relation) {
        return pcsSkuCategoryMapper.insertFrontCategoryRelation(relation);
    }

    @Override
    @Transactional
    public Integer addBackCategoryRelation(PcsSkuCategoryRelation relation) {
        return pcsSkuCategoryMapper.insertBackCategoryRelation(relation);
    }

    @Override
    public Integer addFrontBackCategoryRelation(PcsFrontBackCategoryVO frontBackCategoryVO) {
        return pcsSkuCategoryMapper.insertFrontBackCategoryRelation(frontBackCategoryVO);
    }

    @Override
    @Transactional
    public Integer updateBackCategoryRelation(PcsSkuCategoryRelation relation) {
        return pcsSkuCategoryMapper.updateBackCategoryRelation(relation);
    }

    @Override
    public List<PcsSkuCategoryRelation> findFrontCategoryRelation(PcsSkuCategoryRelationCond cond) {
        return pcsSkuCategoryMapper.findFrontCategoryRelation(cond);
    }

    @Override
    public List<Long> findFrontCategoryIdListByBackCategoryId(Long backCategoryRelationId) {
        return pcsSkuCategoryMapper.findFrontCategoryIdListByBackCategoryId(backCategoryRelationId);
    }

    @Override
    public List<Long> findFrontCategoryIdListByBackCategoryIds(List<Long> backCategoryRelationIds) {
        Map<String, Object> params = new HashMap<>();
        params.put("ids", backCategoryRelationIds);
        return pcsSkuCategoryMapper.findFrontCategoryIdListByBackCategoryIds(params);

    }

    @Override
    public List<Long> findBackCategoryIdListByFrontCategoryId(Long frontCategoryRelationId) {
        return pcsSkuCategoryMapper.findBackCategoryIdListByFrontCategoryId(frontCategoryRelationId);
    }

    @Override
    public Integer countOfFrontCategoryRelation(PcsSkuCategoryRelationCond cond) {
        return pcsSkuCategoryMapper.countOfFrontCategoryRelation(cond);
    }

    @Override
    public Integer updateFrontCategoryRelation(PcsSkuCategoryRelation relation) {
        return pcsSkuCategoryMapper.updateFrontCategoryRelation(relation);
    }

    @Override
    public Integer deleteFrontCategoryRelationList(List<Long> idList) {
        Map<String, Object> params = new HashMap<>();
        if (EmptyUtil.isNotEmpty(idList)) {
            params.put("idList", idList);
            return pcsSkuCategoryMapper.deleteFrontCategoryRelation(params);
        }
        return 0;
    }

    @Override
    public Integer deleteBackCategoryRelationList(List<Long> idList) {
        Map<String, Object> params = new HashMap<>();
        if (EmptyUtil.isNotEmpty(idList)) {
            params.put("idList", idList);
            return pcsSkuCategoryMapper.deleteBackCategoryRelation(params);
        }
        return 0;
    }

    @Override
    public Boolean check(Object obj) {
        PcsSkuCategory record = (PcsSkuCategory) obj;
        if (EmptyUtil.isEmpty(record)) {
            throw new PurchaseException(PurchaseExceptionErrorCode.ILLEGAL_PARAM, "参数异常");
        }
        return true;
    }

    private Boolean check(PcsSkuCategory record, Boolean isCreate) {
        if (isCreate && EmptyUtil.isNotEmpty(record.getId())) {
            throw new PurchaseException(PurchaseExceptionErrorCode.ILLEGAL_PARAM, "参数异常");
        }
        if (!isCreate && EmptyUtil.isEmpty(record.getId())) {
            throw new PurchaseException(PurchaseExceptionErrorCode.ILLEGAL_PARAM, "参数异常");
        }
        PcsSkuCategoryExample example1 = new PcsSkuCategoryExample();
        PcsSkuCategoryExample example2 = new PcsSkuCategoryExample();
        example1.createCriteria().andNameEqualTo(record.getName());
        example2.createCriteria().andCodeEqualTo(record.getCode());
        List<PcsSkuCategory> oldList = null;
        oldList = pcsSkuCategoryMapper.selectByExample(example1);
        if (oldList.size() > 0 && isCreate) {
            throw new PurchaseException(PurchaseExceptionErrorCode.DUPLICATE_FIELD, "name字段数据库已有相同的值不能重名");
        }
        if (oldList.size() > 0 && !isCreate && !oldList.get(0).getId().equals(record.getId())) {
            throw new PurchaseException(PurchaseExceptionErrorCode.DUPLICATE_FIELD, "name字段数据库已有相同的值不能重名");
        }
        oldList = pcsSkuCategoryMapper.selectByExample(example2);
        if (oldList.size() > 0 && isCreate) {
            throw new PurchaseException(PurchaseExceptionErrorCode.DUPLICATE_FIELD, "code字段数据库已有相同的值不能重名");
        }
        if (oldList.size() > 0 && !isCreate && !oldList.get(0).getId().equals(record.getId())) {
            throw new PurchaseException(PurchaseExceptionErrorCode.DUPLICATE_FIELD, "code字段数据库已有相同的值不能重名");
        }
        return true;
    }

    private Boolean check(String code,Long id) {
        PcsSkuCategoryExample example = new PcsSkuCategoryExample();
        example.createCriteria().andCodeEqualTo(code).andIdNotEqualTo(id);
        List<PcsSkuCategory> recordList = pcsSkuCategoryMapper.selectByExample(example);
        if (recordList.size() == 0) {
            return true;
        } else {
            return false;
        }
    }

    /**
     * 生成CODE
     *
     * @param id
     * @param parentId
     * @return
     * @throws Exception
     */
/*
    private String generateCode(long id, long parentId) {
        if (NullUtil.isNull(parentId) || parentId <= 0) {
            return id + "";
        } else {
            String parentCode = "";
            PcsSkuCategory skuCategory = findById(parentId);
            if (EmptyUtil.isNotEmpty(skuCategory)) {
                parentCode = skuCategory.getCode();
            }
            if (EmptyUtil.isEmpty(parentCode)) {
                throw new PurchaseException(PurchaseExceptionErrorCode.ILLEGAL_PARAM, "parentCode 计算出错");
            }
            return parentCode + CODE_SEPARATOR + id;
        }
    }
*/

    /**
     * 得到FULLNAME
     *
     * @param record
     * @return
     */
    private String getFullName(PcsSkuCategory record) {
        if (EmptyUtil.isEmpty(record)) {
            return "";
        } else {
            if (EmptyUtil.isEmpty(record.getParentId())) {
                return record.getName();
            } else {
                return getFullName(findById(record.getParentId())) + NAME_SEPARATOR + record.getName();
            }
        }
    }


    /**
     * 得到后台类目FULLNAME
     *
     * @param record
     * @return
     */
    private String getBackFullName(PcsSkuCategory record) {
        if (EmptyUtil.isEmpty(record)) {
            return "";
        } else {
            if (EmptyUtil.isEmpty(record.getParentId())) {
                return record.getName();
            } else {
                return getBackFullName(findBackCategoryById(record.getParentId())) + NAME_SEPARATOR + record.getName();
            }
        }
    }

	@Override
	public List<PcsSkuCategoryVO> findAllFirstBackCategoryList() {
		return pcsSkuCategoryMapper.findAllFirstBackCategoryList();
	}

    @Override
    public PcsSkuCategoryVO findFirstCategoryByParentId(Long id) {
        PcsSkuCategoryVO vo = pcsSkuCategoryMapper.findFirstCategoryByParentId(id);
        return vo;
    }



    @Override
    public List<PcsFrontBackCategoryVO> findRelationBackCategoryByFrontCategoryId(Long frontId) {
    	Map<String,Object> params = new HashMap<String,Object>();
    	params.put("frontSkuCategoryId", frontId);
    	List<PcsFrontBackCategoryVO> list = pcsSkuCategoryMapper.findFrontBackCategory(params);
        for (int i = 0; i < list.size(); i++) {
            PcsFrontBackCategoryVO fbCategoryVO = list.get(i);
            Long bid = fbCategoryVO.getBackSKuCategoryId();
            if (bid != null) {
                Map<String, Object> bParams = new HashMap<>();
                bParams.put("id", bid);
                List<PcsSkuCategoryVO> backCategories = findBackCategoryList(bParams);
                if (EmptyUtil.isNotEmpty(backCategories)) {
                    fbCategoryVO.setBackSkuCategory(backCategories.get(0));
                }
            }
        }

        return list;
    }

	@Override
	public Long createFrontCategory(PcsSkuCategory record) {
		if (check(record)) {
            if (checkFront(record, true)) {
                record.setFullName(getBackFullName(record));
                record.setIsLeaf(1);
                pcsSkuCategoryMapper.insertFrontCategory(record);
                try {
                    if (check(record.getCode(),record.getId())) {
                        pcsSkuCategoryMapper.updateFrontCategoryByPrimaryKeySelective(record);
                        // 如果父类有关联的后台类目则转移到到新增的子类项目中
                        if(record.getParentId() != null){
                        	List<PcsFrontBackCategoryVO> backVO = findRelationBackCategoryByFrontCategoryId(record.getParentId());
                        	if(CollectionUtils.isNotEmpty(backVO)){
                        		// 删除父类关联，新增子类关联
                        		deleteFrontBackCategoryRelationByFrontId(record.getParentId());
                        		for(PcsFrontBackCategoryVO frontBackCategoryVO : backVO){
                					PcsFrontBackCategoryVO vo = new PcsFrontBackCategoryVO();
                					vo.setFrontSkuCategoryId(record.getId());
                					vo.setBackSKuCategoryId(frontBackCategoryVO.getBackSKuCategoryId());
                					addFrontBackCategoryRelation(vo);
                				}
                        	}
                        }
                        return record.getId();
                    } else {
                        throw new PurchaseException(PurchaseExceptionErrorCode.DUPLICATE_FIELD, "code字段数据库已有相同的值不能重复");
                    }
                } catch (Exception e) {
                    log.error("", e);
                    throw new PurchaseException(PurchaseExceptionErrorCode.ERROR_GENERATE_CODE, e.getLocalizedMessage());
                }
            }
        }
        return null;
	}

	@Override
	public boolean updateFrontCategory(PcsSkuCategory record) {
		if (check(record)) {
            if (checkFront(record, false)) {
                record.setFullName(getBackFullName(record));
                return pcsSkuCategoryMapper.updateFrontCategoryByPrimaryKeySelective(record) != 0;
            }
        }
        return false;
	}

	@Override
	public Integer deleteFrontBackCategoryRelationByFrontId(Long frontId) {
		return pcsSkuCategoryMapper.deleteFrontBackCategoryRelationByFrontId(frontId);
	}

	// 检查前端类目是否有重复项
	private Boolean checkFront(PcsSkuCategory record, Boolean isCreate) {
        if (isCreate && EmptyUtil.isNotEmpty(record.getId())) {
            throw new PurchaseException(PurchaseExceptionErrorCode.ILLEGAL_PARAM, "参数异常");
        }
        if (!isCreate && EmptyUtil.isEmpty(record.getId())) {
            throw new PurchaseException(PurchaseExceptionErrorCode.ILLEGAL_PARAM, "参数异常");
        }
        List<PcsSkuCategory> oldList = null;
        Map<String,Object> params1 = new HashMap<String,Object>();
        Map<String,Object> params2 = new HashMap<String,Object>();
        params1.put("name", record.getName());
        params2.put("code", record.getCode());
        oldList = pcsSkuCategoryMapper.findFrontCategory(params1);
        if (oldList.size() > 0 && isCreate) {
            throw new PurchaseException(PurchaseExceptionErrorCode.DUPLICATE_FIELD, "name字段数据库已有相同的值不能重名");
        }
        if (oldList.size() > 0 && !isCreate && !oldList.get(0).getId().equals(record.getId())) {
            throw new PurchaseException(PurchaseExceptionErrorCode.DUPLICATE_FIELD, "name字段数据库已有相同的值不能重名");
        }
        oldList = pcsSkuCategoryMapper.findFrontCategory(params2);
        if (oldList.size() > 0 && isCreate) {
            throw new PurchaseException(PurchaseExceptionErrorCode.DUPLICATE_FIELD, "code字段数据库已有相同的值不能重名");
        }
        if (oldList.size() > 0 && !isCreate && !oldList.get(0).getId().equals(record.getId())) {
            throw new PurchaseException(PurchaseExceptionErrorCode.DUPLICATE_FIELD, "code字段数据库已有相同的值不能重名");
        }
        return true;
    }

    @Override
    public List<Long> findByCondCategory(PcsSkuCategoryCond cond) {
        return pcsSkuCategoryMapper.findByCondCategory(cond);
    }
}