/*
 * Copyright (C), 上海布鲁爱电子商务有限公司
 */
package com.thebeastshop.pegasus.merchandise.client;

import com.dianping.cat.Cat;
import com.dianping.cat.message.Message;
import com.dianping.cat.message.Transaction;
import com.google.common.base.Strings;
import com.thebeastshop.pegasus.merchandise.cond.DateRange;
import com.thebeastshop.pegasus.merchandise.constants.SearchType;
import com.thebeastshop.pegasus.merchandise.domain.SearchKeyword;
import com.thebeastshop.pegasus.merchandise.exception.PublicException;
import com.thebeastshop.pegasus.merchandise.exception.PublicExceptionErrorCode;
import com.thebeastshop.pegasus.merchandise.util.McReflectionUtil;
import com.thebeastshop.pegasus.merchandise.vo.PsBaseVO;
import com.thebeastshop.pegasus.merchandise.vo.PsUpdateVO;
import com.thebeastshop.pegasus.util.comm.DateUtil;
import com.thebeastshop.pegasus.util.comm.EmptyUtil;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.elasticsearch.action.admin.indices.mapping.delete.DeleteMappingRequest;
import org.elasticsearch.action.admin.indices.mapping.delete.DeleteMappingResponse;
import org.elasticsearch.action.admin.indices.mapping.put.PutMappingRequest;
import org.elasticsearch.action.admin.indices.mapping.put.PutMappingResponse;
import org.elasticsearch.action.bulk.BulkRequestBuilder;
import org.elasticsearch.action.bulk.BulkResponse;
import org.elasticsearch.action.delete.DeleteRequest;
import org.elasticsearch.action.index.IndexResponse;
import org.elasticsearch.action.search.SearchRequestBuilder;
import org.elasticsearch.action.search.SearchResponse;
import org.elasticsearch.action.update.UpdateRequest;
import org.elasticsearch.action.update.UpdateRequestBuilder;
import org.elasticsearch.action.update.UpdateResponse;
import org.elasticsearch.client.Requests;
import org.elasticsearch.client.transport.TransportClient;
import org.elasticsearch.common.lang3.StringUtils;
import org.elasticsearch.common.xcontent.XContentBuilder;
import org.elasticsearch.common.xcontent.XContentFactory;
import org.elasticsearch.index.query.*;
import org.elasticsearch.search.SearchHit;
import org.elasticsearch.search.SearchHits;
import org.elasticsearch.search.sort.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.util.*;

/**
 * @author Roy.Chen
 * @version $Id: AbstractESClient.java, v 0.1 2015-08-05 17:20
 */
public abstract class AbstractESClient<T> {
    private static final Logger log = LoggerFactory.getLogger(AbstractESClient.class);

    protected String index;
    protected String type;

    //default
    private static final String DEFAULT_HOME = "127.0.0.1";
    private static final int DEFAULT_PORT = 9300;

    //default page
    private static final int DEFAULT_FROM = 0;
    private static final int DEFAULT_PAGE_SIZE = 20;
    private static final int DEFAULT_PAGE_MAX_SIZE = Integer.MAX_VALUE;

    protected TransportClient client;

    protected final ElasticsearchClientFactory elasticsearchClientFactory;


    public AbstractESClient(String index, String type, ElasticsearchClientFactory elasticsearchClientFactory) {
        this.index = index;
        this.type = type;
        this.elasticsearchClientFactory = elasticsearchClientFactory;
        client = elasticsearchClientFactory.createClient();
//        try {
//            client = new TransportClient(ImmutableSettings.settingsBuilder().loadFromSource(XContentFactory.jsonBuilder()
//                    .startObject()
//                    .startObject("analysis")
//                    .startObject("analyzer")
//                    .startObject("steak")
//                    .field("type", "custom")
//                    .field("tokenizer", "standard")
//                    .field("filter", new String[]{"snowball", "standard", "lowercase"})
//                    .endObject()
//                    .endObject()
//                    .endObject()
//                    .endObject().string())).addTransportAddress(new InetSocketTransportAddress(host , port));
//        } catch (IOException e) {
//            e.printStackTrace();
//        }
    }


    public ElasticsearchClientFactory getElasticsearchClientFactory() {
        return elasticsearchClientFactory;
    }

    public List<T> findAllByField(String name, String value) {
        StackTraceElement stackTraceElement = getStrackElement();
        return findAllByField(stackTraceElement, name, value);

    }

    public List<T> findAllByField(StackTraceElement stackTraceElement, String name,String value) {
        return findByField(stackTraceElement, name, value, DEFAULT_PAGE_MAX_SIZE);
    }


    private void createMapping() {
        PutMappingRequest mappingRequest = Requests.putMappingRequest(index).type(type);
        PutMappingResponse response = client.admin().indices().putMapping(mappingRequest).actionGet();
    }


    private StackTraceElement getStrackElement() {
        StackTraceElement stackTraceElement = Thread.currentThread().getStackTrace()[3];
        return stackTraceElement;
    }

    /**
     *
     * @param name
     * @param value
     * @param count  需要返回的最多rows
     * @return
     */
    public List<T> findByField(String name,Object value, int count) {
        StackTraceElement stackTraceElement = getStrackElement();
        return findByField(stackTraceElement, name, value, count);
    }

    public List<T> findByField(String name,Object value) {
        StackTraceElement stackTraceElement = getStrackElement();
        return findByField(stackTraceElement, name, value, 0);
    }


    private List<T> findByField(StackTraceElement stackTraceElement, String name,Object value, int count){
        if(count < 1) count = DEFAULT_PAGE_SIZE;
        QueryBuilder builder = null;
        if (StringUtils.isBlank(name)) {
            builder = QueryBuilders.matchAllQuery();
        }else{
            BoolQueryBuilder boolQueryBuilder = QueryBuilders.boolQuery();
            builder = boolQueryBuilder;
            boolQueryBuilder.must(QueryBuilders.multiMatchQuery(value, name).type(MatchQueryBuilder.Type.PHRASE_PREFIX));
//            boolQueryBuilder.should(QueryBuilders.matchQuery(name, value).minimumShouldMatch("1"));
//            boolQueryBuilder.should(QueryBuilders.nestedQuery(name, QueryBuilders.queryStringQuery(value.toString())));
//            QueryBuilders.matchQuery(name, QueryBuilders.queryStringQuery(value.toString()).defaultOperator(QueryStringQueryBuilder.Operator.AND));
//            builder = QueryBuilders.termsQuery(name, value).minimumShouldMatch("1");

        }
        List<T> result = searchByQryBuilderWithSortList(builder, stackTraceElement, null, DEFAULT_FROM, count, null);
        return result;
    }

    private List<T> findByField(StackTraceElement stackTraceElement, String name,Object value) {
        return findByField(stackTraceElement, name, value, 0);
    }

    private void setOrder(SortBuilder sortBuilder, String order) {
        if ("asc".equals(order.toLowerCase())) {
            sortBuilder.order(SortOrder.ASC);
        }
        else if ("desc".equals(order.toLowerCase())) {
            sortBuilder.order(SortOrder.DESC);
        }
    }


    private void setFilterMap(SortBuilder sortBuilder, Map map) {
        String order = (String) map.get("order");
        if (StringUtils.isNotEmpty(order)) {
            setOrder(sortBuilder, order);
        }
        String mode = (String) map.get("mode");
        if (StringUtils.isNotEmpty(mode)) {
            if (sortBuilder instanceof FieldSortBuilder) {
                ((FieldSortBuilder) sortBuilder).sortMode(mode);
            }
            else if (sortBuilder instanceof ScriptSortBuilder) {
                ((ScriptSortBuilder) sortBuilder).sortMode(mode);
            }
        }
        Map nestedFilterMap = (Map) map.get("nested_filter");
        if (EmptyUtil.isNotEmpty(nestedFilterMap)) {
        	if (EmptyUtil.isNotEmpty(nestedFilterMap.get("range"))){
        		Map rangeMap = (Map)nestedFilterMap.get("range");
        		for (Iterator<String> iterator = rangeMap.keySet().iterator(); iterator.hasNext(); ) {
        			String key = iterator.next();
        			RangeFilterBuilder filterBuilder = FilterBuilders.rangeFilter(key);
        			Map value = (Map)rangeMap.get(key);
        			filterBuilder.from(value.get("from"));
        			filterBuilder.to(value.get("to"));
        			setNestedFilterByCond(sortBuilder,filterBuilder);
        		}
        	}else{
        		BoolFilterBuilder filterBuilder =  FilterBuilders.boolFilter();
        		for (Iterator<String> iterator = nestedFilterMap.keySet().iterator(); iterator.hasNext(); ) {
        			String key = iterator.next();
        			Object value = nestedFilterMap.get(key);
        			filterBuilder.must(FilterBuilders.termFilter(key, value));
        		}
        		setNestedFilterByCond(sortBuilder,filterBuilder);
        	}
        }
    }

	private void setNestedFilterByCond(SortBuilder sortBuilder,BaseFilterBuilder filterBuilder) {
		if (sortBuilder instanceof FieldSortBuilder) {
		    ((FieldSortBuilder) sortBuilder).setNestedFilter(filterBuilder);
		}
		else if (sortBuilder instanceof ScriptSortBuilder) {
		    ((ScriptSortBuilder) sortBuilder).setNestedFilter(filterBuilder);
		}
	}


    public List<T> searchByQryBuilderWithSortMap(QueryBuilder queryBuilder, StackTraceElement stackTraceElement, String chnCode, int from, int size, Map sort) {
        List<T> list = new ArrayList<T>();
        String class_method = stackTraceElement.getClassName() + "." + stackTraceElement.getMethodName();
        Transaction t = Cat.newTransaction("ES", class_method);
        try {
                SearchRequestBuilder searchRequestBuilder = client.prepareSearch(index).setTypes(type)
                    .setQuery(queryBuilder)
//                    .setPostFilter(FilterBuilders.boolFilter().should(FilterBuilders.termFilter("channelOffShelfList", "teststststs")))
                    .setFrom(from)
                    .setSize(size)
                    .setQueryCache(false);
            if(sort != null && sort.size() > 0){
                Iterator<String> iterator = sort.keySet().iterator();
                for( ; iterator.hasNext(); ){
                    String key = iterator.next();
                    Object value = sort.get(key);
                    if (key.equals("_script") && value instanceof Map) {
                        Map scriptMap = (Map) value;
                        String script = (String) scriptMap.get("script");
                        ScriptSortBuilder scriptSortBuilder = SortBuilders.scriptSort(script, "number");
                        if (scriptMap.containsKey("params")) {
                            Map params = (Map) scriptMap.get("params");
                            scriptSortBuilder.setParams(params);
                        }
                        setFilterMap(scriptSortBuilder, (Map) value);
                        searchRequestBuilder.addSort(scriptSortBuilder);
                    }
                    else {
                        FieldSortBuilder sortBuilder = SortBuilders.fieldSort(key);
                        if (value instanceof String) {
                            setOrder(sortBuilder, (String) value);
                        } else if (value instanceof Map) {
                            setFilterMap(sortBuilder, (Map) value);
                        }
                        searchRequestBuilder.addSort(sortBuilder);
                    }
                }
            }
            if (sort == null || EmptyUtil.isEmpty(sort)) {
                searchRequestBuilder.addSort(SortBuilders.fieldSort("_id").order(SortOrder.ASC).ignoreUnmapped(true));
            }

            String esQueryString = searchRequestBuilder.toString();
            esQueryString = esQueryString.replaceAll("\\n", "");
            log.info("GET " + index + "/" + type + "/_search \n" + esQueryString);
            Cat.logEvent("ES.Query", "GET " + index + "/" + type + "/_search", Message.SUCCESS, esQueryString);
            SearchResponse searchResponse = searchRequestBuilder
                    .execute()
                    .actionGet();
            SearchHits hits = searchResponse.getHits();
            long total = hits.getTotalHits();
//            log.info("查询到记录数=" + hits.getTotalHits());
            SearchHit[] searchHists = hits.getHits();
            if (searchHists.length > 0) {
                for (SearchHit hit : searchHists) {
                    if (this instanceof ProductClient) {
                        list.add(build(hit, chnCode, total));
                    }
                    else {
                        list.add(build(hit, chnCode));
                    }
                }
            }
            t.setStatus(Message.SUCCESS);
        } catch (RuntimeException e) {
            log.error("Search ES Error: " + e);
            Cat.logError(e);
            e.printStackTrace();
            throw new PublicException(PublicExceptionErrorCode.SEARCH_ERROR, "查询出错！");
        } finally {
            t.complete();
        }
        return list;

    }

    public List<T> searchByQryBuilderWithSortList(QueryBuilder queryBuilder, StackTraceElement stackTraceElement, String chnCode, int from, int size,List<List<String>> sort)  {
        List<T> list = new ArrayList<T>();
        String esQueryString = "";
        String class_method = stackTraceElement.getClassName() + "." + stackTraceElement.getMethodName();
        Transaction t = Cat.newTransaction("ES", class_method);
        try {
            esQueryString = esQueryString.replaceAll("\\n", "");
            log.info("GET " + index + "/" + type + "/_search \n" + esQueryString);
            SearchRequestBuilder searchRequestBuilder = client.prepareSearch(index).setTypes(type)
                    .setQuery(queryBuilder)
//                    .setPostFilter(FilterBuilders.boolFilter().should(FilterBuilders.termFilter("channelOffShelfList", "teststststs")))
                    .setFrom(from)
                    .setSize(size)
                    .setQueryCache(false);
            if(sort != null && sort.size() > 0){
                for(List<String> mySort:sort){
                    if(mySort.size() == 2){
                        searchRequestBuilder.addSort(SortBuilders.fieldSort(mySort.get(0)).order(mySort.get(1).equals(SortOrder.ASC.toString())?SortOrder.ASC:SortOrder.DESC));
                    }
                }
            }
            if (CollectionUtils.isEmpty(sort)) {
                searchRequestBuilder.addSort(SortBuilders.fieldSort("_id").order(SortOrder.ASC));
            }
            esQueryString = searchRequestBuilder.toString();
            Cat.logEvent("ES.Query", "GET " + index + "/" + type + "/_search", Message.SUCCESS, esQueryString);
            SearchResponse searchResponse = searchRequestBuilder
                    .execute()
                    .actionGet();
            SearchHits hits = searchResponse.getHits();
//            log.info("查询到记录数=" + hits.getTotalHits());
            SearchHit[] searchHists = hits.getHits();
            if (searchHists.length > 0) {
                for (SearchHit hit : searchHists) {
                    list.add(build(hit, chnCode));
                }
            }
            t.setStatus(Message.SUCCESS);
        } catch (RuntimeException e) {
            e.printStackTrace();
            Cat.logError(e);
            throw new PublicException(PublicExceptionErrorCode.SEARCH_ERROR, "查询出错！[ES查询]Query: " + esQueryString + "\n" + e);
        } finally {
            t.complete();
        }
        return list;
    }

    //ROYS  don't use it

    private List<T> searchByQryBuilderJoin(QueryBuilder queryBuilder, StackTraceElement stackTraceElement, String chnCode, int from, int size)  {
        List<T> list = new ArrayList<T>();
        BoolQueryBuilder boolQueryBuilder = QueryBuilders.boolQuery();
        QueryBuilders.disMaxQuery();
        QueryBuilders.prefixQuery("","110");

        String class_method = stackTraceElement.getClassName() + "." + stackTraceElement.getMethodName();
        Transaction t = Cat.newTransaction("ES", class_method);
        String esQueryString = queryBuilder.toString();
        try {
            esQueryString = esQueryString.replaceAll("\\n", "");
            log.info("GET " + index + "/" + type + "/_search \n" + esQueryString);
            Cat.logEvent("ES.Query", "GET " + index + "/" + type + "/_search", Message.SUCCESS, esQueryString);
            SearchResponse searchResponse = client.prepareSearch(index).setTypes(type, "pubsku")
                    .setSearchType(org.elasticsearch.action.search.SearchType.DFS_QUERY_THEN_FETCH)
                    .setQuery(queryBuilder)
//                    .setPostFilter(FilterBuilders.boolFilter().should(FilterBuilders.termFilter("channelOffShelfList", "teststststs")))
                    .setFrom(from)
                    .setSize(size)
                    .setQueryCache(false)
                    .addSort(SortBuilders.fieldSort("_id").order(SortOrder.ASC))
                    .execute()
                    .actionGet();
            SearchHits hits = searchResponse.getHits();
//            log.info("查询到记录数=" + hits.getTotalHits());
            SearchHit[] searchHists = hits.getHits();
            if (searchHists.length > 0) {
                for (SearchHit hit : searchHists) {
                    list.add(build(hit, chnCode));
                }
            }
            t.setStatus(Message.SUCCESS);
        } catch (RuntimeException e) {
            e.printStackTrace();
            Cat.logError(e);
            throw new PublicException(PublicExceptionErrorCode.SEARCH_ERROR, "查询出错！[ES查询]Query: " + esQueryString + "\n" + e);
        } finally {
            t.complete();
        }
        return list;
    }



    /**
     *
     * @param params
     * @param type  条件类型
     * @return
     */
    public List<T> findByFields(Map<String, Object> params, SearchType type) {
        StackTraceElement stackTraceElement = getStrackElement();
        return findByFields(stackTraceElement, params, type);
    }


    private List<T> findByFields(StackTraceElement stackTraceElement, Map<String, Object> params, SearchType type) {
        String queryString = params.values().toString();
        QueryStringQueryBuilder queryStringQueryBuilder = QueryBuilders.queryStringQuery(queryString);
        BoolQueryBuilder boolQueryBuilder = QueryBuilders.boolQuery();

        String chnCode = MapUtils.getString(params, "chnCode");
        if (chnCode != null) {
            params.remove("chnCode");
        }

        Integer currage = (Integer) params.remove("currpage");
        Integer pagenum = (Integer) params.remove("pagenum");
        Integer start = (Integer) params.remove("from");
        Integer size = (Integer) params.remove("size");
        List<List<String>> sortList = null;
        Map sortMap = null;
        if(params.get("sort") != null){
            Object sortObj = params.remove("sort");
            if (sortObj instanceof List) {
                sortList = (List<List<String>>) sortObj;
            }
            else if (sortObj instanceof Map) {
                sortMap = (Map) sortObj;
            }
        }
        //ROYS pre-process sth update time
        queryCondPrepare(boolQueryBuilder, params, type);
        for (String key : params.keySet()) {
            Object val = params.get(key);
            if (val instanceof QueryBuilder) {
                if (val instanceof QueryStringQueryBuilder) {
                    ((QueryStringQueryBuilder) val).field(key);
                }
                boolQueryBuilder.must((QueryBuilder) val);
                continue;
            }
            if(key.endsWith("List")) {
                boolQueryBuilder.must(QueryBuilders.queryStringQuery(val.toString()).field(key));
                //boolQueryBuilder.must(QueryBuilders.fuzzyQuery(key, val));
                continue;
            }
            if (val instanceof List) {
                BoolQueryBuilder orBuilder = QueryBuilders.boolQuery();
                for (Object obj : (List) val) {
                    orBuilder.should(QueryBuilders.queryStringQuery(obj.toString()).field(key));
                }
                if (SearchType.isOr(type)) {
                    boolQueryBuilder.should(orBuilder);
                }else {
                    boolQueryBuilder.must(orBuilder);
                }
                continue;
            }

            // channelProdInfo 内嵌查询
            if (val instanceof Map){
            	if ("chanelProdTime".equals(key)){
            		BoolQueryBuilder nestedBoolQuery = QueryBuilders.boolQuery();
            		nestedBoolQuery.must(QueryBuilders.matchQuery("channelProdInfo.channelCode", ((Map) val).get("channelCode")))
            					   .must(QueryBuilders.rangeQuery("channelProdInfo.createTime")
            	            				.gt(((Map) val).get("createTimeStart")).lte(((Map) val).get("createTimeEnd")));

            		boolQueryBuilder.must(QueryBuilders.nestedQuery("channelProdInfo",nestedBoolQuery));
            		continue;
            	}
            }

            if (SearchType.OR.equals(type)) {
                boolQueryBuilder.should(QueryBuilders.termQuery(key, val));
            }else if(SearchType.AND.equals(type)) {
                boolQueryBuilder.must(QueryBuilders.termQuery(key, val));
            }else if (SearchType.ORLIKE.equals(type)) {
                boolQueryBuilder.should(QueryBuilders.multiMatchQuery(val, key).type(MatchQueryBuilder.Type.PHRASE_PREFIX));
            }else if (SearchType.ANDLIKE.equals(type)) {
                boolQueryBuilder.must(QueryBuilders.multiMatchQuery(val, key).type(MatchQueryBuilder.Type.PHRASE_PREFIX));
            }
        }
//        for (Map.Entry entry : params.entrySet()) {
////            queryStringQueryBuilder.field(entry.getKey().toString());
////            boolQueryBuilder.must(QueryBuilders.termQuery((String) entry.getKey(), entry.getValue()));
//            if (SearchType.OR.equals(type)) {
//                boolQueryBuilder.should(QueryBuilders.termQuery((String) entry.getKey(), entry.getValue()));
//            }else if(SearchType.AND.equals(type)) {
//                boolQueryBuilder.must(QueryBuilders.termQuery((String) entry.getKey(), entry.getValue()));
//            }else if (SearchType.ORLIKE.equals(type)) {
//                boolQueryBuilder.should(QueryBuilders.multiMatchQuery(entry.getValue(), (String) entry.getKey()).type(MatchQueryBuilder.Type.PHRASE_PREFIX));
//            }else if (SearchType.ANDLIKE.equals(type)) {
//                boolQueryBuilder.must(QueryBuilders.multiMatchQuery(entry.getValue(), (String) entry.getKey()).type(MatchQueryBuilder.Type.PHRASE_PREFIX));
//            }
//        }

        if (size == null) {
            size = pagenum != null && pagenum > 0 ? pagenum : DEFAULT_PAGE_SIZE;
        }
        if (start == null) {
            start = currage != null && currage > 0 ? (currage - 1) * pagenum : DEFAULT_FROM;
        }
        List<T> result = null;
        if (sortList != null) {
            result = searchByQryBuilderWithSortList(boolQueryBuilder, stackTraceElement, chnCode, start, size, sortList);
        }
        else {
            result = searchByQryBuilderWithSortMap(boolQueryBuilder, stackTraceElement, chnCode, start, size, sortMap);
        }
        return result;
    }

    public boolean create(T vo, StackTraceElement stackTraceElement){
        String primaryKeyName = getPrimaryKeyName();
        String class_method = stackTraceElement.getClassName() + "." + stackTraceElement.getMethodName();
        Transaction t = Cat.newTransaction("ES", class_method);
        updateTime(vo);
        Object primaryKeyValue = McReflectionUtil.getFieldValue(vo, primaryKeyName);
        try {
            XContentBuilder sourceBuilder = getXContentBuilder(vo);
            String esQueryString = sourceBuilder.toString();
            esQueryString = esQueryString.replaceAll("\\n", "");
            log.info("GET " + index + "/" + type + "/_search \n" + esQueryString);
            Cat.logEvent("ES.Query", "POST " + index + "/" + type, Message.SUCCESS, esQueryString);
            IndexResponse response = client.prepareIndex(index, type).setRefresh(true)
                    .setSource(sourceBuilder)
                    .setId(primaryKeyValue.toString())
                    .execute().actionGet();
            t.setStatus(Message.SUCCESS);
            return response.isCreated();
        } catch (Exception ex) {
            Cat.logError(ex);
            throw new PublicException(PublicExceptionErrorCode.UPDATE_ERROR, "更新出错！");
        } finally {
            t.complete();
        }
    }

    /**
     * batch create
     * @param vos
     * @return
     */
    public boolean create(List<T> vos, StackTraceElement stackTraceElement) {
        if (CollectionUtils.isEmpty(vos)) {
            return false;
        }
        String class_method = stackTraceElement.getClassName() + "." + stackTraceElement.getMethodName();
        Transaction t = Cat.newTransaction("ES", class_method);

//        //创建索引库 需要注意的是.setRefresh(true)这里一定要设置,否则第一次建立索引查找不到数据
//        IndexRequestBuilder requestBuilder = client.prepareIndex(index, type).setRefresh(true);
        BulkRequestBuilder bulkBuilder = client.prepareBulk().setRefresh(true);
        String primaryKeyName = getPrimaryKeyName();
        try {
            for (T vo : vos) {
                updateTime(vo);
                Object primaryKeyValue = getPrimaryKeyValue(vo);
                bulkBuilder.add(client.prepareIndex(index, type).setRefresh(true).setSource(getXContentBuilder(vo)).setId(primaryKeyValue.toString()));
            }
            BulkResponse responses = bulkBuilder.execute().actionGet();
            return !responses.hasFailures();
        } catch (Exception ex) {
            Cat.logError(ex);
            throw new PublicException(PublicExceptionErrorCode.UPDATE_ERROR, "更新出错！");
        } finally {
            t.complete();
        }
    }

    public boolean deleteByPrimaryKey(Object id){
        BulkRequestBuilder bulkRequestBuilder = client.prepareBulk().setRefresh(true);
        bulkRequestBuilder.add(new DeleteRequest(index, type, id.toString()));
        return !bulkRequestBuilder.execute().actionGet().hasFailures();
    }

    public boolean deleteByPrimaryKey(List<String> ids){
        BulkRequestBuilder bulkRequestBuilder = client.prepareBulk().setRefresh(true);
        for (String id : ids) {
            bulkRequestBuilder.add(new DeleteRequest(index, type, id.toString()));
        }
        return !bulkRequestBuilder.execute().actionGet().hasFailures();
    }

    public boolean deleteAll(){
        DeleteMappingRequest deleteMappingRequest = new DeleteMappingRequest(index).types(type);
        DeleteMappingResponse deleteMappingResponse = client.admin().indices().deleteMapping(deleteMappingRequest).actionGet();
        return deleteMappingResponse.isAcknowledged();
    }

    public boolean createOrUpdate(StackTraceElement stackTraceElement, T vo) {
        String primaryKeyName = getPrimaryKeyName();
        Object primaryKeyValue = McReflectionUtil.getFieldValue(vo, primaryKeyName);
        List<T> exists = findByField(stackTraceElement, primaryKeyName, primaryKeyValue);
        if (CollectionUtils.isEmpty(exists)) {
            return create(vo, stackTraceElement);
        }
        T exist = exists.get(0);
        return update(exist);
    }

    public boolean createOrUpdate(StackTraceElement stackTraceElement, List<T> vos) {
        if (CollectionUtils.isEmpty(vos)) {
            return Boolean.TRUE;
        }
        for (T vo : vos) {
            createOrUpdate(stackTraceElement, vo);
        }
        return Boolean.TRUE;
    }

    public boolean update(List<T> vos) {
        BulkRequestBuilder bulkBuilder = client.prepareBulk().setRefresh(true);
        for (T vo : vos) {
            UpdateRequest request = buildUpdateRequest(vo);
            bulkBuilder.add(request);
        }
        return !bulkBuilder.execute().actionGet().hasFailures();
    }

    public boolean update(T vo) {
        String primaryKeyName = getPrimaryKeyName();
        Object primaryKeyValue = getPrimaryKeyValue(vo);
        if(primaryKeyValue == null || StringUtils.isBlank(primaryKeyValue.toString())) {
            throw new PublicException(PublicExceptionErrorCode.ILLEGAL_PARAMETER_IS_NULL,"主键不能为空！");
        }
        Map<String, Object> params = McReflectionUtil.getFieldNameValue(vo);
        if (MapUtils.isEmpty(params)) {
            return true;
        }
//        UpdateRequest updateRequest = new UpdateRequest();
//        updateRequest.index(index);
//        updateRequest.type(type);
//        updateRequest.id(primaryKeyValue.toString());
        try {
            //更新去掉 主键更新
//            params.remove(getPrimaryKeyName());
//            XContentBuilder builder = XContentFactory.jsonBuilder().startObject();
//            for (String key : params.keySet()) {
//                builder = builder.field(key, params.get(key));
//            }
//            updateRequest.doc(builder.endObject());
            UpdateResponse response = client.update(buildUpdateRequest(vo)).actionGet();
            return response.getVersion() > 0;
        } catch (Exception e) {
            log.error("", e);
            throw new PublicException(PublicExceptionErrorCode.UPDATE_ERROR, "更新出错！");
        }
    }


    private UpdateRequest buildUpdateRequest(T vo){
        Object primaryKeyValue = getPrimaryKeyValue(vo);
        if(primaryKeyValue == null) {
            throw new PublicException(PublicExceptionErrorCode.ILLEGAL_PARAMETER_IS_NULL, "主键不能为空");
        }

        updateTime(vo);
        Map<String, Object> params = McReflectionUtil.getFieldNameValue(vo);
        UpdateRequest updateRequest = new UpdateRequest();
        updateRequest.index(index);
        updateRequest.type(type);
        updateRequest.id(primaryKeyValue.toString());
        updateRequest.refresh(true);

        UpdateRequestBuilder updateRequestBuilder = client.prepareUpdate(index, type, (String) primaryKeyValue).setRefresh(true);
        params.remove(getPrimaryKeyName());
        XContentBuilder builder = null;

        try {
            builder = XContentFactory.jsonBuilder();
            //ROYS optimize
            String updateTime = (String) params.get("updateTime");
            if (StringUtils.isNotBlank(updateTime)) {
                params.put("updateLong", DateUtil.parse(updateTime, DateUtil.DEFAULT_DATETIME_FORMAT).getTime());
            }
            buildUpdateParams(builder, params);
            updateRequest.doc(builder);
        } catch (IOException e) {
            e.printStackTrace();
            throw new PublicException(PublicExceptionErrorCode.UPDATE_ERROR, "批量更新出错！");
        }
        return updateRequest;
    }

    private void buildUpdateParams(XContentBuilder builder, Map<String, Object> params) {
        try {
            builder.startObject();
            for (String key : params.keySet()) {
                Object val = params.get(key);
                if (val instanceof List) {
                    List vos = (List) val;
                    if (CollectionUtils.isNotEmpty(vos) && vos.get(0) instanceof PsBaseVO) {
                        builder.startArray(key);
                        for (Object obj : vos) {
                            updateTime(obj);
                            buildUpdateParams(builder, McReflectionUtil.getFieldNameValue(obj));
                        }
                        builder.endArray();
                        continue;
                    }
                }
                builder.field(key, val);
            }
            builder.endObject();
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    private Object getPrimaryKeyValue(T vo) {
        return McReflectionUtil.getFieldValue(vo, getPrimaryKeyName());
    }

    private void updateTime(Object obj) {
        if (obj instanceof PsUpdateVO) {
            ((PsUpdateVO)obj).setUpdateTime(getNow());
        }
    }

     String getNow(){
        return DateUtil.format(DateUtil.getNow(), DateUtil.DEFAULT_DATETIME_FORMAT);
    }

    //组装VO
    public abstract T build(SearchHit hit, String chnCode);
    //组装VO + total
    public abstract T build(SearchHit hit, String chnCode, Long total);
    //设置插入数据的名称，类型
    public abstract XContentBuilder getXContentBuilder(T vo);
    //获取字段的主键
    public abstract String getPrimaryKeyName();

    //可以做一些自定义处理，比如日期，范围
    public void queryCondPrepare(BoolQueryBuilder queryBuilder, Map<String, Object> params, SearchType type) {
        String updateTime = (String) params.get("updateTime");
//        String updateTimeEnd = (String) params.get("updateTimeEnd");
        String updateTimeEnd = getNow();
        if (StringUtils.isNoneBlank(updateTime)) {
//            RangeQueryBuilder rangeBuilder = QueryBuilders.rangeQuery("updateTime");
            RangeQueryBuilder rangeBuilder = QueryBuilders.rangeQuery("updateLong");
//            rangeBuilder.from(updateTime).to(updateTimeEnd);
            rangeBuilder.gte(DateUtil.parse(updateTime, DateUtil.DEFAULT_DATETIME_FORMAT).getTime());
            if (SearchType.isOr(type)) {
                queryBuilder.should(rangeBuilder);
            } else {
                queryBuilder.must(rangeBuilder);
            }
        }
        params.remove("updateTime");

        String primaryUpdateTime = (String) params.get("primaryUpdateTime");
//        String updateTimeEnd = (String) params.get("updateTimeEnd");
        String primaryUpdateTimeEnd = getNow();
        if (StringUtils.isNoneBlank(primaryUpdateTime)) {
//            RangeQueryBuilder rangeBuilder = QueryBuilders.rangeQuery("primaryUpdateTime");
            RangeQueryBuilder rangeBuilder = QueryBuilders.rangeQuery("primaryUpdateLong");
//            rangeBuilder.from(updateTime).to(updateTimeEnd);
            rangeBuilder.gte(DateUtil.parse(primaryUpdateTime, DateUtil.DEFAULT_DATETIME_FORMAT).getTime());
            if (SearchType.isOr(type)) {
                queryBuilder.should(rangeBuilder);
            } else {
                queryBuilder.must(rangeBuilder);
            }
        }
        params.remove("primaryUpdateTime");

    }

    //可以做一些自定义处理，比如日期，范围
    public void queryCondPrepareFilter(BoolFilterBuilder filterBuilder, Map<String, Object> params, SearchType type) {
        String updateTime = (String) params.get("updateTime");
//        String updateTimeEnd = (String) params.get("updateTimeEnd");
        String updateTimeEnd = getNow();
        if (StringUtils.isNoneBlank(updateTime)) {
//            RangeQueryBuilder rangeBuilder = QueryBuilders.rangeQuery("updateTime");
            RangeFilterBuilder rangeBuilder = FilterBuilders.rangeFilter("updateLong");
//            rangeBuilder.from(updateTime).to(updateTimeEnd);
            rangeBuilder.gte(DateUtil.parse(updateTime, DateUtil.DEFAULT_DATETIME_FORMAT).getTime());
            if (SearchType.isOr(type)) {
                filterBuilder.should(rangeBuilder);
            } else {
                filterBuilder.must(rangeBuilder);
            }
        }
        params.remove("updateTime");

        String primaryUpdateTime = (String) params.get("primaryUpdateTime");
//        String updateTimeEnd = (String) params.get("updateTimeEnd");
        String primaryUpdateTimeEnd = getNow();
        if (StringUtils.isNoneBlank(primaryUpdateTime)) {
//            RangeQueryBuilder rangeBuilder = QueryBuilders.rangeQuery("primaryUpdateTime");
            RangeFilterBuilder rangeBuilder = FilterBuilders.rangeFilter("primaryUpdateLong");
//            rangeBuilder.from(updateTime).to(updateTimeEnd);
            rangeBuilder.gte(DateUtil.parse(primaryUpdateTime, DateUtil.DEFAULT_DATETIME_FORMAT).getTime());
            if (SearchType.isOr(type)) {
                filterBuilder.should(rangeBuilder);
            } else {
                filterBuilder.must(rangeBuilder);
            }
        }
        params.remove("primaryUpdateTime");

    }



    void addQueryBuilder(BoolQueryBuilder boolQueryBuilder,QueryBuilder queryBuilder,SearchType type) {
        if (SearchType.isOr(type)) {
            boolQueryBuilder.should(queryBuilder);
        }else {
            boolQueryBuilder.must(queryBuilder);
        }
    }


    public void testQuery(){
        client.prepareSearch(index).setTypes(type).setExtraSource("");
    }

    /**
     *
     * @param params
     * @param type  条件类型
     * @return
     */
    public List<T> findByFieldWithProdSearchCondVO(Map<String, Object> params, SearchType type, Integer from, Integer size) {
        StackTraceElement stackTraceElement = getStrackElement();
        String queryString = params.values().toString();
        QueryStringQueryBuilder queryStringQueryBuilder = QueryBuilders.queryStringQuery(queryString);
        BoolQueryBuilder boolQueryBuilder = QueryBuilders.boolQuery();

        String chnCode = MapUtils.getString(params, "chnCode");
        if (chnCode != null) {
            params.remove("chnCode");
        }

        List<List<String>> sortList = null;
        Map sortMap = null;
        if(params.get("sort") != null){
            Object sortObj = params.remove("sort");
            if (sortObj instanceof List) {
                sortList = (List<List<String>>) sortObj;
            }
            else if (sortObj instanceof Map) {
                sortMap = (Map) sortObj;
            }
        }
        //ROYS pre-process sth update time
        queryCondPrepare(boolQueryBuilder, params, type);
        for (String key : params.keySet()) {
            Object val = params.get(key);
            if(key.endsWith("List")) {
                boolQueryBuilder.must(QueryBuilders.queryStringQuery(val.toString()).field(key));
                continue;
            }

            if (val instanceof List) {
                BoolQueryBuilder orBuilder = QueryBuilders.boolQuery();
                for (Object obj : (List) val) {
                    orBuilder.should(QueryBuilders.queryStringQuery(obj.toString()).field(key));
                }
                if (SearchType.isOr(type)) {
                    boolQueryBuilder.should(orBuilder);
                }else {
                    boolQueryBuilder.must(orBuilder);
                }
                continue;
            }

            if(val instanceof DateRange) {
                Date start = ((DateRange) val).getStart();
                Date end = ((DateRange) val).getEnd();
                BoolQueryBuilder nestedBoolQuery = QueryBuilders.boolQuery();
        		nestedBoolQuery.must(QueryBuilders.rangeQuery(key).gt(start).lte(end));
//                RangeQueryBuilder rangeBuilder = QueryBuilders.rangeQuery(key);
//                rangeBuilder.from(start).to(end).includeLower(true).includeUpper(true);  //包括上、下界  ;
//                if (SearchType.isOr(type)) {
//                    boolQueryBuilder.should(rangeBuilder);
//                }else {
//                    boolQueryBuilder.must(QueryBuilders.nestedQuery("channelProdInfo",nestedBoolQuery),RangeQueryBuilder);
//                }
        		boolQueryBuilder.must(QueryBuilders.nestedQuery("channelProdInfo",nestedBoolQuery));
                continue;
            }

            if (SearchType.OR.equals(type)) {
                boolQueryBuilder.should(QueryBuilders.termQuery(key, val));
            }else if(SearchType.AND.equals(type)) {
                boolQueryBuilder.must(QueryBuilders.termQuery(key, val));
            }else if (SearchType.ORLIKE.equals(type)) {
                boolQueryBuilder.should(QueryBuilders.multiMatchQuery(val, key).type(MatchQueryBuilder.Type.PHRASE_PREFIX));
            }else if (SearchType.ANDLIKE.equals(type)) {
                boolQueryBuilder.must(QueryBuilders.multiMatchQuery(val, key).type(MatchQueryBuilder.Type.PHRASE_PREFIX));
            }
        }

//        Integer size = (Integer) params.remove("size");
//        Integer from = (Integer) params.remove("from");
        List<T> result = null;
        if (sortList != null) {
            result = searchByQryBuilderWithSortList(boolQueryBuilder, stackTraceElement, chnCode, from, size, sortList);
        }
        else {
            result = searchByQryBuilderWithSortMap(boolQueryBuilder, stackTraceElement, chnCode, from, size, sortMap);
        }
        return result;

    }

    /**
     *
     * @param params
     * @param type  条件类型
     * @return
     */
    public List<T> findByFieldWithKeyWords(Map<String, Object> params, SearchType type,String keyWords,Integer from, Integer size) {
        StackTraceElement stackTraceElement = Thread.currentThread().getStackTrace()[1];
        BoolQueryBuilder mustQueryBuilder = QueryBuilders.boolQuery();
        BoolQueryBuilder orQueryBuilder = QueryBuilders.boolQuery();

        BoolFilterBuilder andFilterBuilder = FilterBuilders.boolFilter();

        String chnCode = MapUtils.getString(params, "chnCode");
        if (chnCode != null) {
            params.remove("chnCode");
        }

        List<List<String>> sortList = null;
        Map sortMap = null;
        if(params.get("sort") != null){
            Object sortObj = params.remove("sort");
            if (sortObj instanceof List) {
                sortList = (List<List<String>>) sortObj;
            }
            else if (sortObj instanceof Map) {
                sortMap = (Map) sortObj;
            }
        }
        //ROYS pre-process sth update time
        queryCondPrepareFilter(andFilterBuilder, params, type);
        for (String key : params.keySet()) {
            Object val = params.get(key);
            if(key.endsWith("List")) {
                mustQueryBuilder.must(QueryBuilders.queryStringQuery(val.toString()).field(key));
                continue;
            }

            if (val instanceof List) {
                BoolQueryBuilder orBuilder = QueryBuilders.boolQuery();
                for (Object obj : (List) val) {
                    orBuilder.should(QueryBuilders.queryStringQuery(obj.toString()).field(key));
                }
                if (SearchType.isOr(type)) {
                    mustQueryBuilder.should(orBuilder);
                }else {
                    mustQueryBuilder.must(orBuilder);
                }
                continue;
            }

            if(val instanceof DateRange) {
                Date start = ((DateRange) val).getStart();
                Date end = ((DateRange) val).getEnd();
                RangeQueryBuilder rangeBuilder = QueryBuilders.rangeQuery(key);
                rangeBuilder.from(start).to(end).includeLower(true).includeUpper(true);  //包括上、下界  ;
                if (SearchType.isOr(type)) {
                    mustQueryBuilder.should(rangeBuilder);
                }else {
                    mustQueryBuilder.must(rangeBuilder);
                }
                continue;
            }

            if (SearchType.OR.equals(type)) {
                mustQueryBuilder.should(QueryBuilders.termQuery(key, val));
            }else if(SearchType.AND.equals(type)) {
                mustQueryBuilder.must(QueryBuilders.termQuery(key, val));
            }else if (SearchType.ORLIKE.equals(type)) {
                mustQueryBuilder.should(QueryBuilders.multiMatchQuery(val, key).type(MatchQueryBuilder.Type.PHRASE_PREFIX));
            }else if (SearchType.ANDLIKE.equals(type)) {
                mustQueryBuilder.must(QueryBuilders.multiMatchQuery(val, key).type(MatchQueryBuilder.Type.PHRASE_PREFIX));
            }
        }

        if(!Strings.isNullOrEmpty(keyWords)){
//            andFilterBuilder.must(FilterBuilders.termFilter("stocks.soldOut", "0"));
//            mustQueryBuilder.must(QueryBuilders.termQuery("stocks.soldOut", "0"));
            List<SearchKeyword> searchKeywords = new ArrayList<>();
            searchKeywords.add(new SearchKeyword(keyWords, 20.0d, 'n'));

            for (SearchKeyword searchKeyword : searchKeywords) {
                String word = searchKeyword.getKeyword();
                // 过滤近义词词性
                if (searchKeyword.getFlag() != null && searchKeyword.getFlag() != 'n' && searchKeyword.getFlag() != 'v') {
                    continue;
                }
//                    Float boost = searchKeyword.getBoost().floatValue();
//                Float sboost = searchKeyword.getBoost().floatValue();
                orQueryBuilder
                        .should(QueryBuilders.matchQuery("name", word).slop(3).boost(4600).minimumShouldMatch("75%"))
                        .should(QueryBuilders.matchQuery("nameCn", word).slop(3).boost(4600).minimumShouldMatch("75%"))
                        .should(QueryBuilders.matchQuery("frontCategories.categoryName", word).slop(2).boost(1600).minimumShouldMatch("75%"))
                        .should(QueryBuilders.matchQuery("frontCategories.categoryFullName", word).slop(3).boost(1600).minimumShouldMatch("75%"))
                        .should(QueryBuilders.matchQuery("brandList.brandName", word).slop(2).boost(2200).minimumShouldMatch("75%"))
                        .should(QueryBuilders.matchQuery("brandList.brandNameCN", word).slop(3).boost(2200).minimumShouldMatch("75%"))
                        .should(QueryBuilders.matchPhraseQuery("labels.labelName", word).boost(300))
                        .should(QueryBuilders.matchPhraseQuery("labels.labelNameCn", word).boost(300))
                        .should(QueryBuilders.matchPhraseQuery("campaignList.discountTypeName", word).boost(100));

/*
                        .should(QueryBuilders.multiMatchQuery(
                                word,
                                "name_exact^3600",
                                "nameCn_exact^3600",
                                "categoryName_exact^1600",
                                "categoryFullName_exact^1200",
                                "name^360",
                                "nameCn^360",
                                "brandList.brandName^220",
                                "brandList.brandNameCN^220",
                                "labels.labelNameCn^80",
                                "frontCategories.categoryName^100",
                                "frontCategories.categoryFullName^50",
                                "campaignList.discountTypeName^30"))
*/
//                        .should(QueryBuilders.matchPhraseQuery("brandList.brandName", word).slop(1).boost(120))
//                        .should(QueryBuilders.matchPhraseQuery("brandList.brandNameCN", word).slop(1).boost(120));

/*                .should(QueryBuilders.queryStringQuery(word).boost(12  * sboost).field("name"))
                        .should(QueryBuilders.queryStringQuery(word).boost(20 * sboost).field("nameCn"))
                        .should(QueryBuilders.queryStringQuery(word).boost(8 * sboost).field("brandList.brandNameCN"))
                        .should(QueryBuilders.queryStringQuery(word).boost(7 * sboost).field("labels.labelNameCn"))
                        .should(QueryBuilders.queryStringQuery(word).boost(9 * sboost).field("frontCategories.categoryFullName"))
                        .should(QueryBuilders.queryStringQuery(word).boost(6 * sboost).field("campaignList.discountTypeName"));*/
            }
            mustQueryBuilder.must(orQueryBuilder);
        }
//        Integer size = (Integer) params.remove("size");
//        Integer from = (Integer) params.remove("from");
        List<T> result = null;
        result = searchByQryBuilderWithKeyWordsSortMap(mustQueryBuilder, null, stackTraceElement, chnCode, from, size, sortMap);

        return result;

    }

    /**
     *
     * @param keyWords  SKU关键字
     * @return
     */
    public List<T> findSkuByFieldWithKeyWords(String keyWords,Integer from, Integer size) {
        StackTraceElement stackTraceElement = Thread.currentThread().getStackTrace()[1];
        BoolQueryBuilder mustQueryBuilder = QueryBuilders.boolQuery();
        BoolQueryBuilder orQueryBuilder = QueryBuilders.boolQuery();
        if(!Strings.isNullOrEmpty(keyWords)){
            List<SearchKeyword> searchKeywords = new ArrayList<>();
            searchKeywords.add(new SearchKeyword(keyWords, 20.0d, 'n'));

            for (SearchKeyword searchKeyword : searchKeywords) {
                String word = searchKeyword.getKeyword();
                // 过滤近义词词性
                if (searchKeyword.getFlag() != null && searchKeyword.getFlag() != 'n' && searchKeyword.getFlag() != 'v') {
                    continue;
                }
                orQueryBuilder
                        .should(QueryBuilders.matchQuery("skuCode", word).slop(3).boost(4600).minimumShouldMatch("80%"))
                        .should(QueryBuilders.matchQuery("skuName", word).slop(3).boost(4600).minimumShouldMatch("80%"))
                        .should(QueryBuilders.matchQuery("skuNameCN", word).slop(3).boost(4600).minimumShouldMatch("80%"));
//                        .should(QueryBuilders.matchQuery("frontCategories.categoryFullName", word).slop(3).boost(1600).minimumShouldMatch("75%"))
//                        .should(QueryBuilders.matchQuery("brandList.brandName", word).slop(2).boost(2200).minimumShouldMatch("75%"))
//                        .should(QueryBuilders.matchQuery("brandList.brandNameCN", word).slop(3).boost(2200).minimumShouldMatch("75%"))
//                        .should(QueryBuilders.matchPhraseQuery("labels.labelName", word).boost(300))
//                        .should(QueryBuilders.matchPhraseQuery("labels.labelNameCn", word).boost(300))
//                        .should(QueryBuilders.matchPhraseQuery("campaignList.discountTypeName", word).boost(100));
            }
            mustQueryBuilder.must(orQueryBuilder);
        }
        List<T> result = null;
        result = searchByQryBuilderWithKeyWordsSortMap(mustQueryBuilder, null, stackTraceElement, "", from, size, null);

        return result;

    }

    /**
     *
     * @param params
     * @param type  条件类型
     * @return
     */
    public List<T> findByFieldWithSelectMap(Map<String, Object> params, SearchType type,Map<String,Object> selectMap,Integer from, Integer size) {
        StackTraceElement stackTraceElement = Thread.currentThread().getStackTrace()[1];
        String queryString = params.values().toString();
        QueryStringQueryBuilder queryStringQueryBuilder = QueryBuilders.queryStringQuery(queryString);
        BoolQueryBuilder boolQueryBuilder = QueryBuilders.boolQuery();

        String chnCode = MapUtils.getString(params, "chnCode");
        if (chnCode != null) {
            params.remove("chnCode");
        }

        List<List<String>> sortList = null;
        Map sortMap = null;
        if(params.get("sort") != null){
            Object sortObj = params.remove("sort");
            if (sortObj instanceof List) {
                sortList = (List<List<String>>) sortObj;
            }
            else if (sortObj instanceof Map) {
                sortMap = (Map) sortObj;
            }
        }
        //ROYS pre-process sth update time
        queryCondPrepare(boolQueryBuilder, params, type);
        for (String key : params.keySet()) {
            Object val = params.get(key);
            if(key.endsWith("List")) {
                boolQueryBuilder.must(QueryBuilders.queryStringQuery(val.toString()).field(key));
                continue;
            }

            if (val instanceof List) {
                BoolQueryBuilder orBuilder = QueryBuilders.boolQuery();
                for (Object obj : (List) val) {
                    orBuilder.should(QueryBuilders.queryStringQuery(obj.toString()).field(key));
                }
                if (SearchType.isOr(type)) {
                    boolQueryBuilder.should(orBuilder);
                }else {
                    boolQueryBuilder.must(orBuilder);
                }
                continue;
            }

            if(val instanceof DateRange) {
                Date start = ((DateRange) val).getStart();
                Date end = ((DateRange) val).getEnd();
                RangeQueryBuilder rangeBuilder = QueryBuilders.rangeQuery(key);
                rangeBuilder.from(start).to(end).includeLower(true).includeUpper(true);  //包括上、下界  ;
                if (SearchType.isOr(type)) {
                    boolQueryBuilder.should(rangeBuilder);
                }else {
                    boolQueryBuilder.must(rangeBuilder);
                }
                continue;
            }

            Boolean isCrossBorder = (Boolean)selectMap.get("isCrossBorder");
            Object labels = selectMap.get("labels");
            Object brandId = selectMap.get("brandId");
            Object brandIds = selectMap.get("brandIds");
            Object labelIds = selectMap.get("labelIds");

//            if(brandIds != null){
//                List<String> brandIdsObj = (List<String>)brandIds;
//                if(brandIdsObj.size() > 0) {
//                boolQueryBuilder.must(QueryBuilders.termQuery("stocks.soldOut", "0"));
//                    boolQueryBuilder.must(QueryBuilders.termQuery("brandList.brandId", brandIdsObj.get(0)));
//                }
//            }

//            if(labelIds != null){
//                List<String> labelIdsObj = (List<String>)labelIds;
//                if(labelIdsObj.size() > 0) {
//                boolQueryBuilder.must(QueryBuilders.termQuery("stocks.soldOut", "0"));
//                    boolQueryBuilder.must(QueryBuilders.termQuery("labels.labelId", labelIdsObj.get(0)));
//                }
//            }

            if(isCrossBorder != null && isCrossBorder == true){
                boolQueryBuilder.must(QueryBuilders.termQuery("stocks.soldOut", "0"));
                boolQueryBuilder.must(QueryBuilders.termQuery("crossBorderFlag", "1"));
            }

            if(labels != null){
                String labelsObj = labels.toString();
                boolQueryBuilder.must(QueryBuilders.termQuery("stocks.soldOut", "0"));
                boolQueryBuilder.must(QueryBuilders.termQuery("labels.labelId", labelsObj));
            }

            if(brandId != null){
                String brandIdObj = brandId.toString();
//                boolQueryBuilder.must(QueryBuilders.termQuery("stocks.soldOut", "0"));
                boolQueryBuilder.must(QueryBuilders.termQuery("brandList.brandId", brandIdObj));
            }

            if (SearchType.OR.equals(type)) {
                boolQueryBuilder.should(QueryBuilders.termQuery(key, val));
            }else if(SearchType.AND.equals(type)) {
                boolQueryBuilder.must(QueryBuilders.termQuery(key, val));
            }else if (SearchType.ORLIKE.equals(type)) {
                boolQueryBuilder.should(QueryBuilders.multiMatchQuery(val, key).type(MatchQueryBuilder.Type.PHRASE_PREFIX));
            }else if (SearchType.ANDLIKE.equals(type)) {
                boolQueryBuilder.must(QueryBuilders.multiMatchQuery(val, key).type(MatchQueryBuilder.Type.PHRASE_PREFIX));
            }
        }

//        Integer size = (Integer) params.remove("size");
//        Integer from = (Integer) params.remove("from");
        List<T> result = null;
        result = searchByQryBuilderWithKeyWordsSortMap(boolQueryBuilder, null, stackTraceElement, chnCode, from, size, sortMap);
        return result;

    }

    public List<T> searchByQryBuilderWithKeyWordsSortMap(QueryBuilder queryBuilder, FilterBuilder filterBuilder, StackTraceElement stackTraceElement, String chnCode, int from, int size, Map sort) {
        List<T> list = new ArrayList<T>();
        String esQueryString = "";
        String class_method = stackTraceElement.getClassName() + "." + stackTraceElement.getMethodName();
        Transaction t = Cat.newTransaction("ES", class_method);
        try {
            SearchRequestBuilder searchRequestBuilder = client.prepareSearch(index).setTypes(type)
                    .setQuery(queryBuilder)
                    .setFrom(from)
                    .setSize(size)
                    .setQueryCache(false);
            if (filterBuilder != null) {
                searchRequestBuilder.setPostFilter(filterBuilder);
            }
            if (sort != null && sort.size() > 0) {
                Iterator<String> iterator = sort.keySet().iterator();
                for( ; iterator.hasNext(); ){
                    String key = iterator.next();
                    Object value = sort.get(key);
                    if (key.equals("_script") && value instanceof Map) {
                        Map scriptMap = (Map) value;
                        String script = (String) scriptMap.get("script");
                        ScriptSortBuilder scriptSortBuilder = SortBuilders.scriptSort(script, "number");
                        if (scriptMap.containsKey("params")) {
                            Map params = (Map) scriptMap.get("params");
                            scriptSortBuilder.setParams(params);
                        }
                        setFilterMap(scriptSortBuilder, (Map) value);
                        searchRequestBuilder.addSort(scriptSortBuilder);
                    }
                    else {
                        FieldSortBuilder sortBuilder = SortBuilders.fieldSort(key);
                        if (value instanceof String) {
                            setOrder(sortBuilder, (String) value);
                        } else if (value instanceof Map) {
                            setFilterMap(sortBuilder, (Map) value);
                        }
                        searchRequestBuilder.addSort(sortBuilder);
                    }
                }
            }
//            searchRequestBuilder.addSort(SortBuilders.fieldSort("id").order(SortOrder.ASC));
            esQueryString = searchRequestBuilder.toString();
            esQueryString = esQueryString.replaceAll("\\n", "");
            log.info("GET " + index + "/" + type + "/_search \n" + esQueryString);
            Cat.logEvent("ES.Query", "GET " + index + "/" + type + "/_search", Message.SUCCESS, esQueryString);
            SearchResponse searchResponse = searchRequestBuilder //.addSort(SortBuilders.fieldSort("id").order(SortOrder.ASC))
                    .execute()
                    .actionGet();
            SearchHits hits = searchResponse.getHits();
            long total = hits.getTotalHits();
//            log.info("查询到记录数=" + hits.getTotalHits());
            SearchHit[] searchHists = hits.getHits();
            if (searchHists.length > 0) {
                for (SearchHit hit : searchHists) {
                    list.add(build(hit, chnCode, total));
                }
            }
            t.setStatus(Message.SUCCESS);
        } catch (RuntimeException e) {
            e.printStackTrace();
            Cat.logError(e);
            throw new PublicException(PublicExceptionErrorCode.SEARCH_ERROR, "查询出错！[ES查询]Query: " + esQueryString + "\n" + e);
        } finally {
            t.complete();
        }
        return list;

    }

}
